/*
	PropUpdater.cpp
	Property Updater

	The property updater is responsible for updating the properties of all objects within
	the level when property fields are added/deleted
*/

#include "PropUpdater.h"
#include "Next.h"
#include "ConfigData.h"
#include "../UI/ProgressBar.h"
#include "appdata.h"
#include <windows.h>
#include "path.h"

PropUpdater::PropUpdater(HINSTANCE hInstance, HWND hwnd, Interface* ip) :
	ModalDlgWindow(hInstance, MAKEINTRESOURCE(IDD_PROPUPDATER), hwnd)
{
	this->ip = ip;
	this->hInstance = hInstance;
	this->hwndParent = hwndParent;

	RegisterNotification(LoadCB,this,NOTIFY_FILE_POST_OPEN);
}

PropUpdater::~PropUpdater()
{
	UnRegisterNotification(LoadCB,this,NOTIFY_FILE_POST_OPEN);
}

void PropUpdater::LoadCB(void* param,NotifyInfo* ninfo)
{
	PropUpdater* pthis = (PropUpdater*)param;
	pthis->ProcChanges();
}

// Is file 2 newer than file one
bool PropUpdater::FileIsNewer( char* file1, char* file2 )
{
	HANDLE file_handle_1, file_handle_2;
	WIN32_FIND_DATA find_data_1, find_data_2;
	
	file_handle_1 = FindFirstFile( file1, &find_data_1 );  
	if( file_handle_1 == INVALID_HANDLE_VALUE )
	{
		return true;
	}

	file_handle_2 = FindFirstFile( file2, &find_data_2 );  
	if( file_handle_2 == INVALID_HANDLE_VALUE )
	{
		FindClose( file_handle_1 );
		return false;
	}
	
	FindClose( file_handle_1 );
	FindClose( file_handle_2 );
	return( CompareFileTime( &find_data_2.ftLastWriteTime, &find_data_1.ftLastWriteTime ) > 0 );
}

CStr PropUpdater::GetScriptIniName()
{
	CStr strFile=getenv(APP_ENV);

	// Verify that the environment variable is set
	if (strFile==CStr(""))
	{
		char ErrorBuf[256];
		sprintf(ErrorBuf,"Couldn't load '%s' the '%s' environment variable is not set.",SCRIPT_INI,APP_ENV);

		MessageBox(NULL,ErrorBuf,"ParseScriptIni",MB_ICONSTOP|MB_OK);
		return CStr("");
	}

	strFile+=CStr(SCRIPT_PATH)+SCRIPT_INI;

	return strFile;
}

int PropUpdater::CountNodes(INode* node,int num)
{
	if (node == NULL)
		node = ip->GetRootNode();

	int numKids = node->NumberOfChildren();

	for(int i=0;i<numKids;i++)
	{
		INode* childNode = node->GetChildNode(i);
		num = CountNodes(childNode,num);
		num++;
	}

	return num;
}

void PropUpdater::ProcNode(INode* parentNode)
{
	if (!parentNode)
		parentNode = ip->GetRootNode();

	int numKids = parentNode->NumberOfChildren();

	for(int i=0;i<numKids;i++)
	{
		INode* node = parentNode->GetChildNode(i);
		ProcNode(node);
	}

	if (parentNode == ip->GetRootNode())
		return;

	CStr propBuffer;
	parentNode->GetUserPropBuffer(propBuffer);

	CStr name = parentNode->GetName();

	if (propBuffer.Length()>0)
	{
		CStr className = GetClassName(propBuffer);
		CStr typeName  = GetTypeName(propBuffer);

		LinkList<ConfigProp> cprops, DEFprops;
		LinkList<ConfigScript> cscripts;
		DWORD cflags;
		CStr strUnk, strClusterBuf;
		LinkList<ConfigProgram> progs;

		CStr strScript = ParseConfigProps(&cprops,&cflags,propBuffer,&strUnk,&strClusterBuf,&progs,&cscripts);
		GetDefaultProps(&DEFprops,className,typeName);

		Link<ConfigProp>* defprop = DEFprops.GetHead();
		Link<ConfigProp>* curprop;

		while(defprop)
		{
			curprop = cprops.Find(&defprop->data);

			// If the property doesn't exist, add it
			if (!curprop)
			{
				ConfigProp newProp = defprop->data;
				newProp.value = CStr("!");
				cprops.Add(&newProp);
			}

			defprop = defprop->next;
		}

		DumpNode(parentNode,
				 className,
				 typeName,
				 cflags,
				 &cprops,
				 strUnk,
				 strScript,
				 strClusterBuf,
				 &progs,
				 &cscripts);
	}

	pProgress->SetVal(pProgress->GetVal()+1);
}

bool PropUpdater::ShouldUpdate()
{
	ReferenceTarget* scene = ip->GetScenePointer();
	AppDataChunk* appdata;

	HANDLE handle;
	WIN32_FIND_DATA timestampSrc, timestampDest;

	appdata = scene->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_SCRIPTS_INI_TIMESTAMP_ID);

	if (appdata && appdata->data)
	{
		timestampSrc = *((WIN32_FIND_DATA*)appdata->data);

		handle = FindFirstFile( GetScriptIniName(), &timestampDest );  

		if ( CompareFileTime( &timestampDest.ftLastWriteTime, &timestampSrc.ftLastWriteTime ) > 0 )
		{
			int bResult = MessageBox(ip->GetMAXHWnd(),"The scripts.ini file has been updated.  Would you like to update the property data in existing nodes?","Scripts.ini Updated",MB_ICONQUESTION|MB_YESNO);

			if (bResult==IDNO)
			{
				int bResult2 = MessageBox(ip->GetMAXHWnd(),"Do you want to continue to be prompted about this?","Scripts.ini Updated",MB_ICONQUESTION|MB_YESNO);

				if (bResult2==IDYES)
					return false;
			}

			WIN32_FIND_DATA* newFindData = (WIN32_FIND_DATA*)malloc(sizeof(WIN32_FIND_DATA));
			*newFindData = timestampDest;

			scene->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_SCRIPTS_INI_TIMESTAMP_ID);
			scene->AddAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_SCRIPTS_INI_TIMESTAMP_ID,
								   sizeof(WIN32_FIND_DATA),newFindData);	

			if (bResult==IDYES)
				return true;

			return false;
		}

		return false;
	}

	handle = FindFirstFile( GetScriptIniName(), &timestampSrc );

	WIN32_FIND_DATA* newFindData = (WIN32_FIND_DATA*)malloc(sizeof(WIN32_FIND_DATA));
	*newFindData = timestampSrc;

	scene->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_SCRIPTS_INI_TIMESTAMP_ID);
	scene->AddAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_SCRIPTS_INI_TIMESTAMP_ID,
		                   sizeof(WIN32_FIND_DATA),newFindData);	

	return false;
}

void PropUpdater::ProcChanges()
{
	/*
	// If the file doesn't exist, copy it
	CStr strFilename = ip->GetDir(APP_PLUGCFG_DIR);
	strFilename += "\\scripts.bak";
	FILE* fp;

	fp = fopen(strFilename,"r");

	if (!fp)
	{
		// Our backup file doesn't exist, create it and return (no property updates necessary)
		CopyFile(GetScriptIniName(),strFilename,FALSE);
		return;
	}

	fclose(fp);

	if (FileIsNewer(strFilename,GetScriptIniName()))
	*/

	if (ShouldUpdate())
	{
		ParseScriptIni();

		int numNodes = CountNodes(NULL,0);
		pProgress = new ProgressBar(hInstance,hwndParent,"Updating Node Properties",0,numNodes);

		ProcNode(NULL);

		delete pProgress;
	}
}
