/*
	PropBufGen.cpp
	Property Buffer Generator
	2-13-01
*/

#include "../Trigger/Trigger.h"
#include "PropBufGen.h"
#include "PropFlags.h"
#include "ParseFuncs.h"

PropBufGen::PropBufGen()
{
	
}

PropBufGen::~PropBufGen()
{

}

CStr PropBufGen::GetClassProp(INode* node)
{
	CStr ClassProp;

	if (!node->GetUserPropString("Class",ClassProp))
		node->GetUserPropString("class",ClassProp);

	if (ClassProp==CStr(""))
	{
		Object* obj=node->EvalWorldState(0).obj;

		if (!obj)
			return CStr("");

		if(obj->ClassID()==vTRIGGER_CLASS_ID)
		{
			ClassProp=TRIGGER_CLASS;
		}
		else if (obj->ClassID()==Class_ID(SIMPLE_CAM_CLASS_ID,0) ||
			     obj->ClassID()==Class_ID(LOOKAT_CAM_CLASS_ID,0))
		{
			ClassProp=CAMERA_CLASS;
		}
		else
		{
			ClassProp=DEFAULT_CLASS;
		}
	}

	return ClassProp;
}

CStr PropBufGen::GetTypeProp(INode* node)
{
	CStr TypeProp;

	if (!node->GetUserPropString("Type",TypeProp))
	{
		if (!node->GetUserPropString("type",TypeProp))
		{
			CStr ClassProp = GetClassProp(node);
			
			// Find the class and use the first defined type
			ConfigClass cclass;
			cclass.name = ClassProp;
			Link<ConfigClass>* link = ConfigDBFind(&cclass);

			if (!link)
				return CStr("");

			if (link->data.types.GetHead() == NULL)
				return CStr("");

			return link->data.types.GetHead()->data.name;
		}
	}
	
	return TypeProp;
}

CStr PropBufGen::BuildPropBuffer(INode* node,
								 CStr strClass,
						         CStr strType,
						         DWORD flags,
						         LinkList<ConfigProp>* props,
						         CStr strCmds,
						         CStr strScript,
						         CStr strCluster,
								 LinkList<ConfigProgram>* programs,
								 LinkList<ConfigScript>* scripts)
{
	// Dump the node out to the property buffer
	CStr applyBuf;
	CStr eol="\r\n";

	// Dump class/type
	if (strClass.Length()==0 && node)
	{
		strClass=GetClassProp(node);
		strType=GetTypeProp(node);
	}

	applyBuf+=CStr("Class = ")+strClass+eol;
	
	if (strType!=CStr(""))
		applyBuf+=CStr("Type = ")+strType+eol;

	// Dump flags
	if (flags & CCLASS_CREATEDATSTART)
		applyBuf+=CStr("CreatedAtStart")+eol;
	
	if (flags & CCLASS_OCCLUDER)
		applyBuf+=CStr("Occluder")+eol;

	if (flags & CCLASS_ABSENTINNETGAMES)
		applyBuf+=CStr("AbsentInNetGames")+eol;

	if (flags & CCLASS_TRICKOBJECT)
		if (strCluster.Length()>0)
			applyBuf+=CStr("TrickObject Cluster = ")+strCluster+eol;
		else
			applyBuf+=CStr("TrickObject")+eol;

	// Convert properties to default delim where appropriate
	ConvertToDefaultDelim(strClass,strType,props);

	// Dump props

	Link<ConfigProp>* lprop=props->GetHead();

	while(lprop)
	{
		applyBuf+=lprop->data.name+CStr(" = ")+lprop->data.value+eol;

		lprop=lprop->next;
	}

	// Dump property extended data
	lprop=props->GetHead();

	while(lprop)
	{
		// Don't add extended property data if it exists within script.ini
		if (GetDynUICmds(strClass,strType,lprop->data.name).Length()==0)
		{
			if (lprop->data.extdata.Length()>0)
			{
				// Don't add eol unless necessary
				if (lprop->data.extdata.Substr(lprop->data.extdata.Length()-eol.Length(),eol.Length())==eol)
					applyBuf+=lprop->data.extdata;
				else
					applyBuf+=lprop->data.extdata+eol;
			}
		}

		lprop=lprop->next;
	}

	// Dump commands
	applyBuf+=strCmds;

	// Dump program
	if (programs &&
		programs->GetSize() > 0)
	{
		Link<ConfigProgram>* curprog = programs->GetHead();

		while(curprog)
		{
			if (curprog->data.scriptContext.Length()>0)
				applyBuf+=CStr("// @program | ")+curprog->data.filename+CStr(" | ")+curprog->data.scriptContext+eol;
			else
				applyBuf+=CStr("// @program | ")+curprog->data.filename+eol;

			curprog = curprog->next;
		}
	}

	// Dump out script
	applyBuf+=CStr("script")+eol;
	
	if (strScript==CStr(vUNKNOWN))
		strScript="";

	// Don't add eol unless necessary
	if (strScript.Substr(strScript.Length()-eol.Length(),eol.Length())==eol)
		applyBuf+=strScript;
	else
		applyBuf+=strScript+eol;


	applyBuf+=CStr("endscript")+eol;

	// Build the information for the alternate scripts
	if (scripts)
	{	
		Link<ConfigScript>* curscript = scripts->GetHead();

		while(curscript)
		{
			//applyBuf += CStr("altscript ") + curscript->data.name + CStr(" ") + curscript->data.filename + eol;
			applyBuf += CStr("altscript ") + curscript->data.name + eol;
			applyBuf += curscript->data.buffer + eol;

			applyBuf+=CStr("endscript")+eol;
			curscript = curscript->next;
		}
	}

	// Apply the new user properties to the node
	return applyBuf;
}

// Returns true if the buffer already existed
bool PropBufGen::AssignAltBuf(INode* node,CStr bufName,CStr buf)
{
	LinkList<ConfigProp> cprops;
	DWORD flags;
	CStr  unkBuf;
	CStr  clusterBuf;
	LinkList<ConfigProgram> programs;
	CStr  script;
	LinkList<ConfigScript> scripts;
	CStr  propBuffer;

	node->GetUserPropBuffer(propBuffer);

	CStr strClass = GetClassName(propBuffer);
	CStr strType  = GetTypeName(propBuffer);

	script = ParseConfigProps(&cprops,&flags,propBuffer,&unkBuf,&clusterBuf,&programs,&scripts);

	ConfigScript srch;
	srch.name = bufName;
	Link<ConfigScript>* link = scripts.Find(&srch);

	if (!link)
	{
		srch.buffer = buf;
		scripts.Add(&srch);
		PropBufGen::DumpNode(node,strClass,strType,flags,&cprops,unkBuf,script,clusterBuf,&programs,&scripts);
		return false;
	}

	link->data.buffer = buf;
	PropBufGen::DumpNode(node,strClass,strType,flags,&cprops,unkBuf,script,clusterBuf,&programs,&scripts);
	return true;
}

void PropBufGen::RemoveAltBuf(INode* node, CStr bufName)
{
	LinkList<ConfigProp> cprops;
	DWORD flags;
	CStr  unkBuf;
	CStr  clusterBuf;
	LinkList<ConfigProgram> programs;
	CStr  script;
	LinkList<ConfigScript> scripts;
	CStr  propBuffer;

	node->GetUserPropBuffer(propBuffer);

	CStr strClass = GetClassName(propBuffer);
	CStr strType  = GetTypeName(propBuffer);

	script = ParseConfigProps(&cprops,&flags,propBuffer,&unkBuf,&clusterBuf,&programs,&scripts);

	ConfigScript srch;
	srch.name = bufName;
	Link<ConfigScript>* link = scripts.Find(&srch);

	if (link)
		scripts.Remove(link);

	PropBufGen::DumpNode(node,strClass,strType,flags,&cprops,unkBuf,script,clusterBuf,&programs,&scripts);
}

CStr PropBufGen::GetAltBuf(INode* node,CStr bufName)
{
	CStr propBuffer;
	LinkList<ConfigScript> scripts;

	node->GetUserPropBuffer(propBuffer);

	ParseConfigProps(NULL,NULL,propBuffer,NULL,NULL,NULL,&scripts);

	ConfigScript srch;
	srch.name = bufName;

	Link<ConfigScript>* link = scripts.Find(&srch);

	if (!link)
		return CStr("");

	return link->data.buffer;
}

void PropBufGen::BuildList(LinkList<CStr>* list, CStr buf)
{
	int pos = 0;
	int len = buf.Length();

	while (pos < len)
	{
		CStr line = GetRemainLineExact(buf,&pos);
		line = Shave(line);
		list->Add(&line);
	}
}

bool PropBufGen::BuildList(CStr Filename,LinkList<CStr>* list)
{
	FILE* fp;
	char lineBuf[1024];

	list->Clear();

	fp = fopen(Filename,"r");

	if (!fp)
		return false;

	while(!feof(fp))
	{
		if (fgets(lineBuf,1023,fp))
			list->Add(&Shave(lineBuf));
	}

	fclose(fp);

	return true;
}

CStr PropBufGen::BuildBuf(LinkList<CStr>* list)
{
	CStr buf;
	Link<CStr>* curline = list->GetHead();

	while(curline)
	{
		buf += curline->data + CStr("\n");	
		curline = curline->next;
	}

	return buf;
}

CStr PropBufGen::BuildBufCRLF(LinkList<CStr>* list)
{
	CStr buf;
	Link<CStr>* curline = list->GetHead();

	while(curline)
	{
		buf += curline->data + CStr("\r\n");	
		curline = curline->next;
	}

	return buf;
}

CStr PropBufGen::BuildPartialList(LinkList<CStr>* list,Link<CStr>* start,Link<CStr>* end)
{
	CStr buf;

	if (!start)
		start = list->GetHead();

	if (!end)
		end = list->GetTail();

	Link<CStr>* curline = start;

	if (!curline)
		return CStr("");

	do
	{
		buf += curline->data + CStr("\n");	
		curline = curline->next;

	} while(curline && curline != end);

	return buf;
}
