#include "../Next.h"
#include "FaceFlags.h"

Tab<FlagType> g_deleted_face_data;

// The following functions make sure the Pipeline are aware of us and the m_data flows
// up the stack correctly

IFaceDataChannel* FaceFlagsData::CreateChannel( )
{
	return static_cast<IFaceDataChannel*>( new FaceFlagsData() );
}

IFaceDataChannel* FaceFlagsData::CloneChannel( )
{
	IFaceDataChannel* fdc = CreateChannel();
	if ( fdc != NULL )
		fdc->AppendChannel( this );

	return fdc;
}

BOOL FaceFlagsData::AppendChannel( const IFaceDataChannel* fromChan )
{
	int i;
	if ( fromChan == NULL ||
			 fromChan->DataChannelID() != DataChannelID() )
		return FALSE;

	const FaceFlagsData* fromFDChan = dynamic_cast<const FaceFlagsData*>(fromChan);

	int start = m_data.Count();
	for(i = 0; i<fromFDChan->m_data.Count();i++)
		m_data.Insert(i+start,1,&fromFDChan->m_data[i]);
	
	return TRUE;
}

// Called when new faces were created at index 'at' in the object's list of faces.
BOOL FaceFlagsData::FacesCreated( ULONG at, ULONG num )
{
	FlagType val = 0;	 // our default value
	if ( num < 1 ) 
		return FALSE;

	if ( at > m_data.Count() )
		at = m_data.Count();

	for(int i=0;i<num;i++)
	{
		m_data.Insert(i+at,1,&val);
	}
	
	return TRUE;
}

// The following functions are called when the object has deleted some of its faces
BOOL FaceFlagsData::FacesDeleted( BitArray& set )
{
	int j;

	if ( set.GetSize() != m_data.Count() ) 
		set.SetSize( m_data.Count(), TRUE );
	
	if ( set.NumberSet() == 0 ) 
		// nothing to free
		return TRUE;

	int n = set.GetSize();

	g_deleted_face_data.SetCount( set.NumberSet());

	j = 0;
	for ( int i = n-1; i >= 0; i-- )
	{
		if ( set[i] != 0 )
		{
			g_deleted_face_data[j] = m_data[i];
			j++;
			m_data.Delete( i,1 );
		}
	}

	return TRUE;
}

BOOL FaceFlagsData::FacesDeleted( ULONG from, ULONG num )
{
	int i;

	if ((from == m_data.Count()-1 ) || num < 1 )
		return FALSE;

	if ( from + num  == m_data.Count()-1 )
		num = m_data.Count() - from;

	g_deleted_face_data.SetCount( num );
	for( i = 0; i < num; i++ )
	{
		int index;		

		index = from + i;
		g_deleted_face_data[i] = m_data[index];
	}

	m_data.Delete(from,num);

	return TRUE;
}

void FaceFlagsData::AllFacesDeleted()
{
	m_data.SetCount(0);

}

// Called when the object has copied some of its faces 'from' 'to'
BOOL FaceFlagsData::FaceCopied(ULONG from,ULONG to)
{
	if ( from == m_data.Count() )
		return FALSE;
	if ( to == m_data.Count() )
		return FALSE;

	m_data[to] = m_data[from];

	return TRUE;
	
}

// Called when a new face has been created in the owner object based on
// data interpolated from other face
BOOL FaceFlagsData::FaceInterpolated(ULONG numSrc,ULONG *srcFaces,float *coeff,ULONG targetFace)
{
	// if params for interpolation are illegal, try simple copydata
	if ( numSrc < 1 || srcFaces == NULL || numSrc == m_data.Count()  || coeff == NULL )
	{
		return FALSE;
	}

	// For now just use the value in the first source face as the result for the
	// new face
	m_data[targetFace] = m_data[srcFaces[0]];

	/*m_data[targetFace] = m_data[srcFaces[0]]*coeff[0];

	for ( ULONG i = 1; i < numSrc; i++ )
	{
		m_data[targetFace] += m_data[srcFaces[i]]*coeff[i];
	}*/

	return TRUE;
}

// Called when the owner object has cloned some of its faces and appended
// them to its list of faces.
BOOL FaceFlagsData::FacesClonedAndAppended( BitArray& set )
{
	if ( set.GetSize() != m_data.Count() ) 
		set.SetSize( m_data.Count(), TRUE );
	
	if ( set.NumberSet() == 0 ) 
		// nothing to free
		return TRUE;

	int n = set.GetSize();

	for ( int i = 0; i < n; i++ )
	{
		if ( set[i] != 0 )
		{
			m_data.Insert(m_data.Count()+i,1, &m_data[i]);
		}
	}


	return TRUE;
}

// Simple Access to our data
BOOL FaceFlagsData::GetValue( ULONG at, FlagType& val ) const
{
	if( at >= m_data.Count())
	{
		val = 0;
		return TRUE;
	}
	val = m_data[at];
	return TRUE;
}

BOOL FaceFlagsData::GetValue( BitArray& set, FlagType& mask, FlagType& diff )
{
	int i;
	bool first_time;

	mask = 0;
	diff = 0;

	if( set.GetSize() != m_data.Count()) 
	{
		return FALSE;
	}	
	
	first_time = true;
	for( i = 0; i < set.GetSize(); i++ )
	{
		FlagType new_mask;

		if( set[i] )
		{
			if( first_time )
			{
				mask = m_data[i];
				first_time = false;
			}
			else
			{
				new_mask = m_data[i];
				diff |= ( new_mask ^ mask );
				mask &= new_mask;
			}
		}
	}

	return TRUE;
}

void FaceFlagsData::SetValue( ULONG at, const FlagType& val )
{
	m_data[at] = val;
}

void FaceFlagsData::SetValue( BitArray& set, const FlagType& val )
{
	int i;

	if( set.GetSize() != m_data.Count()) 
	{
		return;
	}

	for( i = 0; i < set.GetSize(); i++ )
	{
		if( set[i] )
		{
			m_data[i] = val;
		}
	}
}

bool ValidFlagConfiguration( FlagType mask )
{
	if(( mask & mFD_SKATABLE ) && ( mask & mFD_NOT_SKATABLE ))
	{
		return false;
	}

	if(( mask & mFD_VERT ) && ( mask & mFD_NOT_SKATABLE ))
	{
		return false;
	}

	if(( mask & mFD_VERT ) && ( mask & mFD_SKATABLE ))
	{
		return false;
	}

	if(( mask & mFD_SKATABLE ) && ( mask & mFD_NON_COLLIDABLE ))
	{
		return false;
	}

	if(( mask & mFD_NOT_SKATABLE ) && ( mask & mFD_NON_COLLIDABLE ))
	{
		return false;
	}

	if(( mask & mFD_WALL_RIDABLE ) && ( mask & mFD_NON_COLLIDABLE ))
	{
		return false;
	}

	return true;
}