#include <next.h>
#include <Export/Export.h>
#include <Export/TextureExporter.h>
#include <Texture/NExtTexture.h>
#include <Image/Image.h>
#include <Misc/GenCrc.h>
#include <Misc/Util.h>
#include "texture/bmtex.h"

#include <fstream.h>
#include <stdmat.h>
#include "../PropEdit/ParseFuncs.h"
#include "../UI/OKtoAll.h"

class TextureExporter : public ITextureExporter
{
	bool            bWarn;

public:
	TextureExporter() { bWarn = true; }

	void			Reset( void );	
	bool			AddTexture( Texmap* map, int flags, unsigned long checksum_array[vNUM_PLATFORMS] );
	bool			LoadTextureData( void );
	bool            VerifyTexturePaths( char* path );
	bool			SaveTextureDictionary( char* dict_path, char* usg_path );
	NxTexture*		GetTextureByChecksum( unsigned long checksum );
	
	Tab< NxTexture* >	m_Textures;	
	

private:

	unsigned long	add_texture( INExtTexture* tex, int platform, int flags, int plat_use_flags );

	char			m_tex_buff[1024 * 1024 * 4];
};


static TextureExporter	s_texture_exporter;

ITextureExporter*	GetTextureExporter( void )
{
	return &s_texture_exporter;
}


DWORD WINAPI TextureExportProgressFunc(LPVOID arg) 
{
    return(0);
}

void	TextureExporter::Reset( void )
{
	/*
	int i;

	for( i = 0; i < m_Textures.Count(); i++ )
	{
		delete m_Textures[i];
	}
	*/
	m_Textures.Delete(0,m_Textures.Count());	// aml: Tab class has ownership, it has to do the delete
	m_Textures.ZeroCount();
	m_WarnedAboutDupeMaterials = false;
	bWarn = true;

	MessageBoxResetAll();
}

int	get_max_mip_level( BitmapInfo* bitmap )
{
	int x_power, y_power, larger, smaller;

	if( !IsPowerOfTwo( bitmap->Width()) ||
		!IsPowerOfTwo( bitmap->Height()))
	{
		return 0;
	}

	x_power = GetLog2( bitmap->Width());
	y_power = GetLog2( bitmap->Height());
	larger = ( x_power < y_power ) ? y_power : x_power;
	smaller = ( x_power < y_power ) ? x_power : y_power;
	
	if( smaller >= 2 )
	{
		return ( smaller - 2 );
	}
	else
	{
		return 0;
	}
}

unsigned long	TextureExporter::add_texture( INExtTexture* tex, int platform, int flags, int plat_use_flags )
{
	NxTexture* new_tex;
	BitmapInfo* map;
	int i, num_mip_levels;

	new_tex = new NxTexture;

	map = tex->GetBaseMap( platform );

	// set to generated textures if appropriate
	if ( map->Flags() & MAP_IS_GENERATED )
	{
		flags |= NxTexture::mPOST_GENERATED_TEXTURE;
	}



	if( map == NULL )
	{
		return 0;
	}
	if( tex->GetMipType( platform ) == vMIP_TYPE_NONE )
	{
		num_mip_levels = 0;
	}
	else if( tex->GetMipType( platform ) == vMIP_TYPE_AUTO )
	{
		num_mip_levels = get_max_mip_level( map );
		flags |= NxTexture::mAUTO_GENERATE_MIPMAPS;
	}
	else
	{
		int j;

		num_mip_levels = tex->GetNumMipLevels( platform );
		for( j = 1; j <= num_mip_levels; j++ )
		{
			BitmapInfo* mip_map;

			mip_map = tex->GetMipMap( platform, j );
			if( mip_map == NULL )
			{
				// This shouldn't happen. If so, then it means we don't actually have as many levels as we thought
				// we did. In that case, adjust the number of levels accordingly
				num_mip_levels = j - 1;
				break;
			}

			new_tex->SetName((char*) mip_map->Name(), j );
		}		
	}	

	// Assign the UseBasePal flags
	for( i = 0; i<num_mip_levels; i++)
		new_tex->SetUseBasePal(i, tex->UseBasePal(platform,i));


	if( tex->ChangeTransColor())
	{
		flags |= NxTexture::mCHANGE_FULLY_TRANSPARENT_COLOR;
	}

	if( tex->Force24BitPalette())
	{
		flags |= NxTexture::mFORCE_BYTE_PER_COMPONENT;
	}

	if( tex->ShouldCompress( vPLAT_NGC ))
	{
		flags |= NxTexture::mCOMPRESS_NGC;
	}
	if( tex->ShouldCompress( vPLAT_XBOX ))
	{
		flags |= NxTexture::mCOMPRESS_XBOX;
	}

	new_tex->SetName((char*) map->Name(), 0);	
	new_tex->SetNumMipLevels( num_mip_levels );
	new_tex->SetMap( map );	
	new_tex->SetFlags( flags );
	new_tex->SetTransparentColor( tex->GetTransColor());
	new_tex->SetPlatformFlags( plat_use_flags );
	new_tex->GenerateChecksum();	

	// First check to see if we already have this texture
	for( i = 0; i < m_Textures.Count(); i++ )
	{
		if( *m_Textures[i] == *new_tex )
		{
			// If the new texture has more mip levels than the previous texture with the same base name
			// is the new one instead
			if( m_Textures[i]->GetNumMipLevels() < num_mip_levels )
			{
				char msg[1024];

				sprintf( msg, "Two NxTextures share the same base texture %s but have different mipmaps.  Using the one with more mipmap levels defined", new_tex->GetName( 0 ));
				// Temporarily disabled for THPS4
				//MessageBox( gInterface->GetMAXHWnd(), msg, "Warning!", MB_OK );
				m_Textures.Append( 1, &new_tex );
				m_Textures.Delete( i, 1 );
				return new_tex->GetChecksum();
			}
			else
			{			
				// The final flags should be a combination of all of the operations the user
				// wanted to perform on this texture
				m_Textures[i]->SetFlags( flags );
				delete new_tex;
				return m_Textures[i]->GetChecksum();
			}
		}
	}	
	
	m_Textures.Append( 1, &new_tex );
	return new_tex->GetChecksum();
}

bool	platform_override( INExtTexture* tex, int src_plat, int dst_plat )
{
	BitmapInfo* bm_src, *bm_dst;

	bm_src = tex->GetBaseMap( src_plat );
	assert( bm_src );
	bm_dst = tex->GetBaseMap( dst_plat );
	
	// If there's a different base texture defined, the dst platform is overriding for sure
	if( bm_dst && ( stricmp( bm_src->Name(), bm_dst->Name()) != 0 ))
	{
		return true;
	}

	// Even with the same base texture, the dst platform may want to override mip maps
	if(	tex->GetMipType( src_plat ) != tex->GetMipType( dst_plat ))
	{
		return true;
	}

	if(	src_plat == vPLAT_PS2 )
	{
		// At this point the only way they could be the same is if the source is using manual mips and the
		// dst is sharing them
		if( tex->GetMipType( src_plat ) == vMIP_TYPE_MANUAL )
		{
			return ( tex->UsePS2Mips( dst_plat ) == false );		
		}	
	}

	return false;
}

bool	TextureExporter::AddTexture( Texmap* map, int flags, unsigned long checksum_array[vNUM_PLATFORMS] )
{
	BitmapInfo* bitmap_ps2, *bitmap_ngc, *bitmap_xbox;
	int plat_use_flags;

	if( map == NULL )
	{
		return false;
	}
	
	if( map->ClassID() != NEXT_TEXTURE_CLASS_ID )
	{
		return false;
	}

	INExtTexture* next_tex = dynamic_cast< INExtTexture* >( map );		

	// For now force all NGC and XBOX to use 32-bit RGBA regardless
	bitmap_ps2 = next_tex->GetBaseMap( vPLAT_PS2 );

	if (bitmap_ps2 == NULL)
		return false;

	// Don't add the map if there's no defined device (Bad/Empty map?)
	if (strlen(bitmap_ps2->Device()) == 0)
		return false;

	char bufDrive[_MAX_DRIVE];
	char bufPath[_MAX_DIR];
	char bufFilename[_MAX_FNAME];
	char bufExt[_MAX_EXT];

	_splitpath((char*)bitmap_ps2->Name(),bufDrive,bufPath,bufFilename,bufExt);


	TSTR strOverride = TSTR(bufDrive) + TSTR(bufPath) + TSTR("mips\\") + TSTR(bufFilename);
	strOverride += "_auto32m0.png";

	// Only one texture should be assigned so texture usage isn't duplicated
	// (If XBOX and NGC are both assigned auto they will get the texture object
	//  for the first type used, should be NGC, if they are assigned none or
	//  manual the override does not apply and processing should be done as it
	//  always has been (Manual mips/ or none))
	if (next_tex->GetMipType( vPLAT_NGC ) == vMIP_TYPE_AUTO)
	{
		next_tex->AssignOverrideTexName( vPLAT_NGC, strOverride );
		flags |= NxTexture::mAUTO_GENERATE_NGC_MIPS;
	}
	else if (next_tex->GetMipType( vPLAT_XBOX ) == vMIP_TYPE_AUTO)
	{
		next_tex->AssignOverrideTexName( vPLAT_XBOX, strOverride );
		flags |= NxTexture::mAUTO_GENERATE_XBOX_MIPS;
	}
	

	bitmap_ngc = next_tex->GetBaseMap( vPLAT_NGC );
	bitmap_xbox = next_tex->GetBaseMap( vPLAT_XBOX );

	if( bitmap_ps2 == NULL )
	{
		return false;
	}

	plat_use_flags = mPLAT_PS2;
	if( platform_override( next_tex, vPLAT_PS2, vPLAT_NGC ) == false )
	{
		plat_use_flags |= mPLAT_NGC;
	}
	if( platform_override( next_tex, vPLAT_PS2, vPLAT_XBOX ) == false )
	{
		plat_use_flags |= mPLAT_XBOX;
	}

	checksum_array[vPLAT_PS2] = add_texture( next_tex, vPLAT_PS2, flags, plat_use_flags );

	// If neither NGC nor Xbox share this texture, check if THEY'RE using the same
	// texture as eachother
	if(	(	( plat_use_flags & ( mPLAT_NGC | mPLAT_XBOX )) == 0 ) &&
		( platform_override( next_tex, vPLAT_XBOX, vPLAT_NGC ) == false ))
	{
		
		checksum_array[vPLAT_NGC] = add_texture( next_tex, vPLAT_NGC, flags, mPLAT_NGC | mPLAT_XBOX );		
		checksum_array[vPLAT_XBOX] = checksum_array[vPLAT_NGC];
	}
	else
	{			
		if( platform_override( next_tex, vPLAT_PS2, vPLAT_NGC ))
		{
			checksum_array[vPLAT_NGC] = add_texture( next_tex, vPLAT_NGC, flags, mPLAT_NGC );
		}
		else
		{
			checksum_array[vPLAT_NGC] = checksum_array[vPLAT_PS2];		
		}

		if( platform_override( next_tex, vPLAT_PS2, vPLAT_XBOX ))
		{
			checksum_array[vPLAT_XBOX] = add_texture( next_tex, vPLAT_XBOX, flags, mPLAT_XBOX );
		}
		else
		{
			checksum_array[vPLAT_XBOX] = checksum_array[vPLAT_PS2];		
		}
	}

	return true;
}

bool TextureExporter::LoadTextureData( void )
{	
	int i, num_textures, export_pass;
	unsigned long checksum;
	
	// Ok, we need to export the normal textures, followed by the
	// mPOST_GENERATED_TEXTURE flag

	for( export_pass = 0; export_pass < 2 ; export_pass++)
	{
		gInterface->ProgressStart(_T("Loading Texture Data"), TRUE, 
								TextureExportProgressFunc, NULL );
		num_textures = m_Textures.Count();
		for( i = 0; i < num_textures; i++ )
		{
			NxTexture* texture;
			BitmapInfo* bitmap;	
			int width, height;
			bool bExportPostGenMips;

			texture = m_Textures[i];
			bitmap = texture->GetMap();
			assert( bitmap );
			
			if (export_pass == 0 && !(texture->GetFlags() & NxTexture::mPOST_GENERATED_TEXTURE) ||
				export_pass == 1 && (texture->GetFlags() & NxTexture::mPOST_GENERATED_TEXTURE))
			{
				checksum = texture->GetChecksum();

				// If this is pass 2 we won't have width and height yet
				if (export_pass == 0)
				{
					width = bitmap->Width();
					height = bitmap->Height();
					if( !IsPowerOfTwo( width ) ||
						!IsPowerOfTwo( height ))
					{
						char error_msg[128];
						
						sprintf( error_msg, "Texture %s has a dimensions %d %d. They must be powers of two.", texture->GetName( 0 ),
									width, height );				
						MessageBoxAll( gInterface->GetMAXHWnd(), error_msg, "Texture Dimension Error!", MB_OK );		
						texture->SetValidity( false );
						continue;
					}

					texture->SetWidth( 0, width );
					texture->SetHeight( 0, height );
					
					bExportPostGenMips = true;
				}
				else
				{
					bExportPostGenMips = false;
				}

				if( texture->LoadImage( bExportPostGenMips ))
				{
					switch( texture->GetPixelFormat())
					{
						case NxTexture::v32_BIT:						
						case NxTexture::v24_BIT:						
						case NxTexture::v16_BIT:						
						case NxTexture::v8_BIT:						
						case NxTexture::v4_BIT:
							break;
						default:
							texture->SetValidity( false );
							continue;
					}

					switch( texture->GetPaletteFormat())
					{
						case NxTexture::v32_BIT:
						case NxTexture::v16_BIT:
						case NxTexture::v24_BIT:
							break;
						default:
							texture->SetValidity( false );
							continue;
					}
				}
				else
				{
					texture->SetValidity( false );
				}

				if( gInterface->GetCancel())
				{
					gInterface->ProgressEnd();
					return false;
				}
			}

			gInterface->ProgressUpdate( i * 100 / num_textures );		
		}

		gInterface->ProgressEnd();
	}

	return true;
}

bool TextureExporter::SaveTextureDictionary( char* dict_path, char* usg_path )
{
	fstream file, usg_file;
	int i, j, num_textures, version;
	unsigned long checksum;
	DWORD attribs;
	
	attribs = GetFileAttributes( dict_path );
	if( attribs != -1 )
	{
		if( attribs & FILE_ATTRIBUTE_READONLY )
		{
			char message[256];
			
			sprintf( message, "You are about to save over a Read-Only file, %s, perhaps you should check-out the file first", dict_path );
			if( MessageBoxAll( gInterface->GetMAXHWnd(), message, "Read-only Warning", MB_OKCANCEL ) == IDCANCEL )
			{
				return false;
			}
		}
	}

	// Do the same check for the usage file
	attribs = GetFileAttributes( usg_path );
	if( attribs != -1 )
	{
		if( attribs & FILE_ATTRIBUTE_READONLY )
		{
			char message[256];
			
			sprintf( message, "You are about to save over a Read-Only file, %s, perhaps you should check-out the file first", usg_path );
			if( MessageBoxAll( gInterface->GetMAXHWnd(), message, "Read-only Warning", MB_OKCANCEL ) == IDCANCEL )
			{
				return false;
			}
		}
	}

	// break the lock, if necessary
	SetFileAttributes( dict_path, FILE_ATTRIBUTE_NORMAL );
	SetFileAttributes( usg_path, FILE_ATTRIBUTE_NORMAL );
	file.open( dict_path, ios::out | ios::binary );
	usg_file.open( usg_path, ios::out );
	
	version = vTEXTURE_VERSION_NUMBER;
	file.write((const char*) &version, sizeof( int ));

	num_textures = m_Textures.Count();
	file.write((const char*) &num_textures, sizeof( int ));

	gInterface->ProgressStart(_T("Exporting Texture Dictionary"), TRUE, 
							TextureExportProgressFunc, NULL );
	for( i = 0; i < num_textures; i++ )
	{		
		NxTexture* texture;
		int max_level;
		char basename[_MAX_FNAME], ext[_MAX_EXT];
		char usg_line[128];
		
		texture = m_Textures[i];

		// Verify that the texture names don't have spaces
		CStr texname = texture->GetName( 0 );
		char* nstr = strrchr((char*)texname,'\\');
		if (!nstr)
			nstr = (char*)texname;
		else
			nstr++;

		if (strstr(nstr," "))
		{
			char strErr[256];
			sprintf(strErr,"A texture '%s' contains spaces.  This is unsupported.",nstr);
			MessageBoxAll(gInterface->GetMAXHWnd(),strErr,"Texture contains spaces",MB_ICONWARNING|MB_OK);
		}

		if( texture->IsValid())
		{
			int width, height;
			int pixel_mode, clut_mode, flags, plat_flags, name_len;
	
			name_len = strlen( texture->GetName( 0 ));
			file.write((const char*) &name_len, sizeof( int ));
			file.write((const char*) texture->GetName( 0 ), name_len );

			plat_flags = texture->GetPlatformFlags();
			file.write((const char*) &plat_flags, sizeof( int ));

			checksum = texture->GetChecksum();
			file.write((const char*) &checksum, sizeof( unsigned long ));
			width = texture->GetWidth( 0 );
			height = texture->GetHeight( 0 );
			file.write((const char*) &width, sizeof( int ));
			file.write((const char*) &height, sizeof( int ));
			pixel_mode = texture->GetPixelFormat();
			clut_mode = texture->GetPaletteFormat();
			flags = texture->GetFlags();			
		
			// Get the max mip level
			max_level = texture->GetNumMipLevels();
			
			// Write out pixel/clut storage modes
			file.write((const char*) &flags, sizeof( int ));			
			file.write((const char*) &pixel_mode, sizeof( int ));
			file.write((const char*) &clut_mode, sizeof( int ));			
			file.write((const char*) &max_level, sizeof( int ));

			if( texture->IsPaletted())
			{				
				if( texture->GetPaletteFormat() == NxTexture::v32_BIT )
				{
					if( texture->ShouldChangeTransparentColor())
					{
						unsigned char* src, *alpha;
						int num_entries;
						unsigned char red, green, blue;
						Color trans_color;

						trans_color = texture->GetTransparentColor();
						red = trans_color.r * 255.0f;
						green = trans_color.g * 255.0f;
						blue = trans_color.b * 255.0f;
						num_entries = texture->GetNumPaletteEntries();
						src = (unsigned char*) texture->GetPaletteData();
						for( j = 0; j < num_entries; j++ )
						{
							alpha = src + ( 3 * sizeof( char ));	// skip to alpha val
							// If the alpha is
							if( *alpha == 0 )
							{
								*src++ = red;
								*src++ = green;
								*src++ = blue;
								src++;
							}
							else
							{
								src += ( 4 * sizeof( char ));
							}
						}							
					}
				}
				
				file.write((const char*) texture->GetPaletteData(), texture->GetPaletteDataSize());				
			}

			for( j = 0; j <= texture->GetNumMipLevels(); j++ )
			{
				// Write out the width and height of the mip  aml
				int mipWidth, mipHeight;

				mipWidth  = texture->GetWidth( j );
				mipHeight = texture->GetHeight( j );

				file.write((const char*) &mipWidth, sizeof(int));
				file.write((const char*) &mipHeight, sizeof(int));

				// Write out the texel data
				file.write((const char*) texture->GetTexelData( j ), 
								texture->GetTexelDataSize( j ));									
			}
		}
		else
		{
			int pad;

			pad = 0;

			file.write((const char*) &pad, sizeof( int ));		
			file.write((const char*) &pad, sizeof( int ));

			pad = -1;			
		
			// Write out a header that signifies an empty texture
			checksum = 0xFFFFFFFF;
			file.write((const char*) &checksum, sizeof( unsigned long ));
			
			file.write((const char*) &pad, sizeof( int ));
			file.write((const char*) &pad, sizeof( int ));
			file.write((const char*) &pad, sizeof( int ));
			file.write((const char*) &pad, sizeof( int ));
			file.write((const char*) &pad, sizeof( int ));
			file.write((const char*) &pad, sizeof( int ));
		}

		if( gInterface->GetCancel())
		{
			gInterface->ProgressEnd();
			file.close();
			usg_file.close();
			return false;
		}

		_splitpath( texture->GetName( 0 ), NULL, NULL, basename, ext );
		sprintf( usg_line, "%s%s (0x%x) : %dx%d: %d bpp %d pbpp %d levels\n", basename, ext, 
					texture->GetChecksum(),	texture->GetWidth( 0 ),	texture->GetHeight( 0 ), 
					texture->GetBpp(), texture->GetPaletteBpp(), max_level + 1 );
		usg_file.write( usg_line, strlen( usg_line ));
		
		gInterface->ProgressUpdate( i * 100 / num_textures );
	}	

	gInterface->ProgressEnd();

	file.close();
	usg_file.close();
	return true;
}


NxTexture*	TextureExporter::GetTextureByChecksum( unsigned long checksum )
{
	int i;

	for( i = 0; i < m_Textures.Count(); i++ )
	{
		if( m_Textures[i]->GetChecksum() == checksum )
		{
			return m_Textures[i];
		}
	}
	
	return NULL;
}

bool TextureExporter::VerifyTexturePaths( char* path )
{
	if (!bWarn)
		return true;

	int i;

	// Standardize on using backslashes for paths
	CStr strPath = ReplaceStr(path,"/","\\");
	CStr filePath;

	strPath = CStr("q:\\sk4\\") + strPath + CStr("\\tex");

	for( i = 0; i < m_Textures.Count(); i++ )
	{
		// Standardize on using backslashes for paths
		filePath = ReplaceStr(m_Textures[i]->GetName(0),"/","\\");
		filePath = ReplaceStr(filePath,"\\\\Bruce\\Qqq","q:");		// TODO: Hard coded paths for now, will change later
		filePath.toLower();
		strPath.toLower();

		if( !IsInstr(filePath,strPath) )
		{
			strPath.toLower();

			char buf[256];
			sprintf(buf,"WARNING! The texture '%s' does not reside in the expected directory '%s'.\nWould you like to recieve further warnings of this type?",(char*)m_Textures[i]->GetName(0),(char*)strPath);
			int bResult = MessageBox(gInterface->GetMAXHWnd(),buf,"Directory Warning",MB_ICONWARNING|MB_YESNOCANCEL);

			if (bResult==IDNO)
			{
				bWarn = false;
				return true;
			}

			if (bResult==IDCANCEL)
				return false;
		}
	}

	return true;
}
