#include <Export/Skeleton.h>
#include <Misc/GenCrc.h>
#include <Misc/HelperFuncs.h>

void CSkeletonData::print_name_table()
{
    DebugPrint( "Name table:\n" );

    for ( int i = 0; i < m_numNodes; i++ )
    {
        MaxAssert( mp_nodes[i] );
        DebugPrint( "Name %d %s\n", i, mp_nodes[i]->GetName() );
    }
};

INode* CSkeletonData::GetRootBone( INode* pNode )
{
    MaxAssert( pNode );
    MaxAssert( pNode->GetParentNode() );

    if ( pNode->GetParentNode()->GetParentNode() == NULL )
    {
        // the root object's parent should be "Scene Root".
        // The Scene Root's parent should be NULL.
        return pNode;
    }
    else
    {
        return GetRootBone( pNode->GetParentNode() );
    }
}

void CSkeletonData::recursive_add_node( INode* pNode )
{
    MaxAssert( pNode );

	// ignore nodes with physique meshes, and leaf nodes, and any nodes that have "IK_" in them...
    if ( !FindPhysiqueModifier( pNode ) && !strstr( pNode->GetName(), "IK_" ) )
    {
		if ( (bIgnoreLeafs  && pNode->NumberOfChildren() > 0) || !bIgnoreLeafs )
		{
			MaxAssert( m_numNodes < vMAX_NODES );
			mp_nodes[m_numNodes++] = pNode;
		}
    }

    for ( int i = 0; i < pNode->NumberOfChildren(); i++ )
    {
        recursive_add_node( pNode->GetChildNode(i) );
    }
}

CSkeletonData::CSkeletonData( INode* pNode, bool bIgnoreLeafs )
{
    this->bIgnoreLeafs=bIgnoreLeafs;
	
	m_numNodes = 0;
    
	DebugPrint( "Starting with node %s\n", pNode->GetName() );

    INode* pRootNode = GetRootBone( pNode );
    // should be the "Dummy_Scale" node for skaters

	DebugPrint( "Root node is %s\n", pRootNode->GetName() );

    // given any node in a particular hierarchy,
    // this will walk up the tree until it finds the root
    // ("Dummy_Scale" for skaters), then build a table
    // of children from there.  It will ignore
    // any Physiqued children
    recursive_add_node(pRootNode);

//  print_name_table();

	print_debug_file();
}

INode* CSkeletonData::GetBone( int index )
{
	MaxAssert( index >= 0 && index < m_numNodes );

	return mp_nodes[index];
}

int CSkeletonData::GetCount()
{
	return m_numNodes;
}

int CSkeletonData::GetBoneIndex( INode* pNode )
{
    for ( int i = 0; i < m_numNodes; i++ )
    {
        if ( pNode == mp_nodes[i] )
        {
            return i;
        }
    }
                   
    // node not found
    MaxAssert( 0 );
    return -1;
}

unsigned int CSkeletonData::GetBoneName( int i )
{
	MaxAssert( i >= 0 && i < GetCount() );

	return GenerateCRC( mp_nodes[i]->GetName() );
}

unsigned int CSkeletonData::GetParentName( int i )
{
	MaxAssert( i >= 0 && i < GetCount() );

	if ( GetBone(i) == GetRootBone( GetBone(i) ) )
	{
		// if we're the root node, then the parent is 0
		return 0;
	}
	else
	{
		return GenerateCRC( mp_nodes[i]->GetParentNode()->GetName() );
	}
}

unsigned int CSkeletonData::GetFlipName( int i )
{
	MaxAssert( i >= 0 && i < GetCount() );

	TCHAR* pCurrentBoneName = GetBone(i)->GetName();
	TSTR neutral_name = GetBone(i)->GetName();
	BOOL is_left = FALSE;
	BOOL is_right = FALSE;
	TCHAR* p_temp;

	if ( p_temp = strstr(pCurrentBoneName, "left_"))
	{
		// compensate for "left_" not being at the beginning of the string
		neutral_name = p_temp;
		neutral_name = neutral_name.Substr( strlen("left_"), strlen(pCurrentBoneName) - strlen("left_") );
		is_left = TRUE;

	}
	else if ( p_temp = strstr(pCurrentBoneName, "right_"))
	{
		// compensate for "right_" not being at the beginning of the string
		neutral_name = p_temp;
		neutral_name = neutral_name.Substr( strlen("right_"), strlen(pCurrentBoneName) - strlen("right_") );
		is_right = TRUE;
	}

	TSTR left_name;
	TSTR right_name;

	left_name = TSTR("left_") + neutral_name;
	right_name = TSTR("right_") + neutral_name;

	// if left-hand part
	if (is_left)
	{
		for ( int i = 0; i < GetCount(); i++ )
		{
			TCHAR* pBoneName = GetBone(i)->GetName();
			if ( strstr(pBoneName, right_name.data()) )
			{
				return GenerateCRC(pBoneName);
			}
		}
		return 0;
	}
	else if (is_right)
	{
		for ( int i = 0; i < GetCount(); i++ )
		{
			TCHAR* pBoneName = GetBone(i)->GetName();
			if ( strstr(pBoneName, left_name.data()) )
			{
				return GenerateCRC(pBoneName);
			}
		}
		return 0;
	}
	else
	{
		return 0;
	}
}

unsigned int CSkeletonData::GetChecksum()
{
	unsigned int checksum = (unsigned int)GetCount();

	int i;

	for ( i = 0; i < GetCount(); i++ )
	{
		checksum ^= GetBoneName( i );
	}

	for ( i = 0; i < GetCount(); i++ )
	{
		checksum ^= GetParentName( i );
	}

	for ( i = 0; i < GetCount(); i++ )
	{
		checksum ^= GetFlipName( i );
	}

	return checksum;
}

void CSkeletonData::print_debug_file( void )
{
	unsigned int checksum = GetChecksum();
	DebugPrint("checksum = %08x\n", checksum);

	unsigned int numBones = GetCount();
	DebugPrint("numBones = %08x\n", numBones);

	int i;

	for ( i = 0; i < GetCount(); i++ )
	{
		unsigned int boneName = GetBoneName( i );
	}

	for ( i = 0; i < GetCount(); i++ )
	{
		unsigned int parentName = GetParentName( i );
	}

	for ( i = 0; i < GetCount(); i++ )
	{
		unsigned int flipName = GetFlipName( i );
	}

	for ( i = 0; i < GetCount(); i++ )
	{
		char parentName[256];
		for ( int j = 0; j < GetCount(); j++ )
		{
			if ( GetBone(i) == GetRootBone( GetBone(i) ) )
			{
				// if we're the root node, then the parent is 0
				strcpy( parentName, "Root" );
			}
			else
			{
				strcpy( parentName, mp_nodes[i]->GetParentNode()->GetName() );
			}
		}
		int flipIndex = -1;

		DebugPrint("bone %d name %s %08x parent = %08x (%s)\n", i, GetBone(i)->GetName(),
			GetBoneName(i),
			GetParentName(i),
			parentName);
	}
}

int CSkeletonData::Write( FILE* pFile )
{
	MaxAssert( pFile );

	unsigned int size = 0;

	unsigned int checksum = GetChecksum();
	fwrite(&checksum,sizeof(unsigned int),1,pFile);
	size += sizeof(unsigned int);

	unsigned int numBones = GetCount();
	fwrite(&numBones,sizeof(unsigned int),1,pFile);
	size += sizeof(unsigned int);

	int i;

	for ( i = 0; i < GetCount(); i++ )
	{
		unsigned int boneName = GetBoneName( i );
		fwrite(&boneName,sizeof(unsigned int),1,pFile);
		size += sizeof(unsigned int);
	}

	for ( i = 0; i < GetCount(); i++ )
	{
		unsigned int parentName = GetParentName( i );
		fwrite(&parentName,sizeof(unsigned int),1,pFile);
		size += sizeof(unsigned int);
	}

	for ( i = 0; i < GetCount(); i++ )
	{
		unsigned int flipName = GetFlipName( i );
		fwrite(&flipName,sizeof(unsigned int),1,pFile);
		size += sizeof(unsigned int);
	}

	// return number of bytes written
	return size;
}
