 /**********************************************************************  
 *<
	FILE: PolyModes.cpp

	DESCRIPTION: Editable Polygon Mesh Object - Command modes

	CREATED BY: Steve Anderson

	HISTORY: created April 2000

 *>	Copyright (c) 2000, All Rights Reserved.
 **********************************************************************/

#include "EPoly.h"
#include "PolyEdit.h"
#include "macrorec.h"
#include "decomp.h"

#define EPSILON .0001f

namespace EditPoly
{

CommandMode * EditPolyObject::getCommandMode (int mode) {
	switch (mode) {
	case epmode_create_vertex: return createVertMode;
	case epmode_create_edge: return createEdgeMode;
	case epmode_create_face: return createFaceMode;
	case epmode_divide_edge: return divideEdgeMode;
	case epmode_divide_face: return divideFaceMode;
	case epmode_extrude_vertex:
	case epmode_extrude_edge:
	case epmode_extrude_face:
		return extrudeMode;
	case epmode_chamfer_vertex:
	case epmode_chamfer_edge:
		return chamferMode;
	case epmode_bevel: return bevelMode;
	case epmode_cut_vertex: return cutVertMode;
	case epmode_cut_edge: return cutEdgeMode;
	case epmode_cut_face: return cutFaceMode;
	case epmode_weld: return weldMode;
	case epmode_edit_tri: return editTriMode;
	}
	return NULL;
}

void EditPolyObject::EpActionToggleCommandMode (int mode) {
	if (!ip) return;

	if (mode == epmode_sliceplane) {
		// Special case.
		ip->ClearPickMode();
		if (sliceMode) ExitSliceMode();
		else EnterSliceMode();
		return;
	}

	// Otherwise, make sure we're not in Slice mode:
	if (sliceMode) ExitSliceMode ();

	CommandMode *cmd = getCommandMode (mode);
	if (cmd==NULL) return;

	if (ip->GetCommandMode() == cmd) {
		ip->SetStdCommandMode(CID_OBJMOVE);
	} else {
		EnterCommandMode (mode);
	}
}

void EditPolyObject::EnterCommandMode(int mode) {
	if (!ip) return;
	CommandMode *cmd = getCommandMode (mode);
	if (cmd==NULL) return;

	switch (mode) {
	case epmode_create_vertex:
		if (selLevel != EP_SL_VERTEX) SetSubobjectLevel (EP_SL_VERTEX);
		ip->SetCommandMode(createVertMode);
		break;

	case epmode_create_edge:
		if (meshSelLevel[selLevel] != MNM_SL_EDGE) SetSubobjectLevel (EP_SL_EDGE);
		ip->SetCommandMode (createEdgeMode);
		break;

	case epmode_create_face:
		if (selLevel < EP_SL_FACE) SetSubobjectLevel (EP_SL_FACE);
		ip->SetCommandMode (createFaceMode);
		break;

	case epmode_divide_edge:
		if (meshSelLevel[selLevel] != MNM_SL_EDGE) SetSubobjectLevel (EP_SL_EDGE);
		ip->SetCommandMode (divideEdgeMode);
		break;

	case epmode_divide_face:
		if (selLevel < EP_SL_FACE) SetSubobjectLevel (EP_SL_FACE);
		ip->SetCommandMode (divideFaceMode);
		break;

	case epmode_extrude_vertex:
		if (selLevel != EP_SL_VERTEX) SetSubobjectLevel (EP_SL_VERTEX);
		ip->SetCommandMode (extrudeMode);
		break;

	case epmode_extrude_edge:
		if (meshSelLevel[selLevel] != MNM_SL_EDGE) SetSubobjectLevel (EP_SL_EDGE);
		ip->SetCommandMode (extrudeMode);
		break;

	case epmode_extrude_face:
		if (selLevel < EP_SL_FACE) SetSubobjectLevel (EP_SL_FACE);
		ip->SetCommandMode (extrudeMode);
		break;

	case epmode_chamfer_vertex:
		if (selLevel != EP_SL_VERTEX) SetSubobjectLevel (EP_SL_VERTEX);
		ip->SetCommandMode (chamferMode);
		break;

	case epmode_chamfer_edge:
		if (meshSelLevel[selLevel] != MNM_SL_EDGE) SetSubobjectLevel (EP_SL_EDGE);
		ip->SetCommandMode (chamferMode);
		break;

	case epmode_bevel:
		if (selLevel < EP_SL_FACE) SetSubobjectLevel (EP_SL_FACE);
		ip->SetCommandMode (bevelMode);
		break;

	case epmode_cut_vertex:
		if (selLevel != EP_SL_VERTEX) SetSubobjectLevel (EP_SL_VERTEX);
		ip->SetCommandMode (cutVertMode);
		break;

	case epmode_cut_edge:
		if (meshSelLevel[selLevel] != MNM_SL_EDGE) SetSubobjectLevel (EP_SL_EDGE);
		ip->SetCommandMode (cutEdgeMode);
		break;

	case epmode_cut_face:
		if (selLevel < EP_SL_FACE) SetSubobjectLevel (EP_SL_FACE);
		ip->SetCommandMode (cutFaceMode);
		break;

	case epmode_weld:
		ip->SetCommandMode (weldMode);
		break;

	case epmode_edit_tri:
		if (selLevel < EP_SL_FACE) SetSubobjectLevel (EP_SL_FACE);
		ip->SetCommandMode (editTriMode);
		break;
	}
}

void EditPolyObject::EpActionEnterPickMode (int mode) {
	if (!ip) return;

	// Make sure we're not in Slice mode:
	if (sliceMode) ExitSliceMode ();

	switch (mode) {
	case epmode_attach:
		ip->SetPickMode (attachPickMode);
		break;
	}
}

//------------Command modes & Mouse procs----------------------

HitRecord *PickEdgeMouseProc::HitTestEdges (IPoint2 &m, ViewExp *vpt, float *prop, 
											Point3 *snapPoint) {
	vpt->ClearSubObjHitList();
	ip->SubObHitTest(ip->GetTime(),HITTYPE_POINT,0, 0, &m, vpt);
	if (!vpt->NumSubObjHits()) return NULL;
	HitLog& hitLog = vpt->GetSubObjHitList();
	HitRecord *hr = hitLog.ClosestHit();
	if (!hr) return hr;
	if (!prop) return hr;

	// Find where along this edge we hit
	// Strategy:
	// Get Mouse click, plus viewport z-direction at mouse click, in object space.
	// Then find the direction of the edge in a plane orthogonal to z, and see how far
	// along that edge we are.

	DWORD ee = hr->hitInfo;
	Matrix3 obj2world = hr->nodeRef->GetObjectTM (ip->GetTime ());

	Ray r;
	vpt->MapScreenToWorldRay ((float)m.x, (float)m.y, r);
	if (!snapPoint) snapPoint = &(r.p);
	Point3 Zdir = Normalize (r.dir);

	MNMesh & mm = *(ep->GetMeshPtr());
	Point3 A = obj2world * mm.v[mm.e[ee].v1].p;
	Point3 B = obj2world * mm.v[mm.e[ee].v2].p;
	Point3 Xdir = B-A;
	Xdir -= DotProd(Xdir, Zdir)*Zdir;
	*prop = DotProd (Xdir, *snapPoint-A) / LengthSquared (Xdir);
	if (*prop<.0001f) *prop=0;
	if (*prop>.9999f) *prop=1;
	return hr;
}

int PickEdgeMouseProc::proc (HWND hwnd, int msg, int point, int flags, IPoint2 m) {
	ViewExp *vpt;
	HitRecord *hr;
	float prop;
	Point3 snapPoint;

	switch (msg) {
	case MOUSE_PROPCLICK:
		ip->SetStdCommandMode(CID_OBJMOVE);
		break;

	case MOUSE_POINT:
		ip->SetActiveViewport(hwnd);
		vpt = ip->GetViewport(hwnd);
		snapPoint = vpt->SnapPoint (m, m, NULL);
		snapPoint = vpt->MapCPToWorld (snapPoint);
		hr = HitTestEdges (m, vpt, &prop, &snapPoint);
		if (vpt) ip->ReleaseViewport(vpt);
		if (hr) EdgePick(hr->hitInfo, prop);
		break;

	case MOUSE_MOVE:
	case MOUSE_FREEMOVE:
		vpt = ip->GetViewport(hwnd);
		vpt->SnapPreview (m, m, NULL, SNAP_FORCE_3D_RESULT);//|SNAP_SEL_OBJS_ONLY);
		if (HitTestEdges(m,vpt,NULL,NULL)) SetCursor(ip->GetSysCursor(SYSCUR_SELECT));
		else SetCursor(LoadCursor(NULL,IDC_ARROW));
		if (vpt) ip->ReleaseViewport(vpt);
		break;
	}

	return TRUE;
}

// --------------------------------------------------------

HitRecord *PickFaceMouseProc::HitTestFaces (IPoint2 &m, ViewExp *vpt) {
	vpt->ClearSubObjHitList();
	ip->SubObHitTest(ip->GetTime(),HITTYPE_POINT,0, 0, &m, vpt);
	if (!vpt->NumSubObjHits()) return NULL;
	HitLog& hitLog = vpt->GetSubObjHitList();
	HitRecord *hr = hitLog.ClosestHit();
	return hr;
}

void PickFaceMouseProc::ProjectHitToFace (IPoint2 &m, ViewExp *vpt,
										  HitRecord *hr, Point3 *snapPoint) {
	if (!hr) return ;

	// Find subtriangle, barycentric coordinates of hit in face.
	face = hr->hitInfo;
	Matrix3 obj2world = hr->nodeRef->GetObjectTM (ip->GetTime ());

	Ray r;
	vpt->MapScreenToWorldRay ((float)m.x, (float)m.y, r);
	if (!snapPoint) snapPoint = &(r.p);
	Point3 Zdir = Normalize (r.dir);

	// Find an approximate location for the point on the surface we hit:
	// Get the average normal for the face, thus the plane, and intersect.
	Point3 intersect;
	MNMesh & mm = *(ep->GetMeshPtr());
	Point3 planeNormal = mm.GetFaceNormal (face, TRUE);
	planeNormal = Normalize (obj2world.VectorTransform (planeNormal));
	float planeOffset=0.0f;
	for (int i=0; i<mm.f[face].deg; i++)
		planeOffset += DotProd (planeNormal, obj2world*mm.v[mm.f[face].vtx[i]].p);
	planeOffset = planeOffset/float(mm.f[face].deg);

	// Now we intersect the snapPoint + r.dir*t line with the
	// DotProd (planeNormal, X) = planeOffset plane.
	float rayPlane = DotProd (r.dir, planeNormal);
	float firstPointOffset = planeOffset - DotProd (planeNormal, *snapPoint);
	if (fabsf(rayPlane) > EPSILON) {
		float amount = firstPointOffset / rayPlane;
		intersect = *snapPoint + amount*r.dir;
	} else {
		intersect = *snapPoint;
	}

	Matrix3 world2obj = Inverse (obj2world);
	intersect = world2obj * intersect;

	mm.FacePointBary (face, intersect, bary);
}

int PickFaceMouseProc::proc (HWND hwnd, int msg, int point, int flags, IPoint2 m) {
	ViewExp *vpt;
	HitRecord *hr;
	Point3 snapPoint;

	switch (msg) {
	case MOUSE_PROPCLICK:
		ip->SetStdCommandMode(CID_OBJMOVE);
		break;

	case MOUSE_POINT:
		ip->SetActiveViewport(hwnd);
		vpt = ip->GetViewport(hwnd);
		snapPoint = vpt->SnapPoint (m, m, NULL);
		snapPoint = vpt->MapCPToWorld (snapPoint);
		hr = HitTestFaces (m, vpt);
		ProjectHitToFace (m, vpt, hr, &snapPoint);
		if (vpt) ip->ReleaseViewport(vpt);
		if (hr) FacePick ();
		break;

	case MOUSE_MOVE:
	case MOUSE_FREEMOVE:
		vpt = ip->GetViewport(hwnd);
		vpt->SnapPreview (m, m, NULL, SNAP_FORCE_3D_RESULT);
		if (HitTestFaces(m,vpt)) SetCursor(ip->GetSysCursor(SYSCUR_SELECT));
		else SetCursor(LoadCursor(NULL,IDC_ARROW));
		if (vpt) ip->ReleaseViewport(vpt);
		break;
	}

	return TRUE;
}

// -------------------------------------------------------

HitRecord *ConnectVertsMouseProc::HitTestVertices (IPoint2 & m, ViewExp *vpt) {
	vpt->ClearSubObjHitList();
	ip->SubObHitTest(ip->GetTime(),HITTYPE_POINT,0, 0, &m, vpt);
	if (!vpt->NumSubObjHits()) return NULL;
	HitLog& hitLog = vpt->GetSubObjHitList();
	HitRecord *hr = hitLog.ClosestHit();
	if (v1<0) {
		// can't accept vertices on no faces.
		if (ep->GetMeshPtr()->vfac[hr->hitInfo].Count() == 0) return NULL;
		return hr;
	}

	// Otherwise, we're looking for a vertex on v1's faces - these are listed in neighbors.
	for (int i=0; i<neighbors.Count(); i++) if (neighbors[i] == hr->hitInfo) break;
	if (i>=neighbors.Count()) return NULL;
	return hr;
}

void ConnectVertsMouseProc::DrawDiag (HWND hWnd, const IPoint2 & m) {
	if (v1<0) return;

	HDC hdc = GetDC(hWnd);
	SetROP2(hdc, R2_XORPEN);
	SetBkMode(hdc, TRANSPARENT);
	SelectObject(hdc,CreatePen(PS_DOT, 0, ComputeViewportXORDrawColor()));

	MoveToEx (hdc, m1.x, m1.y, NULL);
	LineTo (hdc, m.x, m.y);

	DeleteObject(SelectObject(hdc,GetStockObject(BLACK_PEN)));
	ReleaseDC(hWnd, hdc);
}

void ConnectVertsMouseProc::SetV1 (int vv) {
	v1 = vv;
	neighbors.ZeroCount();
	Tab<int> & vf = ep->GetMeshPtr()->vfac[vv];
	Tab<int> & ve = ep->GetMeshPtr()->vedg[vv];
	// Add to neighbors all the vertices that share faces (but no edges) with this one:
	int i,j,k;
	for (i=0; i<vf.Count(); i++) {
		MNFace & mf = ep->GetMeshPtr()->f[vf[i]];
		for (j=0; j<mf.deg; j++) {
			// Do not consider v1 a neighbor:
			if (mf.vtx[j] == v1) continue;

			// Filter out those that share an edge with v1:
			for (k=0; k<ve.Count(); k++) {
				if (ep->GetMeshPtr()->e[ve[k]].OtherVert (vv) == mf.vtx[j]) break;
			}
			if (k<ve.Count()) continue;

			// Add to neighbor list.
			neighbors.Append (1, &(mf.vtx[j]), 4);
		}
	}
}

int ConnectVertsMouseProc::proc (HWND hwnd, int msg, int point, int flags, IPoint2 m) {
	ViewExp *vpt;
	HitRecord *hr;
	Point3 snapPoint;

	switch (msg) {
	case MOUSE_INIT:
		v1 = v2 = -1;
		neighbors.ZeroCount ();
		break;

	case MOUSE_PROPCLICK:
		ip->SetStdCommandMode(CID_OBJMOVE);
		break;

	case MOUSE_POINT:
		ip->SetActiveViewport(hwnd);
		vpt = ip->GetViewport(hwnd);
		snapPoint = vpt->SnapPoint (m, m, NULL);
		snapPoint = vpt->MapCPToWorld (snapPoint);
		hr = HitTestVertices (m, vpt);
		ip->ReleaseViewport(vpt);
		if (!hr) break;
		if (v1<0) {
			SetV1 (hr->hitInfo);
			m1 = m;
			lastm = m;
			DrawDiag (hwnd, m);
			break;
		}
		// Otherwise, we've found a connection.
		DrawDiag (hwnd, lastm);	// erase last dotted line.
		v2 = hr->hitInfo;
		VertConnect ();
		v1 = -1;
		return FALSE;	// Done with this connection.

	case MOUSE_MOVE:
	case MOUSE_FREEMOVE:
		vpt = ip->GetViewport(hwnd);
		vpt->SnapPreview (m, m, NULL, SNAP_FORCE_3D_RESULT);
		if (hr=HitTestVertices(m,vpt)) SetCursor(ip->GetSysCursor(SYSCUR_SELECT));
		else SetCursor(LoadCursor(NULL,IDC_ARROW));
		if (vpt) ip->ReleaseViewport(vpt);
		DrawDiag (hwnd, lastm);
		DrawDiag (hwnd, m);
		lastm = m;
		break;
	}

	return TRUE;
}

// -------------------------------------------------------

static HCURSOR hCurCreateVert = NULL;

void CreateVertCMode::EnterMode() {
	HWND hGeom = ep->GetDlgHandle (ep_geom);
	if (!hGeom) return;
	ICustButton *but = GetICustButton (GetDlgItem (hGeom,IDC_CREATE));
	but->SetCheck(TRUE);
	ReleaseICustButton(but);
}

void CreateVertCMode::ExitMode() {
	HWND hGeom = ep->GetDlgHandle (ep_geom);
	if (!hGeom) return;
	ICustButton *but = GetICustButton (GetDlgItem (hGeom,IDC_CREATE));
	but->SetCheck(FALSE);
	ReleaseICustButton(but);
}

int CreateVertMouseProc::proc (HWND hwnd, int msg, int point, int flags, IPoint2 m) {
	if (!hCurCreateVert) hCurCreateVert = LoadCursor(hInstance,MAKEINTRESOURCE(IDC_ADDVERTCUR)); 

	ViewExp *vpt = ip->GetViewport (hwnd);
	Matrix3 ctm;
	Point3 pt;
	IPoint2 m2;

	switch (msg) {
	case MOUSE_PROPCLICK:
		ip->SetStdCommandMode(CID_OBJMOVE);
		break;

	case MOUSE_POINT:
		ip->SetActiveViewport(hwnd);
		vpt->GetConstructionTM(ctm);
		pt = vpt->SnapPoint (m, m2, &ctm);
		pt = pt * ctm;
		theHold.Begin ();
		if (ep->EpfnCreateVertex(pt)<0) {
			theHold.Cancel ();
		} else {
			theHold.Accept (GetString (IDS_CREATE_VERTEX));
			ep->RefreshScreen ();
		}
		break;

	case MOUSE_FREEMOVE:
		SetCursor(hCurCreateVert);
		vpt->SnapPreview(m, m, NULL, SNAP_FORCE_3D_RESULT);
		break;
	}

	if (vpt) ip->ReleaseViewport(vpt);
	return TRUE;
}

//----------------------------------------------------------

void CreateEdgeCMode::EnterMode () {
	HWND hGeom = ep->GetDlgHandle (ep_geom);
	if (hGeom) {
		ICustButton *but = GetICustButton (GetDlgItem (hGeom,IDC_CREATE));
		but->SetCheck(TRUE);
		ReleaseICustButton(but);
	}
	if (!ep->ip->GetShowEndResult()) ep->mm.dispFlags |= MNDISP_VERTTICKS;
	ep->NotifyDependents (FOREVER, PART_DISPLAY, REFMSG_CHANGE);
	ep->ip->RedrawViews(ep->ip->GetTime());
}

void CreateEdgeCMode::ExitMode() {
	HWND hGeom = ep->GetDlgHandle (ep_geom);
	if (hGeom) {
		ICustButton *but = GetICustButton (GetDlgItem (hGeom,IDC_CREATE));
		but->SetCheck (FALSE);
		ReleaseICustButton(but);
	}
	if (!ep->ip->GetShowEndResult()) ep->mm.dispFlags &= ~(MNDISP_VERTTICKS);
	ep->NotifyDependents (FOREVER, PART_DISPLAY, REFMSG_CHANGE);
	ep->ip->RedrawViews(ep->ip->GetTime());
}

void CreateEdgeMouseProc::VertConnect () {
	theHold.Begin();
	if (ep->EpfnCreateEdge (v1, v2) < 0) {
		theHold.Cancel ();
		return;
	}
	theHold.Accept (GetString (IDS_CREATE_EDGE));
	ep->RefreshScreen ();
}

//----------------------------------------------------------

void CreateFaceCMode::EnterMode() {
	HWND hGeom = ep->GetDlgHandle (ep_geom);
	if (hGeom) {
		ICustButton *but = GetICustButton (GetDlgItem (hGeom,IDC_CREATE));
		but->SetCheck(TRUE);
		ReleaseICustButton(but);
	}
	if (!ep->ip->GetShowEndResult()) ep->mm.dispFlags |= (MNDISP_VERTTICKS);
	ep->NotifyDependents (FOREVER, PART_DISPLAY, REFMSG_CHANGE);
	ep->ip->RedrawViews(ep->ip->GetTime());
}

void CreateFaceCMode::ExitMode() {
	HWND hGeom = ep->GetDlgHandle (ep_geom);
	if (hGeom) {
		ICustButton *but = GetICustButton (GetDlgItem (hGeom,IDC_CREATE));
		but->SetCheck (FALSE);
		ReleaseICustButton(but);
	}
	if (!ep->ip->GetShowEndResult()) {
		ep->mm.dispFlags &= ~(MNDISP_VERTTICKS|MNDISP_SELVERTS);
	}
	ep->NotifyDependents (FOREVER, PART_DISPLAY, REFMSG_CHANGE);
	ep->ip->RedrawViews(ep->ip->GetTime());
}

CreateFaceMouseProc::CreateFaceMouseProc (EditPolyObject* e, IObjParam *i) {
	ep = e;
	ip = i;	
	pt = 0;
}

// We assume the transform, color, render style, etc, has been set up in advance.
void CreateFaceMouseProc::DrawEstablishedFace (GraphicsWindow *gw) {
	if (pt<2) return;
	Tab<Point3> rverts;
	rverts.SetCount (pt+1);
	for (int j=0; j<pt; j++) {
		rverts[j] = ep->GetMeshPtr()->v[vts[j]].p;
	}
	gw->polyline (pt, rverts.Addr(0), NULL, NULL, FALSE, NULL);
}

void CreateFaceMouseProc::DrawCreatingFace (HWND hWnd, const IPoint2 & m) {
	if (pt<1) return;

	HDC hdc = GetDC(hWnd);
	SetROP2(hdc, R2_XORPEN);
	SetBkMode(hdc, TRANSPARENT);
	SelectObject(hdc,CreatePen(PS_DOT, 0, ComputeViewportXORDrawColor()));

	MoveToEx (hdc, mpts[0].x, mpts[0].y, NULL);
	LineTo (hdc, m.x, m.y);
	if (pt>1) LineTo (hdc, mpts[pt-1].x, mpts[pt-1].y);

	DeleteObject(SelectObject(hdc,GetStockObject(BLACK_PEN)));
	ReleaseDC(hWnd, hdc);
}

BOOL CreateFaceMouseProc::HitTestVerts (IPoint2 m, ViewExp *vpt, int &v) {
	vpt->ClearSubObjHitList();
	ip->SubObHitTest(ip->GetTime(),HITTYPE_POINT,0, 0, &m, vpt);
	if (!vpt->NumSubObjHits()) return FALSE;
	HitLog& hitLog = vpt->GetSubObjHitList();
	HitRecord *hr = hitLog.ClosestHit();
	assert(hr);
	/*
	if (ep->selLevel != SL_POLY) {
		for (int i=0; i<pt; i++) if (vts[i]==(int)hr->hitInfo) return FALSE;
	}
	*/
	v = hr->hitInfo;
	return TRUE;
}

int CreateFaceMouseProc::proc (HWND hwnd, int msg, int point, int flags, IPoint2 m) {
	if (!hCurCreateVert) hCurCreateVert = LoadCursor(hInstance,MAKEINTRESOURCE(IDC_ADDVERTCUR));

	ViewExp *vpt = ip->GetViewport(hwnd);
	int dummyVert;
	int nv, lpt;

	switch (msg) {
	case MOUSE_PROPCLICK:
		ip->SetStdCommandMode(CID_OBJMOVE);
		vts.ZeroCount ();
		mpts.ZeroCount ();
		break;

	case MOUSE_POINT:
		if (point==1) break;
		ip->SetActiveViewport(hwnd);
		bool done;
		done=FALSE;

		if (HitTestVerts(m, vpt, nv)) {
			HitLog& hitLog = vpt->GetSubObjHitList();
			HitRecord *hr = hitLog.ClosestHit();
			MaxAssert (hr);
			for (int j=0; j<pt; j++) if (vts[j] == nv) break;
			if (j<pt) done=TRUE;
			else {
				vts.Append (1, &nv, 20);
				mpts.Append (1, &m, 20);
				pt++;
				ep->LocalDataChanged (PART_DISPLAY);
				ip->RedrawViews (ip->GetTime());
				oldm = m;
				DrawCreatingFace(hwnd, m);
			}
		} else {
			Matrix3 ctm;
			vpt->GetConstructionTM(ctm);
			Point3 newpt = vpt->SnapPoint(m,m,&ctm, SNAP_FORCE_3D_RESULT|SNAP_SEL_SUBOBJ_ONLY);
			newpt = newpt * ctm;
			nv = ep->GetMeshPtr()->numv;

			vts.Append (1, &nv, 20);
			mpts.Append (1, &m, 20);
			pt++;
			theHold.Begin ();
			if (ep->EpfnCreateVertex(newpt) < 0) {
				theHold.Cancel ();
			} else {
				theHold.Accept (GetString (IDS_CREATE_VERTEX));
				ip->RedrawViews (ip->GetTime());
			}
			oldm = m;
			DrawCreatingFace(hwnd, m);
		}

		if (done) {
			// We're done collecting verts - build a face
			lpt = pt;
			pt = 0;	// so the redraw gets that we're done.
			theHold.Begin ();
			if ((lpt>2) && (ep->EpfnCreateFace(vts.Addr(0), lpt) < 0)) {
				theHold.Cancel ();
				InvalidateRect(vpt->getGW()->getHWnd(),NULL,FALSE);
				TSTR buf1 = GetString(IDS_ILLEGAL_NEW_FACE);
				TSTR buf2 = GetString(IDS_SCA_E_POLY);
				MessageBox(ip->GetMAXHWnd(),buf1,buf2,MB_OK|MB_ICONINFORMATION);						
				ep->LocalDataChanged (PART_DISPLAY);
			} else {
				theHold.Accept (GetString (IDS_CREATE_FACE));
			}
			ip->RedrawViews (ip->GetTime());
			vts.ZeroCount();
			mpts.ZeroCount();
			ip->ReleaseViewport (vpt);
			return FALSE;
		}
		break;

	case MOUSE_MOVE:
		if (pt) DrawCreatingFace (hwnd, oldm);	// Erase old outline
		if (HitTestVerts(m,vpt,dummyVert)) {
			SetCursor(ip->GetSysCursor(SYSCUR_SELECT));
		} else {
			SetCursor(hCurCreateVert);
			vpt->SnapPreview (m, m, NULL, SNAP_FORCE_3D_RESULT|SNAP_SEL_SUBOBJ_ONLY);
			ip->RedrawViews (ip->GetTime());
		}
		if (pt) {
			oldm = m;
			DrawCreatingFace (hwnd, oldm);
		}
		break;

	case MOUSE_FREEMOVE:
		if (HitTestVerts(m,vpt,dummyVert))
			SetCursor(ip->GetSysCursor(SYSCUR_SELECT));
		else {
			SetCursor(hCurCreateVert);
			vpt->SnapPreview (m, m, NULL, SNAP_SEL_SUBOBJ_ONLY|SNAP_FORCE_3D_RESULT);
		}
		break;

	case MOUSE_ABORT:
		pt = 0;
		vts.ZeroCount ();
		mpts.ZeroCount ();
		ep->LocalDataChanged (PART_DISPLAY);
		ip->RedrawViews (ip->GetTime());
		break;
	}

	ip->ReleaseViewport(vpt);
	return TRUE;
}

void CreateFaceMouseProc::Backspace () {
	pt--;
	mpts.Delete (pt-1, 1);
	vts.Delete (pt-1, 1);
}

//-----------------------------------------------------------------------/

BOOL AttachPickMode::Filter(INode *node) {
	if (!node) return FALSE;

	// Make sure the node does not depend on us
	node->BeginDependencyTest();
	ep->NotifyDependents(FOREVER,0,REFMSG_TEST_DEPENDENCY);
	if (node->EndDependencyTest()) return FALSE;

	ObjectState os = node->GetObjectRef()->Eval(ip->GetTime());
	if (os.obj->IsSubClassOf(polyObjectClassID)) return TRUE;
	if (os.obj->CanConvertToType(polyObjectClassID)) return TRUE;
	return FALSE;
}

BOOL AttachPickMode::HitTest(IObjParam *ip, HWND hWnd, ViewExp *vpt, IPoint2 m,int flags) {
	return ip->PickNode(hWnd,m,this) ? TRUE : FALSE;
}

BOOL AttachPickMode::Pick(IObjParam *ip,ViewExp *vpt) {
	INode *node = vpt->GetClosestHit();
	if (!Filter(node)) return FALSE;

	ModContextList mcList;
	INodeTab nodes;
	ip->GetModContexts(mcList,nodes);

	BOOL ret = TRUE;
	if (nodes[0]->GetMtl() && node->GetMtl() && (nodes[0]->GetMtl()!=node->GetMtl())) {
		ret = DoAttachMatOptionDialog (ip, ep);
	}
	if (!ret) {
		nodes.DisposeTemporary ();
		return FALSE;
	}

	bool canUndo = TRUE;
	ep->EpfnAttach (node, canUndo, nodes[0], ip->GetTime());
	if (!canUndo) GetSystemSetting (SYSSET_CLEAR_UNDO);
	ep->RefreshScreen ();
	nodes.DisposeTemporary ();
	return FALSE;
}

void AttachPickMode::EnterMode(IObjParam *ip) {
	HWND hGeom = ep->GetDlgHandle (ep_geom);
	if (!hGeom) return;
	ICustButton *but = GetICustButton (GetDlgItem (hGeom, IDC_ATTACH2));
	but->SetCheck(TRUE);
	ReleaseICustButton(but);
}

void AttachPickMode::ExitMode(IObjParam *ip) {
	HWND hGeom = ep->GetDlgHandle (ep_geom);
	if (!hGeom) return;
	ICustButton *but = GetICustButton (GetDlgItem (hGeom, IDC_ATTACH2));
	but->SetCheck(FALSE);
	ReleaseICustButton(but);
}

// -----------------------------

int AttachHitByName::filter(INode *node) {
	if (!node) return FALSE;

	// Make sure the node does not depend on this modifier.
	node->BeginDependencyTest();
	ep->NotifyDependents(FOREVER,0,REFMSG_TEST_DEPENDENCY);
	if (node->EndDependencyTest()) return FALSE;

	ObjectState os = node->GetObjectRef()->Eval(ep->ip->GetTime());
	if (os.obj->IsSubClassOf(polyObjectClassID)) return TRUE;
	if (os.obj->CanConvertToType(polyObjectClassID)) return TRUE;
	return FALSE;
}

void AttachHitByName::proc(INodeTab &nodeTab) {
	if (inProc) return;
	if (!ep->ip) return;
	inProc = TRUE;
	ModContextList mcList;
	INodeTab nodes;
	ep->ip->GetModContexts (mcList, nodes);
	BOOL ret = TRUE;
	if (nodes[0]->GetMtl()) {
		for (int i=0; i<nodeTab.Count(); i++) {
			if (nodeTab[i]->GetMtl() && (nodes[0]->GetMtl()!=nodeTab[i]->GetMtl())) break;
		}
		if (i<nodeTab.Count()) ret = DoAttachMatOptionDialog ((IObjParam *)ep->ip, ep);
		if (!ep->ip) ret = FALSE;
	}
	inProc = FALSE;
	if (ret) ep->EpfnMultiAttach (nodeTab, nodes[0], ep->ip->GetTime ());
	nodes.DisposeTemporary ();
}

//----------------------------------------------------------

// Divide edge modifies two faces; creates a new vertex and a new edge.
void DivideEdgeProc::EdgePick (int edge, float prop) {
	theHold.Begin ();
	ep->EpfnDivideEdge (edge, prop);
	theHold.Accept (GetString (IDS_DIVIDE_EDGE));
	ep->RefreshScreen ();
}

void DivideEdgeCMode::EnterMode() {
	HWND hGeom = ep->GetDlgHandle (ep_geom);
	if (hGeom) {
		ICustButton *but = GetICustButton (GetDlgItem (hGeom,IDC_DIVIDE));
		but->SetCheck(TRUE);
		ReleaseICustButton(but);
	}
	if (!ep->ip->GetShowEndResult()) ep->mm.dispFlags |= (MNDISP_VERTTICKS);
}

void DivideEdgeCMode::ExitMode() {
	HWND hGeom = ep->GetDlgHandle (ep_geom);
	if (hGeom) {
		ICustButton *but = GetICustButton (GetDlgItem (hGeom,IDC_DIVIDE));
		but->SetCheck(FALSE);
		ReleaseICustButton(but);
	}
	if (!ep->ip->GetShowEndResult()) ep->mm.dispFlags &= (~MNDISP_VERTTICKS);
}

//----------------------------------------------------------

void DivideFaceProc::FacePick () {
	theHold.Begin ();
	ep->EpfnDivideFace (face, bary);
	theHold.Accept (GetString (IDS_FACE_DIVIDE));
	ep->RefreshScreen ();
}

void DivideFaceCMode::EnterMode() {
	HWND hGeom = ep->GetDlgHandle (ep_geom);
	if (!hGeom) return;
	ICustButton *but = GetICustButton (GetDlgItem (hGeom,IDC_DIVIDE));
	but->SetCheck(TRUE);
	ReleaseICustButton(but);		
}

void DivideFaceCMode::ExitMode() {
	HWND hGeom = ep->GetDlgHandle (ep_geom);
	if (!hGeom) return;
	ICustButton *but = GetICustButton (GetDlgItem (hGeom,IDC_DIVIDE));
	but->SetCheck(FALSE);
	ReleaseICustButton(but);
}

// ------------------------------------------------------

int ExtrudeProc::proc (HWND hwnd, int msg, int point, int flags, IPoint2 m ) {	
	ViewExp *vpt=ip->GetViewport (hwnd);
	Point3 p0, p1;
	float amount;
	IPoint2 m2;
	ISpinnerControl *spin;
	HWND hWnd;

	switch (msg) {
	case MOUSE_PROPCLICK:
		ip->SetStdCommandMode(CID_OBJMOVE);
		break;

	case MOUSE_POINT:
		if (!point) {
			eo->EpfnBeginExtrude(meshSelLevel[eo->selLevel], MN_SEL, ip->GetTime());
			om = m;
		} else {
			ip->RedrawViews(ip->GetTime(),REDRAW_END);
			eo->EpfnEndExtrude (true, ip->GetTime());
		}
		break;

	case MOUSE_MOVE:
		p0 = vpt->MapScreenToView (om,float(-200));
		m2.x = om.x;
		m2.y = m.y;
		p1 = vpt->MapScreenToView (m2,float(-200));
		amount = Length (p1-p0);
		if (m.y > om.y) amount *= -1.0f;
		eo->EpfnDragExtrude (amount, ip->GetTime());

		hWnd = eo->GetDlgHandle (ep_geom);
		if (hWnd) {
			spin = GetISpinner (GetDlgItem (hWnd,IDC_EXTRUDESPINNER));
			if (spin) {
				spin->SetValue (amount, FALSE);
				ReleaseISpinner(spin);
			}
		}

		ip->RedrawViews(ip->GetTime(),REDRAW_INTERACTIVE);
		break;

	case MOUSE_ABORT:
		eo->EpfnEndExtrude (false, ip->GetTime ());
		ip->RedrawViews (ip->GetTime(), REDRAW_END);
		break;
	}

	if (vpt) ip->ReleaseViewport(vpt);
	return TRUE;
}

HCURSOR ExtrudeSelectionProcessor::GetTransformCursor() { 
	static HCURSOR hCur = NULL;
	if ( !hCur ) hCur = LoadCursor(hInstance,MAKEINTRESOURCE(IDC_EXTRUDECUR));
	return hCur; 
}

void ExtrudeCMode::EnterMode() {
	HWND hGeom = eo->GetDlgHandle (ep_geom);
	if (!hGeom) return;
	ICustButton *but = GetICustButton (GetDlgItem (hGeom,IDC_EXTRUDE));
	but->SetCheck(TRUE);
	ReleaseICustButton(but);
}

void ExtrudeCMode::ExitMode() {
	HWND hGeom = eo->GetDlgHandle (ep_geom);
	if (!hGeom) return;
	ICustButton *but = GetICustButton (GetDlgItem (hGeom,IDC_EXTRUDE));
	but->SetCheck(FALSE);
	ReleaseICustButton(but);
}

// ------------------------------------------------------

int BevelMouseProc::proc (HWND hwnd, int msg, int point, int flags, IPoint2 m ) {	
	ViewExp *vpt=ip->GetViewport (hwnd);
	Point3 p0, p1;
	float amount;
	IPoint2 m2;
	HWND hWnd;
	ISpinnerControl *spin=NULL;

	switch (msg) {
	case MOUSE_PROPCLICK:
		ip->SetStdCommandMode(CID_OBJMOVE);
		break;

	case MOUSE_POINT:
		switch (point) {
		case 0:
			m0 = m;
			m0set = TRUE;
			eo->EpfnBeginBevel (meshSelLevel[eo->selLevel], MN_SEL, true, ip->GetTime());
			break;
		case 1:
			m1 = m;
			m1set = TRUE;
			p0 = vpt->MapScreenToView (m0, float(-200));
			m2.x = m0.x;
			m2.y = m.y;
			p1 = vpt->MapScreenToView (m2, float(-200));
			height = Length (p0-p1);
			if (m1.y > m0.y) height *= -1.0f;
			eo->EpfnDragBevel (0.0f, height, ip->GetTime());
			break;
		case 2:
			ip->RedrawViews(ip->GetTime(),REDRAW_END);
			eo->EpfnEndBevel (true, ip->GetTime());
			m1set = m0set = FALSE;
			break;
		}
		break;

	case MOUSE_MOVE:
		if (!m0set) break;
		m2.y = m.y;
		hWnd = eo->GetDlgHandle (ep_geom);
		if (!m1set) {
			p0 = vpt->MapScreenToView (m0, float(-200));
			m2.x = m0.x;
			p1 = vpt->MapScreenToView (m2, float(-200));
			amount = Length (p1-p0);
			if (m.y > m0.y) amount *= -1.0f;
			eo->EpfnDragBevel (0.0f, amount, ip->GetTime());
			if (hWnd) spin = GetISpinner(GetDlgItem(hWnd,IDC_EXTRUDESPINNER));
		} else {
			p0 = vpt->MapScreenToView (m1, float(-200));
			m2.x = m1.x;
			p1 = vpt->MapScreenToView (m2, float(-200));
			amount = Length (p1-p0);
			if (m.y > m1.y) amount *= -1.0f;
			eo->EpfnDragBevel (amount, height, ip->GetTime());
			if (hWnd) spin = GetISpinner(GetDlgItem(hWnd,IDC_OUTLINESPINNER));
		}
		if (spin) {
			spin->SetValue(amount,FALSE);
			ReleaseISpinner(spin);
		}

		ip->RedrawViews(ip->GetTime(),REDRAW_INTERACTIVE);
		break;

	case MOUSE_ABORT:
		eo->EpfnEndBevel (false, ip->GetTime());
		ip->RedrawViews(ip->GetTime(),REDRAW_END);
		m1set = m0set = FALSE;
		break;
	}

	if (vpt) ip->ReleaseViewport(vpt);
	return TRUE;
}

HCURSOR BevelSelectionProcessor::GetTransformCursor() { 
	static HCURSOR hCur = NULL;
	if ( !hCur ) hCur = LoadCursor(hInstance,MAKEINTRESOURCE(IDC_BEVELCUR));
	return hCur;
}

void BevelCMode::EnterMode() {
	HWND hGeom = eo->GetDlgHandle (ep_geom);
	if (!hGeom) return;
	ICustButton *but = GetICustButton (GetDlgItem (hGeom,IDC_BEVEL));
	but->SetCheck(TRUE);
	ReleaseICustButton(but);
}

void BevelCMode::ExitMode() {
	HWND hGeom = eo->GetDlgHandle (ep_geom);
	if (!hGeom) return;
	ICustButton *but = GetICustButton (GetDlgItem (hGeom,IDC_BEVEL));
	but->SetCheck(FALSE);
	ReleaseICustButton(but);
}

// ------------------------------------------------------

int ChamferMouseProc::proc (HWND hwnd, int msg, int point, int flags, IPoint2 m ) {	
	ViewExp *vpt=ip->GetViewport (hwnd);
	Point3 p0, p1;
	ISpinnerControl *spin;

	switch (msg) {
	case MOUSE_PROPCLICK:
		ip->SetStdCommandMode(CID_OBJMOVE);
		break;

	case MOUSE_POINT:
		if (!point) {
			eo->EpfnBeginChamfer (eo->GetMNSelLevel(), ip->GetTime());
			om = m;
		} else {
			ip->RedrawViews (ip->GetTime(),REDRAW_END);
			eo->EpfnEndChamfer (true, ip->GetTime());
		}
		break;

	case MOUSE_MOVE:
		p0 = vpt->MapScreenToView(om,float(-200));
		p1 = vpt->MapScreenToView(m,float(-200));
		eo->EpfnDragChamfer (Length(p1-p0), ip->GetTime());

		spin = GetISpinner (GetDlgItem (eo->GetDlgHandle (ep_geom),
			IDC_OUTLINESPINNER));
		if (spin) {
			spin->SetValue(Length(p1-p0),FALSE);
			ReleaseISpinner(spin);
		}

		ip->RedrawViews(ip->GetTime(),REDRAW_INTERACTIVE);
		break;

	case MOUSE_ABORT:
		eo->EpfnEndChamfer (false, ip->GetTime());			
		ip->RedrawViews (ip->GetTime(),REDRAW_END);
		break;
	}

	if (vpt) ip->ReleaseViewport(vpt);
	return TRUE;
}

HCURSOR ChamferSelectionProcessor::GetTransformCursor() { 
	static HCURSOR hECur = NULL;
	static HCURSOR hVCur = NULL;
	if (eto->selLevel == EP_SL_VERTEX) {
		if ( !hVCur ) hVCur = LoadCursor(hInstance,MAKEINTRESOURCE(IDC_VCHAMFERCUR));
		return hVCur;
	}
	if ( !hECur ) hECur = LoadCursor(hInstance,MAKEINTRESOURCE(IDC_ECHAMFERCUR));
	return hECur;
}

void ChamferCMode::EnterMode() {
	HWND hGeom = eo->GetDlgHandle (ep_geom);
	if (!hGeom) return;
	ICustButton *but = GetICustButton(GetDlgItem(hGeom,IDC_BEVEL));
	but->SetCheck(TRUE);
	ReleaseICustButton(but);
}

void ChamferCMode::ExitMode() {
	HWND hGeom = eo->GetDlgHandle (ep_geom);
	if (!hGeom) return;
	ICustButton *but = GetICustButton(GetDlgItem(hGeom,IDC_BEVEL));
	but->SetCheck(FALSE);
	ReleaseICustButton(but);
}

// --- Slice: not really a command mode, just looks like it.--------- //

void EditPolyObject::EnterSliceMode () {
	sliceMode = TRUE;
	HWND hGeom = GetDlgHandle (ep_geom);
	if (hGeom) {
		ICustButton *but = GetICustButton (GetDlgItem(hGeom,IDC_SLICE));
		but->Enable ();
		ReleaseICustButton (but);
		but = GetICustButton (GetDlgItem (hGeom, IDC_RESET_PLANE));
		but->Enable ();
		ReleaseICustButton (but);
		but = GetICustButton (GetDlgItem (hGeom, IDC_SLICEPLANE));
		but->SetCheck (TRUE);
		ReleaseICustButton (but);
	}

	if (!sliceInitialized) {
		EpResetSlicePlane ();
		sliceInitialized = true;
	}

	if (ip->GetCommandMode()->ID() >= CID_USER) ip->SetStdCommandMode (CID_OBJMOVE);
	NotifyDependents(FOREVER, PART_DISPLAY, REFMSG_CHANGE);
	RefreshScreen ();
}

void EditPolyObject::ExitSliceMode () {
	sliceMode = FALSE;
	HWND hGeom = GetDlgHandle (ep_geom);
	if (hGeom) {
		ICustButton *but = GetICustButton (GetDlgItem(hGeom,IDC_SLICE));
		but->Disable ();
		ReleaseICustButton (but);
		but = GetICustButton (GetDlgItem (hGeom, IDC_RESET_PLANE));
		but->Disable ();
		ReleaseICustButton (but);
		but = GetICustButton (GetDlgItem (hGeom, IDC_SLICEPLANE));
		but->SetCheck (FALSE);
		ReleaseICustButton (but);
	}
	NotifyDependents(FOREVER, PART_DISPLAY, REFMSG_CHANGE);
	RefreshScreen ();
}

// -- Cut Edge proc/mode -------------------------------------

HitRecord *CutVertProc::HitTestVerts (IPoint2 &m, ViewExp *vpt) {
	vpt->ClearSubObjHitList();
	ip->SubObHitTest (ip->GetTime(), HITTYPE_POINT, 0, 0, &m, vpt);
	if (!vpt->NumSubObjHits()) return NULL;
	HitLog& hitLog = vpt->GetSubObjHitList();
	HitRecord *ret = hitLog.ClosestHit();

	if (v1 == (int)ret->hitInfo) return NULL;
	return ret;
}

void CutVertProc::DrawCutter (HWND hWnd,IPoint2 &m) {
	HDC hdc = GetDC(hWnd);
	SetROP2(hdc, R2_XORPEN);
	SetBkMode(hdc, TRANSPARENT);	
	SelectObject(hdc,CreatePen(PS_DOT, 0, ComputeViewportXORDrawColor()));
	MoveToEx(hdc,m1.x,m1.y,NULL);
	LineTo(hdc,m.x,m.y);
	DeleteObject(SelectObject(hdc,GetStockObject(BLACK_PEN)));
	ReleaseDC(hWnd, hdc);
}

int CutVertProc::proc (HWND hwnd, int msg, int point, int flags, IPoint2 m ) {
	ViewExp *vpt;
	HitRecord *hr;
	IPoint2 betterM;
	Point3 A, B, snapPoint, Xdir, Zdir;
	Matrix3 obj2world, world2obj;
	Ray r;

	switch (msg) {
	case MOUSE_ABORT:
		// Erase last cutter line:
		if (v1>-1) DrawCutter (hwnd, oldm2);
		v1 = -1;
		return FALSE;

	case MOUSE_PROPCLICK:
		// Erase last cutter line:
		if (v1>-1) DrawCutter (hwnd, oldm2);
		ip->SetStdCommandMode(CID_OBJMOVE);
		return FALSE;

	case MOUSE_DBLCLICK:
		return false;

	case MOUSE_POINT:
		ip->SetActiveViewport (hwnd);
		vpt = ip->GetViewport (hwnd);
		hr = HitTestVerts(m,vpt);
		if (!hr) {
			ip->ReleaseViewport (vpt);
			return true;
		}
		int vv;
		vv = hr->hitInfo;

		// Find where along this edge we hit
		// Strategy:
		// Get Mouse click, plus viewport z-direction at mouse click, in object space.
		// Then find the direction of the edge in a plane orthogonal to z, and see how far
		// along that edge we are.
		snapPoint = vpt->SnapPoint (m, betterM, NULL);
		snapPoint = vpt->MapCPToWorld (snapPoint);
		vpt->MapScreenToWorldRay ((float)betterM.x, (float)betterM.y, r);
		Zdir = Normalize (r.dir);
		ip->ReleaseViewport (vpt);

		if (v1<0) {
			v1 = vv;
			m1 = betterM;
			DrawCutter (hwnd, m);
			oldm2=m;
			break;
		}

		// Erase last cutter line:
		DrawCutter (hwnd, oldm2);

		// Do the cut:
		obj2world = hr->nodeRef->GetObjectTM (ip->GetTime ());
		world2obj = Inverse (obj2world);
		Zdir = Normalize (VectorTransform (world2obj, Zdir));

		theHold.Begin ();
		v1 = ep->EpfnCutVertex (v1, ep->GetMeshPtr()->P(vv), -Zdir);
		theHold.Accept (GetString (IDS_CUT));
		ep->RefreshScreen ();

		if (v1<0) return FALSE;

		// Otherwise, start on next cut.
		m1 = betterM;
		DrawCutter (hwnd, m);
		oldm2 = m;
		return true;

	case MOUSE_MOVE:
		vpt = ip->GetViewport (hwnd);
		vpt->SnapPoint(m,m,NULL);
		if (v1>-1) {
			DrawCutter (hwnd,oldm2);
			oldm2 = m;
		}
		if (HitTestVerts(m,vpt)) SetCursor(ip->GetSysCursor(SYSCUR_SELECT));
		else SetCursor(LoadCursor(NULL,IDC_ARROW));
		ip->ReleaseViewport (vpt);
		ip->RedrawViews (ip->GetTime());
		if (v1>-1) DrawCutter (hwnd,oldm2);
		break;

	case MOUSE_FREEMOVE:
		vpt = ip->GetViewport (hwnd);
		vpt->SnapPreview(m,m,NULL);//, SNAP_SEL_OBJS_ONLY);
		if (HitTestVerts(m,vpt)) SetCursor(ip->GetSysCursor(SYSCUR_SELECT));
		else SetCursor(LoadCursor(NULL,IDC_ARROW));
		ip->ReleaseViewport (vpt);
		break;
	}

	return TRUE;
}

void CutVertCMode::EnterMode() {
	HWND hGeom = ep->GetDlgHandle (ep_geom);
	if (!hGeom) return;
	ICustButton *but = GetICustButton (GetDlgItem (hGeom,IDC_CUT));
	but->SetCheck(TRUE);
	ReleaseICustButton(but);
	proc.v1 = -1;
}

void CutVertCMode::ExitMode() {
	HWND hGeom = ep->GetDlgHandle (ep_geom);
	if (!hGeom) return;
	ICustButton *but = GetICustButton (GetDlgItem (hGeom,IDC_CUT));
	but->SetCheck(FALSE);
	ReleaseICustButton(but);
}

void CutVertCMode::AbandonCut () {
	proc.v1 = -1;
}

HitRecord *CutEdgeProc::HitTestEdges (IPoint2 &m, ViewExp *vpt) {
	vpt->ClearSubObjHitList();
	ip->SubObHitTest (ip->GetTime(), HITTYPE_POINT, 0, 0, &m, vpt);
	if (!vpt->NumSubObjHits()) return NULL;
	HitLog& hitLog = vpt->GetSubObjHitList();
	HitRecord *ret = hitLog.ClosestHit();

	if (e1set && (e1 == (int)ret->hitInfo)) return NULL;
	return ret;
}

void CutEdgeProc::DrawCutter (HWND hWnd,IPoint2 &m) {
	HDC hdc = GetDC(hWnd);
	SetROP2(hdc, R2_XORPEN);
	SetBkMode(hdc, TRANSPARENT);	
	SelectObject(hdc,CreatePen(PS_DOT, 0, ComputeViewportXORDrawColor()));
	MoveToEx(hdc,m1.x,m1.y,NULL);
	LineTo(hdc,m.x,m.y);
	DeleteObject(SelectObject(hdc,GetStockObject(BLACK_PEN)));
	ReleaseDC(hWnd, hdc);
}

int CutEdgeProc::proc (HWND hwnd, int msg, int point, int flags, IPoint2 m ) {
	ViewExp *vpt;
	HitRecord *hr;
	IPoint2 betterM;
	Point3 A, B, snapPoint, Xdir, Zdir;
	Matrix3 obj2world, world2obj;
	float prop;
	Ray r;

	switch (msg) {
	case MOUSE_ABORT:
		// Erase last cutter line:
		if (e1set) DrawCutter (hwnd, oldm2);
		e1set = FALSE;
		return FALSE;

	case MOUSE_PROPCLICK:
		// Erase last cutter line:
		if (e1set) DrawCutter (hwnd, oldm2);
		ip->SetStdCommandMode(CID_OBJMOVE);
		return FALSE;

	case MOUSE_DBLCLICK:
		if (!e1set) break;
		// Erase last cutter line:
		DrawCutter (hwnd, oldm2);

		theHold.Begin ();
		if (ep->EpfnDivideEdge (e1, prop1) > -1) {
			theHold.Accept (GetString (IDS_CUT));
			ep->RefreshScreen ();
		} else {
			theHold.Cancel ();
		}

		e1set = FALSE;
		return FALSE;

	case MOUSE_POINT:
		ip->SetActiveViewport (hwnd);
		vpt = ip->GetViewport (hwnd);
		hr = HitTestEdges(m,vpt);
		if (!hr) {
			ip->ReleaseViewport (vpt);
			return true;
		}
		int ee;
		ee = hr->hitInfo;

		// Find where along this edge we hit
		// Strategy:
		// Get Mouse click, plus viewport z-direction at mouse click, in object space.
		// Then find the direction of the edge in a plane orthogonal to z, and see how far
		// along that edge we are.
		snapPoint = vpt->SnapPoint (m, betterM, NULL);
		snapPoint = vpt->MapCPToWorld (snapPoint);
		vpt->MapScreenToWorldRay ((float)betterM.x, (float)betterM.y, r);
		Zdir = Normalize (r.dir);
		obj2world = hr->nodeRef->GetObjectTM (ip->GetTime ());
		A = obj2world * ep->GetMeshPtr()->v[ep->GetMeshPtr()->e[ee].v1].p;
		B = obj2world * ep->GetMeshPtr()->v[ep->GetMeshPtr()->e[ee].v2].p;
		Xdir = B-A;
		Xdir -= DotProd(Xdir, Zdir)*Zdir;
		prop = DotProd (Xdir, snapPoint-A) / LengthSquared (Xdir);
		if (prop<.0001f) prop=0;
		if (prop>.9999f) prop=1;

		ip->ReleaseViewport (vpt);

		if (!e1set) {
			e1 = hr->hitInfo;
			prop1 = prop;
			e1set = TRUE;
			m1 = betterM;
			DrawCutter (hwnd, m);
			oldm2=m;
			break;
		}

		// Erase last cutter line:
		DrawCutter (hwnd, oldm2);

		// Do the cut:
		world2obj = Inverse (obj2world);
		Zdir = Normalize (VectorTransform (world2obj, Zdir));

		int vret;
		theHold.Begin ();
		vret = ep->EpfnCutEdge (e1, prop1, ee, prop, -Zdir);
		theHold.Accept (GetString (IDS_CUT));
		ep->RefreshScreen ();

		if (vret==-1) {
			e1set = FALSE;
			return FALSE;
		}

		// (Turn our vert result into an edge result:)
		e1 = ep->GetMeshPtr()->vedg[vret][0];
		prop1 = (ep->GetMeshPtr()->e[e1].v1 == vret) ? 0.0f : 1.0f;
		m1 = betterM;
		DrawCutter (hwnd, m);
		oldm2 = m;
		break;

	case MOUSE_MOVE:
		vpt = ip->GetViewport (hwnd);
		vpt->SnapPoint(m,m,NULL);
		if (e1set) {
			DrawCutter (hwnd,oldm2);
			oldm2 = m;
		}
		if (HitTestEdges(m,vpt)) SetCursor(ip->GetSysCursor(SYSCUR_SELECT));
		else SetCursor(LoadCursor(NULL,IDC_ARROW));
		ip->ReleaseViewport (vpt);
		ip->RedrawViews (ip->GetTime());
		if (e1set) DrawCutter (hwnd,oldm2);
		break;

	case MOUSE_FREEMOVE:
		vpt = ip->GetViewport (hwnd);
		vpt->SnapPreview(m,m,NULL);//, SNAP_SEL_OBJS_ONLY);
		if (HitTestEdges(m,vpt)) SetCursor(ip->GetSysCursor(SYSCUR_SELECT));
		else SetCursor(LoadCursor(NULL,IDC_ARROW));
		ip->ReleaseViewport (vpt);
		break;
	}

	return TRUE;	
}

void CutEdgeCMode::EnterMode() {
	HWND hGeom = ep->GetDlgHandle (ep_geom);
	if (!hGeom) return;
	ICustButton *but = GetICustButton (GetDlgItem (hGeom,IDC_CUT));
	but->SetCheck(TRUE);
	ReleaseICustButton(but);
	proc.e1set = FALSE;
}

void CutEdgeCMode::ExitMode() {
	HWND hGeom = ep->GetDlgHandle (ep_geom);
	if (!hGeom) return;
	ICustButton *but = GetICustButton (GetDlgItem (hGeom,IDC_CUT));
	but->SetCheck(FALSE);
	ReleaseICustButton(but);
}

void CutEdgeCMode::AbandonCut () {
	proc.e1set = FALSE;
}

HitRecord *CutFaceProc::HitTestFaces (IPoint2 &m, ViewExp *vpt) {
	vpt->ClearSubObjHitList();
	ip->SubObHitTest (ip->GetTime(), HITTYPE_POINT, 0, 0, &m, vpt);
	if (!vpt->NumSubObjHits()) return NULL;
	HitLog& hitLog = vpt->GetSubObjHitList();
	HitRecord *ret = hitLog.ClosestHit();

	if (f1 == (int)ret->hitInfo) return NULL;
	return ret;
}

void CutFaceProc::DrawCutter (HWND hWnd,IPoint2 &m) {
	HDC hdc = GetDC(hWnd);
	SetROP2(hdc, R2_XORPEN);
	SetBkMode(hdc, TRANSPARENT);	
	SelectObject(hdc,CreatePen(PS_DOT, 0, ComputeViewportXORDrawColor()));
	MoveToEx(hdc,m1.x,m1.y,NULL);
	LineTo(hdc,m.x,m.y);
	DeleteObject(SelectObject(hdc,GetStockObject(BLACK_PEN)));
	ReleaseDC(hWnd, hdc);
}

int CutFaceProc::proc (HWND hwnd, int msg, int point, int flags, IPoint2 m ) {
	ViewExp *vpt;
	HitRecord *hr;
	IPoint2 betterM;
	Point3 A, B, snapPoint, Xdir, Zdir;
	Matrix3 obj2world, world2obj;
	int ff;
	Point3 hitPoint;
	Ray r;

	switch (msg) {
	case MOUSE_ABORT:
		// Erase last cutter line:
		if (f1>-1) DrawCutter (hwnd, oldm2);
		f1 = -1;
		return FALSE;

	case MOUSE_PROPCLICK:
		// Erase last cutter line:
		if (f1>-1) DrawCutter (hwnd, oldm2);
		ip->SetStdCommandMode(CID_OBJMOVE);
		return FALSE;

	case MOUSE_POINT:
		ip->SetActiveViewport (hwnd);
		vpt = ip->GetViewport (hwnd);
		hr = HitTestFaces(m,vpt);
		if (!hr) {
			ip->ReleaseViewport (vpt);
			return true;
		}
		snapPoint = vpt->SnapPoint (m, betterM, NULL);
		snapPoint = vpt->MapCPToWorld (snapPoint);
		ProjectHitToFace (m, vpt, hr, &snapPoint, ff, hitPoint);

		vpt->MapScreenToWorldRay ((float)betterM.x, (float)betterM.y, r);
		Zdir = Normalize (r.dir);
		ip->ReleaseViewport (vpt);

		if (f1<0) {
			f1 = ff;
			p1 = hitPoint;
			m1 = betterM;
			DrawCutter (hwnd, m);
			oldm2=m;
			break;
		}

		// Erase last cutter line:
		DrawCutter (hwnd, oldm2);

		// Do the cut:
		obj2world = hr->nodeRef->GetObjectTM (ip->GetTime ());
		world2obj = Inverse (obj2world);
		Zdir = Normalize (VectorTransform (world2obj, Zdir));

		theHold.Begin ();
		f1 = ep->EpfnCutFace (f1, p1, hitPoint, -Zdir);
		theHold.Accept (GetString (IDS_CUT));
		ep->RefreshScreen ();

		if (f1<0) return FALSE;

		// Otherwise, f1 represents the final vertex.  Turn it into a face index.
		if (ep->GetMeshPtr()->vfac[f1].Count() == 0) return false;
		f1 = ep->GetMeshPtr()->vfac[f1][0];

		// Start on next cut.
		p1 = hitPoint;
		m1 = betterM;
		DrawCutter (hwnd, m);
		oldm2 = m;
		return true;

	case MOUSE_MOVE:
		vpt = ip->GetViewport (hwnd);
		vpt->SnapPoint(m,m,NULL);
		if (f1>-1) {
			DrawCutter (hwnd,oldm2);
			oldm2 = m;
		}
		if (HitTestFaces(m,vpt)) SetCursor(ip->GetSysCursor(SYSCUR_SELECT));
		else SetCursor(LoadCursor(NULL,IDC_ARROW));
		ip->ReleaseViewport (vpt);
		ip->RedrawViews (ip->GetTime());
		if (f1>-1) DrawCutter (hwnd,oldm2);
		break;

	case MOUSE_FREEMOVE:
		vpt = ip->GetViewport (hwnd);
		vpt->SnapPreview(m,m,NULL);//, SNAP_SEL_OBJS_ONLY);
		if (HitTestFaces(m,vpt)) SetCursor(ip->GetSysCursor(SYSCUR_SELECT));
		else SetCursor(LoadCursor(NULL,IDC_ARROW));
		ip->ReleaseViewport (vpt);
		break;
	}

	return TRUE;
}

void CutFaceProc::ProjectHitToFace (IPoint2 &m, ViewExp *vpt, HitRecord *hr,
									Point3 *snapPoint, int & face, Point3 & hitPoint) {
	if (!hr) return;

	// Find subtriangle, barycentric coordinates of hit in face.
	face = hr->hitInfo;
	Matrix3 obj2world = hr->nodeRef->GetObjectTM (ip->GetTime ());

	Ray r;
	vpt->MapScreenToWorldRay ((float)m.x, (float)m.y, r);
	if (!snapPoint) snapPoint = &(r.p);
	Point3 Zdir = Normalize (r.dir);

	// Find an approximate location for the point on the surface we hit:
	// Get the average normal for the face, thus the plane, and intersect.
	MNMesh & mm = *(ep->GetMeshPtr());
	Point3 planeNormal = mm.GetFaceNormal (face, TRUE);
	planeNormal = Normalize (obj2world.VectorTransform (planeNormal));
	float planeOffset=0.0f;
	for (int i=0; i<mm.f[face].deg; i++)
		planeOffset += DotProd (planeNormal, obj2world*mm.v[mm.f[face].vtx[i]].p);
	planeOffset = planeOffset/float(mm.f[face].deg);

	// Now we intersect the snapPoint + r.dir*t line with the
	// DotProd (planeNormal, X) = planeOffset plane.
	float rayPlane = DotProd (r.dir, planeNormal);
	float firstPointOffset = planeOffset - DotProd (planeNormal, *snapPoint);
	if (fabsf(rayPlane) > EPSILON) {
		float amount = firstPointOffset / rayPlane;
		hitPoint = *snapPoint + amount*r.dir;
	} else {
		hitPoint = *snapPoint;
	}

	Matrix3 world2obj = Inverse (obj2world);
	hitPoint = world2obj * hitPoint;
}

void CutFaceCMode::EnterMode() {
	HWND hGeom = ep->GetDlgHandle (ep_geom);
	if (!hGeom) return;
	ICustButton *but = GetICustButton (GetDlgItem (hGeom,IDC_CUT));
	but->SetCheck(TRUE);
	ReleaseICustButton(but);
	proc.f1 = -1;
}

void CutFaceCMode::ExitMode() {
	HWND hGeom = ep->GetDlgHandle (ep_geom);
	if (!hGeom) return;
	ICustButton *but = GetICustButton (GetDlgItem (hGeom,IDC_CUT));
	but->SetCheck(FALSE);
	ReleaseICustButton(but);
}

void CutFaceCMode::AbandonCut () {
	proc.f1 = -1;
}

//-------------------------------------------------------

void WeldCMode::EnterMode () {
	mproc.targ1 = -1;
	HWND hGeom = ep->GetDlgHandle (ep_geom);
	if (!hGeom) return;
	ICustButton *but = GetICustButton (GetDlgItem (hGeom, IDC_WELD_TO));
	but->SetCheck(TRUE);
	ReleaseICustButton(but);
}

void WeldCMode::ExitMode () {
	HWND hGeom = ep->GetDlgHandle (ep_geom);
	if (!hGeom) return;
	ICustButton *but = GetICustButton (GetDlgItem (hGeom, IDC_WELD_TO));
	but->SetCheck(FALSE);
	ReleaseICustButton(but);
}

void WeldMouseProc::DrawWeldLine (HWND hWnd,IPoint2 &m) {
	if (targ1<0) return;
	HDC hdc = GetDC(hWnd);
	SetROP2(hdc, R2_XORPEN);
	SetBkMode(hdc, TRANSPARENT);	
	SelectObject(hdc,CreatePen(PS_DOT, 0, ComputeViewportXORDrawColor()));
	MoveToEx(hdc,m1.x,m1.y,NULL);
	LineTo(hdc,m.x,m.y);
	DeleteObject(SelectObject(hdc,GetStockObject(BLACK_PEN)));
	ReleaseDC(hWnd, hdc);
}

HitRecord *WeldMouseProc::HitTest (IPoint2 &m, ViewExp *vpt) {
	vpt->ClearSubObjHitList();
	// Substitute in the weld pixel value for hit-testing.
	ep->pblock->GetValue (ep_weld_pixels, TimeValue(0), ep->pickBoxSize, FOREVER);
	ip->SubObHitTest (ip->GetTime(), HITTYPE_POINT, 0, 0, &m, vpt);
	// Return to normal value:
	ep->pickBoxSize = DEF_PICKBOX_SIZE;
	if (!vpt->NumSubObjHits()) return NULL;
	HitLog& hitLog = vpt->GetSubObjHitList();
	HitRecord *hr = hitLog.ClosestHit();
	if ((targ1>-1) && (targ1 == hr->hitInfo)) return NULL;
	return hr;
}

int WeldMouseProc::proc (HWND hwnd, int msg, int point, int flags, IPoint2 m) {
	ViewExp *vpt = NULL;
	int res = TRUE;
	HitRecord *hr;

	switch (msg) {
	case MOUSE_ABORT:
		// Erase last weld line:
		if (targ1>-1) DrawWeldLine (hwnd, oldm2);
		targ1 = -1;
		return FALSE;

	case MOUSE_PROPCLICK:
		// Erase last weld line:
		if (targ1>-1) DrawWeldLine (hwnd, oldm2);
		ip->SetStdCommandMode(CID_OBJMOVE);
		return FALSE;

	case MOUSE_POINT:
		ip->SetActiveViewport (hwnd);
		vpt = ip->GetViewport (hwnd);
		hr = HitTest (m, vpt);
		if (!hr) break;
		if (targ1 < 0) {
			targ1 = hr->hitInfo;
			m1 = m;
			DrawWeldLine (hwnd, m);
			oldm2 = m;
			break;
		}

		// Otherwise, we're completing the weld.
		// Erase the last weld-line:
		DrawWeldLine (hwnd, oldm2);

		// Do the weld:
		theHold.Begin();
		bool ret;
		ret = false;
		int stringID;
		if (ep->selLevel == EP_SL_VERTEX) {
			Point3 destination = ep->mm.P(hr->hitInfo);
			ret = ep->EpfnWeldVerts (targ1, hr->hitInfo, destination)?true:false;
			stringID = IDS_WELD_VERTS;
		} else if (meshSelLevel[ep->selLevel] == MNM_SL_EDGE) {
			ret = ep->EpfnWeldEdges (targ1, hr->hitInfo)?true:false;
			stringID = IDS_WELD_EDGES;
		}
		if (ret) {
			theHold.Accept (GetString(stringID));
			ep->RefreshScreen ();
		} else {
			theHold.Cancel ();
		}
		targ1 = -1;
		ip->ReleaseViewport(vpt);
		return false;

	case MOUSE_MOVE:
	case MOUSE_FREEMOVE:
		vpt = ip->GetViewport(hwnd);
		if (targ1 > -1) {
			DrawWeldLine (hwnd, oldm2);
			oldm2 = m;
		}
		if (HitTest (m,vpt)) SetCursor(ip->GetSysCursor(SYSCUR_SELECT));
		else SetCursor (LoadCursor (NULL, IDC_ARROW));
		ip->RedrawViews (ip->GetTime());
		if (targ1 > -1) DrawWeldLine (hwnd, m);
		break;
	}
	
	if (vpt) ip->ReleaseViewport(vpt);
	return true;	
}

//-------------------------------------------------------

void EditTriCMode::EnterMode() {
	HWND hSurf = ep->GetDlgHandle (ep_face);
	if (hSurf) {
		ICustButton *but = GetICustButton (GetDlgItem (hSurf, IDC_FS_EDIT_TRI));
		but->SetCheck(TRUE);
		ReleaseICustButton(but);
	}
	if (!ep->ip->GetShowEndResult()) ep->mm.dispFlags |= MNDISP_DIAGONALS;
	ep->NotifyDependents (FOREVER, PART_DISPLAY, REFMSG_CHANGE);
	ep->ip->RedrawViews(ep->ip->GetTime());
}

void EditTriCMode::ExitMode() {
	HWND hSurf = ep->GetDlgHandle (ep_face);
	if (hSurf) {
		ICustButton *but = GetICustButton (GetDlgItem (hSurf, IDC_FS_EDIT_TRI));
		but->SetCheck (FALSE);
		ReleaseICustButton(but);
	}
	if (!ep->ip->GetShowEndResult()) ep->mm.dispFlags &= ~(MNDISP_DIAGONALS);
	ep->NotifyDependents (FOREVER, PART_DISPLAY, REFMSG_CHANGE);
	ep->ip->RedrawViews(ep->ip->GetTime());
}

void EditTriProc::VertConnect () {
	DbgAssert (v1>=0);
	DbgAssert (v2>=0);
	Tab<int> v1fac = ep->GetMeshPtr()->vfac[v1];
	int i, j, ff, v1pos, v2pos=-1;
	for (i=0; i<v1fac.Count(); i++) {
		MNFace & mf = ep->GetMeshPtr()->f[v1fac[i]];
		for (j=0; j<mf.deg; j++) {
			if (mf.vtx[j] == v2) v2pos = j;
			if (mf.vtx[j] == v1) v1pos = j;
		}
		if (v2pos<0) continue;
		ff = v1fac[i];
		break;
	}

	if (v2pos<0) return;

	theHold.Begin();
	ep->EpfnSetDiagonal (ff, v1pos, v2pos);
	theHold.Accept (GetString (IDS_EDIT_TRIANGULATION));
	ep->RefreshScreen ();
}

}	// end namespace EditPoly