/* 
Makefont.exe, V2.0
by Ryan McMahon

This program produces a FNT file from a BMP for use on the PS2

Usage:
	Makefont infile.bmp outfile.fnt
*/


#include <stdio.h>
#include <stdlib.h>
#include <fstream.h>
#include <memory.h>
#include <stdarg.h>

#include "Makefont.h"

char Update[9] = "8/13/00";
int OptPrint;

BMPInfo BMP;

// character separation
// (very large value to start with)
int Separation = 100000;
// the guesstimated halfway to the next character, relative to the baseline of any current one
int Halfway;


Character CharTable[256];
int NumChars;
int MaxCharHeight = 0;
int MaxCharHeightAboveBase = 0;
int MaxCharHeightBelowBase = 0;

OutInfo Out;



void Assert(int condition, char *pMessage, ...)
{
	char pad[128];
	
	if (condition) return;
	
	va_list args;
	
	va_start( args, pMessage );
	vsprintf( pad, pMessage, args);
	va_end( args );

	printf("ERROR: ");
	printf( pad );
	printf("\n");
	exit(1);
}



int LoadBMP(char* file) 
{
	ifstream from(file, ios::nocreate | ios::binary | ios::in);
	
	// get 54 characters, enough to read the BMP header and figure out how
	// much space we need

	// read in file header
	char *pHeader = new char[54];
	if (!from.read(pHeader, 54)) 
		Assert(0, "oh no, error opening %s!", file); 

	BMP.w = (int) (*(Uint16 *) (pHeader + 18));
	BMP.h = (int) (*(Uint16 *) (pHeader + 22));

	Assert(!(BMP.w & 1), "Error: width must be multiple of 2");  
	
	BMP.bpp = *(Uint16 *) (pHeader + 28);
	Assert(BMP.bpp == 8, "Not 8 bits per pixel in %s", file);

	if (BMP.bpp == 24)
		BMP.bpp = 32;

	BMP.byteWidth = (BMP.w * BMP.bpp) >> 3;
	// each line will be padded	to a 4-byte boundary in input file
	BMP.byteWidthPadded = (BMP.byteWidth + 3) & (~3);

	if (OptPrint) printf("BMP width, height, depth are %d,%d,%d\n", BMP.w, BMP.h, BMP.bpp);
	
	// how many colors?
	BMP.numColors = 1 << BMP.bpp;
	
	int colors = (int) (*(Uint16 *) (pHeader + 46));
	if (OptPrint) printf("There are %d used colors\n", colors);
	if (colors == 0) colors = 256;
	BMP.numColors = colors;

	// read in actual color data
	if (BMP.numColors <= 256)
	{
		BMP.pColorTab = new RGBA[BMP.numColors];
		if (!from.read((char *) BMP.pColorTab, BMP.numColors * 4))
			Assert(0, "error reading colors of %s", file);
	
		// find transparent color and set alpha component
		BMP.transparentColor = -1;
		for (int i = 0; i < BMP.numColors; i++)
		{
			// switch red and blue
			Uint8 red = BMP.pColorTab[i].r;
			BMP.pColorTab[i].r = BMP.pColorTab[i].b;
			BMP.pColorTab[i].b = red;
			
			if (BMP.pColorTab[i].r == 0 && BMP.pColorTab[i].g == 0 && BMP.pColorTab[i].b == 0)
			{
				// we want to use very first transparent color found
				if (BMP.transparentColor == -1)
				{
					if (OptPrint)
						printf("transparent color at %d\n", i);
					BMP.transparentColor = i;
				}
				BMP.pColorTab[i].a = 0;
			}
			else
				BMP.pColorTab[i].a = 128;
		}
		Assert(BMP.transparentColor != -1, "no transparent color entry found");
	}

	// read in image data
	Uint8 *pInBuf = new Uint8[BMP.byteWidthPadded * BMP.h];
	if (!from.read(pInBuf, BMP.byteWidthPadded * BMP.h))
		Assert(0, "error reading data of %s", file);

	// flip it over
	Uint8 *pOutBuf = new Uint8[BMP.byteWidth * BMP.h];
	Uint8 *pIn = pInBuf + BMP.byteWidthPadded * (BMP.h - 1);
	Uint8 *pOut = pOutBuf;
	for (int j = 0; j < BMP.h; j++) 
	{
		Uint8 *pLineIn = pIn;
		for (int i = 0; i < BMP.byteWidth; i++)
			*(pOut++) = *(pLineIn++);
		pIn -= BMP.byteWidthPadded;
	}
	
	// add vertical padding
	// added to bottom of image
	/*
	Height += Vert_padding;
	for (j = 0; j < Vert_padding; j++)
		for (int i = 0; i < Width; i++)
			*(out++) = transparent_color;
	*/
		
	// set up global pointer
	BMP.pData = pOutBuf;

	return(0);
}


/*
	Loads a file to set the ascii value for each character
*/
void LoadAsciiTable(char *file)
{
	ifstream from(file, ios::nocreate | ios::in);
	if (!from.good())
		Assert(0, "error opening file %s\n", file);

	// Fill with zeros in case there are fewer characters in the ascii map file
	// than in the BMP. Extra BMP characters won't be exported.
	for (int n = 0; n < 256; n++)
		CharTable[n].ascii = 0;

	n = 0;
	while(1) 
	{
		unsigned char c = from.get();
		if (from.eof())
			break;
		if (c != '\n') 
			CharTable[n++].ascii = c;
	}
}


/*
	Scans line. Returns true if non-blank pixel found. Also returns first and last pixel x coordinates.
*/
bool scanForPixel(int line, int *pFirst, int *pLast)
{
	Uint8 *pBuf = (Uint8 *) (BMP.pData + BMP.byteWidth * line);
	
	int first, last;
	bool isBlank = true;
	
	for (int i = 0; i < BMP.byteWidth; i++) 
	{
		if (*pBuf != BMP.transparentColor)
		{
			// line not blank
			if (isBlank) first = i;
			last = i + 1;
			isBlank = false;
		}
		pBuf++;
	}

	if (pFirst) *pFirst = first;
	if (pLast) *pLast = last;
	
	return !isBlank;
}


int scanForLine(int start, int step, bool needBlank) 
{
	// for 8 bit only
	
	/*
	int end_top = start - Separation;
	if (end_top < 0) end_top = 0;
	int end_bottom = start + Separation;
	if (end_bottom > BMP.h) end_bottom = BMP.h;
	*/
	int end_top = 0;
	int end_bottom = BMP.h;

	int j;
	for (j = start; j < end_bottom && j >= end_top; j += step) 
	{
		bool isOccupied = scanForPixel(j, NULL, NULL);
		if (!needBlank && isOccupied)
			return j;
		else if (needBlank && !isOccupied)
			return j;
	}

	// if desired line can't be found, indicate error
	return(-1);
}


int scanForBlankLine(int start, int step)
{
	return scanForLine(start, step, true);
}


int scanForNonBlankLine(int start, int step)
{
	return scanForLine(start, step, false);
}


/*
	Return baseline of first character (second A)
*/
int getCharHeight()
{
	int top1, top2;
	int base;

	top1 = scanForNonBlankLine(0, 1);
	base = scanForBlankLine(top1, 1);
	top2 = scanForNonBlankLine(base, 1);

	if (OptPrint)
		printf("top1 at %d, base at %d, top2 %d\n", top1, base, top2);
	Assert(top1 != -1 && top2 != -1 && base != -1, "couldn't calibrate character separation: make sure input file is correct");

	Separation = top2 - top1;
	Halfway = (top2 - base + 1) / 2;
	
	return base + Separation - 1;
}


/*
	'line' should point to the character baseline
*/
bool getCharacterDims(int n, int line) 
{
	/*
	// find top of this character by beginning at baseline of last character
	int top	= scanForNonBlankLine(scanForBlankLine(line - Separation, 1), 1);
	// find bottom of this character by beginning at baseline and scanning down
	int bottom = scanForBlankLine(line, 1);
	*/

	// find top of this character by beginning at baseline of last character
	int top	= scanForNonBlankLine(scanForBlankLine(line - Separation + Halfway, 1), 1);
	// find bottom of this character by scanning back to baseline and scanning down
	int bottom = scanForBlankLine(scanForNonBlankLine(line + Halfway, -1), 1);

	if (top == -1 || bottom == -1) return false;
	
	CharTable[n].y = top;
	CharTable[n].h = bottom - top + 1;
	CharTable[n].baseline = line - top;

	CharTable[n].x = BMP.w;
	CharTable[n].w = 0;
	
	for (int j = top; j < bottom; j++)
	{
		int l, r;
		if (scanForPixel(j, &l, &r))
		{
			if (l < CharTable[n].x) CharTable[n].x = l;
			// CharTable[n].w isn't width just yet, but will be 
			if (r > CharTable[n].w) CharTable[n].w = r;
		}
	}
	CharTable[n].w -= CharTable[n].x - 1;
	CharTable[n].byteW = (CharTable[n].w * BMP.bpp) >> 3;
	
	if (OptPrint)
	{
		printf("character %d has x, y, w, h, baseline = %d,%d,%d,%d\n", n, 
			CharTable[n].x, CharTable[n].y, CharTable[n].w, CharTable[n].h);
		printf("                 baseline is %d\n", CharTable[n].baseline);
	}

	if (bottom - top > MaxCharHeight) MaxCharHeight = bottom - top;
	if (CharTable[n].baseline > MaxCharHeightAboveBase) 
		MaxCharHeightAboveBase = CharTable[n].baseline;
	if (CharTable[n].h - CharTable[n].baseline > MaxCharHeightBelowBase) 
		MaxCharHeightBelowBase = CharTable[n].h - CharTable[n].baseline;
	
	return true;
}



void sendOut(Uint8 *pImage, int out_x, int out_y, int n)
{
	Uint8 *pOut = pImage + out_y * Out.byteW + out_x;
	Uint8 *pIn = BMP.pData + CharTable[n].y * BMP.byteWidth + CharTable[n].x;

	for (int j = 0; j < CharTable[n].h; j++)
	{
		for (int i = 0; i < CharTable[n].byteW; i++)
			*(pOut++) = *(pIn++);
		pOut += Out.byteW - CharTable[n].byteW;
		pIn += BMP.byteWidth - CharTable[n].byteW;
	}
}


void saveToFile(char *file) 
{
	// create output buffer

	Out.w = 256;
	Out.h = 256;
	// this number will probably increase
	Out.usedH = 32;
	Out.byteW = (Out.w * BMP.bpp) >> 3;

	// We reserve space for a 256X256 image buffer, although we may output
	// fewer rows than 256 to disk, as determined by Out.usedH
	Out.size = Out.byteW * Out.h;
	Out.pData = new Uint8[Out.size];
	// fill with zeros
	memset(Out.pData, 0, Out.size);

	// reserve space for sub-texture descriptors
	Out.pChar = new OutCharInfo[NumChars];

	// reserve space for extra character info
	// (which isn't included in the texture part of the FNT file)
	Out.pExtra = new ExtraCharInfo[NumChars];

	int out_x = 0, out_y = 0;
	// used to count characters successfully written to disk
	// (some might be incorrect)
	int written_char = 0;

	// create image data
	for (int n = 0; n < NumChars; n++)
	{
		// skip any characters that have been read incorrectly
		if (CharTable[n].w <= 0 || CharTable[n].h <= 0 || CharTable[n].w > Out.w)
		{
			printf("Character %d not exported. Dimensions incorrect:\n", n);
			printf("    w = %d, h = %d\n", CharTable[n].w, CharTable[n].h);
			continue;
		}

		// skip any characters not in ascii map file
		if (!CharTable[n].ascii)
		{
			printf("Character %d not in ascii map file, skipped\n", n);
			continue;
		}
		
		if (out_x + CharTable[n].w > Out.w)
		{
			out_x = 0;
			out_y += MaxCharHeight;
			if (out_y + CharTable[n].h > Out.h)
			{
				printf("not enough room in output bitmap\n");
				exit(1);
			}
			while (out_y + CharTable[n].h > Out.usedH) 
				Out.usedH *= 2;
		}
		
		Out.pChar[written_char].x = out_x;
		Out.pChar[written_char].y = out_y;
		Out.pChar[written_char].w = CharTable[n].w;
		Out.pChar[written_char].h = CharTable[n].h;
		sendOut(Out.pData, out_x, out_y, n);

		Out.pExtra[written_char].baseline = CharTable[n].baseline;
		Out.pExtra[written_char].ascii = CharTable[n].ascii;
		
		written_char++;

		out_x += CharTable[n].w;
	}
	int num_written_chars = written_char;
	// for accounting purposes
	if (out_x > 0)
		out_y += MaxCharHeight;
	
	if (OptPrint)
		printf("Final texture size: %d by %d by %dbpp (used height = %d)\n",
			Out.w, Out.usedH, BMP.bpp, out_y);	
	
	/*
	=======================================
	Do actual file writing now
	=======================================
	*/

	ofstream to(file, ios::out | ios::binary);

	// for writing garbage characters
	char blank[256];
	
	// compute sizes
	int out_size = (Out.byteW * Out.usedH + 3) & (~3);
	int palette_size = 1024;
	int texture_size = 16 + out_size + palette_size + 4 + num_written_chars * 8;
	
	int font_size = 16 + 4 * num_written_chars + texture_size;
	
	/*
	=======================================
	Write FNT file specific section of file
	=======================================
	*/

	int default_height = MaxCharHeightAboveBase + MaxCharHeightBelowBase;
	
	to.write((char *) &font_size, 4);
	to.write((char *) &num_written_chars, 4);
	to.write((char *) &default_height, 4);
	to.write((char *) &MaxCharHeightAboveBase, 4);
	to.write((char *) Out.pExtra, 4 * num_written_chars);
	
	// should be nicely padded at this point

	/*
	=======================================
	Write Image::Texture section of file
	=======================================
	*/
	
	// write header
	to.write((char *) &texture_size, 4);
	to.write((char *) &Out.w, 2);
	to.write((char *) &Out.usedH, 2);
	to.write((char *) &BMP.bpp, 2);
	// padding
	to.write(blank, 6);

	// write image data
	to.write((char *) Out.pData, out_size);

	// write palette data
	if (palette_size)
		to.write((char *) BMP.pColorTab, palette_size);

	// write total number of subtextures in texture
	to.write((char *) &num_written_chars, 4);

	// write subtexture descriptors
	to.write((char *) Out.pChar, num_written_chars * 8);
	
	to.close();
}



void error()
{
	printf("\nUsage:\n");
	printf("makefont [options] infile.bmp asciifile.map outfile.fnt\n");
	printf("options: <-p>\n"); 
}



int main(int argc, char *argv[]) 
{
	if (argc < 3) 
	{
		error();
		return 1;
	}

	OptPrint = 0; 
	
	// scan for options
	int i;
	for (i = 1; i < argc; i++)
	{
		if (*argv[i] == '-') 
		{
			switch(*(argv[i] + 1)) 
			{
			case 'p':
				OptPrint = 1;
				printf("Font Grabber, last updated %s\n", Update);
				break;
			default:
				printf("\nError: invalid option\n");
				error();
				return(1);
			} // end switch
		} // end if

		else 
			break;
	}
	
	if (argc < i+3)
	{
		printf("\nError: not enough options\n");
		error();
		return(1);
	}
	
	printf("creating font %s from BMP %s and map file %s\n", argv[i+2], argv[i], argv[i+1]);
	
	// load file
	LoadBMP(argv[i]);

	LoadAsciiTable(argv[i+1]);
	
	int currentLine = getCharHeight();

	int j;
	for (j = 0; j < 200; j++)
	{
		if (currentLine >= BMP.h) break;
		if (!getCharacterDims(j, currentLine)) break;
		
		currentLine += Separation;
	}
	NumChars = j;

	saveToFile(argv[i+2]);

	return 0;
}
