#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <math.h>
#include <assert.h>


int		textureWidth	= 128;
int		textureHeight	= 128;
int		density			= 6000;
int		numLibraries	= 32;
int		numSlices		= 6;
float	layerSpacing	= 1.0f;
char	*filename		= "grass";
void	*pSliceTextures[32];

// Important to ensure that bytes get packed as the file format expects.
#pragma pack( 1 )

typedef unsigned char	byte;
typedef unsigned short	word;

struct tgaHeader
{
	byte	m_size;
	byte	m_colormap_type;
	byte	m_image_type;
	word	m_colormap_start;
	word	m_colormap_length;
	byte	m_colormap_depth;
	word	m_x_offset;
	word	m_y_offset;
	word	m_width;
	word	m_height;
	byte	m_pixel_depth;
	byte	m_image_descriptor;
};

// A fuzz is a single hair follicle, blade of grass, etc.
struct fuzz
{
	float	dp[3];			// Velocity
	float	ddp[3];			// Acceleration
	float	colorBase[4];
    float	colorTip[4];
	float	time[32];		// Calculated values for each layer.

};

struct fuzz_instance
{
	float	x, z;
	int		lib;
};



fuzz fuzzCenter;
fuzz fuzzRandom;


#define irand( a )	((rand() * ( a )) >> 15 )
#define frand( a )	((float)rand() * ( a ) / 32768.0f )


void createTextures( void )
{
	int i, s;
	
	// First allocate memory for the texture slices.
	for( s = 0; s < numSlices; ++s )
	{
		pSliceTextures[s] = new unsigned int[textureWidth * textureHeight];
		memset( pSliceTextures[s], 0, sizeof( unsigned int ) * textureWidth * textureHeight );
	}

	// Generate the fuzz libraries.
	fuzz *p_fuzz = new fuzz[numLibraries];
	for( i = 0; i < numLibraries; ++i )
	{
		p_fuzz[i].dp[0]			= fuzzCenter.dp[0] + fuzzRandom.dp[0] * ( 2 * frand( 1.0f ) - 1.0f );
		p_fuzz[i].dp[1]			= fuzzCenter.dp[1] + fuzzRandom.dp[1] * ( 2 * frand( 1.0f ) - 1.0f );
		p_fuzz[i].dp[2]			= fuzzCenter.dp[2] + fuzzRandom.dp[2] * ( 2 * frand( 1.0f ) - 1.0f );

		p_fuzz[i].ddp[0]		= fuzzCenter.ddp[0] + fuzzRandom.ddp[0] * ( 2 * frand( 1.0f ) - 1.0f );
		p_fuzz[i].ddp[1]		= fuzzCenter.ddp[1] + fuzzRandom.ddp[1] * ( 2 * frand( 1.0f ) - 1.0f );
		p_fuzz[i].ddp[2]		= fuzzCenter.ddp[2] + fuzzRandom.ddp[2] * ( 2 * frand( 1.0f ) - 1.0f );
	
		p_fuzz[i].colorBase[0]	= fuzzCenter.colorBase[0] + ( 2 * frand( 1.f ) - 1.f ) * fuzzRandom.colorBase[0];
		p_fuzz[i].colorBase[1]	= fuzzCenter.colorBase[1] + ( 2 * frand( 1.f ) - 1.f ) * fuzzRandom.colorBase[1];
		p_fuzz[i].colorBase[2]	= fuzzCenter.colorBase[2] + ( 2 * frand( 1.f ) - 1.f ) * fuzzRandom.colorBase[2];
		p_fuzz[i].colorBase[3]	= fuzzCenter.colorBase[3] + ( 2 * frand( 1.f ) - 1.f ) * fuzzRandom.colorBase[3];
	
		p_fuzz[i].colorTip[0]	= fuzzCenter.colorTip[0] + ( 2 * frand( 1.f ) - 1.f ) * fuzzRandom.colorTip[0];
		p_fuzz[i].colorTip[1]	= fuzzCenter.colorTip[1] + ( 2 * frand( 1.f ) - 1.f ) * fuzzRandom.colorTip[1];
		p_fuzz[i].colorTip[2]	= fuzzCenter.colorTip[2] + ( 2 * frand( 1.f ) - 1.f ) * fuzzRandom.colorTip[2];
		p_fuzz[i].colorTip[3]	= fuzzCenter.colorTip[3] + ( 2 * frand( 1.f ) - 1.f ) * fuzzRandom.colorTip[3];

		for( int c = 0; c < 4; ++c )
		{
			if( p_fuzz[i].colorBase[c] < 0.0f )
				p_fuzz[i].colorBase[c] = 0.0f;
			else if( p_fuzz[i].colorBase[c] > 1.0f )
				p_fuzz[i].colorBase[c] = 1.0f;

			if( p_fuzz[i].colorTip[c] < 0.0f )
				p_fuzz[i].colorTip[c] = 0.0f;
			else if( p_fuzz[i].colorTip[c] > 1.0f )
				p_fuzz[i].colorTip[c] = 1.0f;
		}
		
		p_fuzz[i].time[0]		= 0.0f;
		for( s = 1; s < numSlices; ++s )
		{
			// For each fuzz and each slice, figure the time for the blade to reach height h...
			float y_dist = s * layerSpacing;

			float a				= 0.5f * p_fuzz[i].ddp[1];
			float b				= p_fuzz[i].dp[1];
			float c				= -y_dist;
			p_fuzz[i].time[s]	= ( -b + sqrtf(( b * b ) - ( 4.0f * a * c ))) / ( 2.0f * a );
		}
	}

	// Now generate the blade representations.
	fuzz_instance *p_fuzz_instances = new fuzz_instance[density];
	for( i = 0; i < density; ++i )
	{
		p_fuzz_instances[i].x	= frand(( textureWidth ));
		p_fuzz_instances[i].z	= frand(( textureHeight ));
		p_fuzz_instances[i].lib	= irand( numLibraries );
	}


	for( s = 0; s < numSlices; ++s )
	{
		// For each blade, figure the time for the blade to reach height h...
		float y_dist = s * layerSpacing;

		for( int blade = 0; blade < density; ++blade )
		{
			int		lib			= p_fuzz_instances[blade].lib;
			float	time		= p_fuzz[lib].time[s];
			float	max_time	= p_fuzz[lib].time[numSlices - 1];
			
			float	xs		= ( p_fuzz[lib].dp[0] * time ) + ( 0.5f * p_fuzz[lib].ddp[0] * time * time );
			float	zs		= ( p_fuzz[lib].dp[2] * time ) + ( 0.5f * p_fuzz[lib].ddp[2] * time * time );
			
			float	x_pos	= p_fuzz_instances[blade].x + xs;
			float	z_pos	= p_fuzz_instances[blade].z + zs;

			// Wrap pixels round.
			if( x_pos >= (float)textureWidth )
				x_pos -= (float)textureWidth;
			else if( x_pos < 0.0f )
				x_pos += (float)textureWidth;

			if( z_pos >= (float)textureHeight )
				z_pos -= (float)textureHeight;
			else if( z_pos < 0.0f )
				z_pos += (float)textureHeight;

			float color_delta[4];
			for( int d = 0; d < 4; ++d )
			{
				color_delta[d] = p_fuzz[lib].colorTip[d] - p_fuzz[lib].colorBase[d];
				color_delta[d] = p_fuzz[lib].colorBase[d] + (( color_delta[d] * time ) / max_time );
				if( color_delta[d] > 1.0f )
				{
					color_delta[d] = 1.0f;
				}
			}
			
			assert(( x_pos >= 0.0f ) && ( x_pos < (float)textureWidth ));
			assert(( z_pos >= 0.0f ) && ( z_pos < (float)textureHeight ));
			
			int offset = (int)( textureHeight * ( textureWidth - 0.0001f ));
			
			unsigned int *p_pixel = ((unsigned int*)pSliceTextures[s] ) + (int)(( textureWidth * (int)z_pos ) + x_pos );

			unsigned int red = (int)( color_delta[0] * 0xFF );
			unsigned int grn = (int)( color_delta[1] * 0xFF );
			unsigned int blu = (int)( color_delta[2] * 0xFF );
			unsigned int alp = (int)( color_delta[3] * 0xFF );
		
			*p_pixel = ( alp << 24 ) | ( red << 16 ) | ( grn << 8 ) | blu;
		}
	}

	// Cleanup the arrays we're finished with.
	delete [] p_fuzz_instances;
	delete [] p_fuzz;


}


void cleanupTextures( void )
{
	// First allocate memory for the texture slices.
	for( int s = 0; s < numSlices; ++s )
	{
		delete [] pSliceTextures[s];
	}
}



void saveTextures( void )
{
	size_t bytes_written;

	for( int s = 0; s < numSlices; ++s )
	{
		char fname[256];
		sprintf( fname, "%s%d.tga", filename, s );

		FILE *stream;
		stream = fopen( fname, "wb" );
		if( stream == NULL )
		{
			printf( "Failed to open file: %s\n", filename );
			exit( 1 );
		}

		tgaHeader th;
		memset( &th, 0, sizeof( tgaHeader ));
		th.m_image_type			= 2;	// Truecolor, no encoding.
		th.m_width				= textureWidth;
		th.m_height				= textureHeight;
		th.m_pixel_depth		= 32;
		th.m_image_descriptor	= 8;	// Number of alpha bits per pixel.
		bytes_written = fwrite( &th, sizeof( tgaHeader ), 1, stream );

		// Save bitmap data.
		bytes_written = fwrite( pSliceTextures[s], sizeof( unsigned int ) * textureWidth * textureHeight, 1, stream );

		// Close the file.
		fclose( stream );
	}
}



void main( int argc, char* argv[] )
{
	fuzzCenter.dp[0]	= 0.0f;
	fuzzCenter.dp[1]	= 4.0f;
	fuzzCenter.dp[2]	= 0.25f;
	fuzzCenter.ddp[0]	= 0.0f;
	fuzzCenter.ddp[1]	= 0.0f;
	fuzzCenter.ddp[2]	= 0.0f;
	
	fuzzRandom.dp[0]	= 0.25f;
	fuzzRandom.dp[1]	= 0.25f;
	fuzzRandom.dp[2]	= 0.25f;
	fuzzRandom.ddp[0]	= 0.5f;
	fuzzRandom.ddp[1]	= 0.5f;
	fuzzRandom.ddp[2]	= 0.5f;
	
	fuzzCenter.colorBase[0]	= 0.150980f;
	fuzzCenter.colorBase[1]	= 0.401961f;
	fuzzCenter.colorBase[2]	= 0.1f;
	fuzzCenter.colorBase[3]	= 1.0f;

	fuzzCenter.colorTip[0]	= 0.2f;
	fuzzCenter.colorTip[1]	= 0.3f;
	fuzzCenter.colorTip[2]	= 0.2f;
	fuzzCenter.colorTip[3]	= 0.2f;

	fuzzRandom.colorBase[0]	= 0.2f;
	fuzzRandom.colorBase[1]	= 0.2f;
	fuzzRandom.colorBase[2]	= 0.2f;
	fuzzRandom.colorBase[3]	= 0.0f;

	fuzzRandom.colorTip[0]	= 0.1f;
	fuzzRandom.colorTip[1]	= 0.1f;
	fuzzRandom.colorTip[2]	= 0.1f;
	fuzzRandom.colorTip[3]	= 0.1f;
	
	if( argc > 1 )
	{
		for( int i = 1; i < argc; ++i )
		{
			if( argv[i][0] == '-' )
			{
				switch( argv[i][1] )
				{
					case 'R':
					case 'r':
					{
						// Random base color.
						switch( argv[i][2] )
						{
							case 'R':
							case 'r':
							{
								int sscanf_result = sscanf( &argv[i][3], "%f", &fuzzRandom.colorBase[0] );
								break;
							}
							case 'G':
							case 'g':
							{
								int sscanf_result = sscanf( &argv[i][3], "%f", &fuzzRandom.colorBase[1] );
								break;
							}
							case 'B':
							case 'b':
							{
								int sscanf_result = sscanf( &argv[i][3], "%f", &fuzzRandom.colorBase[2] );
								break;
							}
							case 'A':
							case 'a':
							{
								int sscanf_result = sscanf( &argv[i][3], "%f", &fuzzRandom.colorBase[3] );
								break;
							}
							default:
							{
								printf( "Unrecognized option: %s\n", argv[i] );
								exit( 0 );
							}
						}
						break;
					}

					case 'Q':
					case 'q':
					{
						// Random tip color.
						switch( argv[i][2] )
						{
							case 'R':
							case 'r':
							{
								int sscanf_result = sscanf( &argv[i][3], "%f", &fuzzRandom.colorTip[0] );
								break;
							}
							case 'G':
							case 'g':
							{
								int sscanf_result = sscanf( &argv[i][3], "%f", &fuzzRandom.colorTip[1] );
								break;
							}
							case 'B':
							case 'b':
							{
								int sscanf_result = sscanf( &argv[i][3], "%f", &fuzzRandom.colorTip[2] );
								break;
							}
							case 'A':
							case 'a':
							{
								int sscanf_result = sscanf( &argv[i][3], "%f", &fuzzRandom.colorTip[3] );
								break;
							}
							default:
							{
								printf( "Unrecognized option: %s\n", argv[i] );
								exit( 0 );
							}
						}
						break;
					}

					case 'B':
					case 'b':
					{
						// Base color.
						switch( argv[i][2] )
						{
							case 'R':
							case 'r':
							{
								int sscanf_result = sscanf( &argv[i][3], "%f", &fuzzCenter.colorBase[0] );
								break;
							}
							case 'G':
							case 'g':
							{
								int sscanf_result = sscanf( &argv[i][3], "%f", &fuzzCenter.colorBase[1] );
								break;
							}
							case 'B':
							case 'b':
							{
								int sscanf_result = sscanf( &argv[i][3], "%f", &fuzzCenter.colorBase[2] );
								break;
							}
							case 'A':
							case 'a':
							{
								int sscanf_result = sscanf( &argv[i][3], "%f", &fuzzCenter.colorBase[3] );
								break;
							}
							default:
							{
								printf( "Unrecognized option: %s\n", argv[i] );
								exit( 0 );
							}
						}
						break;
					}
				
					case 'T':
					case 't':
					{
						// Tip color.
						switch( argv[i][2] )
						{
							case 'R':
							case 'r':
							{
								int sscanf_result = sscanf( &argv[i][3], "%f", &fuzzCenter.colorTip[0] );
								break;
							}
							case 'G':
							case 'g':
							{
								int sscanf_result = sscanf( &argv[i][3], "%f", &fuzzCenter.colorTip[1] );
								break;
							}
							case 'B':
							case 'b':
							{
								int sscanf_result = sscanf( &argv[i][3], "%f", &fuzzCenter.colorTip[2] );
								break;
							}
							case 'A':
							case 'a':
							{
								int sscanf_result = sscanf( &argv[i][3], "%f", &fuzzCenter.colorTip[3] );
								break;
							}
							default:
							{
								printf( "Unrecognized option: %s\n", argv[i] );
								exit( 0 );
							}
						}
						break;
					}

					case 'W':
					case 'w':
					{
						int sscanf_result = sscanf( &argv[i][2], "%d", &textureWidth );
						break;
					}

					case 'H':
					case 'h':
					{
						int sscanf_result = sscanf( &argv[i][2], "%d", &textureHeight );
						break;
					}

					case 'D':
					case 'd':
					{
						int sscanf_result = sscanf( &argv[i][2], "%d", &density );
						break;
					}

					case 'S':
					case 's':
					{
						int sscanf_result = sscanf( &argv[i][2], "%d", &numSlices );
						break;
					}

					case 'F':
					case 'f':
					{
						filename = &argv[i][2];
						break;
					}

					default:
					{
						printf( "Unrecognized option: %s\n", argv[i] );
						exit( 0 );
					}
				}
			}
		}	
	}
	else
	{
		printf( "Usage: makegrass <options>\n" );
		printf( "-d<fuzz density>\n" );
		printf( "-s<number of slices>\n" );
		printf( "-w<texture width>\n" );
		printf( "-h<texture height>\n" );
		printf( "-f<filename root>\n" );
		printf( "-b<r/g/b/a><base color value between 0.0 and 1.0>\n" );
		printf( "-t<r/g/b/a><tip color value between 0.0 and 1.0>\n" );
		printf( "-r<r/g/b/a><base color random variance value between 0.0 and 1.0>\n" );
		printf( "-q<r/g/b/a><tip color random variance value between 0.0 and 1.0>\n" );
		exit( 0 );
	}

	// Okay, we have all the details, now create the textures.
	createTextures();

	// Now save the textures.
	saveTextures();

	// Shutdown.
	cleanupTextures();
}
