/*
	NxTexture.cpp
	Class for managing textures in standard game TEX format
*/

#include "stdafx.h"
#include "NxTexture.h"

NxTexture::NxTexture( void )
{
	int i;

	for( i = 0; i < vMAX_MIP_LEVELS; i++ )
	{
		m_TexelData[i] = NULL;
		m_MipLevels = 0;
	}

	m_AlreadyExported = false;
	m_GroupFlags = 0;
	m_Flags = 0;
	m_PlatFlags = 0;
	m_TotalPaletteDataSize = 0;
	m_TotalTexelDataSize = 0;
	m_PaletteData = NULL;
	m_Bpp = 0;
	m_PaletteBpp = 0;
	m_LastGroupIndex = vNO_GROUP;
	m_LastGroupId = vNO_GROUP;
	m_LastDrawOrder = 0;
	strcpy( m_Name, "None" );
}

NxTexture::~NxTexture( void )
{
	int i;

	for( i = 0; i <= m_MipLevels; i++ )
	{
		if ( m_TexelData[i] )
		{
			delete [] m_TexelData[i];
		}
	}

	if ( m_PaletteData )
	{
		delete [] m_PaletteData;
	}
}

bool NxTexture::IsPaletted( void )
{
	return (( m_PixelFormat == v8_BIT ) || ( m_PixelFormat == v4_BIT ));
}



int	NxTexture::GetTotalDataSize( void )
{
	return( m_TotalPaletteDataSize + m_TotalTexelDataSize );
}



bool NxTexture::Convert4BitPixelFormatTo8BitPixelFormat( void )
{
	if( !IsPaletted() || ( m_PixelFormat != v4_BIT ))
	{
		return false;
	}

	// Reset this so we count up from scratch.
	m_TotalTexelDataSize = 0;

	for( int i = 0; i <= m_MipLevels; ++i )
	{
		int	src, dst;
		unsigned char*	p_8bit_texel_data;
		int	num_texels, bytes_per_row, row_count;

		num_texels = m_Width[i] * m_Height[i];
		p_8bit_texel_data = new unsigned char[num_texels];

		/*int texel_index = 0;
		for( int p = 0; p < m_TexelDataSize[i]; ++p )
		{
			// Read indices.
			unsigned int index0 = ((unsigned char)m_TexelData[i][p] >> 0 ) & 0xF;
			unsigned int index1 = ((unsigned char)m_TexelData[i][p] >> 4 ) & 0xF;

			p_8bit_texel_data[texel_index++] = index0;
			if( texel_index < num_texels )
			{
				p_8bit_texel_data[texel_index++] = index1;
			}
		}*/

		// I'm sure this loop could be more straight-forward, but
		// for now, I'm just concerned with correctness. -SG
		src = 0;
		dst = 0;
		bytes_per_row = (( m_Width[i] * 4 ) + 7 ) >> 3;
		for( int j = 0; j < m_Height[i]; ++j )
		{
			row_count = 0;
			for( int k = 0; k < bytes_per_row; ++k )
			{
				// Read indices.
				unsigned int index0 = ((unsigned char)m_TexelData[i][src] >> 0 ) & 0xF;
				unsigned int index1 = ((unsigned char)m_TexelData[i][src] >> 4 ) & 0xF;

				p_8bit_texel_data[dst++] = index0;
				row_count++;
				if( row_count < m_Width[i] )
				{
					p_8bit_texel_data[dst++] = index1;
					row_count++;
				}
				src++;
			}
		}

		delete [] m_TexelData[i];

		m_TexelData[i]			= (char*)p_8bit_texel_data;
		m_TexelDataSize[i]		= num_texels;
		m_TotalTexelDataSize	+= m_TexelDataSize[i];
	}

	// Change the format.
	m_PixelFormat = v8_BIT;

	return true;
}



bool NxTexture::Convert16BitPaletteFormatTo32BitPaletteFormat( void )
{
	if( !IsPaletted() || ( m_PaletteFormat != v16_BIT ))
	{
		return false;
	}

	unsigned int*	p_32bit_palette_data	= new unsigned int[m_NumPaletteEntries];

	for( int p = 0; p < m_NumPaletteEntries; ++p )
	{
		unsigned int entry	= ((unsigned short*)m_PaletteData )[p];
		entry				= (( entry & 0x1F ) << 3 ) | ((( entry >> 5 ) & 0x1F ) << 11 ) | ((( entry >> 10 ) & 0x1F ) << 19 ) | (( entry & 0x8000 ) ? 0xFF000000 : 0x00 );

		p_32bit_palette_data[p] = entry;
	}

	delete [] m_PaletteData;

	m_PaletteData			= (char*)p_32bit_palette_data;
	m_TotalPaletteDataSize	*= 2;

	m_PaletteFormat = v32_BIT;

	return true;
}



bool NxTexture::Convert24BitPaletteFormatTo32BitPaletteFormat( void )
{
	if( !IsPaletted() || ( m_PaletteFormat != v24_BIT ))
	{
		return false;
	}

	unsigned int	*p_32bit_palette_data	= new unsigned int[m_NumPaletteEntries];
	unsigned char	*p_24bit_palette_data	= (unsigned char*)m_PaletteData;
	for( int p = 0; p < m_NumPaletteEntries; ++p )
	{
		unsigned int red	= *p_24bit_palette_data++;
		unsigned int grn	= *p_24bit_palette_data++;
		unsigned int blu	= *p_24bit_palette_data++;
		unsigned int entry	= red | ( grn << 8 ) | ( blu << 16 ) | 0xFF000000;

		p_32bit_palette_data[p] = entry;
	}

	delete [] m_PaletteData;

	m_PaletteData			= (char*)p_32bit_palette_data;
	m_TotalPaletteDataSize	= ( m_TotalPaletteDataSize * 4 ) / 3;

	m_PaletteFormat = v32_BIT;

	return true;
}



bool NxTexture::Convert32BitRGBAPaletteFormatTo32BitBGRAPaletteFormat( void )
{
	if( !IsPaletted() || ( m_PaletteFormat != v32_BIT ))
	{
		return false;
	}

	unsigned int *p_32bit_palette_data	= (unsigned int*)m_PaletteData;
	for( int p = 0; p < m_NumPaletteEntries; ++p )
	{
		unsigned int old		= *p_32bit_palette_data;
		unsigned int red		= old & 0xFF;
		unsigned int grn		= ( old >> 8 ) & 0xFF;
		unsigned int blu		= ( old >> 16 ) & 0xFF;
		unsigned int alp		= ( old >> 24 ) & 0xFF;
		*p_32bit_palette_data++	= blu | ( grn << 8 ) | ( red << 16 ) | ( alp << 24 );
	}

	return true;
}


void AdjustChrominance( unsigned int &red, unsigned int &grn, unsigned int &blu, float s )
{
	float r		= ((float)red ) / 255.0f;
	float g		= ((float)grn ) / 255.0f;
	float b		= ((float)blu ) / 255.0f;

	// Approximate values for each component's contribution to luminance.
    // Based upon the NTSC standard described in ITU-R Recommendation BT.709.
	float grey	= ( r * 0.2125f ) + ( g * 0.7154f ) + ( b * 0.0721f );

	float new_r	= grey + s * ( r - grey );
	float new_g	= grey + s * ( g - grey );
	float new_b	= grey + s * ( b - grey );

	new_r		= ( new_r > 1.0f ) ? 1.0f : ( new_r < 0.0f ) ? 0.0f : new_r;
	new_g		= ( new_g > 1.0f ) ? 1.0f : ( new_g < 0.0f ) ? 0.0f : new_g;
	new_b		= ( new_b > 1.0f ) ? 1.0f : ( new_b < 0.0f ) ? 0.0f : new_b;

	red			= (int)( new_r * 255.0f );
	grn			= (int)( new_g * 255.0f );
	blu			= (int)( new_b * 255.0f );
}


bool NxTexture::Convert32BitRGBAPixelFormatTo32BitBGRAPixelFormat( void )
{
	if( m_PixelFormat != v32_BIT )
	{
		return false;
	}

	for( int i = 0; i <= m_MipLevels; ++i )
	{
		unsigned int *p_32bit_texel_data	= (unsigned int*)m_TexelData[i];
		for( int p = 0; p < ( m_Width[i] * m_Height[i] ); ++p )
		{
			unsigned int old		= *p_32bit_texel_data;
			unsigned int red		= old & 0xFF;
			unsigned int grn		= ( old >> 8 ) & 0xFF;
			unsigned int blu		= ( old >> 16 ) & 0xFF;
			unsigned int alp		= ( old >> 24 ) & 0xFF;

			AdjustChrominance( red, grn, blu, 1.115f );

			*p_32bit_texel_data++	= blu | ( grn << 8 ) | ( red << 16 ) | ( alp << 24 );
		}
	}

	return true;
}



bool NxTexture::ConvertTo32BitPixelFormat( void )
{
	// Currently only supports 16 and 32 bit palettes (if a palette is present).
	if( IsPaletted() && ( m_PaletteFormat != v32_BIT ) && ( m_PaletteFormat != v24_BIT ) && ( m_PaletteFormat != v16_BIT ))
	{
		return false;
	}

	switch( m_PixelFormat )
	{
		case v4_BIT:
		{
			// Just convert to 8 bit pixel format, and fall through to next case.
			if( !Convert4BitPixelFormatTo8BitPixelFormat())
			{
				return false;
			}
		}

		case v8_BIT:
		{
			// Reset this so we count up from scratch.
			m_TotalTexelDataSize = 0;

			// Convert to 32 bit palette if not already.
			if( m_PaletteFormat == v16_BIT )
			{
				if( !Convert16BitPaletteFormatTo32BitPaletteFormat())
				{
					return false;
				}
			}
			if( m_PaletteFormat == v24_BIT )
			{
				if( !Convert24BitPaletteFormatTo32BitPaletteFormat())
				{
					return false;
				}
			}

			for( int i = 0; i <= m_MipLevels; ++i )
			{
				unsigned int	texel_index			= 0;
				unsigned int*	p_32bit_texel_data	= new unsigned int[m_Width[i] * m_Height[i]];

				unsigned char*	p_texel_data	= (unsigned char*)m_TexelData[i];
				for( int p = 0; p < m_TexelDataSize[i]; ++p )
				{
					// Read indices.
					unsigned int entry;
					unsigned int index0					= *p_texel_data++;

					// Must be in 32 bit palette format at this stage.
					entry								= ((unsigned int*)m_PaletteData )[index0];
					p_32bit_texel_data[texel_index++]	= entry;
				}

				delete [] m_TexelData[i];

				m_TexelData[i]			= (char*)p_32bit_texel_data;
				m_TexelDataSize[i]		*= 4;
				m_TotalTexelDataSize	+= m_TexelDataSize[i];
			}

			// Now we can free up the palette data...
			delete [] m_PaletteData;
			m_PaletteData = NULL;

			m_TotalPaletteDataSize		= 0;

			// ...and change format..
			m_PixelFormat = v32_BIT;

			return true;
		}

		case v24_BIT:
		{
			// Reset this so we count up from scratch.
			m_TotalTexelDataSize = 0;

			for( int i = 0; i <= m_MipLevels; ++i )
			{
				unsigned int*	p_32bit_texel_data	= new unsigned int[m_Width[i] * m_Height[i]];
				unsigned char*	p_24bit_texel_data	= (unsigned char*)m_TexelData[i];

				for( int p = 0; p < ( m_Width[i] * m_Height[i] ); ++p )
				{
					// Read indices.
					unsigned int red		= *p_24bit_texel_data++;
					unsigned int grn		= *p_24bit_texel_data++;
					unsigned int blu		= *p_24bit_texel_data++;
					p_32bit_texel_data[p]	= red | ( grn << 8 ) | ( blu << 16 ) | 0xFF000000UL;
				}

				delete [] m_TexelData[i];

				m_TexelData[i]			= (char*)p_32bit_texel_data;
				m_TexelDataSize[i]		= ( m_TexelDataSize[i] * 4 ) / 3;
				m_TotalTexelDataSize	+= m_TexelDataSize[i];
			}

			// Change the format.
			m_PixelFormat = v32_BIT;

			return true;
		}

		default:
		{
			break;
		}
	}

	// Cannot convert.
	return false;
}
