//****************************************************************************
//* MODULE:         Tools/FindWorstCAS
//* FILENAME:       main.cpp
//* OWNER:          Gary Jesdanun
//* CREATION DATE:  7/17/2002
//****************************************************************************

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <io.h>

#define __USE_OLD_STREAMS__
#include <fstream.h>

#include <core/defines.h>

#include <gel/scripting/scriptcache.h> 
#include <gel/scripting/tokens.h>
#include <gel/scripting/init.h>
#include <gel/scripting/script.h>
#include <gel/scripting/symboltable.h>
#include <gel/scripting/parse.h>
#include <gel/scripting/checksum.h>
#include <gel/scripting/struct.h>
#include <gel/scripting/array.h>
#include <gel/scripting/utils.h>

#include <sys/config/config.h>

#include "Utility.h"
#include "VirtualFile.h"
#include "FileDatabase.h"

#define	vMAJOR_REV		1
#define vMINOR_REV	 	0
#define vEXE_NAME		"FindWorstCAS"

int g_platform = Utils::vPLATFORM_PS2;
uint32 g_current_part = 0;
uint32 g_t_part = 0;
uint32 g_t_desc_id = 0;

const int vMAX_PARTS = 100;
bool s_dependency_table[vMAX_PARTS][vMAX_PARTS];

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

int get_part_index( uint32 part_checksum )
{
	Script::CArray* pArray = Script::GetArray( "master_editable_list" );
	for ( uint32 i = 0; i < pArray->GetSize(); i++ )
	{
		Script::CStruct* pStruct = pArray->GetStructure( i );
		uint32 part_name;
		pStruct->GetChecksum( "part", &part_name, Script::ASSERT );
		if ( part_name == part_checksum )
		{
			return i;
		}
	}

	Utils::Assert( 0, "Couldn't find part" );
	return -1;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

uint32 get_part_by_index( int index )
{
	Script::CArray* pArray = Script::GetArray( "master_editable_list" );
	Script::CStruct* pStruct = pArray->GetStructure( index );

	uint32 part_name;
	pStruct->GetChecksum( "part", &part_name, Script::ASSERT );

	return part_name;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void add_dependency( uint32 part_a, uint32 part_b )
{
//	printf( "%s is dependent on %s\n", Script::FindChecksumName(part_a), Script::FindChecksumName(part_b) );

	s_dependency_table[ get_part_index(part_a) ][ get_part_index(part_b) ] = true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void print_dependency_graph()
{
	Script::CArray* pArray = Script::GetArray( "master_editable_list" );
	for ( int i = 0; i < pArray->GetSize(); i++ )
	{
		int dependency_count = 0;
		for ( int j = 0; j < pArray->GetSize(); j++ )
		{
			if ( s_dependency_table[i][j] )
			{
				dependency_count++;
			}
		}
	
		if ( dependency_count > 0 )
		{
			printf( "%s is dependent on:\n", Script::FindChecksumName( get_part_by_index(i) ) );

			for ( int j = 0; j < pArray->GetSize(); j++ )
			{
				if ( s_dependency_table[i][j] )
				{
					printf( "\t%s\n", Script::FindChecksumName( get_part_by_index(j) ) );
				}
			}
		}
	}
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void print_usage()
{
#ifdef _DEBUG
	printf( "\n%s v%d.%d (Debug) Built %s %s\n", vEXE_NAME, vMAJOR_REV, vMINOR_REV ,__DATE__, __TIME__ );
#else
	printf( "\n%s v%d.%d (Release) Built %s %s\n", vEXE_NAME, vMAJOR_REV, vMINOR_REV ,__DATE__, __TIME__ );
#endif
	printf( "Neversoft Entertainment, 2003\n" );
	printf( "\nUsage: FindWorstCas -p[p | g | x]\n" );
	printf( "Where p = PS2, g = GameCube, x = Xbox\n" );	
	printf( "Ex: FindWorstCas -pp\n" );
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

uint32 get_desc_id_from_structure( Script::CStruct* pStructure )
{
	Dbg_Assert( pStructure );

	uint32 descId;
	pStructure->GetChecksum( CRCD(0x4bb2084e,"desc_id"), &descId, true );
	return descId;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool ScriptSetPart( Script::CStruct* pParams, Script::CScript* pScript )
{
	uint32 part_name;
	pParams->GetChecksum( "part", &part_name, Script::ASSERT );

	uint32 desc_id;
	pParams->GetChecksum( "desc_id", &desc_id, Script::ASSERT );

	add_dependency( g_current_part, part_name );
	add_dependency( part_name, g_current_part );

	return true;
}

bool ScriptClearPart( Script::CStruct* pParams, Script::CScript* pScript )
{
	uint32 part_name;
	pParams->GetChecksum( "part", &part_name, Script::ASSERT );

	add_dependency( g_current_part, part_name );
	add_dependency( part_name, g_current_part );

	return true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool ScriptChecksumEquals(Script::CStruct *pParams, Script::CScript *pScript)
{
	uint32 a=0;
	pParams->GetChecksum("a",&a);
	uint32 b=0;
	pParams->GetChecksum("b",&b);

	return a==b;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool ScriptGotParam(Script::CStruct *pParams, Script::CScript *pScript)
{	
	uint32 FlagChecksum=0;
	pParams->GetChecksum(NONAME,&FlagChecksum);
	Dbg_MsgAssert(FlagChecksum,("GotParam command requires a flag name"));
	
	// Get the script's parameters, and see if they contain the aforementioned flag therein.
	Script::CStruct *pScriptParams=pScript->GetParams();
	Dbg_MsgAssert(pScriptParams,("NULL pScriptParams"));
	return pScriptParams->ContainsComponentNamed(FlagChecksum);
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool ScriptEditPlayerAppearance(Script::CStruct *pParams, Script::CScript *pScript)
{
	uint32 targetChecksum;
	pParams->GetChecksum( "target", &targetChecksum, Script::ASSERT );

	Script::CStruct* pTargetParams;
	pParams->GetStructure( "targetParams", &pTargetParams, Script::ASSERT );

	Script::RunScript( targetChecksum, pTargetParams, NULL );

	return true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

Script::CStruct* GetOptionStructure( uint32 partChecksum, uint32 lookFor, bool assertOnFail = true )
{
	Script::CArray* pArray;
	pArray = Script::GetArray( partChecksum, Script::ASSERT );

	Script::CStruct* pReturnStructure = NULL;

	if ( lookFor == 0 )
	{
		// special checksum, meaning to return NULL
		// (handy when preprocessing random descs)
		return NULL;
	}

	for ( uint32 i = 0; i < pArray->GetSize(); i++ )
	{
		Script::CStruct* pCurrDesc = pArray->GetStructure( i );
		uint32 desc_id = get_desc_id_from_structure( pCurrDesc );
		if ( desc_id == lookFor )
		{
			pReturnStructure = pCurrDesc;
			break;
		}
	}

	if ( assertOnFail )
	{
		Dbg_MsgAssert( pReturnStructure, ( "Couldn't find %s %s", Script::FindChecksumName(partChecksum), Script::FindChecksumName(lookFor) ) );
	}

	return pReturnStructure;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool ScriptGetActualCASOptionStruct(Script::CStruct *pParams, Script::CScript *pScript)
{
	uint32 partChecksum;
	pParams->GetChecksum(CRCD(0xb6f08f39,"part"), &partChecksum, Script::ASSERT);
	
	uint32 descChecksum;
	pParams->GetChecksum(CRCD(0x4bb2084e,"desc_id"), &descChecksum, Script::ASSERT);

//	printf( "%s\n", pScript->GetScriptInfo() );
//	Script::PrintContents( pParams );

	Script::CStruct* pActualStruct = GetOptionStructure( partChecksum, descChecksum );

	pScript->GetParams()->AppendStructure( pActualStruct );

	return true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool ScriptGetCurrentSkaterProfileIndex(Script::CStruct *pParams, Script::CScript *pScript)
{
	// do nothing

	return true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool ScriptGetPlayerAppearancePart(Script::CStruct *pParams, Script::CScript *pScript)
{
	pScript->GetParams()->AddChecksum( "part", g_t_part );
	pScript->GetParams()->AddChecksum( "desc_id", g_t_desc_id );

	return false;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void RegisterCFuncs()
{
	Script::CSymbolTableEntry *p_new=NULL;
	
	p_new=Script::CreateNewSymbolEntry( Crc::GenerateCRCFromString( "SetPart" ) );
	p_new->mType=ESYMBOLTYPE_CFUNCTION;
	p_new->mpCFunction=ScriptSetPart;

	p_new=Script::CreateNewSymbolEntry( Crc::GenerateCRCFromString( "ClearPart" ) );
	p_new->mType=ESYMBOLTYPE_CFUNCTION;
	p_new->mpCFunction=ScriptClearPart;

	p_new=Script::CreateNewSymbolEntry( Crc::GenerateCRCFromString( "ChecksumEquals" ) );
	p_new->mType=ESYMBOLTYPE_CFUNCTION;
	p_new->mpCFunction=ScriptChecksumEquals;

	p_new=Script::CreateNewSymbolEntry( Crc::GenerateCRCFromString( "GotParam" ) );
	p_new->mType=ESYMBOLTYPE_CFUNCTION;
	p_new->mpCFunction=ScriptGotParam;

	p_new=Script::CreateNewSymbolEntry( Crc::GenerateCRCFromString( "EditPlayerAppearance" ) );
	p_new->mType=ESYMBOLTYPE_CFUNCTION;
	p_new->mpCFunction=ScriptEditPlayerAppearance;

	p_new=Script::CreateNewSymbolEntry( Crc::GenerateCRCFromString( "GetActualCASOptionStruct" ) );
	p_new->mType=ESYMBOLTYPE_CFUNCTION;
	p_new->mpCFunction=ScriptGetActualCASOptionStruct;

	p_new=Script::CreateNewSymbolEntry( Crc::GenerateCRCFromString( "GetCurrentSkaterProfileIndex" ) );
	p_new->mType=ESYMBOLTYPE_CFUNCTION;
	p_new->mpCFunction=ScriptGetCurrentSkaterProfileIndex;

	p_new=Script::CreateNewSymbolEntry( Crc::GenerateCRCFromString( "GetPlayerAppearancePart" ) );
	p_new->mType=ESYMBOLTYPE_CFUNCTION;
	p_new->mpCFunction=ScriptGetPlayerAppearancePart;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

/*
void run_utility( AnimConverter* pAnimConverter, const char* pSrcName, CompressTable* pQCompressTable, CompressTable* pTCompressTable, int platform )
{
	pAnimConverter->Reset();

	if ( !pAnimConverter->Load( pSrcName ) )
	{
		printf( "Error!  %s is in bad format\n", pSrcName );
		exit(1);
	}

	char pDstName[_MAX_PATH];
	sprintf( pDstName, "%s.%s", pSrcName, Utils::GetPlatformExt(platform) );

	bool reverseByteOrder = false;
	switch( platform )
	{
		case Utils::vPLATFORM_PS2:
		case Utils::vPLATFORM_XBOX:
			reverseByteOrder = false;
			break;
		case Utils::vPLATFORM_NGC:
			reverseByteOrder = true;
			break;
	}

	// from now on, all of the platforms generate the compressed version
	pAnimConverter->SaveCompressed(pDstName, pQCompressTable, pTCompressTable, reverseByteOrder);

	if ( s_generate_uncompressed )
	{
		// the uncompressedStream gets exported so that
		// we can run the animation compressor on it
		strcpy( pDstName, pSrcName );
		pDstName[strlen(pDstName)-3] = 's';
		pDstName[strlen(pDstName)-2] = 'k';
		pDstName[strlen(pDstName)-1] = 'u';
		strcat( pDstName, "." );
		strcat( pDstName, Utils::GetPlatformExt(platform) );
		pAnimConverter->Save(pDstName, reverseByteOrder);
	}

	// success
	printf( "." );
}
*/

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

uint8 *LoadFileIntoMemory(const char *p_fileName)
{
	IoUtils::CVirtualInputFile theInputFile;
	int size = theInputFile.Load(p_fileName);
	if (!size)
	{
		//printf("File '%s' does not exist!\n",p_fileName);
		return NULL;
	}
	uint8* p_file = (uint8*)malloc(size);
	theInputFile.Read( (char*)p_file, size );
	return p_file;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void LoadQB(const char *p_qb_name)
{
	uint8* p_qb=LoadFileIntoMemory(p_qb_name);
	if (p_qb)
	{
		Script::ParseQB(p_qb_name,p_qb);
		// All the globals and scripts defined in the q file are now loaded, 
		// so no need to keep the qb any more.
		free(p_qb);
	}
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

int generate_part_list( const char* p_sex_filter, uint32* p_return_list )
{
//	printf( "Generating part list with filter '%s'!\n", p_sex_filter );

	int num_found = 0;

	Script::CArray* pArray = Script::GetArray( "master_editable_list" );
	for ( uint32 i = 0; i < pArray->GetSize(); i++ )
	{
		Script::CStruct* pStruct = pArray->GetStructure( i );

		if ( pStruct->ContainsComponentNamed( "use_pedpre" ) )
		{
			// skip ped parts (probably worth finding the worst case ped at some point)
			continue;
		}
	
		uint32 part_name;
		pStruct->GetChecksum( "part", &part_name, Script::ASSERT );

		bool include_in_list = true;

		// remove the part if it's the wrong sex
		int filter_matches;
		pStruct->GetInteger( p_sex_filter, &filter_matches, Script::ASSERT );
		if ( !filter_matches )
		{
			include_in_list = false;
		}

		if ( include_in_list )
		{
			*p_return_list = part_name;
			p_return_list++;
			num_found++;
		}
	}

	return num_found;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

// This function is platform-specific...  on the PS2/NGC,
// both SKIN and TEX files go on the skater geom heap, but
// on the Xbox, only the SKIN file goes on the skater geom heap
int get_mesh_size( const char* pFileName )
{
	char full_name[_MAX_PATH];

	int file_size = 0;

	switch ( g_platform )
	{
		case Utils::vPLATFORM_PS2:
		case Utils::vPLATFORM_NGC:
			{
				sprintf( full_name, "%s\\data\\%s.%s", Utils::GetProjRoot(), pFileName, Utils::GetPlatformExt(g_platform) );
				file_size += Utils::GetFileSize( full_name );
				if ( file_size == -1 )
				{
					// file not found
					char msg[_MAX_PATH];
					sprintf( msg, "CAS file %s not found", full_name );
					Utils::Assert( 0, msg );
					file_size = 0;
				}

				// now find the TEX file
				char* pExt = strstr( full_name, "." );
				if ( pExt )
				{
					sprintf( pExt, ".tex.%s", Utils::GetPlatformExt(g_platform) );
				}

				file_size += Utils::GetFileSize( full_name );
				if ( file_size == -1 )
				{
					// file not found
					char msg[_MAX_PATH];
					sprintf( msg, "TEX file %s not found", full_name );
					Utils::Assert( 0, msg );
					file_size = 0;
				}
			}
			break;
		case Utils::vPLATFORM_XBOX:
			{
				sprintf( full_name, "%s\\data\\%s.%s", Utils::GetProjRoot(), pFileName, Utils::GetPlatformExt(g_platform) );
				file_size += Utils::GetFileSize( full_name );
				if ( file_size == -1 )
				{
					// file not found
					char msg[_MAX_PATH];
					sprintf( msg, "SKIN file %s not found", full_name );
					Utils::Assert( 0, msg );
					file_size = 0;
				}
			}
			break;
		default:
			Utils::Assert( 0, "Unrecognized platform" );
	}

	return file_size;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

int generate_option_list( uint32* part_list, int num_parts, Script::CStruct** p_return_list )
{
	Utils::Assert( part_list != NULL, "No part list"  );
	Utils::Assert( p_return_list != NULL, "No return list for options" );

	int num_found = 0;
	int total_size = 0;

	for ( int i = 0; i < num_parts; i++ )
	{
		uint32 part_name = part_list[i];

//		printf( "Generating option list for '%s'!\n", Script::FindChecksumName(part_name) );
	
		Script::CArray* pPartArray = Script::GetArray( part_name, Script::NO_ASSERT );
		
		if ( pPartArray )
		{
			int max_size = -1;
			Script::CStruct* pMaxStruct = NULL;

			for ( int j = 0; j < pPartArray->GetSize(); j++ )
			{
				Script::CStruct* pSubStruct = pPartArray->GetStructure( j );

				int mesh_size = 0;
				
				// TODO:  Hidden parts shouldn't be factored into
				// the equation, because they are reserved for
				// pro skaters who are non-editable
				if ( pSubStruct->ContainsComponentNamed( "hidden" ) )
				{
					continue;
				}

				const char* pMeshName;

				//--------------------------------------------
				int first_mesh_size = 0;
				int first_facemapped_mesh_size = 0;
				if ( pSubStruct->GetString( "mesh", &pMeshName, Script::NO_ASSERT ) )
				{
					first_mesh_size = get_mesh_size( pMeshName );
				}
				if ( pSubStruct->GetString( "mesh_if_facemapped", &pMeshName, Script::NO_ASSERT ) )
				{
					first_facemapped_mesh_size = get_mesh_size( pMeshName );
				}

				// take the larger of the two mesh sizes
				mesh_size += ( ( first_mesh_size > first_facemapped_mesh_size ) ? first_mesh_size : first_facemapped_mesh_size );
				//--------------------------------------------

				if ( pSubStruct->GetString( "mesh1", &pMeshName, Script::NO_ASSERT ) )
				{
					mesh_size += get_mesh_size( pMeshName );
				}

				if ( pSubStruct->GetString( "mesh2", &pMeshName, Script::NO_ASSERT ) )
				{
					mesh_size += get_mesh_size( pMeshName );
				}

				if ( pSubStruct->GetString( "mesh3", &pMeshName, Script::NO_ASSERT ) )
				{
					mesh_size += get_mesh_size( pMeshName );
				}

				if ( pSubStruct->GetString( "mesh4", &pMeshName, Script::NO_ASSERT ) )
				{
					mesh_size += get_mesh_size( pMeshName );
				}

				if ( mesh_size )
				{
					*p_return_list = pSubStruct;
					p_return_list++;
					num_found++;
				}

				if ( mesh_size > max_size )
				{
					pMaxStruct = pSubStruct;
					max_size = mesh_size;
				}
			}

			if ( max_size > 0 )
			{
				uint32 max_desc_id;
				pMaxStruct->GetChecksum( "desc_id", &max_desc_id, Script::ASSERT );
				printf( "\t%s = { desc_id=#\"%s\" } // (%d bytes) (%d options)\n", Script::FindChecksumName(part_name), Script::FindChecksumName(max_desc_id), max_size, pPartArray->GetSize() );
				total_size += max_size;
			}
		}
	}

//	printf( "Combined largest option is approximately %d bytes.\n", total_size );

	return num_found;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void lowercase( char* pSrc )
{
	while ( *pSrc )
	{
		if ( *pSrc >= 'A' && *pSrc <= 'Z' )
		{
			*pSrc = *pSrc - 'A' + 'a';
		}
		pSrc++;
	}
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool tag_dependencies( uint32 part_a, uint32 part_b )
{
	g_current_part = part_a;

	Script::CArray* pPartArray = Script::GetArray( part_a, Script::NO_ASSERT );
		
	if ( pPartArray )
	{
		for ( int j = 0; j < pPartArray->GetSize(); j++ )
		{
			Script::CStruct* pSubStruct = pPartArray->GetStructure( j );

			Script::SStructScript* pStructScript = new Script::SStructScript;
			
			if ( pSubStruct->GetScript( "disqualify_script", pStructScript, Script::NO_ASSERT ) )
			{
				// construct a script and run it
				Script::CScript* pScript = new Script::CScript;
				pScript->SetScript( pStructScript, NULL );	
				int ret_val;
				while ((ret_val = pScript->Update()) != Script::ESCRIPTRETURNVAL_FINISHED)
				{
				// Script must not get blocked, otherwise it'll hang in this loop forever.
					Dbg_MsgAssert(ret_val != Script::ESCRIPTRETURNVAL_BLOCKED,("\n%s\nScript got blocked when being run by RunScript.",pScript->GetScriptInfo()));
				}
				delete pScript;
			}

			delete pStructScript;
		}
	}	

	return true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void handle_cas_dependency_pairs( uint32* part_list, int num_parts )
{
	Script::CArray* pDependencyArray = Script::GetArray( "cas_dependency_pairs", Script::ASSERT );

	if ( pDependencyArray )
	{
		for ( int i = 0; i < pDependencyArray->GetSize(); i++ )
		{
			Script::CStruct* pDependencyStruct = pDependencyArray->GetStructure( i );

			uint32 flag;
			pDependencyStruct->GetChecksum("flag",&flag,Script::ASSERT);

			uint32 part;
			pDependencyStruct->GetChecksum("part",&part,Script::ASSERT);

			for ( int j = 0; j < num_parts; j++ )
			{
				Script::CArray* pSubArray = Script::GetArray( part_list[j], Script::ASSERT );
				for ( int k = 0; k < pSubArray->GetSize(); k++ )
				{
					Script::CStruct* pSubStruct = pSubArray->GetStructure( k );
					if ( pSubStruct->ContainsFlag( flag ) )
					{
						add_dependency( part_list[j], part );
						add_dependency( part, part_list[j] );
					}
				}
			}
		}
	}
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void create_dependency_graph( uint32* part_list, int num_parts )
{
	for ( int i = 0; i < num_parts; i++ )
	{
//		printf( "Generating dependency graph for '%s'!\n", Script::FindChecksumName(part_list[i]) );

		for ( int j = 0; j < num_parts; j++ )
		{
//			for now, don't do dependencies...  it's a tricky thing with all the disqualifications...
//			tag_dependencies( part_list[i], part_list[j] );
		}
	}

//	handle_cas_dependency_pairs( part_list, num_parts );

//	print_dependency_graph();
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

int main( int argc, char* argv[] )
{
	char* pArg = NULL;

	if ( pArg = Utils::TestArg( argc, argv, '?' ) )
	{
		print_usage();
		return 0;
	}

	if ( pArg = Utils::TestArg( argc, argv, 'P' ) )
	{
		switch ( *pArg )
		{
			case 'P':
			case 'p':
				g_platform = Utils::vPLATFORM_PS2;
				break;
			case 'G':
			case 'g':
				g_platform = Utils::vPLATFORM_NGC;
				break;
			case 'X':
			case 'x':
				g_platform = Utils::vPLATFORM_XBOX;
				break;
		}
	}

	// check for debug mode
	bool debugOutput = false;
	if ( pArg = Utils::TestArg( argc, argv, 'D' ) )
	{
		debugOutput = true;
	}

	// A few essential calls for the script code.
	Config::Init(0,NULL);
	Mem::Manager::sSetUp();
	Script::CScriptCache::Create();

	Script::CreateSymbolHashTable();
	Script::AllocatePools();
	RegisterCFuncs();

#if 0
	// TODO:  The following asserts due to a possible undeleted pointer...		// when i have more time, i'll track down what's the deal
	char scripts_path[_MAX_PATH];
	sprintf( scripts_path, "%s\\data\\scripts", Utils::GetProjRoot() );
	IoUtils::CFileDatabase theFileDatabase;
	theFileDatabase.RefreshFileList( scripts_path );
	theFileDatabase.IterateStart();
	IoUtils::SFileInfo* pInfo = NULL;
	while ( pInfo = theFileDatabase.IterateNext() )
	{
		lowercase( pInfo->pFileName );
		if ( strstr( pInfo->pFileName, ".qb" ) )
		{
			LoadQB( pInfo->pFileName );
		}
	}
#else
	// TODO:  Read list of QB files in from a text file (or from qdir.txt)
	char* p_all_qb_names[] = {
		"game\\cas_parts.qb",
		"game\\cas_skater_m.qb",
		"game\\cas_skater_f.qb",
		"game\\cas_skater_shared.qb",
		"game\\cas_logos.qb",
		"game\\decks.qb",
		"game\\deck_layers.qb",
		"game\\skater\\iniskater_exceptions.qb",
		NULL
	};

	char** pp_current_qb_name = p_all_qb_names;
	while ( *pp_current_qb_name )
	{
		char full_qb_name[_MAX_PATH];
		sprintf( full_qb_name, "%s\\data\\scripts\\%s", Utils::GetProjRoot(), *pp_current_qb_name );
		LoadQB(full_qb_name);
		pp_current_qb_name++;
	}
#endif

	// combine the hat and the helmet lists,
	// because they're really the same thing.

	// remove all the non-flagged items
	// that are small

	// actually create an appearance
	// and make it

	const int MAX_PARTS = 100;
	uint32 part_list[MAX_PARTS];
	int num_parts = 0;

	const int MAX_OPTIONS = 2000;
	Script::CStruct* option_list[MAX_OPTIONS];
	int num_options = 0;

	printf( "Finding worst-case male skater...\n" );
	num_parts = generate_part_list( "male", &part_list[0] );
	create_dependency_graph( part_list, num_parts );
	printf( "\t%s = { desc_id=#\"%s\" } // (%d bytes)\n", "body", "MaleBody", get_mesh_size("models//skater_male//skater_male.skin") );
	num_options = generate_option_list( part_list, num_parts, &option_list[0] );

	printf( "Finding worst-case female skater...\n" );
	num_parts = generate_part_list( "female", &part_list[0] );
	create_dependency_graph( part_list, num_parts );
	printf( "\t%s = { desc_id=#\"%s\" } // (%d bytes)\n", "body", "FemaleBody", get_mesh_size("models//skater_female//skater_female.skin") );
	num_options = generate_option_list( part_list, num_parts, &option_list[0] );

	printf( "Success!\n" );

	return 0;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/
