#include "stdafx.h"
#include <stdio.h>
#include <conio.h>
#include <windows.h>

#pragma comment (lib, "c:/skate5/tools/lib/ps2tmapi.lib")

#include "c:/skate5/tools/include/ps2tmapi/ps2tmapi.h"

#define LOG_INFO_LOCATION 0x06E00000

typedef UINT64 uint64;

/////////////////////////////////////////////////////////////////////////////////////////
// This lot should be in a header file shared with the game code ...

namespace Tmr
{
typedef uint64 CPUCycles;				// CPUCycles, also needs higher resolution
}

// Chosen just to make sizeof(SLogEntry) a nice round 128
#define MAX_MESSAGE_LENGTH 107
struct SLogEntry
{
	Tmr::CPUCycles mCPUTime;
	const char *mpSourceFileName;
	const char *mpFunctionName;
	int mLineNumber;
	char mpMessage[MAX_MESSAGE_LENGTH+1];
};

struct SLogBufferInfo
{
	// Pointer to the start of the big log buffer, which will be in the debug heap somewhere. 
	SLogEntry *mpBuffer;
	// The total number of entries in the buffer.
	int mTotalEntries;
	// The number of entries written to so far. This starts at 0 when the game starts,
	// and increases till it hits mTotalEntries, then stays at that value.
	int mNumEntries;
	
	// Points to just after the last entry that got added to the buffer.
	// So it may not point to a valid SLogEntry, for example when the buffer is filling
	// up, or when pTop points to the end of pBuffer.
	SLogEntry *mpTop;
};
/////////////////////////////////////////////////////////////////////////////////////////

SLogBufferInfo gLogInfo;

SNPS2TargetInfo gTargetInfo;
bool gFoundTarget=false;
const char *gpTargetName="";

void CleanUp()
{
	/*
	if (gFoundTarget)
	{
		printf("Disconnecting from '%s'\n",gpTargetName);
		if (SNPS2Disconnect(gTargetInfo.hTarget)<0)
		{
			printf("SNPS2Disconnect failed !\n");
		}
	}
	*/

    SNPS2UnInitialise();
}

void Assert(int condition, char *p_message)
{
	if (!condition)
	{
		printf("DumpLog assertion failed:\n%s",p_message);
		CleanUp();
		exit(1);
	}
}

#define MSGASSERT_BUFFER_SIZE 1000
static char spMsgAssertBuffer[MSGASSERT_BUFFER_SIZE];
void MsgAssert_PrintfIntoBuffer(const char* p_text, ...)
{
	spMsgAssertBuffer[0]=0;
	if (p_text)
	{
		va_list args;
		va_start( args, p_text );

		sprintf( spMsgAssertBuffer, p_text, args);
		
		va_end( args );
	}
}

#define Dbg_MsgAssert( _c, _params )							\
																\
if( !( _c ))													\
{																\
	MsgAssert_PrintfIntoBuffer _params;							\
	Assert( 0, spMsgAssertBuffer );								\
}

int EnumCallBack(SNPS2TargetInfo* p_targetInfo)
{
	if (strcmp(p_targetInfo->pszName,gpTargetName)==0)
	{
		gTargetInfo=*p_targetInfo;
		gFoundTarget=true;
		// Return 1 to terminate enumeration since we've found a match.
		return 1;
	}
	// Return 0 to continue enumeration.
    return 0;
}

int ListTargets(SNPS2TargetInfo* p_targetInfo)
{
	printf("'%s'\n",p_targetInfo->pszName);
	// Return 0 to continue enumeration.
    return 0;
}

void PrintEntry(SLogEntry *p_entry)
{
	static Tmr::CPUCycles s_last_time=0;

	SLogEntry entry;
    Assert(SNPS2GetMemory(gTargetInfo.hTarget,
						  UI_EECPU,
						  MEMSPACE_MAIN,
						  (unsigned long)p_entry,
						  sizeof(SLogEntry),
						  &entry)>=0,"SNPS2GetMemory failed");

	#define STRING_BUF_SIZE 200
	char p_function_name_buf[STRING_BUF_SIZE];
	char p_source_file_name_buf[STRING_BUF_SIZE];

    Assert(SNPS2GetMemory(gTargetInfo.hTarget,
						  UI_EECPU,
						  MEMSPACE_MAIN,
						  (unsigned long)entry.mpFunctionName,
						  STRING_BUF_SIZE,
						  p_function_name_buf)>=0,"SNPS2GetMemory failed");

    Assert(SNPS2GetMemory(gTargetInfo.hTarget,
						  UI_EECPU,
						  MEMSPACE_MAIN,
						  (unsigned long)entry.mpSourceFileName,
						  STRING_BUF_SIZE,
						  p_source_file_name_buf)>=0,"SNPS2GetMemory failed");

	printf("Clock ticks = %lu, ",entry.mCPUTime);
	printf("elapsed time = %lu ticks ",entry.mCPUTime-s_last_time);
	double milliseconds=(double)(entry.mCPUTime-s_last_time)/150000.0f;
	printf("(=%.3f ms",milliseconds);
	printf(", %.2f%% frame)\n",100.0f*milliseconds/16.66666666f);
	
	printf("Line %d of %s\n",entry.mLineNumber,p_source_file_name_buf);
	printf("%s\n",p_function_name_buf);
	if (entry.mpMessage[0])
	{
		printf("%s\n",entry.mpMessage);
	}

	s_last_time=entry.mCPUTime;
}

int main(int argc, char* argv[])
{
	char *p_path=getenv("SKATE4_PATH");
	Assert(p_path!=NULL,"DumpLog requires that the environment variable SKATE4_PATH be set.");

	gpTargetName=getenv("DUMPLOG_TARGET");
	Assert(gpTargetName!=NULL,"DumpLog requires that the environment variable DUMPLOG_TARGET be set.");

	Assert(SNPS2Initialise()>=0,"SNPS2 failed to initialise");
    Assert(SNPS2EnumTargets((SNPS2EnumTargetsCB)EnumCallBack)>=0,"Failed to enumerate targets");

	if (!gFoundTarget)
	{
		printf("The only targets found were:\n");
		SNPS2EnumTargets((SNPS2EnumTargetsCB)ListTargets);
		printf("Could not find the target '%s'\n",gpTargetName);
		Assert(0,"Could not find target");
	}

    printf("Connecting to '%s' ... \n", gpTargetName);
    Assert(SNPS2Connect(gTargetInfo.hTarget, NULL)>=0,"Failed to connect !");

	//printf("sizeof(uint64)=%d\n",sizeof(uint64));
	//printf("sizeof(SLogEntry)=%d\n",sizeof(SLogEntry));

    Assert(SNPS2GetMemory(gTargetInfo.hTarget,
						  UI_EECPU,
						  MEMSPACE_MAIN,
						  LOG_INFO_LOCATION,
						  sizeof(SLogBufferInfo), &gLogInfo)>=0,"SNPS2GetMemory failed");
	
	//printf("%d,%d,%x,%x\n",gLogInfo.mNumEntries,gLogInfo.mTotalEntries,gLogInfo.mpBuffer,gLogInfo.mpTop);

	int n=10;

	if (n>gLogInfo.mNumEntries)
	{
		n=gLogInfo.mNumEntries;
	}

	SLogEntry *p_entry=gLogInfo.mpTop-n;
	if (p_entry < gLogInfo.mpBuffer)
	{
		p_entry+=gLogInfo.mTotalEntries;
	}

	for (int i=0; i<n; ++i)
	{
		PrintEntry(p_entry);
		printf("\n");

		++p_entry;
		if (p_entry >= gLogInfo.mpBuffer+gLogInfo.mTotalEntries)
		{
			p_entry=gLogInfo.mpBuffer;
		}
	}

    CleanUp();
	return 0;
}




/*
// -----------------------------------------------------------------------------
//   Copyright (c) 2001 SN Systems Software Limited
//   All Rights Reserved
//
//   Author:  Dan Farley
//     File:  ps2api.cpp
//     Desc:  Simple command-line tool to test PS2TM SDK features
// -----------------------------------------------------------------------------

#include <stdio.h>
#include <conio.h>
#include <windows.h>
#include <vector>
using namespace std;

#pragma comment (lib, "../dll/ps2tmapi.lib")

#include "../ps2tmapi.h"

// Global defines
#define UPDATE_NOTIFY  (WM_USER+0)
#define UPDATE_TTY     (WM_USER+1)
#define BUFFER_SIZE    (32768)

// Forward declaration of target functions
bool SelectTarget(void);
bool Connect(void);
bool Disconnect(void);
bool LoadElf(void);
bool ResetTarget(void);
bool StartTarget(void);
bool StopTarget(void);
bool GetMemory(void);
bool SetMemory(void);
bool GetRegisters(void);
bool SetRegisters(void);
bool ReceiveTTY(void);
bool ReceiveUPD(void);

// Global variables
vector<SNPS2TargetInfo*> g_Targets;
UINT  g_NumTargets = 0;
UINT  g_CurrTarget = 0;
HWND  g_hWndHidden = 0;
BYTE* g_pBuffer = NULL;

// -----------------------------------------------------------------------------
//   Func: HiddenWndProc
//   Desc: Hidden window procedure, receives target notifications
// -----------------------------------------------------------------------------

LRESULT CALLBACK HiddenWndProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
    switch(uMsg)
    {
        case UPDATE_NOTIFY:
            printf("Unit status change, Status=%d\n", (UINT)lParam);
        return 0;

        case UPDATE_TTY:
        {
            HTARGET  hTarget = (HTARGET) HIWORD(wParam);
            UINT     nStream = LOWORD(wParam);
            int      nLen = (int)lParam;
            char*    g_pBuffer = new char[nLen+1];
            UINT     recv_size = 0;
            UINT     nResult;

            nResult = SNPS2GetTTY(hTarget, hwnd, nStream, nLen, g_pBuffer, &recv_size);

            if(SN_SUCCEEDED( nResult ))
            {
                g_pBuffer[nLen] = '\0';
                printf("%s", g_pBuffer);
            }
            else
            {
                char* pszMsg;
                SNTranslateError(nResult, &pszMsg);
                printf("GetTTY failed, Error = %s\n", pszMsg);
                SNFreeErrorStr(pszMsg);
            }

            delete[] g_pBuffer;
        }
        return 0;

        default:
        break;
    }

    return DefWindowProc(hwnd, uMsg, wParam, lParam);
}

// -----------------------------------------------------------------------------
//   Func: CreateHiddenWnd
//   Desc: Creates an invisible window to receive update notifications
// -----------------------------------------------------------------------------

bool CreateHiddenWnd()
{
    // CREATE THE HIDDEN WINDOW
	WNDCLASS wc;
    HINSTANCE hInstance = GetModuleHandle(NULL);

    memset(&wc, 0, sizeof(WNDCLASS));
	wc.lpszClassName = "HiddenClass";
	wc.lpfnWndProc = HiddenWndProc;
	wc.hInstance = hInstance;

	RegisterClass(&wc);

    g_hWndHidden = CreateWindow("HiddenClass", "HiddenWindow", 0, 0, 0, 0, 0,
        NULL, NULL, hInstance, 0);

    return (g_hWndHidden != NULL);
}

// -----------------------------------------------------------------------------
//   Func: EnumCallBack
//   Desc: Callback function to enumerate available targets
// -----------------------------------------------------------------------------

int EnumCallBack(SNPS2TargetInfo* pTargetInfo)
{
    SNPS2TargetInfo ti;
    SNPS2TargetInfo* pti = new SNPS2TargetInfo;

    if(pti)
    {
        ti.hTarget = pTargetInfo->hTarget;
        ti.nFlags = SNPS2_TI_TARGETID;

        if(SN_S_OK == SNPS2GetTargetInfo(&ti))
        {
            // STORE TARGET PARAMETERS
            pti->hTarget = pTargetInfo->hTarget;
            pti->pszName = strdup(ti.pszName);
            pti->pszHomeDir = strdup(ti.pszHomeDir);
            pti->pszFSDir = strdup(ti.pszFSDir);

            // OUTPUT SOME DEBUG INFO
            printf("Target:    %d\n"
                   "Name:      %s\n"
                   "HomeDir:   %s\n"
                   "RootDir    %s\n",
                   pti->hTarget, pti->pszName, pti->pszHomeDir, pti->pszFSDir);

            // STORE THIS TARGET
            g_Targets.push_back(pti);
            g_NumTargets++;
        }
        else
        {
            // TERMINATE ENUMERATION
            return 1;
        }
    }

    // CARRY ON WITH ENUMERATION
    return 0;
}

// -----------------------------------------------------------------------------
//   Func: PrintMem
//   Desc: Displays a block of memory with corresponding ANSI characters
// -----------------------------------------------------------------------------

void PrintMem(BYTE* g_pBuffer, DWORD dwLength)
{
    int i = 0;
    char ch;
    DWORD dwBytesRemaining = dwLength;

    while(dwBytesRemaining--)
    {
        printf("%02X ", g_pBuffer[i++]);
        if(!(dwBytesRemaining % 16))
        {
            printf("  [ ");

            for(int j = 16; j > 0; j--)
            {
                ch = g_pBuffer[i - j - 2];

                if((ch < 0x20) || (ch > 0x7F))
                    printf(".");
                else
                    printf("%c", ch);
            }

            printf(" ]\n");
        }
    }
}

// -----------------------------------------------------------------------------
//   Func: main
//   Desc: Main entry point for the application
// -----------------------------------------------------------------------------

void main()
{
    int nChoice = 1;
    g_pBuffer = new BYTE[BUFFER_SIZE];

    // INITIALISE PS2TMAPI
    if(SN_FAILED( SNPS2Initialise() ))
    {
        printf("Failed to initialise PS2TM!\n");
        exit(0);
    }
    printf("Initialised PS2TM\n");

    // CREATE A HIDDEN WINDOW TO RECEIVE NOTIFICATIONS
    if(!CreateHiddenWnd())
    {
        printf("Failed to create hidden window!\n");
        exit(0);
    }
    printf("Created hidden window\n");

    // GET NUMBER OF TARGETS
    if(SN_FAILED( SNPS2GetNumTargets(&g_NumTargets) ))
    {
        printf("Failed to get number of targets!\n");
        exit(0);
    }

    // CHECK THERE IS AT LEAST ONE TARGET
    if(g_NumTargets <= 0)
    {
        printf("Couldn't find any targets!\n");
        exit(0);
    }

    // ENUMERATE AVAILABLE TARGETS
    if(SN_FAILED( SNPS2EnumTargets((SNPS2EnumTargetsCB)EnumCallBack) ))
    {
        printf("Failed to enumerate targets!\n");
        exit(0);
    }

    // ENTER MAIN MENU LOOP
    while(nChoice > 0)
    {
        printf("\nMain Menu:\n");
        printf(" [1] Select Target\n");
        printf(" [2] Connect to Target\n");
        printf(" [3] Disconnect from Target\n");
        printf(" [4] Load and Run ELF\n");
        printf(" [5] Reset Target\n");
        printf(" [6] Start Target\n");
        printf(" [7] Stop Target\n");
        printf(" [8] Get Memory\n");
        printf(" [9] Set Memory\n");
        printf("[10] Get Register(s)\n");
        printf("[11] Set Register(s)\n");
        printf("[12] Receive TTY Output\n");
        printf("[13] Receive Update Notifications\n");

        printf("\nChoose a number: ");
        scanf("%d", &nChoice);
        printf("\n");

        switch(nChoice)
        {
            case 0:
                printf("\nBye bye...\n");
            break;

            case 1: SelectTarget(); break;
            case 2: Connect(); break;
            case 3: Disconnect(); break;
            case 4: LoadElf(); break;
            case 5: ResetTarget(); break;
            case 6: StartTarget(); break;
            case 7: StopTarget(); break;
            case 8: GetMemory(); break;
            case 9: SetMemory(); break;
            case 10: GetRegisters(); break;
            case 11: SetRegisters(); break;
            case 12: ReceiveTTY(); break;
            case 13: ReceiveUPD(); break;

            default:
                nChoice = 0;
                printf("\nInvalid number, try again\n");
        }
    }

    // UNINITIALISE PS2TMAPI
    SNPS2UnInitialise();
    printf("Uninitialised PS2TM\n");
    delete[] g_pBuffer;
}

// -----------------------------------------------------------------------------
//   Func: FailCommand
//   Desc: Displays an error message for failed commands
// -----------------------------------------------------------------------------

void FailCommand(SNRESULT snResult)
{
    char* pszError = NULL;

    if(SN_FAILED( SNTranslateError(snResult, &pszError) ))
    {
        printf("Unknown error occurred\n");
        return;
    }

    printf("%s\n", pszError);
}

// -----------------------------------------------------------------------------
//   Func: SelectTarget
//   Desc: List available targets and prompt user to select one
// -----------------------------------------------------------------------------

bool SelectTarget(void)
{
    UINT nNumTargets = 0;

    // GET NUMBER OF TARGETS
    if(SN_FAILED( SNPS2GetNumTargets(&nNumTargets) ))
    {
        printf("Failed to get number of targets!\n");
        return false;
    }
    printf("Number of targets = %d\n", nNumTargets);

    // ENUM TARGETS
    printf("Enumerating targets...\n");
    if(SN_FAILED( SNPS2EnumTargets((SNPS2EnumTargetsCB)EnumCallBack) ))
    {
        printf("Failed!\n");
        return false;
    }
    printf("Complete\n");

    printf("Choose a target: ");
    scanf("%d", &g_CurrTarget);

    if(g_CurrTarget < 0 || g_CurrTarget > nNumTargets)
    {
        printf("Invalid target, setting to 0\n");
        g_CurrTarget = 0;
    }

    return true;
}

// -----------------------------------------------------------------------------
//   Func: Connect
//   Desc: Connect to the currently selected target
// -----------------------------------------------------------------------------

bool Connect(void)
{
    // CONNECTING
    printf("Connecting to %s... ", g_Targets[g_CurrTarget]->pszName);

    SNRESULT snResult = SNPS2Connect(g_Targets[g_CurrTarget]->hTarget, NULL);

    if(SN_FAILED(snResult))
    {
        FailCommand(snResult);
        return false;
    }

    printf("Complete\n");
    return true;
}

// -----------------------------------------------------------------------------
//   Func: Disconnect
//   Desc: Disconnect from the currently selected target
// -----------------------------------------------------------------------------

bool Disconnect(void)
{
    // DISCONNECTING
    printf("Disconnecting from %s... ", g_Targets[g_CurrTarget]->pszName);

    SNRESULT snResult = SNPS2Disconnect(g_Targets[g_CurrTarget]->hTarget);

    if(SN_FAILED(snResult))
    {
        printf("Failed!\n");
        return false;
    }

    printf("Complete\n");
    return true;
}

// -----------------------------------------------------------------------------
//   Func: LoadElf
//   Desc: Download .elf file to target and execute
// -----------------------------------------------------------------------------

bool LoadElf(void)
{
    char chFile[_MAX_PATH];

    printf("Enter filename: ");
    scanf("%s", chFile);

    // LOAD AND RUN ELF
    printf("Loading %s... ", chFile);
    if(SN_FAILED( SNPS2LoadRunELF(g_Targets[g_CurrTarget]->hTarget, chFile) ))
    {
        printf("Failed!\n");
        return false;
    }

    printf("Complete\n");
    return true;
}

// -----------------------------------------------------------------------------
//   Func: ResetTarget
//   Desc: Reset the target
// -----------------------------------------------------------------------------

bool ResetTarget(void)
{
    // RESET TARGET
    printf("Resetting target... ");
    if(SN_FAILED( SNPS2Reset(g_Targets[g_CurrTarget]->hTarget, 0xffffffffffffffffui64, 0xffffffffffffffffui64) ))
    {
        printf("Failed!\n");
        return false;
    }

    printf("Complete\n");
    return true;
}

// -----------------------------------------------------------------------------
//   Func: StartTarget
//   Desc: Begin execution on the target
// -----------------------------------------------------------------------------

bool StartTarget(void)
{
    // START TARGET
    if(SN_FAILED( SNPS2StartTarget(g_Targets[g_CurrTarget]->hTarget, UI_EECPU) ))
    {
        printf("Failed to start target!\n");
        return false;
    }

    printf("Target started\n");
    return true;
}

// -----------------------------------------------------------------------------
//   Func: StartTarget
//   Desc: Suspend execution of the target
// -----------------------------------------------------------------------------

bool StopTarget(void)
{
    // STOP TARGET
    if(SN_FAILED( SNPS2StopTarget(g_Targets[g_CurrTarget]->hTarget, UI_EECPU) ))
    {
        printf("Failed to stop target!\n");
        return false;
    }

    printf("Target stopped\n");
    return true;
}

// -----------------------------------------------------------------------------
//   Func: GetMemory
//   Desc: Query target for memory, prompt user for address and length
// -----------------------------------------------------------------------------

bool GetMemory(void)
{
    DWORD dwAddr = 0;
    DWORD dwLength = 0;

    printf("Enter address: ");
    scanf("%x", &dwAddr);
    printf("Enter length: ");
    scanf("%d", &dwLength);

    memset(g_pBuffer, 42, dwLength);

    // GET TARGET MEMORY
    printf("Trying GetMemory... ");
    if(SN_FAILED( SNPS2GetMemory(g_Targets[g_CurrTarget]->hTarget, UI_EECPU, MEMSPACE_MAIN, dwAddr, dwLength, g_pBuffer) ))
    {
        printf("Failed!\n");
        return false;
    }
    printf("Complete\n");
    PrintMem(g_pBuffer, dwLength);

    return true;
}

// -----------------------------------------------------------------------------
//   Func: SetMemory
//   Desc: Alter target memory, prompt user for address and length
// -----------------------------------------------------------------------------

bool SetMemory(void)
{
    DWORD dwAddr = 0;
    DWORD dwLength = 0;
    DWORD dwValue = 0;

    printf("Enter address: ");
    scanf("%x", &dwAddr);
    printf("Enter length: ");
    scanf("%d", &dwLength);
    printf("Enter value: ");
    scanf("%x", &dwValue);

    memset(g_pBuffer, dwValue, dwLength);

    // SET TARGET MEMORY
    printf("Trying SetMemory... ");
    if(SN_FAILED( SNPS2SetMemory(g_Targets[g_CurrTarget]->hTarget, UI_EECPU, MEMSPACE_MAIN, dwAddr, dwLength, g_pBuffer) ))
    {
        printf("Failed!\n");
        return false;
    }
    printf("Complete\n");
    PrintMem(g_pBuffer, dwLength);

    return true;
}

// -----------------------------------------------------------------------------
//   Func: GetRegisters
//   Desc: Query register value(s), prompt user for address and length
// -----------------------------------------------------------------------------

bool GetRegisters(void)
{
    const UINT nRegNum = 10;

    UINT nRegWidth = 16; // Bytes
    UINT nRegMem = nRegNum * nRegWidth;

    UINT Registers[nRegNum] = 
    {
        REG_8, REG_9, REG_10, REG_11, REG_12, REG_13, REG_14, REG_15, REG_16, REG_17
    };

    BYTE* RegBuffer = new BYTE[nRegMem];

    // GET REGISTERS
    printf("Trying GetRegisters... ");
    if(SN_FAILED( SNPS2GetRegisters(g_Targets[g_CurrTarget]->hTarget, UI_EECPU, nRegNum, (UINT*)&Registers, RegBuffer) ))
    {
        printf("Failed!\n");
        return false;
    }
    printf("Complete\n");
    PrintMem(RegBuffer, nRegMem);

    delete [] RegBuffer;

    return true;
}

// -----------------------------------------------------------------------------
//   Func: SetRegisters
//   Desc: Alter register value(s), prompt user for address and length
// -----------------------------------------------------------------------------

bool SetRegisters(void)
{
    const UINT nRegNum = 10;

    UINT nRegWidth = 16; // Bytes
    UINT nRegMem = nRegNum * nRegWidth;

    UINT Registers[nRegNum] = 
    {
        REG_8, REG_9, REG_10, REG_11, REG_12, REG_13, REG_14, REG_15, REG_16, REG_17
    };

    BYTE* RegBuffer = new BYTE[nRegMem];

    memset(RegBuffer, 0xAB, nRegMem);

    // SET REGISTERS
    printf("Trying SetRegisters... ");
    if(SN_FAILED( SNPS2SetRegisters(g_Targets[g_CurrTarget]->hTarget, UI_EECPU, nRegNum, (UINT*)&Registers, RegBuffer) ))
    {
        printf("Failed!\n");
        return false;
    }
    printf("Complete\n");
    PrintMem(RegBuffer, nRegMem);

    delete [] RegBuffer;

    return true;
}

// -----------------------------------------------------------------------------
//   Func: ReceiveTTY
//   Desc: Request TTY notification, then enter message loop reciving output
// -----------------------------------------------------------------------------

bool ReceiveTTY(void)
{
    PTTYSTREAM pTTYStreams;
    UINT nNumTTYStreams;

    // GET NUMBER OF AVAILABLE TTY CHANNELS
    if(SN_FAILED( SNPS2ListTTYStreams(g_Targets[g_CurrTarget]->hTarget, &nNumTTYStreams, NULL) ))
    {
        printf("Failed to get count of TTY streams!\n");
        return false;
    }
    pTTYStreams = new TTYSTREAM[nNumTTYStreams];

    // GET LIST OF AVAILABLE TTY CHANNELS
    if(SN_FAILED( SNPS2ListTTYStreams(g_Targets[g_CurrTarget]->hTarget, &nNumTTYStreams, pTTYStreams) ))
    {
        printf("Failed to list TTY streams!\n");
        return false;
    }

    printf("Available TTY streams:\n");

    for(int i = 0; i < nNumTTYStreams; i++)
    {
        printf("%d: %s\n", pTTYStreams[i].nStreamIdx, pTTYStreams[i].szName);
    }

    // REQUEST TTY NOTIFICATION FOR ALL STREAMS
    for(i = 0; i < nNumTTYStreams; i++)
    {
        if(SN_FAILED( SNPS2RequestTTYNotification(g_Targets[g_CurrTarget]->hTarget, pTTYStreams[i].nStreamIdx, g_hWndHidden, UPDATE_TTY) ))
        {
            printf("Failed to register for TTY notifications on %s!\n", pTTYStreams[i].szName);
            return false;
        }
    }

    MSG msg;
    printf("Entering message loop, Press any key to quit...\n");
    while(1)
    {
        if(PeekMessage(&msg, 0, 0, 0, PM_REMOVE) != 0)
        {
            TranslateMessage(&msg);
            DispatchMessage(&msg);
        }
        if(_kbhit()) break;
    }

    return true;
}

// -----------------------------------------------------------------------------
//   Func: ReceiveUPD
//   Desc: Request update notification, then enter message loop reciving output
// -----------------------------------------------------------------------------

bool ReceiveUPD(void)
{
    // REQUEST STATUS NOTIFICATION
    if(SN_FAILED( SNPS2RequestUPDNotification(g_Targets[g_CurrTarget]->hTarget, g_hWndHidden, UPDATE_NOTIFY) ))
    {
        printf("Failed to register for update notifications!\n");
        return false;
    }

    MSG msg;
    printf("Entering message loop, Press any key to quit...\n");
    while(1)
    {
        if(PeekMessage(&msg, 0, 0, 0, PM_REMOVE) != 0)
        {
            TranslateMessage(&msg);
            DispatchMessage(&msg);
        }
        if(_kbhit()) break;
    }

    return true;
}
*/