/*
	AnimExportFmt.cpp
	Animation Export Format for THPS4/5
*/

typedef unsigned long uint32;
#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <assert.h>
#include "../include/AnimExportFmt.h"
#include <windows.h>

//////////////////////// SHeader
bool SHeader::Write(FILE* fp)
{
	if (!fwrite(this,sizeof(SHeader),1,fp))
		return false;

	return true;
}

bool SHeader::Write(unsigned char* pData, unsigned int* pos)
{
	memcpy(pData, this, sizeof(SHeader));
	*pos = sizeof(SHeader);
	return true;
}

bool SHeader::Read(FILE* fp)
{
	if (!fread(this,sizeof(SHeader),1,fp))
		return false;

	return true;
}

/////////////////////// SSkeletonData
SSkeletonData::SSkeletonData(SHeader* header)
{
	boneNames=NULL;
	parentData=NULL;
	flipTable=NULL;
	mask=NULL;

	version = header->version;
}

SSkeletonData::SSkeletonData(SHeader* header, int numBones)
{
	this->numBones=numBones;

	if (numBones==0)
	{
		boneNames=NULL;
		parentData=NULL;
	}
	else
	{
		boneNames=new uint32[numBones];
		parentData=new uint32[numBones];
		flipTable=new uint32[numBones];
		mask=new uint32[numBones];
	}

	version = header->version;
}

bool SSkeletonData::Read(FILE* fp,int numBones)
{
	this->numBones=numBones;

	if (!fread(&checksum,sizeof(uint32),1,fp))
		return false;

	if (!fread(&numBones,sizeof(uint32),1,fp))
		return false;

	if (boneNames)
		delete [] boneNames;

	if (parentData)
		delete [] parentData;

	if (flipTable)
		delete [] flipTable;

	if (numBones==0)
	{
		boneNames=NULL;
		parentData=NULL;
		flipTable=NULL;
	}
	else
	{
		boneNames=new uint32[numBones];
		parentData=new uint32[numBones];
		flipTable=new uint32[numBones];
		mask=new uint32[numBones];

		if (!fread(boneNames,sizeof(uint32),numBones,fp))
			return false;

		if (!fread(parentData,sizeof(uint32),numBones,fp))
			return false;

		if (!fread(flipTable,sizeof(uint32),numBones,fp))
			return false;
	}

#ifdef ANIMVER3SUPPORT
	if (version >= 3)
	{
		if (!fread(mask,sizeof(uint32),numBones,fp))
			return false;
	}
#endif

	return true;
}

SSkeletonData::~SSkeletonData()
{
	if (boneNames!=NULL)
		delete [] boneNames;

	if (parentData!=NULL)
		delete [] parentData;
	
	if (flipTable!=NULL)
		delete [] flipTable;

	if (mask!=NULL)
		delete [] mask;
}

bool SSkeletonData::Write(FILE* fp)
{
	if (!fwrite(&checksum,sizeof(uint32),1,fp))
		return false;

	if (!fwrite(&numBones,sizeof(uint32),1,fp))
		return false;

	if (boneNames)
		if (!fwrite(boneNames,sizeof(uint32),numBones,fp))
			return false;

	if (parentData)
		if (!fwrite(parentData,sizeof(uint32),numBones,fp))
			return false;

	if (flipTable)
		if (!fwrite(flipTable,sizeof(uint32),numBones,fp))
			return false;

#ifdef ANIMVER3SUPPORT
	if (mask && version >= 3)
	{
		if (!fwrite(&numBones, sizeof(uint32),1,fp))
			return false;

		if (!fwrite(mask,sizeof(uint32),numBones,fp))
			return false;
	}
#endif

	return true;
}

bool SSkeletonData::Write(unsigned char* pData, unsigned int* pos)
{
	int size = 0;
	memcpy(pData, &checksum, sizeof(uint32));
	pData += sizeof(uint32);
	size  += sizeof(uint32);
	memcpy(pData, &numBones, sizeof(uint32));
	pData += sizeof(uint32);
	size  += sizeof(uint32);
	
	if (boneNames)
	{
		memcpy(pData, boneNames, sizeof(uint32) * numBones);
		pData += sizeof(uint32) * numBones;
		size  += sizeof(uint32) * numBones;
	}

	if (parentData)
	{
		memcpy(pData, parentData, sizeof(uint32) * numBones);
		pData += sizeof(uint32) * numBones;
		size  += sizeof(uint32) * numBones;
	}
	
	if (flipTable)
	{
		memcpy(pData, flipTable, sizeof(uint32) * numBones);
		pData += sizeof(uint32) * numBones;
		size  += sizeof(uint32) * numBones;
	}

#ifdef ANIMVER3SUPPORT
	if (mask && version >= 3)
	{
		memcpy(pData, &numBones, sizeof(uint32));
		pData += sizeof(uint32);
		size  += sizeof(uint32);

		memcpy(pData, mask, sizeof(uint32) * numBones);
		pData += sizeof(uint32) * numBones;
		size  += sizeof(uint32) * numBones;
	}
#endif

	*pos = size;
	return true;
}

//////////////////////////// SQKey
bool SQKey::Write(FILE* fp)
{
	if (!fwrite(this,sizeof(SQKey),1,fp))
		return false;

	return true;
}

bool SQKey::Write(unsigned char* pData, unsigned int* pos)
{
	int size = 0;
	memcpy(pData, this, sizeof(SQKey));
	*pos = sizeof(SQKey);

	return true;
}

bool SQKey::Read(FILE* fp)
{
	if (!fread(this,sizeof(SQKey),1,fp))
		return false;

	return true;
}

/////////////////////////// SQData
SQData::SQData()
{
	numKeysPerBone=NULL;
	theQKeys=NULL;
}

SQData::SQData(int numBones,int totalNumKeys)
{
	Init(numBones,totalNumKeys);
}

SQData::~SQData()
{
	if (numKeysPerBone!=NULL)
		delete [] numKeysPerBone;

	if (theQKeys!=NULL)
		delete [] theQKeys;
}

void SQData::Init(int numBones,int totalNumKeys)
{
	this->numBones=numBones;
	this->totalNumKeys=totalNumKeys;

	if (numBones==0)
		numKeysPerBone=NULL;
	else
		numKeysPerBone=new uint32[numBones];

	if (totalNumKeys==0)
		theQKeys=NULL;
	else
		theQKeys=new SQKey[totalNumKeys];
}

bool SQData::Read(FILE* fp,int numBones,int totalNumKeys)
{
	if (numKeysPerBone)
		delete [] numKeysPerBone;

	if (theQKeys)
		delete [] theQKeys;

	Init(numBones,totalNumKeys);

	if (!fread(numKeysPerBone,sizeof(uint32),numBones,fp))
		return false;

	for(int i=0;i<totalNumKeys;i++)
	{
		if (!theQKeys[i].Read(fp))
			return false;
	}

	return true;
}

bool SQData::Write(FILE* fp)
{
	if (numKeysPerBone)
		if (!fwrite(numKeysPerBone,sizeof(uint32),numBones,fp))
			return false;

	for(int i=0;i<totalNumKeys;i++)
	{
		if (!theQKeys[i].Write(fp))
			return false;
	}

	return true;
}

bool SQData::Write(unsigned char* pData, unsigned int* pos)
{
	int size = 0;
	if (numKeysPerBone)
	{
		memcpy(pData, numKeysPerBone, sizeof(uint32) * numBones);
		pData += sizeof(uint32) * numBones;
		size  += sizeof(uint32) * numBones;
	}

	for(int i = 0; i < totalNumKeys; i++)
	{
		unsigned int nsize = 0;
		if (!theQKeys[i].Write(pData,&nsize))
		{
			*pos = size;
			return false;
		}

		pData += nsize;
		size  += nsize;
	}

	*pos = size;
	return true;
}

//////////////////////////// STKey
bool STKey::Write(FILE* fp)
{
	if (!fwrite(this,sizeof(STKey),1,fp))
		return false;

	return true;
}

bool STKey::Write(unsigned char* pData, unsigned int* pos)
{
	memcpy(pData, this, sizeof(STKey));
	*pos = sizeof(STKey);
	return true;
}

bool STKey::Read(FILE* fp)
{
	if (!fread(this,sizeof(STKey),1,fp))
		return false;

	return true;
}

////////////////////////////// STData
STData::STData()
{
	numKeysPerBone=NULL;
	theTKeys=NULL;
}

STData::STData(int numBones,int totalNumKeys)
{
	Init(numBones,totalNumKeys);
}

STData::~STData()
{
	if (numKeysPerBone!=NULL)
		delete [] numKeysPerBone;

	if (theTKeys!=NULL)
		delete [] theTKeys;
}

void STData::Init(int numBones,int totalNumKeys)
{
	this->numBones=numBones;
	this->totalNumKeys=totalNumKeys;

	if (numBones==0)
		numKeysPerBone=NULL;
	else
		numKeysPerBone=new uint32[numBones];

	if (totalNumKeys==0)
		theTKeys=NULL;
	else
		theTKeys=new STKey[totalNumKeys];
}

bool STData::Read(FILE* fp,int numBones,int totalNumKeys)
{
	if (numKeysPerBone)
		delete [] numKeysPerBone;

	if (theTKeys)
		delete [] theTKeys;

	Init(numBones,totalNumKeys);

	if (!fread(numKeysPerBone,sizeof(uint32),numBones,fp))
		return false;

	for(int i=0;i<totalNumKeys;i++)
	{
		if (!theTKeys[i].Read(fp))
			return false;
	}

	return true;
}

bool STData::Write(FILE* fp)
{
	if (numKeysPerBone)
		if (!fwrite(numKeysPerBone,sizeof(uint32),numBones,fp))
			return false;

	for(int i=0;i<totalNumKeys;i++)
	{
		if (!theTKeys[i].Write(fp))
			return false;
	}

	return true;
}

bool STData::Write(unsigned char* pData, unsigned int* pos)
{
	int size = 0;
	if (numKeysPerBone)
	{
		memcpy(pData, numKeysPerBone, sizeof(uint32) * numBones);
		pData += sizeof(uint32) * numBones;
		size  += sizeof(uint32) * numBones;
	}

	for(int i = 0; i < totalNumKeys; i++)
	{
		unsigned int npos = 0;
		if (!theTKeys[i].Write(pData, &npos))
		{
			*pos = size;
			return false;
		}

		pData += npos;
		size  += npos;
	}

	*pos = size;
	return true;
}

///////////////////////////////// SChangeParentKey
bool SChangeParentKey::Write(FILE* fp)
{
	if (!fwrite(this,sizeof(SChangeParentKey),1,fp))
		return false;

	return true;
}

bool SChangeParentKey::Write(unsigned char* pData, unsigned int* pos)
{
	memcpy(pData, this, sizeof(SChangeParentKey));
	*pos = sizeof(SChangeParentKey);
	return true;
}

bool SChangeParentKey::Read(FILE* fp)
{
	if (!fread(this,sizeof(SChangeParentKey),1,fp))
		return false;

	return true;
}

////////////////////////////////// SUserDefinedKey
bool SUserDefinedKey::Write(FILE* fp)
{
	if (!fwrite(this,size,1,fp))
		return false;

	return true;
}

bool SUserDefinedKey::Write(unsigned char* pData, unsigned int* pos)
{
	memcpy(pData, this, size);
	*pos = size;
	return true;
}

bool SUserDefinedKey::Read(FILE* fp)
{
	if (!fread(this,sizeof(SUserDefinedKey),1,fp))
		return false;

	return true;
}

///////////////////////////////////// ChangeParentKey
ChangeParentKey::ChangeParentKey()
{
	keyType=KEYTYPE_CHANGEPARENTKEY;
	size=KEYSIZE_CHANGEPARENTKEY;
}

ChangeParentKey::ChangeParentKey(float time,uint32 id)
{
	keyType=KEYTYPE_CHANGEPARENTKEY;
	size=KEYSIZE_CHANGEPARENTKEY;		
	timeStamp=time;
	newParentBoneId=id;
}

bool ChangeParentKey::Write(FILE* fp)
{
	if (!fwrite(this,size,1,fp))
		return false;

	return true;
}

bool ChangeParentKey::Write(unsigned char* pData, unsigned int* pos)
{
	memcpy(pData, this, size);
	*pos = size;
	return true;
}

bool ChangeParentKey::Read(FILE* fp)
{
	if (!SUserDefinedKey::Read(fp))
		return false;

	if (!fread(&newParentBoneId,sizeof(uint32),1,fp))
		return false;

	return true;
}

//////////////////////////////////////// ChangeFovKey
ChangeFovKey::ChangeFovKey()
{
	keyType=KEYTYPE_CHANGEFOV;
	size=KEYSIZE_CHANGEFOV;
}

ChangeFovKey::ChangeFovKey(float time,float fov)
{
	keyType=KEYTYPE_CHANGEFOV;
	size=KEYSIZE_CHANGEFOV;
	newFov=fov;
	timeStamp=time;
}

bool ChangeFovKey::Write(FILE* fp)
{
	if (!fwrite(this,size,1,fp))
		return false;

	return true;
}

bool ChangeFovKey::Write(unsigned char* pData, unsigned int* pos)
{
	memcpy(pData, this, size);
	*pos = size;
	return true;
}

bool ChangeFovKey::Read(FILE* fp)
{
	if (!SUserDefinedKey::Read(fp))
		return false;

	if (!fread(&newFov,size,1,fp))
		return false;

	return true;
}

////////////////////////////////////////////// RunScriptKey
RunScriptKey::RunScriptKey()
{
	keyType=KEYTYPE_RUNSCRIPT;
	size=KEYSIZE_RUNSCRIPT;
}

RunScriptKey::RunScriptKey(float time,uint32 name)
{
	keyType=KEYTYPE_RUNSCRIPT;
	size=KEYSIZE_RUNSCRIPT;
	scriptName=name;
	timeStamp=time;
}

bool RunScriptKey::Write(FILE* fp)
{
	if (!fwrite(this,size,1,fp))
		return false;

	return true;
}

bool RunScriptKey::Write(unsigned char* pData, unsigned int* pos)
{
	memcpy(pData, this, size);
	*pos = size;
	return true;
}

bool RunScriptKey::Read(FILE* fp)
{
	if (!SUserDefinedKey::Read(fp))
		return false;

	if (!fread(&scriptName,sizeof(uint32),1,fp))
		return false;

	return true;
}

///////////////////////////////////////////// EnableObjKey
EnableObjKey::EnableObjKey()
{
	keyType = KEYTYPE_ENABLEOBJ;
	size    = KEYSIZE_ENABLEOBJ;
}

EnableObjKey::EnableObjKey(float time, uint32 name)
{
	keyType    = KEYTYPE_ENABLEOBJ;
	size       = KEYSIZE_ENABLEOBJ;
	arrayName = name;
}

bool EnableObjKey::Write(unsigned char* pData, unsigned int* pos)
{
	memcpy(pData, this, size);
	*pos = size;
	return true;
}

bool EnableObjKey::Write(FILE* fp)
{
	if (!fwrite(this,size,1,fp))
		return false;

	return true;
}

bool EnableObjKey::Read(FILE* fp)
{
	if (!SUserDefinedKey::Read(fp))
		return false;

	if (!fread(&arrayName,sizeof(uint32),1,fp))
		return false;

	return true;
}
//////////////////////////////////////////// DisableObjKey
DisableObjKey::DisableObjKey()
{
	keyType = KEYTYPE_DISABLEOBJ;
	size    = KEYSIZE_DISABLEOBJ;
}

DisableObjKey::DisableObjKey(float time, uint32 name)
{
	keyType    = KEYTYPE_DISABLEOBJ;
	size       = KEYSIZE_DISABLEOBJ;
	arrayName = name;
}

bool DisableObjKey::Write(unsigned char* pData, unsigned int* pos)
{
	memcpy(pData, this, size);
	*pos = size;
	return true;
}

bool DisableObjKey::Write(FILE* fp)
{
	if (!fwrite(this,size,1,fp))
		return false;

	return true;
}

bool DisableObjKey::Read(FILE* fp)
{
	if (!SUserDefinedKey::Read(fp))
		return false;

	if (!fread(&arrayName,sizeof(uint32),1,fp))
		return false;

	return true;
}

////////////////////////////////////// ChangeCamKey
ChangeCamKey::ChangeCamKey()
{
	x  = y  = z  = 0.0f;  w  = 1.0f;
	rx = ry = rz = 0.0f;  rw = 1.0f;
	
	keyType = KEYTYPE_CHANGECAM;
	size    = KEYSIZE_CHANGECAM; // 12 (header) + 4 * 8 (float fields)
}

ChangeCamKey::ChangeCamKey(float time,
						   float posX, float posY, float posZ,
			               float rotX, float rotY, float rotZ, float rotW)
{
	timeStamp = time;

	x  = posX; y  = posY; z  = posZ; w  = 1.0f;
	rx = rotX; ry = rotY; rz = rotZ; rw = rotW;

	keyType = KEYTYPE_CHANGECAM;
	size    = KEYSIZE_CHANGECAM; // 12 (header) + 4 * 8 (float fields)
}

bool ChangeCamKey::Write(FILE* fp)
{
	if (!fwrite(&x, sizeof(float), 4, fp))
		return false;

	if (!fwrite(&rx, sizeof(float), 4, fp))
		return false;

	return true;
}

bool ChangeCamKey::Write(unsigned char* pData, unsigned int* pos)
{
	memcpy(pData, this, size);
	*pos = size;
	return true;

}

bool ChangeCamKey::Read(FILE* fp)
{
	if (!SUserDefinedKey::Read(fp))
		return false;

	if (!fread(&x, sizeof(float), 4, fp))
		return false;

	if (!fread(&rx, sizeof(float), 4, fp))
		return false;

	return true;
}

////////////////////////// Helper Functions
int CompareUserDefinedKeys(const void *pElem1, const void *pElem2)
{
	SUserDefinedKey* pItemA = *((SUserDefinedKey**)pElem1);
	SUserDefinedKey* pItemB = *((SUserDefinedKey**)pElem2);

	if (*pItemA == *pItemB)
		return 0;

	if (*pItemA < *pItemB)
		return -1;

	return 1;
}

unsigned long GetUserKeyMemoryUsage(SUserDefinedKey* keys, int nKeys)
{
	int nTotalBytes = 0;
	unsigned char* offset = (unsigned char*)keys;

	for(int i = 0; i < nKeys; i++)
	{
		SUserDefinedKey* key = (SUserDefinedKey*)offset;
		nTotalBytes += key->size;
		offset += key->size;
	}

	return nTotalBytes;
}

void SortUserKeys(SUserDefinedKey* keys, int nKeys)
{
#ifdef MEM_DEBUG
	VerifyMemory();
#endif

	int i;

	// Determine the total amount of memory used by the user defined keys
	unsigned long nTotalBytes = GetUserKeyMemoryUsage(keys, nKeys);

	// Build an array of pointers to the user defined keys
	SUserDefinedKey** pOrigKeys = (SUserDefinedKey**)malloc(sizeof(SUserDefinedKey*) * nKeys);

	unsigned char* pKeyOffsetRead = (unsigned char*)keys;
	unsigned char* pKeyOffsetWrite;

	for(i = 0; i < nKeys; i++)
	{
		pOrigKeys[i]      =  (SUserDefinedKey*)pKeyOffsetRead;
		pKeyOffsetRead   += ((SUserDefinedKey*)pKeyOffsetRead)->size;
	}

	// Sort our pointer list
	qsort(pOrigKeys, nKeys, sizeof(SUserDefinedKey*), CompareUserDefinedKeys);

	// Allocate a temporary buffer to store out the sorted data
	unsigned char* pBuf = (unsigned char*)malloc(nTotalBytes);

	// Fill temp buffer w/ keys in correct order
	//pKeyOffsetRead  = keys;
	pKeyOffsetWrite = pBuf;

	int nDbgBytes = 0;

	for(i = 0; i < nKeys; i++)
	{
		memcpy(pKeyOffsetWrite, pOrigKeys[i], pOrigKeys[i]->size);
		//memset(pKeyOffsetWrite, 0, pOrigKeys[i]->size);
		pKeyOffsetWrite += pOrigKeys[i]->size;
		
		nDbgBytes += pOrigKeys[i]->size;
		
		assert(pOrigKeys[i]->size == 16 ||
			   pOrigKeys[i]->size == 44);
	}

	// Copy our temp buffer back into the original
	memcpy(keys, pBuf, nTotalBytes);

	free(pBuf);
	free(pOrigKeys);
}
