#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <io.h>
#include <fstream.h>

#include "AnimComp.h"

typedef enum
{
	vPLAT_PS2,
	vPLAT_NGC,
	vPLAT_XBOX,
	vNUM_PLATFORMS,
	vPLAT_NONE,
};

#define	vVERSION_STR	"1.0"
#define vEXE_NAME		"AnimComp"

char*	GetExt(int platform)
{
	switch (platform)
	{
		case vPLAT_PS2:
			return "ps2";
		case vPLAT_XBOX:
			return "xbx";
		case vPLAT_NGC:
			return "ngc";
		default:
			return "none";
	}
}

bool	UpToDate(char* pSrcFileName, char* pDstFileName)
{
	struct _finddata_t file_info, dst_file_info, exe_file_info;
	long handle_src, handle_dst, handle_exe;		
	char pExeFileName[_MAX_PATH];

	char *skate_4_path = getenv( "SKATE4_PATH" );
	if( skate_4_path == NULL )
	{		
		fprintf(stderr, "You must first define your SKATE4_PATH environment variable\n");
		exit(1);
	}

	handle_src = _findfirst( pSrcFileName, &file_info );
	if( handle_src < 0 )
	{
		printf( "%s: Does not exist.\n", pSrcFileName );
		exit(1);
	}

	sprintf( pExeFileName, "%s\\bin\\win32\\%s.exe", skate_4_path, vEXE_NAME);

	handle_dst = _findfirst( pDstFileName, &dst_file_info );
	if( handle_dst >= 0 )
	{
		handle_exe = _findfirst( pExeFileName, &exe_file_info );
		if ( handle_exe < 0 )
		{
			fprintf( stderr, "Cannot find .exe file: %s\n", pExeFileName );
			exit(1);
		}
		if( (file_info.time_write < dst_file_info.time_write) &&
			(exe_file_info.time_write < dst_file_info.time_write) )
		{
//			printf( "Up to date.\n" );
			_findclose( handle_dst );
					_findclose( handle_src );
					_findclose( handle_exe );
			return true;
		}
		_findclose( handle_dst );
		_findclose( handle_exe );
	}

	_findclose( handle_src );
	return false;
}

void	main( int argc, char* argv[] )
{
	bool success = false;

	char pSrcName[_MAX_PATH];
	strcpy( pSrcName, "" );

	char pDstName[_MAX_PATH];
	strcpy( pDstName, "" );

	int platform;
	platform = vPLAT_NONE;

	if( argc > 2 )
	{
		int i;

		for( i = 1; i < argc; i++ )
		{
			if( argv[i][0] == '-' )
			{
				switch( argv[i][1] )
				{
					case 'P':
					case 'p':
						switch( argv[i][2] )
						{
							case 'P':
							case 'p':
								platform = vPLAT_PS2;
								break;
							case 'G':
							case 'g':
								platform = vPLAT_NGC;
								break;
							case 'X':
							case 'x':
								platform = vPLAT_XBOX;
								break;
						}
						break;

					case 'I':
					case 'i':
						strcpy( pSrcName, &argv[i][2] );
						break;

					case 'O':
					case 'o':
						strcpy( pDstName, &argv[i][2] );
						break;
				}
			}
		}	
	}

	char pQDstName[_MAX_PATH];
	char pTDstName[_MAX_PATH];

	if( ( platform == vPLAT_NONE ) ||
		( pSrcName == NULL ) ||
		( !strcmp(pDstName, "") ) )
	{	
		printf( "AnimComp v%s\n", vVERSION_STR );
		printf( "Neversoft Entertainment, 2002\n" );
		printf( "\nDescription:  Generates a lookup-table of frequently-used animation keys" );
		printf( "\nUsage: AnimComp -i<inputfile> -o<outputname> -p[p | g | x]\n" );
		printf( "Where p = PS2, g = GameCube, x = Xbox\n" );	
		printf( "Ex: AnimComp -ic:/skate4/bin/win32/anims.txt -oc:/skate4/data/anims/standardkey -pp\n" );
		return;
	}

	if ( UpToDate(pSrcName, pDstName) )
	{
		printf( "%s is up to date.\n", pDstName );
		return;
	}
	else
	{
		printf( "%s is not up to date.\n", pDstName );
	}

	// TODO:  This is not NGC-friendly...
	// Need to change the parsing and output functions

	char buf[512];
	char filename[512];
	AnimCompressor theAnimCompressor;

	fstream inStream;
	fstream animStream;
	fstream tOutStream;
	fstream qOutStream;
	
	int numParsed = 0;

	// make sure the output files are writeable
	strcpy( pQDstName, pDstName );
	strcat( pQDstName, "q.bin" );
//	strcat( pQDstName, "." );
//	strcat( pQDstName, GetExt(platform) );
	qOutStream.open( pQDstName, ios::out | ios::binary );
	if ( !qOutStream.is_open() )
	{
		printf( "Error!  Could not open %s\n", pQDstName );
		goto cleanup;
	}

	// make sure the output files are writeable
	strcpy( pTDstName, pDstName );
	strcat( pTDstName, "t.bin" );
//	strcat( pTDstName, "." );
//	strcat( pTDstName, GetExt(platform) );
	tOutStream.open( pTDstName, ios::out | ios::binary );
	if ( !tOutStream.is_open() )
	{
		printf( "Error!  Could not open %s\n", pTDstName );
		goto cleanup;
	}

	inStream.open( pSrcName, ios::in | ios::nocreate );
	if ( !inStream.is_open() )
	{
		printf( "Error!  Could not open %s\n", pSrcName );
		goto cleanup;
	}

	numParsed = 0;

	// FIRST PASS, FOR 48-BIT LOOKUP TABLE
	while ( !inStream.eof() )
	{
		inStream.getline( buf, 512 );

		if ( inStream.eof() )
		{
			// gets the final newline at the end of the file
			break;
		}

		printf( "Parsing %d (1st pass):  %s...\n", numParsed++, buf );

		animStream.open( buf, ios::in | ios::binary | ios::nocreate );
		if ( !animStream.is_open() )
		{
			printf( "Error!  Could not open %s\n", filename );
			goto cleanup;
		}
		else if ( theAnimCompressor.LoadAnimFirstPass( animStream, platform ) )
		{
			printf( "Error!  Could not parse %s\n", filename );
			goto cleanup;
		}

		animStream.close();
	}

	inStream.close();

	// SECOND PASS, FOR 8-BIT LOOKUP TABLE
	inStream.open( pSrcName, ios::in | ios::nocreate );
	if ( !inStream.is_open() )
	{
		printf( "Error!  Could not open %s for second-pass\n", pSrcName );
		goto cleanup;
	}

	numParsed = 0;

	// starts the second pass here,
	// which sorts the elements so
	// that only the most frequently-used
	// items go in
	theAnimCompressor.StartSecondPass();

	while ( !inStream.eof() )
	{
		inStream.getline( buf, 512 );

		if ( inStream.eof() )
		{
			// gets the final newline at the end of the file
			break;
		}

		printf( "Parsing %d (2nd pass):  %s...\n", numParsed++, buf );

		animStream.open( buf, ios::in | ios::binary | ios::nocreate );
		if ( !animStream.is_open() )
		{
			printf( "Error!  Could not open %s\n", filename );
			goto cleanup;
		}
		else if ( theAnimCompressor.LoadAnimSecondPass( animStream, platform ) )
		{
			printf( "Error!  Could not parse %s\n", filename );
			goto cleanup;
		}

		animStream.close();
	}

	inStream.close();

	// write out Q-table
	if ( theAnimCompressor.SaveQ48Table( qOutStream, platform ) )
	{
		printf( "Error!  Could not write out %s\n", pQDstName );
		goto cleanup;
	}

	// write out T-table
	if ( theAnimCompressor.SaveT48Table( tOutStream, platform ) )
	{
		printf( "Error!  Could not write out %s\n", pTDstName );
		goto cleanup;
	}

	printf("Successfully generated compression tables from %d files!\n", numParsed );
	success = true;

cleanup:
	animStream.close();
	inStream.close();
	qOutStream.close();
	tOutStream.close();

	if ( !success )
	{
		exit(1);
	}
}