// -----------------------------------------------------------------------------
//   Copyright (c) 2001 SN Systems Software Limited
//   All Rights Reserved
//
//   Author:  Matt Bush, Dan Farley
//   File:    ps2tmapi.h
//   Desc:    Common command interface to the Target Manager for PS2
// -----------------------------------------------------------------------------

#ifndef __SN_PS2_H_INCLUDED
#define __SN_PS2_H_INCLUDED

// EXPORT ALL FUNCTIONS WITH PREFIX SNPS2API
#ifdef PS2TMAPI_STATIC
    #define SNPS2API
#else
    #define SNPS2API __declspec(dllexport)
#endif

#if defined __cplusplus
extern "C"
{
#endif

// -----------------------------------------------------------------------------
//   All functions return an error code in the form of a 32 bit value with the 
//   following format.
//
//    32..28........20........10........0
//   |S|            error code           |
// 
//   Where S is a 1 bit value representing the severity of the error code given 
//   by the lower 31 bits.
//
//   The following macros are defined for testing the success of a function:
// 
//   SN_SUCCEEDED(snr) returns TRUE if snr contains a success code
//                     or FALSE if snr contains a failure code.
//   SN_FAILED(snr)    returns TRUE if snr contains a failure code
//                     or FALSE if snr contains a success code.
//
//   If a function returns SN_E_ERROR, then the error code will hold additional 
//   information pertaining to the error. A textual representation of this 
//   error may be obtained by calling
//
//   SNRESULT SNTranslateError(SNRESULT snr, char** ppszErrorString)
// 
//   If the error code given by snr exists, this function will return SNR_OK 
//   and pszErrorString will point to a character string. The memory allocated 
//   to the error string must be freed with a call to
//    
//   void SNFreeErrorStr(char* pszErrorString)
// -----------------------------------------------------------------------------

typedef long SNRESULT;

#define SN_SUCCEEDED(snr)   ((SNRESULT)(snr) >= 0)
#define SN_FAILED(snr)      ((SNRESULT)(snr) < 0)

SNPS2API SNRESULT SNTranslateError(SNRESULT snr, char** ppszErrorString);
SNPS2API void SNFreeErrorStr(char* pszErrorString);

// -----------------------------------------------------------------------------
//   Errors codes
//
//    The following error codes are defined. It is possible to get a textual
//    description of the error by calling SNTranslateError(...).
//
//    SN_E_NOT_IMPL       
//        Function not implemented
//
//    SN_E_TM_NOT_RUNNING
//        Initialisation of DLL failed becuase the Target Manager could
//        not be started.
//
//    SN_E_BAD_TARGET 
//        Specified target does not exist    
//
//    SN_E_NOT_CONNECTED  
//        Function failed because target was not connected
//
//    SN_E_PS2_COMMS_ERR  
//        Function failed because of a communications error with the PS2
//
//    SN_E_TM_COMMS_ERR
//        Function failed because of a communications error with the target 
//        manager. When this error occurs, you should call SNPS2UnInitialise() 
//        and then SNPS2Initialise() to reinitialise target communications.
//
//    SN_E_TIMEOUT        
//        A timeout occurred during the execution of the function
//
//    SN_E_HOST_NOT_FOUND 
//        During a connect operation, the host could not be found
//
//    SN_E_TARGET_IN_USE  
//        A connect operation failed because the specified target was in use
//
//    SN_E_LOAD_ELF_FAILED 
//        ELF load failure
//    
//    SN_E_BAD_UNIT       
//        Specified unit is not valid
//    
//    SN_E_OUT_OF_MEM
//        Internal memory allocation failure.
//
//    SN_E_NOT_LISTED
//        Specified client is not in list for either unit update notification
//        or tty update notification. Can be recieved in response to 
//        CancelUPDNotfication() or CancelTTYNotification.
//
//    SN_E_TM_VERSION
//        Version of Target Manager is incorrect. Contact SN Systems Support 
//        for latest version (www.snsys.com or support@snsys.com)
//
//    SN_E_DLL_NOT_INITIALISED
//        The DLL has not been successfully initialised. Call SNPS2Initalise() 
//        to init comms.
//
//    SN_E_TARGET_RUNNING
//        Operation could not be completed because specified unit is in RUN 
//        state.
//
//    SN_E_BAD_MEMSPACE
//        Specified memory space was not valid.
//
//    SN_E_NO_TARGETS
//        Pick Target operation failed because there are no targets defined in 
//        the Target Manager.
//
//    SN_E_NO_SEL
//        Pick Target operation failed because no target was selected *OR* the 
//        user pressed cancel.
// -----------------------------------------------------------------------------

#define SN_S_OK                     0
#define SN_E_ERROR                  0x80000000
#define SN_E_NOT_IMPL               -1
#define SN_E_TM_NOT_RUNNING         -2
#define SN_E_BAD_TARGET             -3
#define SN_E_NOT_CONNECTED          -4
#define SN_E_PS2_COMMS_ERR          -5
#define SN_E_TM_COMMS_ERR           -6
#define SN_E_TIMEOUT                -7
#define SN_E_HOST_NOT_FOUND         -8
#define SN_E_TARGET_IN_USE          -9
#define SN_E_LOAD_ELF_FAILED        -10
#define SN_E_BAD_UNIT               -11
#define SN_E_OUT_OF_MEM             -12
#define SN_E_NOT_LISTED             -13
#define SN_E_TM_VERSION             -14
#define SN_E_DLL_NOT_INITIALISED    -15
#define SN_E_TARGET_RUNNING         -17
#define SN_E_BAD_MEMSPACE           -18
#define SN_E_NO_TARGETS             -19
#define SN_E_NO_SEL                 -20

// -----------------------------------------------------------------------------
//  UNIT IDS
// -----------------------------------------------------------------------------

typedef enum EUNIT
{
    UI_ALL      = -1,
    UI_EECPU    = 0,
    UI_EEVU0    = 1,
    UI_EEVU1    = 2,
    UI_IOP      = 3,
    UI_COUNT    = 4
} EUNIT;

// -----------------------------------------------------------------------------
//  MEMORY SPACES
// -----------------------------------------------------------------------------

typedef enum EMEMSPACE
{
    MEMSPACE_MAIN       = 0,
    MEMSPACE_VU_MICRO   = 1,
    MEMSPACE_VU_DATA    = 2
} EMEMSPACE;

// -----------------------------------------------------------------------------
//  UNIT STATES
// -----------------------------------------------------------------------------

typedef enum EUNITSTATUS
{
    US_UNKNOWN,
    US_RUNNING,
    US_STOPPED,
    US_SIGNALLED,
    US_RESETTING,
    US_MISSING,
    US_RESET,
    US_NOT_CONNECTED,
    US_CONNECTED,
    US_UNIT_STATUS_CHANGE
} EUNITSTATUS;

// DEFAULTS
#define EE_DEF_BOOT_PARAM       (-1)
#define IOP_DEF_BOOT_PARAM      (-1)
#define SNPS2_DEF_PORT_NO       (8510)

typedef int HTARGET;

#define SNPS2_TI_TARGETID       0x00000001
#define SNPS2_TI_NAME           0x00000002
#define SNPS2_TI_IP             0x00000004
#define SNPS2_TI_PORT           0x00000008
#define SNPS2_TI_HOMEDIR        0x00000010
#define SNPS2_TI_FILESERVEDIR   0x00000020
#define SNPS2_TI_BOOT           0x00000040

typedef struct SNPS2TargetInfo
{
    UINT nFlags;
    HTARGET hTarget;
    const char *pszName;
    const char *pszIPAddress;
    unsigned int nPort;
    const char *pszHomeDir;
    const char *pszFSDir;
    UINT64 ee_boot;
    UINT64 iop_boot;
} SNPS2TargetInfo;

typedef struct _TTYSTREAM
{
    UINT nStreamIdx;
    char szName[64];
} TTYSTREAM, *PTTYSTREAM;

/* -----------------------------------------------------------------------------
//
//  SNRESULT SNPS2Initialise();
//
//  Description:
//      Initialises target communications and launches Target Manager.
//
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2Initialise();

/* -----------------------------------------------------------------------------
//
//  void SNPS2UnInitialise();
//
//  Description:
//      Shuts down internal communications (does not close Target Manager) and 
//      frees resources.
//
----------------------------------------------------------------------------- */
SNPS2API void SNPS2UnInitialise();

/* -----------------------------------------------------------------------------
//
//  int (*SNPS2EnumTargetsCB) (SNPS2TargetInfo*);
//
//  Description:
//      EnumTargets Callback function pointer definition.
//
//      The EnumTargets member of SNPS2TM takes a function pointer to a Callback 
//      function as an argument of the type defined below. During operation, 
//      EnumTargets will call this function repeatedly for each target in the 
//      Target Manager passing a pointer to an SNPS2TargetInfo structure. The 
//      implementation of the callback function should store the contents of this
//      structure each time. The callback function returns 0 to continue the 
//      enumeration or non-zero to cancel it.
//
//      For example:
//
//      int EnumCallBack(SNPS2TargetInfo* pTargetInfo)
//      {
//          // Copy contents of pTargetInfo!
//          strcpy(TgtInfo.pszName, pTargetInfo->pszName);
//          // etc...
//      }
//
//      In main() or such...
//
//      pSNPS2TM->EnumTargets( (SNPS2EnumTargetCB) EnumCallBack );
//
----------------------------------------------------------------------------- */
typedef int (*SNPS2EnumTargetsCB) (SNPS2TargetInfo*);

/* -----------------------------------------------------------------------------
//
//  SNRESULT SNPS2Connect(HTARGET hTarget, const char* pszApplication);
//
//  Description:
//      Connect to specified target.
//  Parameters:
//      HTARGET hTarget - Handle to valid target
//      char* pszApplication - Name of connecting application, should be NULL
//
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2Connect(HTARGET hTarget, const char* pszApplication);    

/* -----------------------------------------------------------------------------
//
//  SNRESULT SNPS2Disconnect(HTARGET hTarget);
//
//  Description:
//      Disconnect from specified target.
//  Parameters:
//      HTARGET hTarget - Handle to valid target
//
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2Disconnect(HTARGET hTarget);
    
/* -----------------------------------------------------------------------------
//
//  SNPS2API SNRESULT SNPS2Reset(HTARGET hTarget, UINT64 ee_boot, UINT64 iop_boot);
// 
//  DESCRIPTION:                                           
//     Reset specified target
//  PARAMETERS:                                            
//     HTARGET hTarget        - Handle to valid target
//     UINT64 ee_boot         - ee boot param. or EE_DEF_BOOT_PARAM
//     UINT64 iop_boot        - iop boot param. or IOP_DEF_BOOT_PARAM
//
----------------------------------------------------------------------------- */
SNPS2API SNRESULT  SNPS2Reset(HTARGET hTarget, UINT64 ee_boot, UINT64 iop_boot);
    
/* -----------------------------------------------------------------------------
//
//  SNPS2API SNRESULT SNPS2Run(HTARGET hTarget, int unit, unsigned int nStartAddr,
//                             unsigned int nGPRegister, unsigned int argc,
//                             const char** argv);
//
//  DESCRIPTION:                                           
//     Runs a program at a specified address on a specified target.
//  PARAMETERS:                                            
//     HTARGET hTarget            - Handle to valid target.
//     int unit                   - Processor
//     unsigned int nStartAddr    - Start address of program
//     unsigned int nGPRegister   - Initial contents of GP Register
//     unsigned int argc          - Argument count
//     char** argv                - Argument list
//
----------------------------------------------------------------------------- */
SNPS2API SNRESULT  SNPS2Run(HTARGET hTarget, int unit, unsigned int nStartAddr,
                            unsigned int nGPRegister, unsigned int argc,
                            const char** argv);
    
/* -----------------------------------------------------------------------------
*
* SNPS2API SNRESULT SNPS2AddTarget(const char *pszName, const char *pszIPAddress,
*                                  unsigned int nPort, HTARGET* pnTarget);
* 
* DESCRIPTION:                                           
*    Add a new target to the target manager.
*    This function will fail if the target already exists
*    and return SN_E_BAD_TARGET.
*    To ensure that a target is always added, regardless of
*    whether it already exists, call SNPS2DeleteTarget prior
*    to calling SNPS2AddTarget, ignoring the return code.
* PARAMETERS:                                            
*    char *pszName          - Name of target
*    char *pszIPAddress     - IP Address (either by hostname,
*                             or address).
*    unsigned int nPort     - Port number *OR* SNPS2_DEF_PORT_NO
*    HTARGET* pnTarget      - Address of variable to hold target
*                             token.
*
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2AddTarget(const char *pszName, const char *pszIPAddress,
                                 unsigned int nPort, HTARGET* pnTarget);
    
/* -----------------------------------------------------------------------------
*
* SNPS2API SNRESULT SNPS2DeleteTarget(HTARGET nTarget);
* 
* DESCRIPTION:                                           
*    Delete a specified target from the Target Manager.
*    (this operation is irreversible).
*    Delete target will fail if the target does not
*    exist return SN_E_BAD_TARGET.
* PARAMETERS:                                            
*    HTARGET nTarget        - Handle to valid target.
*
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2DeleteTarget(HTARGET nTarget);
    
/* -----------------------------------------------------------------------------
*
* SNPS2API SNRESULT SNPS2SetTargetInfo(SNPS2TargetInfo* pTargetInfo);
* 
* DESCRIPTION:                                           
*    Sets target information based on members of passed in SNPS2TargetInfo
*    structure. Only members identified by nFlags member will be set. 
*    The desired target to change is indicated by the hTarget member for
*    which nFlags *must* include the SNPS2_TI_TARGETID flag.
* PARAMETERS:                                            
*    SNPS2TargetInfo* pTargetInfo   - address of target info struct.
*
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2SetTargetInfo(SNPS2TargetInfo* pTargetInfo);

/* -----------------------------------------------------------------------------
*
* SNPS2API SNRESULT SNPS2GetTargetInfo(SNPS2TargetInfo* pTargetInfo);
* 
* DESCRIPTION:                                           
*    Retrieves all information for a target specified by hTarget 
*    member of SNPS2TargetInfo structure. SNPS2_TI_TARGETID flag in
*    nFlags member *must* be set.
*    Note: All members are filled in when structure returns and
*    setting of other flags has no effect. Contents of structure
*    returned are temporary and must be copied to a permanent location.
*
* PARAMETERS:                                            
*    SNPS2TargetInfo* pTargetInfo   - address of target info struct.
*
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2GetTargetInfo(SNPS2TargetInfo* pTargetInfo);

/* -----------------------------------------------------------------------------
*
* SNPS2API SNRESULT SNPS2EnumTargets(SNPS2EnumTargetsCB pfnCallBack);
* 
* DESCRIPTION:                                           
*    Enumerate Targets via a callback function. See prototype 
*    of callback function above.
*    EnumTargets calls the callback function for each target 
*    in system passing an SNPS2TargetInfo structure with only the 
*    hTarget member filled in. The callback function can then call
*    SNPS2GetTargetInfo() to retrieve relevant information for that
*    target and store it.
* PARAMETERS:                                            
*    SNPS2EnumTargetsCB pfnCallBack - pointer to callback function
*
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2EnumTargets(SNPS2EnumTargetsCB pfnCallBack);

/* -----------------------------------------------------------------------------
*
* SNPS2API SNRESULT SNPS2GetNumTargets(UINT32 *nNumTargets);
* 
* DESCRIPTION:                                           
*    Returns total number of registered targets in Target Manager
*
* PARAMETERS:                                            
*    UINT32 *nNumTargets - pointer to buffer to receive
*                          number of targets.
*
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2GetNumTargets(UINT32 *nNumTargets);

/* -----------------------------------------------------------------------------
*
* SNPS2API SNRESULT SNPS2GetMemory(HTARGET hTarget, int unit, int memspace,
*                                  UINT32 addr, int count, void *buffptr);
* 
* DESCRIPTION:                                           
*    Gets a range of memory from a 'unit' on the PS2 given
*    by hTarget.
* PARAMETERS:                                            
*    HTARGET hTarget        - Handle to target
*    int unit               - processor    (see enum EUNIT)
*    int memspace           - memory space (see enum EMEMSPACE)
*    UINT32 addr            - Address to read from
*    int count              - Count in bytes to read.
*    void *buffptr          - pointer to buffer for memory.
*
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2GetMemory(HTARGET hTarget, int unit, int memspace,
                                 UINT32 addr, int count, void *buffptr);
    
/* -----------------------------------------------------------------------------
*
* SNPS2API SNRESULT SNPS2SetMemory(HTARGET hTarget, int unit, int memspace,
*                                  UINT32 addr, int count, void *buffptr);
* 
* DESCRIPTION:                                           
*    Sets a range of memory for a 'unit' on the PS2 given by
*    hTarget.
* PARAMETERS:                                            
*    HTARGET hTarget        - Handle to target
*    int unit               - processor     (see enum EUNIT)
*    int memspace           - memory space  (see enum EMEMSPACE)
*    UINT32 addr            - Address to write to.
*    int count              - Size of memory to write (in bytes)
*    void *buffptr          - pointer to buffer containing memory
*                           - to write.
*
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2SetMemory(HTARGET hTarget, int unit, int memspace,
                                  UINT32 addr, int count, void *buffptr);
    
/* -----------------------------------------------------------------------------
*
* SNPS2API SNRESULT SNPS2GetRegisters(HTARGET hTarget, int unit,
*                                     UINT32 lNumRegisters, UINT32* plNum,
*                                     BYTE* pRegBuffer);
* 
* DESCRIPTION:                                           
*    Reads an array of registers from a particular processor
*    (given by unit).
*    Registers are designated by number (snps2regs.h) as
*    assigned by sony. 
* PARAMETERS:                                            
*    HTARGET hTarget        - handle to target
*    int unit               - processor (see enum EUNIT)
*    UINT32 lNumRegisters   - number of registers to read
*    UINT32* plNum          - array of register numbers
*    BYTE* pRegBuffer       - pointer to buffer for registers
*                             to be read back.
*    This should be lNumRegisters * (size_of_register) bytes.
*
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2GetRegisters(HTARGET hTarget, int unit,
                                    UINT32 lNumRegisters, UINT32* plNum,
                                    BYTE* pRegBuffer);
    
/* -----------------------------------------------------------------------------
*
* SNPS2API SNRESULT SNPS2SetRegisters(HTARGET hTarget, int unit,
*                                     UINT32 lNumRegisters, UINT32* plNum,
*                                     BYTE* pRegBuffer);
* 
* DESCRIPTION:                                           
*    Sets a group of registers to particular values.
* PARAMETERS:                                            
*    HTARGET hTarget        - handle to target
*    int unit               - processor (see enum EUNIT)
*    UINT32 lNumRegisters   - number of registers to read
*    UINT32* plNum          - array of register numbers
*    BYTE* pRegBuffer       - pointer to buffer containing
*                             contents of registers in order
*                             specified in array.
*
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2SetRegisters(HTARGET hTarget, int unit,
                                    UINT32 lNumRegisters, UINT32* plNum,
                                    BYTE* pRegBuffer);

/* -----------------------------------------------------------------------------
*
* SNPS2API SNRESULT SNPS2StartTarget(HTARGET hTarget, int unit);
* 
* DESCRIPTION:                                             
*    Starts execution in a given unit. (unit status = US_RUNNING).
* PARAMETERS:                                            
*    HTARGET hTarget        - handle to target
*    int unit               - processor (see enum EUNIT)
*
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2StartTarget(HTARGET hTarget, int unit);
    
/* -----------------------------------------------------------------------------
* SNPS2API SNRESULT SNPS2StopTarget(HTARGET hTarget, int unit);
* 
* DESCRIPTION:                                           
*    Halts execution in a given unit. (unit status = US_HALTED).
* PARAMETERS:                                            
*    HTARGET hTarget        - handle to target
*    int unit               - processor (see enum EUNIT)
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2StopTarget(HTARGET hTarget, int unit);
        
/* -----------------------------------------------------------------------------
*
* SNPS2API SNRESULT SNPS2GetStatus(HTARGET hTarget, int unit, long* pUnitStatus,
*                                  long* pReasonCode);
* 
* DESCRIPTION:                                           
*    Retrieves current status of a particular unit on a
*    given target.
* PARAMETERS:                                            
*    HTARGET hTarget        - handle to target
*    int unit               - processor (see enum EUNIT)
*    long* pUnitStatus      - pointer to variable for status
*                             (see enum EUNITSTATUS)
*    long* pReasonCode      - not used for PlayStation2,
*                             can be NULL.
*
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2GetStatus(HTARGET hTarget, int unit, long* pUnitStatus,
                                 long* pReasonCode);

/* -----------------------------------------------------------------------------
*
* SNPS2API SNRESULT SNPS2LoadRunELF(HTARGET hTarget, const char* pszFileName);
* 
* DESCRIPTION:                                           
*    Loads and runs an ELF file. This is equivalent to calling
*    LoadElf() followed by Run()
* PARAMETERS:                                            
*    HTARGET hTarget        - handle to target
*    char* pszFileName      - filename of ELF file.
*
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2LoadRunELF(HTARGET hTarget, const char* pszFileName);

/* -----------------------------------------------------------------------------
*
* SNPS2API SNRESULT SNPS2LoadElf(HTARGET hTarget, const char* pszFileName,
*                                UINT32 * pnStartAddress, UINT32 * pnGPRegister);
* 
* DESCRIPTION:                                           
*    Loads an ELF file returning the start address and contents
*    of GP register.
* PARAMETERS:                                            
*    HTARGET hTarget            - handle to target
*    char* pszFileName          - filename of ELF file
*    UINT32 * pnStartAddress    - ptr. to buffer for start address.
*    UINT32 * pnGPRegister      - ptr. to buffer for GP Register.
*
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2LoadElf(HTARGET hTarget, const char* pszFileName,
                               UINT32 * pnStartAddress, UINT32 * pnGPRegister);

/* -----------------------------------------------------------------------------
*
* SNPS2API SNRESULT SNPS2RequestUPDNotification(HTARGET hTarget, HWND hWnd, UINT nMsg);
* 
* DESCRIPTION:                                           
*    Registers a window with the target manager to receive 
*    unit status update notifications from Target Manager.
*
*    Messages are posted to a specified window and are received
*    through it's WindowProc function.
*
*       LRESULT CALLBACK WindowProc(
*           HWND hwnd,       // handle to window
*           UINT uMsg,       // id of message.
*           WPARAM wParam,   // unit number
*           LPARAM lParam    // status of target.
*       );
*   
* PARAMETERS:                                            
*    HTARGET hTarget        - handle to target to receive 
*                             notifications for.
*    HWND hWnd              - handle of window that will
*                             recieve notifications
*    UINT nMsg              - id to use for notification
*                             messages.
*
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2RequestUPDNotification(HTARGET hTarget, HWND hWnd, UINT nMsg);

/* -----------------------------------------------------------------------------
*
* SNPS2API SNRESULT SNPS2CancelUPDNotification(HTARGET hTarget, HWND hWnd);
* 
* DESCRIPTION:                                           
*    Unregisters a window that was previously registered for
*    receiving unit status update notifications from the 
*    Target Manager.
* PARAMETERS:                                            
*    HTARGET hTarget        - handle to target to receive 
*                             notifications for
*    HWND hWnd              - Window handle of window that
*                             was receiving notifications.
*
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2CancelUPDNotification(HTARGET hTarget, HWND hWnd);

/* -----------------------------------------------------------------------------
*
* SNPS2API SNRESULT  SNPS2ListTTYStreams(HTARGET hTarget, UINT* nNumTTYStreams,
*                                        PTTYSTREAM pStreams);
* 
* DESCRIPTION:                                           
*    List all available TTY streams for a particular target.
*
*    To obtain a list of TTY Streams, call this function
*    passing NULL for pStreams to get the number of TTY streams
*    
*    Allocate a block of TTYSTREAM structures and call this
*    function again passing in a pointer to this buffer.
*
* PARAMETERS:                                            
*    HTARGET hTarget        - handle to target
*    UINT* nNumTTYStreams   - pointer to buffer to receive no.
*                             of TTY Streams.
*    PTTYSTREAMS pStreams   - pointer to buffer to receive
*                             array of stream info structures.
*
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2ListTTYStreams(HTARGET hTarget, UINT* nNumTTYStreams,
                                      PTTYSTREAM pStreams);

/* -----------------------------------------------------------------------------
*
* SNPS2API SNRESULT SNPS2RequestTTYNotification(HTARGET hTarget, UINT nStream,
*                                               HWND hWnd, UINT nMsg);
* 
* DESCRIPTION:                                           
*    Registers a window to receive TTY Notification messages
*
*    Messages are posted to a specified window and are received
*    through it's WindowProc function.
*
*       LRESULT CALLBACK WindowProc(
*           HWND hwnd,       // handle to window
*           UINT uMsg,       // id of message.
*           WPARAM wParam,   // Target number and stream
*           LPARAM lParam    // length of TTY text to get.
*       );
*   
*     wParam - LOWORD(wParam) = Target number, HIWORD(wParam) = Stream no.
*     lParam - Length of TTY text in chars.
*
*    When a window receives a message as a result of a call
*    to this function, it must call SNPS2GetTTY() to retrieve
*    the text from the target manager.
*           
* PARAMETERS:                                            
*    HTARGET hTarget        - handle to target
*    UINT nStream           - number of stream to receive 
*                             notifications for.
*    HWND hWnd              - handle to window to receive
*                             notification messages.
*    UINT nMsg              - ID of message that will be 
*                             sent to the window.
*
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2RequestTTYNotification(HTARGET hTarget, UINT nStream,
                                              HWND hWnd, UINT nMsg);

/* -----------------------------------------------------------------------------
*
* SNPS2API SNRESULT SNPS2CancelTTYNotification(HTARGET hTarget, UINT nStream,
*                                              HWND hWnd);
* 
* DESCRIPTION:                                           
*    Cancels TTY Notification in the Target Manager for a
*    specified window for a given target and stream.
* PARAMETERS:                                            
*    HTARGET hTarget        - handle to target
*    UINT nStream           - identifier of stream.
*    HWND hWnd              - identifier of window that
*                             was receiving notifications
*
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2CancelTTYNotification(HTARGET hTarget, UINT nStream,
                                             HWND hWnd);

/* -----------------------------------------------------------------------------
*
* SNPS2API SNRESULT SNPS2GetTTY(HTARGET hTarget, HWND hWnd, UINT nStream,
*                               UINT nLength, char* ppszText, UINT* pnSize);
* 
* DESCRIPTION:                                           
*    Retrieves TTY text from the Target Manager. This function
*    should only be called in response to a notification message
*    which is set up using SNPS2RequestTTYNotification().
*    
*    Allocate a buffer for the text to be received using the length
*    passed in the notification message for the size of the buffer.
*    
*    If you pass in a length smaller than the actual amount of text 
*    available, a further message will be posted to the window
*    registered during the call to SNPS2RequestTTYNotification.
*
*    If the amount of text available is smaller than the amount of 
*    text available, then pnSize will be different to nLength.
*   
* PARAMETERS:                                            
*    HTARGET hTarget        - handle to target
*    HWND hWnd              - handle to window that registered
*                             for notification messages. The
*                             target manager uses this to identify
*                             which buffer to fetch the text from.
*    UINT nStream           - identifier of stream.
*    UINT nLength           - length of text to fetch
*    char* pszText          - pointer to buffer to receive text.
*    UINT* pnSize           - actual size of text got.
*
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2GetTTY(HTARGET hTarget, HWND hWnd, UINT nStream,
                              UINT nLength, char* pszText, UINT* pnSize);

/* -----------------------------------------------------------------------------
*
* SNPS2API SNRESULT SNPS2PickTarget(HWND hWndOwner, HTARGET* phTarget)
* 
* DESCRIPTION:                                           
*    Displays a dialog centered in the window passed or to the desktop
*    if NULL is passed in which displays a list of the targets available
*    in the Target Manager.
*    The user can choose a target which is returned through the buffer
*    passed in.
*
*    If NO targets are available, the dialog will not be displayed and
*    the function will return SN_E_NO_TARGETS
*
*    If the user presses cancel or does not select a target before
*    clicking OK, the function will return SN_E_NO_SEL
*
*    If the function succeeds, the function will return SN_S_OK and
*    the contents of the buffer pointed to by phTarget will be set to 
*    the handle for the selected target.
*    This handle can then be used in subsequent operations.
*
* PARAMETERS:                                            
*    HWND hWndOwner         - handle to owner window or NULL for desktop.
*    HTARGET* phTarget      - pointer to buffer to receive target handle.
*
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2PickTarget(HWND hWndOwner, HTARGET* phTarget);

/* -----------------------------------------------------------------------------
*
* SNPS2API SNRESULT SNPS2RegisterProtocolHandler(HTARGET hTarget,
*                                                UINT nProtocolId, HWND hWnd,
*                                                UINT nMsg);
* 
* DESCRIPTION:                                           
*    Registers a custom DECI2 protocol handler with the target manager 
*    allowing usage of custom protocols.
*    This may be called while the target is either connected or disconnected
*    and will remain in the target manager until SNPS2DeRegisterProtocolHandler is
*    called on the same protocol.
*
*    Call SNPS2RegisterProtocolHandler to register a custom protocol with
*    Target Manager which will take care of registering this with NETMP
*    When data arrives on this protocol, Target Manager will send a message
*    to the window specified during the call to SNPS2RegisterProtocolHandler 
*    and is received through it's WndProc function:
*
*       LRESULT CALLBACK WindowProc(
*           HWND hwnd,       // handle to window
*           UINT uMsg,       // id of message.
*           WPARAM wParam,   // Target number and stream
*           LPARAM lParam    // length of TTY text to get.
*       );
*   
*     wParam - LOWORD(wParam) = Target id (HTARGET), 
*            - HIWORD(wParam) = Protocol Id
*     lParam - Length of TTY text in chars.
*
*    You must then call SNPS2ProtocolRecv(...) to obtain the data.
*
*    Target Manager does not modify the data received on a registered protocol in 
*    any way and hands over the entire packet including the DECI2 header.
*    When replying or sending data, you must include the DECI2 header 
*    with the protocol id, destination, source and length fields set correctly.
*    
* PARAMETERS:                                            
*    HTARGET hTarget            - identifier of target
*    UINT nProtocolId           - protocol id to register
*    HWND hWnd                  - handle to window to receive notifications of incoming data.
*    UINT nMsg                  - identifier of message (WM_USER+x) to receive when notifications
*                                 appear
*
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2RegisterProtocolHandler(HTARGET hTarget,
                                               UINT nProtocolId, HWND hWnd,
                                               UINT nMsg);

/* -----------------------------------------------------------------------------
*
* SNPS2API SNRESULT SNPS2DeRegisterProtocolHandler(HTARGET hTarget,
*                                                  UINT nProtocolId, HWND hWnd);
* 
* DESCRIPTION:                                           
*    Deregisters a previously registered protocol from Target Manager.
*
* PARAMETERS:                                            
*    HTARGET hTarget            - identifier of target
*    UINT nProtocolId           - protocol id to de-register
*    HWND hWnd                  - previously registered window handle
*
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2DeRegisterProtocolHandler(HTARGET hTarget,
                                                 UINT nProtocolId, HWND hWnd);

/* -----------------------------------------------------------------------------
*
* SNPS2API SNRESULT SNPS2ProtocolSend(HTARGET hTarget, UINT nProtocolId,
*                                     int count, void *buffptr);
* 
* DESCRIPTION:                                           
*    Call SNPS2ProtocolSend to transmit data to a particular protocol id.
*    
*    You must include a DECI2 header with the appropriate fields set correctly
*    including the protocol field.
*
* PARAMETERS:                                            
*    HTARGET hTarget            - identifier of target
*    UINT nProtocolId           - protocol id
*    int count                  - count of bytes to transmit
*    void *buffptr              - pointer to buffer containing data to be transmitted
*
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2ProtocolSend(HTARGET hTarget, UINT nProtocolId,
                                    int count, void *buffptr);

/* -----------------------------------------------------------------------------
*
* SNPS2API SNRESULT SNPS2ProtocolRecv(HTARGET hTarget, UINT nProtocolId,
*                                     HWND hWnd, int count, void *buffptr,
*                                     UINT* pnSize);
* 
* DESCRIPTION:                                           
*    Retrieves data for a registered protocol for which data
*    has previously arrived.
*
*    If you request less data than is available a subsequent message will be
*    sent to the previously registered window containing the remaining amount
*    of data available.
*
* PARAMETERS:                                            
*    HTARGET hTarget            - identifier of target
*    UINT nProtocolId           - protocol id
*    int count                  - count of bytes to receive
*    void *buffptr              - pointer to buffer to receive data
*    UINT* pnSize               - pointer to buffer to receive actual size of data got!
*
----------------------------------------------------------------------------- */
SNPS2API SNRESULT SNPS2ProtocolRecv(HTARGET hTarget, UINT nProtocolId,
                                    HWND hWnd, int count, void *buffptr,
                                    UINT* pnSize);

#ifdef __cplusplus
}
#endif // #ifdef __cplusplus


/* -----------------------------------------------------------------------------
 Register Number Definitions for PlayStation2.
 (For use with SNPS2GetRegister(...) and SNPS2SetRegister(...)
----------------------------------------------------------------------------- */

/* EE Registers */
 
#define REG_0			0
#define REG_1 			1
#define REG_2 			2
#define REG_3 			3
#define REG_4 			4
#define REG_5 			5
#define REG_6 			6
#define REG_7 			7
#define REG_8 			8
#define REG_9 			9
#define REG_10			10
#define REG_11			11
#define REG_12			12
#define REG_13 			13
#define REG_14 			14
#define REG_15 			15
#define REG_16 			16
#define REG_17 			17
#define REG_18 			18
#define REG_19 			19
#define REG_20 			20
#define REG_21 			21
#define REG_22 			22
#define REG_23 			23
#define REG_24 			24
#define REG_25 			25
#define REG_26 			26
#define REG_27 			27
#define REG_28 			28
#define REG_29 			29
#define REG_30 			30
#define REG_31 			31
#define REG_hi 			100
#define REG_lo 			101
#define REG_hi1 		102
#define REG_lo1			103
#define REG_sa 			104
#define REG_index 		200
#define REG_random 		201
#define REG_entrylo0 	202
#define REG_entrylo1 	203
#define REG_context		204
#define REG_pagemask 	205
#define REG_wired		206
#define REG_badvaddr	208
#define REG_count		209
#define REG_entryhi 	210
#define REG_compare 	211
#define REG_status 		212
#define REG_cause 		213
#define REG_epc 		214
#define REG_prid		215
#define REG_config		216
#define REG_badpaddr	223
#define REG_debug		224
#define REG_perf		225
#define REG_taglo		228
#define REG_taghi		229
#define REG_errorepc 	230

#define REG_ccr			300
#define REG_ctr0		301
#define REG_ctr1		302

#define REG_bpc 400
#define REG_iab 402
#define REG_iabm 403
#define REG_dab 404
#define REG_dabm 405
#define REG_dvb 406
#define REG_dvbm 407
#define REG_fpr0 500
#define REG_fpr1 501
#define REG_fpr2 502
#define REG_fpr3 503
#define REG_fpr4 504
#define REG_fpr5 505
#define REG_fpr6 506
#define REG_fpr7 507
#define REG_fpr8 508
#define REG_fpr9 509
#define REG_fpr10 510
#define REG_fpr11 511
#define REG_fpr12 512
#define REG_fpr13 513
#define REG_fpr14 514
#define REG_fpr15 515
#define REG_fpr16 516
#define REG_fpr17 517
#define REG_fpr18 518
#define REG_fpr19 519
#define REG_fpr20 520
#define REG_fpr21 521
#define REG_fpr22 522
#define REG_fpr23 523
#define REG_fpr24 524
#define REG_fpr25 525
#define REG_fpr26 526
#define REG_fpr27 527
#define REG_fpr28 528
#define REG_fpr29 529
#define REG_fpr30 530
#define REG_fpr31 531
#define REG_fcr0 600
#define REG_fcr31 601

/* IOP Registers */

#define IOP_0  		200
#define IOP_1  		201
#define IOP_2  		202
#define IOP_3  		203
#define IOP_4  		204
#define IOP_5  		205
#define IOP_6  		206
#define IOP_7  		207
#define IOP_8  		208
#define IOP_9  		209
#define IOP_10 		210
#define IOP_11 		211
#define IOP_12 		212
#define IOP_13 		213
#define IOP_14 		214
#define IOP_15 		215
#define IOP_16 		216
#define IOP_17 		217
#define IOP_18 		218
#define IOP_19 		219
#define IOP_20 		220
#define IOP_21 		221
#define IOP_22 		222
#define IOP_23 		223
#define IOP_24 		224
#define IOP_25 		225
#define IOP_26 		226
#define IOP_27 		227
#define IOP_28 		228
#define IOP_29 		229
#define IOP_30 		230
#define IOP_31 		231

#define IOP_hi 		100
#define IOP_lo 		101

#define IOP_bpc		303
#define IOP_bda 	305
#define IOP_tar		307
#define IOP_bada 	308
#define IOP_bdam 	309
#define IOP_bpcm 	311
#define IOP_status	312
#define IOP_cause	313
#define IOP_epc		314
#define IOP_prid	315

/* VU0 Registers */

#define VU0VF00		700
#define VU0VF01		701
#define VU0VF02		702
#define VU0VF03		703
#define VU0VF04		704
#define VU0VF05		705
#define VU0VF06		706
#define VU0VF07		707
#define VU0VF08		708
#define VU0VF09		709
#define VU0VF10		710
#define VU0VF11		711
#define VU0VF12		712
#define VU0VF13		713
#define VU0VF14		714
#define VU0VF15		715
#define VU0VF16		716
#define VU0VF17		717
#define VU0VF18		718
#define VU0VF19		719
#define VU0VF20		720
#define VU0VF21		721
#define VU0VF22		722
#define VU0VF23		723
#define VU0VF24		724
#define VU0VF25		725
#define VU0VF26		726
#define VU0VF27		727
#define VU0VF28		728
#define VU0VF29		729
#define VU0VF30		730
#define VU0VF31		731

#define VU0VI00		800
#define VU0VI01		801
#define VU0VI02		802
#define VU0VI03		803
#define VU0VI04		804
#define VU0VI05		805
#define VU0VI06		806
#define VU0VI07		807
#define VU0VI08		808
#define VU0VI09		809
#define VU0VI10		810
#define VU0VI11		811
#define VU0VI12		812
#define VU0VI13		813
#define VU0VI14		814
#define VU0VI15		815
#define VU0VI16		816		// status flag
#define VU0VI17		817		// MAC flag
#define VU0VI18		818		// clipping flag
#define VU0VI19		819		// reserved
#define VU0VI20		820		// R
#define VU0VI21		821		// I
#define VU0VI22		822		// Q
#define VU0VI23		823		// reserved
#define VU0VI24		824		// reserved
#define VU0VI25		825		// reserved
#define VU0VI26		826		// TPC
#define VU0VI27		827		// CMSAR0
#define VU0VI28		828		// FBRST
#define VU0VI29		829		// VPU-STAT
#define VU0VI30		830		// reserved
#define VU0VI31		831		// CMSAR1

/* VU1 Registers */

#define VU1VF00		900
#define VU1VF01		901
#define VU1VF02		902
#define VU1VF03		903
#define VU1VF04		904
#define VU1VF05		905
#define VU1VF06		906
#define VU1VF07		907
#define VU1VF08		908
#define VU1VF09		909
#define VU1VF10		910
#define VU1VF11		911
#define VU1VF12		912
#define VU1VF13		913
#define VU1VF14		914
#define VU1VF15		915
#define VU1VF16		916
#define VU1VF17		917
#define VU1VF18		918
#define VU1VF19		919
#define VU1VF20		920
#define VU1VF21		921
#define VU1VF22		922
#define VU1VF23		923
#define VU1VF24		924
#define VU1VF25		925
#define VU1VF26		926
#define VU1VF27		927
#define VU1VF28		928
#define VU1VF29		929
#define VU1VF30		930
#define VU1VF31		931

#define VU1VI00		1000
#define VU1VI01		1001
#define VU1VI02		1002
#define VU1VI03		1003
#define VU1VI04		1004
#define VU1VI05		1005
#define VU1VI06		1006
#define VU1VI07		1007
#define VU1VI08		1008
#define VU1VI09		1009
#define VU1VI10		1010
#define VU1VI11		1011
#define VU1VI12		1012
#define VU1VI13		1013
#define VU1VI14		1014
#define VU1VI15		1015
#define VU1VI16		1016		// status flag
#define VU1VI17		1017		// MAC flag
#define VU1VI18		1018		// clipping flag
#define VU1VI19		1019		// reserved
#define VU1VI20		1020		// R
#define VU1VI21		1021		// I
#define VU1VI22		1022		// Q
#define VU1VI23		1023		// reserved
#define VU1VI24		1024		// reserved
#define VU1VI25		1025		// reserved
#define VU1VI26		1026		// TPC
#define VU1VI27		1027		// CMSAR0
#define VU1VI28		1028		// FBRST
#define VU1VI29		1029		// VPU-STAT
#define VU1VI30		1030		// reserved
#define VU1VI31		1031		// CMSAR1

#endif // #ifndef __SN_PS2_H_INCLUDED
