// []-----------------------------------------------------------------------------------------------------------[]
//	|	File				:	ui.c																																												|
//	|																																																						|
//	|								Copyright(c) 2000 Sega Europe.																															|
//	|																																																						|
//	|	Author			:	Elliott Martin, Sega Europe (martine@soe.sega.co.uk).																				|
//	|																																																						|
//	|	Description	:	Public API functions used to control and process the state machine.													|
// []-----------------------------------------------------------------------------------------------------------[]

// TAB WIDTH == 2

// Header files:
// -------------
#include <shinobi.h>
#include <transport.h>
#include <ngppp.h>
#include <ngdns.h>
#include <ngeth.h>

#ifdef TR_USE_DHCP
#include <ngbootpc.h>	// For 'TAG_HOST_NAME' definition.
#include <ngdhcpc.h>
#endif

#ifdef TR_USE_PPPOE
#include <ngpppoe.h>
#endif

#include "states.h"
#include "probe.h"
#include "utils.h"
#include "debug.h"


// Macro definitions:
// ------------------
// Macros for adding the user-defined callback functions to the state data structure, or for pointing
// the state handling function to a void function if no callback is specified (NULL pointer)...
#define TR_SET_CALLBACK(state_func, callback_func)	if(callback_func) state_func = callback_func;	\
																										else							state_func = VoidFunc;

// Private function prototypes:
// ----------------------------
static void VoidFunc(void);


// External variables:
// -------------------
extern NGcfgent			g_trDCLanStackCfg[];
extern NGcfgent			g_trSerDevStackCfg[];
extern NGcfgent			g_trIntMdmStackCfg[];
extern NGdev				g_trDev;
extern NGdevcb			*g_trDevcb;
extern NGifnet			g_trPppIfnet;
extern NGifnet			*g_trIfnet;
extern NGmdmstate		g_trMdmstate;
extern NGethifnet		g_trDhcpIfnet;

#ifdef TR_USE_DHCP
extern NGuint				g_trDhcpUp;
#endif

#ifdef TR_USE_PPPOE
extern NGpppoeifnet	g_trPppoeIfnet;
#endif

extern TR_FUNC_PTR	g_trfpStateHandlingFunc;
extern TR_FUNC_PTR	g_trfpProbeFinishedCallback;
extern TR_FUNC_PTR	g_trfpDialFailureCallback;
extern TR_FUNC_PTR  g_trfpConnectionCallback;
extern TR_FUNC_PTR	g_trfpDisconnectCallback;
extern TR_FUNC_PTR	g_trfpDeviceResetCallback;
extern bool					g_trStackInit;
extern int					g_trTransport;
extern int					g_trDisconnectStatus;
extern char					g_trConnectSpeedSerial[];
extern char					*g_trConnectSpeedDCLan[];


// Private global variables:
// -------------------------
const		char	g_trTransport_Build[]	= "\nTransport Ver 1.01 Build:"__DATE__" "__TIME__"\n";	// Lib version string.
static	bool	g_trLibraryInit				= false;	// Flags when the library has been initialised.


// Public global variables:
// ------------------------
NetworkInfo3		*g_trNetInfo;									// Used to store the info if connection is triggered during probing.
TR_DIAL_INFO		g_trDialInfo;									// Used to store information relative to each dial attempt.
struct in_addr	g_trFlashDns[2];							// Used to store the DNS settings retrieved from flash RAM.
bool						g_trInstantConnect;						// If not false, connect straight after a device has been found.
int							g_trProbeDevices;							// Bit-field indicating which devices the user wishes to probe for.
int							g_trDevicesFound;							// Bit-field of all the transport devices found during probing.
#ifdef TR_USE_DHCP
struct in_addr	g_trFlashGateway;   					// Used to store the Gateway settings retrieved from flash RAM.
char						g_trFlashHost[MAX_HOST_STR];	// Used to store the machine's host name retrieved from flash RAM.
#endif
#ifdef TR_USE_PPPOE
char g_trAcName[MAX_AC_SERV_STR];							// Used to store the access concentrator name from flash RAM.
char g_trServName[MAX_AC_SERV_STR];						// Used to store the service name from flash RAM.
#endif




// ============================================================================================================ //




// ---------- //
//  trInit()  //
// ---------- //
void trInit(int probe_devices)
{
// Check if it is okay to initialise the library...
	if(g_trLibraryInit) return;
	else								g_trLibraryInit = true;

// Set-up the callbacks to point to void functions until the user configures them...
	TR_SET_CALLBACK(g_trfpProbeFinishedCallback,	NULL);
	TR_SET_CALLBACK(g_trfpDialFailureCallback,		NULL);
	TR_SET_CALLBACK(g_trfpConnectionCallback,			NULL);
	TR_SET_CALLBACK(g_trfpDeviceResetCallback,		NULL);
	TR_SET_CALLBACK(g_trfpDisconnectCallback,			NULL);

// Set-up the initial state of the state machine...
	g_trDevicesFound			= 0;									// No transport devices have been found yet.
	g_trProbeDevices			=	probe_devices;			// Store the bit-field of devices to probe for.
	g_trDialInfo.attempt	= TR_DIAL_PRIMARY;		// Initialise the dial attempt status.
	g_trInstantConnect		= false;							// Do not connect straight after probing has finished.
	g_trDisconnectStatus	= NG_EOK;							// Initialise the disconnection status flag.

// Set-up to start probing for a valid transport device...
	ProbeForNextDevice(NULL);
}




// -------------------------------- //
//  trSetOnProbeFinishedCallback()  //
// -------------------------------- //
void trSetOnProbeFinishedCallback(TR_FUNC_PTR user_cb)
{
// Copy the user-defined callback into the state machine...
	TR_SET_CALLBACK(g_trfpProbeFinishedCallback, user_cb);

// This callback is called to inform the application that all device probing has 
// finished and the stack is about to enter an idle state.  At this point the
// application may call trGetDevicesFound() to check which devices were detected
// during probing and, if need be, call trChooseTransportDevice() to have the
// stack select an appropriate transport device for connection.
}




// ------------------------------ //
//  trSetOnDialFailureCallback()  //
// ------------------------------ //
void trSetOnDialFailureCallback(TR_FUNC_PTR user_cb)
{
// Copy the user-defined callback into the state machine...
	TR_SET_CALLBACK(g_trfpDialFailureCallback, user_cb);

// This callback is called to inform the application that a dial attempt has failed
// so that it may update the user interface.  If the user has configured the flash
// RAM settings to attempt dialing of a backup number then the stack will proceed to
// do so once this function returns, until either a connection has succeeded or all
// attempts have failed (whereupon the disconnect status callback will be called).
}




// --------------------------- //
//  trSetConnectionCallback()  //
// --------------------------- //
void trSetConnectionCallback(TR_FUNC_PTR user_cb)
{
// Copy the user-defined callback into the state machine...
	TR_SET_CALLBACK(g_trfpConnectionCallback, user_cb);

// This callback is called repeatedly once the state machine has successfully
// established a connection to an ISP.  It is not necessary to use this callback
// for your own application specific code as you may wish to call a seperate
// function in the main program loop.
}




// ----------------------------- //
//  trSetOnDisconnectCallback()  //
// ----------------------------- //
void trSetOnDisconnectCallback(TR_FUNC_PTR user_cb)
{
// Copy the user-defined callback into the state machine...
	TR_SET_CALLBACK(g_trfpDisconnectCallback, user_cb);

// This callback is called every time a connection dies or a connection attempt
// fails.  At this point it is recommended that 'trGetDisconnectReason()' is called
// to find out the reason for the disconnection.
}




// ------------------------------ //
//  trSetOnDeviceResetCallback()  //
// ------------------------------ //
void trSetOnDeviceResetCallback(TR_FUNC_PTR user_cb)
{
// Copy the user-defined callback into the state machine...
	TR_SET_CALLBACK(g_trfpDeviceResetCallback, user_cb);

// This callback is called straight after a device reset to indicate to the
// application that a connection is now dead and the state machine is idle.
}




// --------------- //
//  trGetDevice()  //
// --------------- //
const NGdev *trGetDevice(void)
{
// Return a pointer to the device structure...
	return &g_trDev;
}




// ----------------- //
//  trGetDeviceCb()  //
// ----------------- //
const NGdevcb *trGetDeviceCb(void)
{
// Return a pointer to the device control block structure...
	return g_trDevcb;
}




// ------------------ //
//  trGetInterface()  //
// ------------------ //
const NGifnet *trGetInterface(void)
{
// Return a pointer to the interface structure...
	return g_trIfnet;
}




// ------------------- //
//  trGetModemState()  //
// ------------------- //
const NGmdmstate *trGetModemState(void)
{
// Return a pointer to the modem state structure...
	return &g_trMdmstate;
}




// --------------------- //
//  trGetDevicesFound()  //
// --------------------- //
int trGetDevicesFound(void)
{
// Return a bit-field containing all the devices found so far...
	return g_trDevicesFound;
}




// ---------------------- //
//  trGetCurrTransport()  //
// ---------------------- //
int trGetCurrTransport(void)
{
//	g_trTransport == TR_TRANSPORT_NONE : A transport device hasn't been selected by the stack yet.
//	g_trTransport >  TR_TRANSPORT_NONE : The stack has selected the specified transport device.

	return g_trTransport;
}




// ----------- //
//  trYield()  //
// ----------- //
void trYield(void)
{
// Check if the stack has been initialised and is therefore safe to poll...
	if(!g_trStackInit) return;

// Poll the NexGenIP stack...
	ngYield();

// Process the current stack state...
	switch(trGetStackState()) {
	case TR_STATE_CONNECTED:	// Currently in the user-defined connection loop:
		if(g_trTransport & TR_COMPONENT_PPP) {
			if(ngPppGetState(g_trIfnet) == NG_PIFS_DEAD) {	// The remote peer has dropped the connection...

			#ifdef TR_USE_PPPOE
			// If DC LAN is being used, shutdown the PPPoE system (do not close the interface yet)...
				if(g_trTransport & TR_COMPONENT_ETHERNET) ngPppoeStop(g_trIfnet);
        else 
			#endif	// TR_USE_PPPOE
				
				ngIfClose(g_trIfnet);
				
			// Reset the device...
				SetDisconnectStatus(TR_STATUS_LINE_DISCONNECTED);
				SetStackState(TR_STATE_RESET_DEV);
				return;
			}
		}

		break;

	case TR_STATE_IDLE: return;	// Ignore these states as they don't require any extra processing:
	}

// Call the current state handling function:
	g_trfpStateHandlingFunc();
}




// ------------------- //
//  trGetStackState()  //
// ------------------- //
int trGetStackState(void)
{
// Return an integer representing the current stack state...
	if(g_trfpStateHandlingFunc == StateProbeDCLan)					return TR_STATE_PROBE_DCLAN;
	if(g_trfpStateHandlingFunc == StateProbeExtModem)				return TR_STATE_PROBE_EXTMODEM;
	if(g_trfpStateHandlingFunc == StateProbeSerialPPP)			return TR_STATE_PROBE_SERIALPPP;
	if(g_trfpStateHandlingFunc == StateProbeIntModem)				return TR_STATE_PROBE_INTMODEM;
	if(g_trfpStateHandlingFunc == StatePollDev)							return TR_STATE_POLL_DEV;
	if(g_trfpStateHandlingFunc == StateResetDevice)					return TR_STATE_RESET_DEV;
	if(g_trfpStateHandlingFunc == StatePollPpp)							return TR_STATE_POLL_PPP;
	if(g_trfpStateHandlingFunc == g_trfpConnectionCallback)	return TR_STATE_CONNECTED;

// If none of the above apply then the stack is idle...
	else return TR_STATE_IDLE;
}




// --------------------- //
//  trGetFlashNetInfo()  //
// --------------------- //
int trGetFlashNetInfo(void *work_area, NetworkInfo3 *net_info)
{
	int err;

// Initialise the network info library (reads the data from flash RAM)...
	err = ntInfInit(net_info, work_area);

	switch(err) {
	case NTD_ERR_NOTINIT:
		DEBUG_PRINT(("Could not read any network info from flash RAM ('NTD_ERR_NOTINIT')\n"));
		return err;

	case NTD_ERR_INVALIDPARAM:
		DEBUG_PRINT(("Could not read any network info from flash RAM ('NTD_ERR_INVALIDPARAM')\n"));
		return err;

	case NTD_ERR_NOINFO:
		DEBUG_PRINT(("Could not read any network info from flash RAM ('NTD_ERR_NOINFO')\n"));
		return err;

	default: return err;	// 'NTD_OK'
	}
}




// ---------------------- //
//  trFreeFlashNetInfo()  //
// ---------------------- //
void trFreeFlashNetInfo(void)
{
// Shutdown the network info library.  This will free any memory that may have
// been allocated internally by the library (rather than by the application)...
	ntInfExit();
}




// --------------------------- //
//  trChooseTransportDevice()  //
// --------------------------- //
int trChooseTransportDevice(void)
{
	bool		use_ext_modem = ntInfExternalModem();	// Enable use of external modem if true (override built-in modem).
	Uint32	flag;																	// Flag in flash RAM to check which ISP info is enabled.

// Choose which transport device to use based on the criteria below...
	if(ntInfGetFlag(&flag) != NTD_OK) return TR_TRANSPORT_NONE;

// Test the DC LAN card...
	if(g_trDevicesFound & TR_DEVICE_DCLAN) {
		if(flag & USE_LAN) return TR_TRANSPORT_DCLAN_STATIC;
	}

// Test the external modem...
	if(g_trDevicesFound & TR_DEVICE_EXTMODEM) {
		if((!(flag & ISP_USE1) || (flag & ISP_USE2)) && use_ext_modem) return TR_TRANSPORT_EXTMODEM;
	}

// Test the RAS serial PPP connection...
	if(g_trDevicesFound & TR_DEVICE_RASSERIALPPP) return TR_TRANSPORT_RASSERIALPPP;

// Test the serial PPP connection...
	if(g_trDevicesFound & TR_DEVICE_SERIALPPP) return TR_TRANSPORT_SERIALPPP;
	
// Test the internal modem...
	if(g_trDevicesFound & TR_DEVICE_INTMODEM) {
		if((!(flag & ISP_USE1) || (flag & ISP_USE2)) && !use_ext_modem) return TR_TRANSPORT_INTMODEM;
	}

// No transport device was chosen...
	return TR_TRANSPORT_NONE;
}




// ------------- //
//  trConnect()  //
// ------------- //
int trConnect(int device, NetworkInfo3 *net_info)
{
	Uint32	flag;	// Stores the flag field read from flash RAM areas.
	int			val;	// Used to change the stack configuration options.
	int			err;

// If the user has chosen to connect, but we are still probing, then set-up to do so the moment a device is found...
	if(g_trProbeDevices != TR_DEVICE_PROBE_FINISHED) {
		g_trInstantConnect	= true;
		g_trNetInfo					= net_info;
		g_trTransport				= device;
		return NG_EOK;
	}

// Check if the stack should pick an appropriate transport device...
	if(device == TR_TRANSPORT_AUTO) g_trTransport = trChooseTransportDevice();
	else														g_trTransport = device;

// Initialise the selected transport device...
	switch(g_trTransport) {
	case TR_TRANSPORT_DCLAN_STATIC:
	case TR_TRANSPORT_DCLAN_DHCP:
	case TR_TRANSPORT_DCLAN_PPPOE:
		err = InitStack(g_trDCLanStackCfg, (NGifnet *)&g_trDhcpIfnet, "DC LAN");
		break;

	case TR_TRANSPORT_EXTMODEM:
		err = InitStack(g_trSerDevStackCfg, (NGifnet *)&g_trPppIfnet, "External Modem");
		break;

	case TR_TRANSPORT_RASSERIALPPP:
		err = InitStack(g_trSerDevStackCfg, (NGifnet *)&g_trPppIfnet, "RAS Serial");
		break;

	case TR_TRANSPORT_SERIALPPP:
		err = InitStack(g_trSerDevStackCfg, (NGifnet *)&g_trPppIfnet, "PPP Serial");
		break;

	case TR_TRANSPORT_INTMODEM:
		err = InitStack(g_trIntMdmStackCfg, (NGifnet *)&g_trPppIfnet, "Internal Modem");
		break;

	default: err = NG_EINVAL;	// No transport device could be established.
	}

	if(err) goto error;

// If this is a modem connection then set-up the AT init and dial strings in the modem script...
  if(g_trTransport & TR_COMPONENT_MODEM) {
		char dial[MAX_DIAL_STR];	// Stores the AT command and dial strings in flash RAM.

	// Get the correct number to dial from flash RAM...
		switch(g_trDialInfo.attempt) {
		case TR_DIAL_PRIMARY:
			err = ntInfGetDialString(0, TR_DIAL_PRIMARY, dial, sizeof(dial));

			if(err) { DEBUG_PRINT(("Could not retrieve the primary dial string from flash RAM (%d)\n", err)); }
			
			else {
				g_trDialInfo.net_info	= net_info;	// Dereference pointer to net info in case of more connection attempts.
				SetDisconnectStatus(NG_EOK);			// New connection - reset the disconnection error status.
			}

			break;

		case TR_DIAL_BACKUP1:
			err = ntInfGetDialString(0, TR_DIAL_BACKUP1, dial, sizeof(dial));

			if(err) { DEBUG_PRINT(("Could not retrieve the first backup dial string from flash RAM (%d)\n", err)); }
			else			SetDisconnectStatus(NG_EOK);	// Reset the disconnection error status for this attempt.
			break;

		case TR_DIAL_BACKUP2:
			err = ntInfGetDialString(0, TR_DIAL_BACKUP2, dial, sizeof(dial));

			if(err) { DEBUG_PRINT(("Could not retrieve the second backup dial string from flash RAM (%d)\n", err)); }
			else			SetDisconnectStatus(NG_EOK);	// Reset the disconnection error status for this attempt.
			
			break;
		}

	// If there was no error then pass the dial string to the stack...
		if(!err) err = SetupDialParams(g_trTransport, dial, sizeof(dial));

		if(err) {
			err = NG_EINVAL;
  		SetDisconnectStatus(TR_STATUS_ERR_DIAL_PARAMS);
			goto error;
		}
	}

// If this is a PPP connection then set-up the login details...
	if(g_trTransport & TR_COMPONENT_PPP) {
		struct in_addr ip;	// Stores the static IP address in flash RAM.

	// If this is a direct serial connection then the disconnection error status has NOT been reset...
		if((g_trTransport & TR_COMPONENT_SERIAL) && !(g_trTransport & TR_COMPONENT_MODEM)) {
			g_trDialInfo.attempt = TR_DIAL_NONE;	// This counter is not relevant to direct serial connections.
			SetDisconnectStatus(NG_EOK);
		}

	// Enter the user login details into the stack...
		err = SetupLoginParams(g_trIfnet, 0);

		if(err) {
			DEBUG_PRINT(("Could not set-up the PPP login params (%d)\n", err));
  		SetDisconnectStatus(TR_STATUS_ERR_LOGIN_PARAMS);
			goto error;
		}

	// Get the flag for the extended info area of flash RAM...
		err = ntInfGetNetworkAccessInfo2Flag(0, &flag);

		if(err) { DEBUG_PRINT(("Could not retrieve the extended info flag from flash RAM (%d)\n", err)); }

		else {	// Extended info may have been configured...
		// Check if static IP addressing should be used...
			ntInfGetIPAddr(0, &ip);

			if(ip.s_addr && (flag & NA2_FIXED_ADDRESS)) {
				val = NG_CFG_FALSE;

			// Set-up to reject the IP address returned by the PPP server and use the one stored in flash RAM...
				err = ngIfSetOption(g_trIfnet, NG_PPPIFO_IPCP_ACCEPT_LOCAL, (void *)&val);
				if(err)	{ DEBUG_PRINT(("Could not disable NG_PPPIFO_IPCP_ACCEPT_LOCAL (%d)\n", err)); }

				err = ngIfSetOption(g_trIfnet, NG_IFO_ADDR,									(void *)&ip.s_addr);
				if(err)	{ DEBUG_PRINT(("Could not store the static IP address in the stack (%d)\n", err)); }
			}

		// Check if Van Jacobson TCP header compression should be used...
			if(flag & NA2_NO_HEADERCOMP) {
				val	= NG_CFG_FALSE;
				err	= ngIfSetOption(g_trIfnet, NG_PPPIFO_IPCP_VJCOMP, (void *)&val);
				if(err)	{ DEBUG_PRINT(("Could not disable Van Jacobson TCP header compression (%d)\n", err)); }
			}
		}

	// Store the flash RAM DNS servers for use later...
		ntInfGetPrimaryDnsAddress(0,		&g_trFlashDns[0]);
		ntInfGetSecondaryDnsAddress(0,	&g_trFlashDns[1]);
	}

// If DHCP is enabled then set-up the host name before connecting...
	if(g_trTransport & TR_COMPONENT_ETHERNET) {
		struct in_addr	gateway1, gateway2;															// Used to store the static gateways.
		char						user[MAX_USERNAME_STR], pass[MAX_PASSWORD_STR];	// Used to store the PPPoE login details.

		g_trDialInfo.attempt = TR_DIAL_NONE;	// This counter is not relevant to DC LAN connections.
		SetDisconnectStatus(NG_EOK);					// New connection - reset the disconnection error status.

	// Get the flag for the TCP/IP area of flash RAM...
		err = ntInfGetTCPIPInfoFlag(&flag);

    if(err) {
			DEBUG_PRINT(("Could not retrieve the TCP/IP info flag from flash RAM (%d)\n", err));
  		SetDisconnectStatus(TR_STATUS_ERR_DCLAN_FAILED);
			err = NG_ENOENT;
			goto error;
		}

	// Check if Van Jacobson TCP header compression is enabled in flash RAM...
	#ifdef TR_USE_PPPOE
		if(flag & TCP_DYNAMIC_NO_HEADERCOMP) {
			val	= NG_CFG_FALSE;
			err = ngIfSetOption((NGifnet *)&g_trPppoeIfnet, NG_PPPIFO_IPCP_VJCOMP, (void *)&val);
			if(err)	{ DEBUG_PRINT(("Could not disable VJ TCP header compression on DC LAN (%d)\n", err)); }
		}
	#endif	// TR_USE_PPPOE

	// Check the flags in flash RAM to see if we should use a purely static configuration...
		if((flag & TCP_DYNAMIC_NO_DHCP) && !(flag & TCP_DYNAMIC_PPPoE)) {
			DEBUG_PRINT(("DHCP and PPPoE are disabled in flash RAM.  Trying a static configuration...\n"));

		// Set-up the static DC LAN configuration...
			err = ConfigStaticDCLan(g_trIfnet, flag);

			if(err) {
				SetDisconnectStatus(TR_STATUS_ERR_DCLAN_FAILED);
				goto error;
			}

		// Flag that the stack is now connected...
			SetStackState(TR_STATE_CONNECTED);
			return NG_EOK;
		}

	#ifdef TR_USE_PPPOE
    if(flag & TCP_DYNAMIC_PPPoE) {
			char *ac = g_trAcName, *serv = g_trServName;

		  err = SetupLoginParams((NGifnet *)&g_trPppoeIfnet, 3);
			if(err) { DEBUG_PRINT(("Could not set-up the PPPoE login params (%d)\n", err)); }

		// Set-up the access concentrator and service names...
			else {
				ntInfGetAcName(g_trAcName,				sizeof(g_trAcName));
				ntInfGetServiceName(g_trServName,	sizeof(g_trServName));

				if(g_trAcName[0]) {
					err = ngIfSetOption((NGifnet *)&g_trPppoeIfnet, NG_PPPOEIFO_ACNAME, (void *)&ac);
					if(err) { DEBUG_PRINT(("Could not set the access concentrator name (%d)\n", err)); }
				}

				if(g_trServName[0]) {
					err = ngIfSetOption((NGifnet *)&g_trPppoeIfnet, NG_PPPOEIFO_SERVNAME, (void *)&serv);
					if(err) { DEBUG_PRINT(("Could not set the required service name (%d)\n", err)); }
				}
			}

		  if(err) {
    		SetDisconnectStatus(TR_STATUS_ERR_PPPOE_INIT);
				goto error;
		  }
    }
	#endif	// TR_USE_PPPOE

	#ifdef TR_USE_DHCP
	// Store all the flash settings into the globals...
		err = ntInfGetPrimaryGateway(&g_trFlashGateway);
		if(err) { DEBUG_PRINT(("Could not retrieve the DC LAN static gateway from flash RAM (%d)\n", err)); }

		err = ntInfGetPrimaryDnsAddress(3, &g_trFlashDns[0]);
		if(err) { DEBUG_PRINT(("Could not retrieve the DNS address from flash RAM (%d)\n", err)); }
    ntInfGetSecondaryDnsAddress(3, &g_trFlashDns[1]);

		err = ntInfGetHost(g_trFlashHost, sizeof(g_trFlashHost));
    if(err) { DEBUG_PRINT(("Could not retrieve the Host Name from flash RAM (%d)\n", err)); }

	#endif	// TR_USE_DHCP
	}

// Start the connection procedure...
	StartDev();
	return NG_EOK;

// An error occured in the code above...
error:
	g_trTransport = TR_TRANSPORT_NONE;
	SetStackState(TR_STATE_RESET_DEV);
  return err;
}




// ---------------- //
//  trDisconnect()  //
// ---------------- //
void trDisconnect(void)
{
	int err;

// Start the disconnection procedure depending on which state the stack is in...
	switch(trGetStackState()) {
	case TR_STATE_POLL_DEV:	// Cancelling connection during dial-up:
		if(g_trTransport & TR_COMPONENT_ETHERNET) {	// If the DC LAN card is being used...

		#ifdef TR_USE_DHCP
			err = ShutdownDhcp((NGifnet *)&g_trDhcpIfnet, &g_trDhcpUp);	// Shutdown the DHCP system.
			if(err) { DEBUG_PRINT(("Could not shutdown the DHCP service (%d)\n", err)); }
		#endif

		#ifdef TR_USE_PPPOE
			err = ngPppoeStop((NGifnet *)&g_trPppoeIfnet);	// Shutdown the PPPoE system.
			if(err) { DEBUG_PRINT(("Could not shutdown the PPPoE service (%d)\n", err)); }
		#endif
		}
		
		else {
			ngDevioClose(g_trDevcb);							// Close the modem/serial device.
			g_trDialInfo.attempt	= TR_DIAL_NONE;	// Prevent further dial attempts.
		}

	// Reset the system...
		SetDisconnectStatus(TR_STATUS_USER_CANCELLED);
		SetStackState(TR_STATE_RESET_DEV);
		break;

	case TR_STATE_POLL_PPP:	// Cancelling connection during PPP negotiation:
		SetDisconnectStatus(TR_STATUS_USER_CANCELLED);
		ngPppStop(g_trIfnet);
		break;

	case TR_STATE_CONNECTED:	// Normal disconnection:
		SetDisconnectStatus(TR_STATUS_USER_DISCONNECTED);

		if(g_trTransport == TR_TRANSPORT_DCLAN_STATIC)	// If this is a static LAN connection...
			SetStackState(TR_STATE_RESET_DEV);

		else if(g_trTransport & TR_COMPONENT_DHCP) {	// If this is a DHCP based connection...

		#ifdef TR_USE_DHCP
			err = ShutdownDhcp(g_trIfnet, &g_trDhcpUp);
			if(err) { DEBUG_PRINT(("Could not shutdown the DHCP service (%d)\n", err)); }
		#endif	// TR_USE_DHCP
		
			SetStackState(TR_STATE_RESET_DEV);
		}

		else {	// Shutdown the PPP/PPPoE connection...
    // We must wait for PPP to shutdown properly before resetting...
			SetStackState(TR_STATE_POLL_PPP);
			ngPppStop(g_trIfnet);
		}

		break;
	}
}




// --------------------- //
//  trGetConnectSpeed()  //
// --------------------- //
const char *trGetConnectSpeed(void)
{
// If a DC LAN card is being used then return the connection speed...
	if(g_trTransport & TR_COMPONENT_ETHERNET) {
		int index = GetDCLanConnectSpeed();	// Index into the string array of the DC LAN card's connection speed.
		return g_trConnectSpeedDCLan[index];
	}

// If a modem is being used then return the down/up link speed...
// If a serial connection is being used so return the baud rate...
	else if(*g_trConnectSpeedSerial != '\0')
		return g_trConnectSpeedSerial;	// This was already retrieved during the dial-up process/initialsation.

// If an error occured above then return a NULL pointer...
	return NULL;
}




// ------------------- //
//  trGetDnsServers()  //
// ------------------- //
int trGetDnsServers(NGuint *dns1, NGuint *dns2)
{
// If DNS has been disabled then return an error ('ngDns_data' is externally declared in ngdns.h)...
	if(!ngDns_data.dns_addr1) {
		if(dns1)	*dns1 = 0;
		if(dns2)	*dns2 = 0;
	
		return NG_EFAULT;
	}

	else {	// Return the valid DNS server addresses...
		if(dns1)	*dns1 = ngDns_data.dns_addr1;
		if(dns2)	*dns2 = ngDns_data.dns_addr2;

		return NG_EOK;
	}
}




// ------------------------- //
//  trGetDisconnectReason()  //
// ------------------------- //
int trGetDisconnectReason(void)
{
  return g_trDisconnectStatus;
}




// ---------- //
//  trExit()  //
// ---------- //
int trExit(void)
{
// We can only shutdown the stack if it is in an idle state...
	if(trGetStackState() != TR_STATE_IDLE) return NG_EBUSY;

// Reset the state machine...
  g_trIfnet					= NULL;
	g_trDevicesFound	= TR_TRANSPORT_NONE;
	g_trTransport			= TR_TRANSPORT_NONE;
	g_trLibraryInit		= false;
	
	SetDisconnectStatus(NG_EOK);
	return NG_EOK;
}




// ------------ //
//  VoidFunc()  //
// ------------ //
static void VoidFunc(void)
{
	// This function is used to ensure that any state callback functions that have not been
	// defined by the application (i.e. NULL pointers) still go somewhere when they are called.
	// This prevents the need to check for a NULL function pointer every time the callback is used.
}