// []-----------------------------------------------------------------------------------------------------------[]
//	|	File				:	probe.c																																											|
//	|																																																						|
//	|								Copyright(c) 2000 Sega Europe.																															|
//	|																																																						|
//	|	Author			:	Elliott Martin, Sega Europe (martine@soe.sega.co.uk).																				|
//	|																																																						|
//	|	Description	:	State machine functions to probe for valid transport devices.																|
// []-----------------------------------------------------------------------------------------------------------[]

// TAB WIDTH == 2

// Header files:
// -------------
#include <shinobi.h>
#include <transport.h>
#include <ngappp.h>
#include <ngeth.h>
#include "states.h"
#include "utils.h"
#include "probe.h"
#include "debug.h"


// Function prototypes:
// --------------------
static void DeviceProbingFinished(void);


// Enumerated types:
// -----------------
enum {	// Probe states...
	PROBE_STATE_INIT_SCR = 1,	// Start the probing procedure for a particular device.
	PROBE_STATE_POLL_SCR,			// Poll the device script until complete.
  PROBE_STATE_START_PPP,		// Start PPP negotiation.
  PROBE_STATE_POLL_PPP,			// Poll the PPP interface until the required state is reached.
};


// External variables:
// -------------------
extern TR_FUNC_PTR	g_trfpStateHandlingFunc;
extern TR_FUNC_PTR	g_trfpProbeFinishedCallback;
extern TR_FUNC_PTR  g_trfpConnectionCallback;
extern NGcfgent			g_trDCLanStackCfg[];
extern NGcfgent			g_trSerDevStackCfg[];
extern NGcfgent			g_trIntMdmStackCfg[];
extern NGdevcb			*g_trDevcb;
extern NGifnet			*g_trIfnet;
extern NGifnet			g_trDhcpIfnet;
extern NGifnet			g_trPppIfnet;
extern NGmdmstate		g_trMdmstate;
extern NGmdmscript	g_trSerRasScript[];
extern NetworkInfo3	*g_trNetInfo;
extern bool					g_trInstantConnect;
extern int					g_trDevicesFound;
extern int					g_trTransport;
extern int					g_trProbeDevices;

#ifdef DEBUG
extern	NGdev		g_trSerDbgDev;
				NGdevcb	*g_trDevcbStdout = NULL;	// Used to output debug over a serial link.
#endif


// Private global variables:
// -------------------------
static TR_PROBE_FUNC_PTR *g_trProbeAutomaton[] = {	// Stores the order in which call the device probing functions.
  StateProbeDCLan,
  StateProbeExtModem,
  StateProbeSerialPPP,
  StateProbeIntModem,
	DeviceProbingFinished,
  NULL
};

static NGmdmscript g_trMdmProbeScript[] =	{	// Modem script (simple test for the existence of a modem).
// Action			Answer		Output		Goto		Retval		Timeout(secs)
	{"ATE0",		"OK",			NULL,			-1,			NG_EOK,			1},
	{NULL,			"ERROR",	NULL,			-1,			NG_EINVAL,	1},
	{NULL,			NULL,			NULL,			-1,			0,					0}
};

static int	g_trCurrProbeScriptState;	// Stores the current state of the external modem and serial probe scripts.


// Public global variables:
// ------------------------
bool g_trStackInit = false;	// Flags when the stack has been initialised.




// ============================================================================================================ //




// ---------------------- //
//  ProbeForNextDevice()  //
// ---------------------- //
void ProbeForNextDevice(TR_PROBE_FUNC_PTR *just_probed)
{
  TR_PROBE_FUNC_PTR **probe_next = g_trProbeAutomaton, *last_probed = NULL;

// Reset the probe script state (only needed by external modem and serial connection)...
	g_trCurrProbeScriptState = PROBE_STATE_INIT_SCR;

// Find out which function in the automaton to call next...
  while(*probe_next) {
    if(last_probed == just_probed)	break;
		else														last_probed = *probe_next++;
  };

// Set-up the probe function to be called by the state machine...
  g_trfpStateHandlingFunc = *probe_next;
  
// Call the next probe function immediately in order to initialise it before it is polled by the state machine...
	if(g_trfpStateHandlingFunc) g_trfpStateHandlingFunc();
}




// ------------------- //
//  StateProbeDCLan()  //
// ------------------- //
void StateProbeDCLan(void)
{
// Probe for the DC LAN card if required...
	if(g_trProbeDevices & TR_DEVICE_DCLAN) {
		int err;

	// Probe for a DC LAN card.  If an error occurs, a bit-field of all NG error values is returned...
		err = InitStack(g_trDCLanStackCfg, (NGifnet *)&g_trDhcpIfnet, NULL);
		
		if(err) { DEBUG_PRINT(("Could not find a Dreamcast LAN card (%d)\n", err)); }
		else			g_trDevicesFound |= TR_DEVICE_DCLAN;

		ngExit(0);							// Free the current stack resources in use.
		g_trStackInit = false;	// Flag that the stack has been shutdown.
	}

// Probe for the next transport device...
	ProbeForNextDevice(StateProbeDCLan);
}




// ---------------------- //
//  StateProbeExtModem()  //
// ---------------------- //
void StateProbeExtModem(void)
{
// Don't probe for an external modem if we have already found a serial connection...
// (This is in case the code is changed to probe for a serial connection before and external modem.)
	if(g_trDevicesFound & (TR_DEVICE_RASSERIALPPP | TR_DEVICE_SERIALPPP))
		g_trProbeDevices &= ~TR_DEVICE_EXTMODEM;

// Probe for an external modem if required...
	if(g_trProbeDevices & TR_DEVICE_EXTMODEM) {
		int	err;

	// Run the probing procedure...
		switch(g_trCurrProbeScriptState) {
		case PROBE_STATE_INIT_SCR:
		// Probe for an external modem.  If an error occurs, a bit-field of all NG error values is returned:
			err = InitStack(g_trSerDevStackCfg, (NGifnet *)&g_trPppIfnet, NULL);

			if(!err) {
				err = InitModemScript(g_trMdmProbeScript);
				
				if(!err) {
					SetupSerialPort();
					g_trCurrProbeScriptState = PROBE_STATE_POLL_SCR;	// Start polling the probe script.
					return;
				}

				DEBUG_PRINT(("Could not initialise the modem probe script (%d)\n", err));
			}

			DEBUG_PRINT(("Could not initialise the stack for the external modem (%d)\n", err));
			break;

		case PROBE_STATE_POLL_SCR:	// Poll the modem script:
			err = ngModemPoll(&g_trMdmstate);
			
			switch(err) {
			case NG_EWOULDBLOCK: return;	// Continue polling the script:

			case NG_EOK:	// An external modem was found!
        g_trDevicesFound |= TR_DEVICE_EXTMODEM;
				break;

			default:	// An undefined state (script timeout?):
				DEBUG_PRINT(("Could not find an external modem (%d)\n", err));
				break;
			}
			
			break;
		}

	// Clean up...
		ngDevioClose(g_trDevcb);
		ngExit(0);							// Free the current stack resources in use.
		g_trStackInit = false;	// Flag that the stack has been shutdown.
	}

// Probe for the next transport device...
	ProbeForNextDevice(StateProbeExtModem);
}




// ---------------------- //
//  StateProbeIntModem()  //
// ---------------------- //
void StateProbeIntModem(void)
{
// Probe for an internal modem if required...
	if(g_trProbeDevices & TR_DEVICE_INTMODEM) {
		int err;

	// Probe for an internal modem.  If an error occurs, a bit-field of all NG error values is returned...
		err = InitStack(g_trIntMdmStackCfg, (NGifnet *)&g_trPppIfnet, NULL);

		if(err) { DEBUG_PRINT(("Could not find an internal modem (%d)\n", err)); }
		else			g_trDevicesFound |= TR_DEVICE_INTMODEM;
		
		ngExit(0);							// Free the current stack resources in use.
		g_trStackInit = false;	// Flag that the stack has been shutdown.
	}

// Probe for the next transport device...
	ProbeForNextDevice(StateProbeIntModem);
}




// ----------------------- //
//  StateProbeSerialPPP()  //
// ----------------------- //
void StateProbeSerialPPP(void)
{
	static	bool					ppp_up					= false;							// Flags when a PPP session has been established.
	static	unsigned int	t_start;															// Used to time the PPP session.
	static	unsigned int	t_timeout;														// Stores the timeout value for PPP negotiation.
	static	int						curr_transport	= TR_TRANSPORT_NONE;	// Stores the current transport device found.
					int						err;

// Don't probe for a serial connection if we have already found an external modem...
	if(g_trDevicesFound & TR_DEVICE_EXTMODEM)
		g_trProbeDevices &= ~(TR_DEVICE_RASSERIALPPP | TR_DEVICE_SERIALPPP);

// Probe for a direct serial PPP connection if required...
	if((g_trProbeDevices & TR_DEVICE_RASSERIALPPP) || (g_trProbeDevices & TR_DEVICE_SERIALPPP)) {

		switch(g_trCurrProbeScriptState) {
		case PROBE_STATE_INIT_SCR:
		// Probe for a serial connection.  If an error occurs, a bit-field of all NG error values is returned...
			err = InitStack(g_trSerDevStackCfg, (NGifnet *)&g_trPppIfnet, NULL);

			if(!err) {
				err = InitModemScript(g_trSerRasScript);
				
				if(!err) {
					SetupSerialPort();
					g_trCurrProbeScriptState = PROBE_STATE_POLL_SCR;	// Start polling the serial RAS script.
					return;
				}

				DEBUG_PRINT(("Could not initialise the RAS modem script for the serial PPP connection (%d)\n", err));
			}

			DEBUG_PRINT(("Could not initialise the stack for the serial PPP connection (%d)\n", err));
			break;

		case PROBE_STATE_POLL_SCR:
			err	= ngModemPoll(&g_trMdmstate);
			
			switch(err) {
			case NG_EWOULDBLOCK: return;	// Continue polling the script:

			case TR_RAS_SCR_CONNECTED:	// A RAS server was found!
				curr_transport = TR_DEVICE_RASSERIALPPP;
				break;

			default:	// The RAS script failed:
				DEBUG_PRINT(("Could not find a RAS server (%d).  Trying for a generic PPP server...\n", err));
				break;
			}	
			
		// Store the baud rate of the serial port...
			err = StoreSerialBaudRate();

			if(err)	{ DEBUG_PRINT(("Could not get the baud rate of the serial port (%d)\n", err)); }

			ngDevioClose(g_trDevcb);
			g_trCurrProbeScriptState = PROBE_STATE_START_PPP;
			return;

		case PROBE_STATE_START_PPP:	// Start a PPP session to reset the RAS connection:
			err = ngIfOpen(g_trIfnet);

			if(err) {
				DEBUG_PRINT(("Could not open the serial PPP interface (%d)\n", err));
				break;
			}

		// Start the PPP session (reset RAS connection)...
			err = ngPppStart(g_trIfnet);

			if(err) {
				DEBUG_PRINT(("Could not start a PPP session on the serial PPP interface (%d)\n", err));
				ngIfClose(g_trIfnet);
				break;
			}

		// Add a dummy username and password to the stack configuration (this is required
		// for serial PPP connections to work with a RAS server under Windows)...
			if(curr_transport == TR_DEVICE_RASSERIALPPP) {
				char *dummy = "dummy";

				err = ngIfSetOption(g_trIfnet, NG_PPPIFO_AUTH_USER, &dummy);
				if(err) { DEBUG_PRINT(("Could not set dummy username for serial PPP connection (%d)\n", err)); }
				
				err = ngIfSetOption(g_trIfnet, NG_PPPIFO_AUTH_SECRET, &dummy);
				if(err) { DEBUG_PRINT(("Could not set dummy password for serial PPP connection (%d)\n", err)); }
			}

			g_trCurrProbeScriptState	= PROBE_STATE_POLL_PPP;
			t_start										= ngOSClockGetTime();	// Start the timer.
			t_timeout									=	4 * NG_CLOCK_FREQ;	// PPP init timeout (4 secs).
			return;

		case PROBE_STATE_POLL_PPP:	// Process the current PPP state:
			switch(ngPppGetState(g_trIfnet)) {		
			case NG_PIFS_AUTH:
			case NG_PIFS_NETWORK:	
				ngPppStop(g_trIfnet);
				ppp_up = true;
				return;

			case NG_PIFS_DEAD:
			// If we haven't already found a RAS server then this must be a generic (Unix style) PPP server...
				if(ppp_up) {
					if(!curr_transport) curr_transport = TR_DEVICE_SERIALPPP;
					g_trDevicesFound |= curr_transport;
					ppp_up						= false;
				}
				else { DEBUG_PRINT(("Could not establish a serial PPP session (timeout?)\n")); }

				ngIfClose(g_trIfnet);
				break;

			default:	// Check to see if the PPP connection attempt has timed out:
				if((ngOSClockGetTime() - t_start) >= t_timeout) ngPppStop(g_trIfnet);
				return;
			}

			break;
		}

		ngExit(0);														// Free the current stack resources in use.
		g_trStackInit		= false;							// Flag that the stack has been shutdown.
		curr_transport	= TR_TRANSPORT_NONE;	// Reset the selected transport device.
	}

// Probe for the next transport device...
	ProbeForNextDevice(StateProbeSerialPPP);
}




// ------------------------- //
//  DeviceProbingFinished()  //
// ------------------------- //
static void DeviceProbingFinished(void)
{
// Display some debug info showing the devices found...
	if(g_trDevicesFound & TR_DEVICE_DCLAN)				{ DEBUG_PRINT(("Found TR_DEVICE_DCLAN\n"				)); }
	if(g_trDevicesFound & TR_DEVICE_EXTMODEM)			{ DEBUG_PRINT(("Found TR_DEVICE_EXTMODEM\n"			)); }
	if(g_trDevicesFound & TR_DEVICE_RASSERIALPPP)	{ DEBUG_PRINT(("Found TR_DEVICE_RASSERIALPPP\n"	)); }
	if(g_trDevicesFound & TR_DEVICE_SERIALPPP)		{ DEBUG_PRINT(("Found TR_DEVICE_SERIALPPP\n"		)); }
	if(g_trDevicesFound & TR_DEVICE_INTMODEM)			{ DEBUG_PRINT(("Found TR_DEVICE_INTMODEM\n"			)); }

// If serial debug is defined, a serial device has not been found and the serial
// device can be opened, then set-up to output debug to the serial port...
#ifdef DEBUG
  if(!(g_trDevicesFound & (TR_DEVICE_EXTMODEM | TR_DEVICE_RASSERIALPPP | TR_DEVICE_SERIALPPP))) {
		int err;

    err = ngDevioOpen(&g_trSerDbgDev, NG_O_NONBLOCK, &g_trDevcbStdout);

		if(err) {
			DEBUG_PRINT(("Could not open the serial device for debug output (%d)\n", err));
			g_trDevcbStdout = NULL;
		}
		else ngPrintf(" \n**** Serial port available for debug output ****\n \n");
	}
#endif

// Reset the device bit-field (used to check when probing has finished)...
	g_trProbeDevices = TR_DEVICE_PROBE_FINISHED;

// Call the application defined finished probing callback...
	g_trfpProbeFinishedCallback();

// Either start connecting straight away (if the user wishes to), or go into an idle state...
	if(g_trInstantConnect)	trConnect(g_trTransport, g_trNetInfo);
	else										SetStackState(TR_STATE_IDLE);
}




// ------------- //
//  InitStack()  //
// ------------- //
int InitStack(NGcfgent *cfg, NGifnet *ifnet, char *ifname)
{
	int err;

// Initialise the stack configuration...
	err = ngInit(cfg);

	if(err) { DEBUG_PRINT(("Could not initialise the stack for '%s' (%d)\n", ifname, err)); }

	else {
		g_trStackInit = true;		// Flag that the stack has been initialised.
		g_trIfnet			=	ifnet;	// Store the interface pointer into a global.

	// Set the interface name in the stack...
		if(ifname) {
			err = ngIfSetOption(ifnet, NG_IFO_NAME, (void *)&ifname);
			if(err) { DEBUG_PRINT(("Could not set the interface name '%s' (%d)\n", err)); }
		}
	}

	return err;
}