#include "weapon.h"
#include "globals.h"
#include "anim.h"
#include "debug4g.h"
#include "actor.h"
#include "db.h"
#include "actor.h"
#include "gameutil.h"
#include "trig.h"
#include "view.h"
#include "options.h"
#include "names.h"
#include "triggers.h"
#include "error.h"
#include "misc.h"
#include "screen.h"
#include "tile.h"

#define kShotgunBarrels		2
#define kVectorsPerBarrel	16
#define kMaxShotgunVectors	(kShotgunBarrels * kVectorsPerBarrel)

enum {
	kVoodooStabChest	= 0,
	kVoodooStabShoulder,
	kVoodooStabEye,
	kVoodooStabGroin,
	kVoodooStabSelf,
};

// Weapon States
enum {
	kFindWeapon			= -1,	// flag to find a loaded weapon

	kForkInitialize		= 0,
	kForkRaise,
	kForkIdle,
	kForkAttack,
	kForkLower,

	kTNTInitialize		= 0,
	kTNTBicOpen,
	kTNTBicIdle,
	kTNTRaise,
	kTNTIdle,
	kTNTLower,

	kCanInitialize		= 0,
	kCanBicOpen,
	kCanBicIdle,
	kCanIdle,
	kCanLower,
	kCanFire1,
	kCanFire2,
	kCanFire3,

	kShotInitialize		= 0,	// indicates first-time reload required
	kShotLoaded,
	kShotRaise,
	kShotIdle1,
	kShotIdle2,
 	kShotIdle3,
	kShotFire1,
	kShotFire2,
	kShotReload,
	kShotLower,

	kFlareInitialize	= 0,
	kFlareRaise,
	kFlareIdle,
	kFlareFire,
	kFlareReload,
	kFlareLower,

	kTommyInitialize	= 0,
	kTommyRaise,
	kTommyIdle,
	kTommyFire1,
	kTommyFire2,
	kTommyFire3,
	kTommyFire4,
	kTommyLower,

	kSpearInitialize	= 0,
	kSpearRaise,
	kSpearIdle1,
	kSpearIdle2,
	kSpearFire,
	kSpearReload,
	kSpearLower,

	kShadowInitialize	= 0,
	kShadowRaise,
	kShadowIdle,
	kShadowFire,
	kShadowLower,

	kBeastInitialize	= 0,
	kBeastRaise,
	kBeastIdle,
	kBeastAttack,
	kBeastLower,

	kDollInitialize	= 0,
	kDollRaise,
	kDollIdleStill,
	kDollIdleMoving,
	kDollStab1,
	kDollStab2,
	kDollStab3,
	kDollStab4,
	kDollStab5,
	kDollLower
};

enum {
	kAnimForkUp = 0,
	kAnimForkIdle,
	kAnimForkStab,
	kAnimForkDown,

	kAnimBicUp,			// raise lighter and open
	kAnimBicFlame,		// light lighter
	kAnimBicIdle,		// burning lighter
	kAnimBicDown,		// lower burning lighter
	kAnimBicDown_NoFlame,	// lower unlit lighter

	kAnimSprayUp,
	kAnimSprayIdle,
	kAnimSprayFire1,
	kAnimSprayFire2,
	kAnimSprayFire3,
	kAnimSprayDown,

	kAnimStickUp,
	kAnimStickIdle,
	kAnimStickLight,
	kAnimStickAim,
	kAnimStickDown,
	kAnimStickThrow,

	kAnimBundleUp,
	kAnimBundleIdle,
	kAnimBundleLight,
	kAnimBundleAim,
	kAnimBundleDown,
	kAnimBundleThrow,

	kAnimFlareUp,
	kAnimFlareIdle,
	kAnimFlareFire,
//	kAnimFlareReload,
	kAnimFlareDown,

	kAnimShotUp,
	kAnimShotIdle1,
	kAnimShotIdle2,
	kAnimShotIdle3,
	kAnimShotFireL,
	kAnimShotFireR,
	kAnimShotReload,
	kAnimShotDown,

	kAnimTommyUp,
	kAnimTommyIdle,
	kAnimTommyFire1,
	kAnimTommyFire2,
 	kAnimTommyFire3,
 	kAnimTommyFire4,
 	kAnimTommyDown,

	kAnimSpearUp,
	kAnimSpearIdle1,
	kAnimSpearIdle2,
	kAnimSpearFire,
	kAnimSpearDown,

	kAnimShadowUp,
	kAnimShadowIdle,
	kAnimShadowFire,
	kAnimShadowDown,

	kAnimBeastUp,
	kAnimBeastIdle,
	kAnimBeastAttack1,
	kAnimBeastAttack2,
 	kAnimBeastAttack3,
 	kAnimBeastAttack4,
	kAnimBeastDown,

	kAnimDollUp,
	kAnimDollIdleStill,
	kAnimDollIdleMoving,
	kAnimDollStab1,
	kAnimDollStab2,
	kAnimDollStab3,
	kAnimDollStab4,
	kAnimDollStab5,
	kAnimDollDown,

	kAnimEnd
};

static char * weaponAnimNames[kAnimEnd] =
{
	"FORKUP",  		//kAnimForkUp = 0,
	"FORKIDLE",     //kAnimForkIdle,
	"PFORK",        //kAnimForkStab,
	"FORKDOWN",     //kAnimForkDown,

	"LITEOPEN",     //kAnimBicUp,
	"LITEFLAM",     //kAnimBicFlame,
	"LITEIDLE",     //kAnimBicIdle,
	"LITECLO2",     //kAnimBicDown,
	"LITECLOS",     //kAnimBicDown_NoFlame,

	"CANPREF",      //kAnimSprayUp,
	"CANIDLE",      //kAnimSprayIdle,
	"CANFIRE1",     //kAnimSprayFire1,
	"CANFIRE2",     //kAnimSprayFire2,
	"CANFIRE3",     //kAnimSprayFire3,
	"CANDOWN",      //kAnimSprayDown,

	"DYNPRE",       //kAnimStickUp,
	"DYNIDLE1",     //kAnimStickIdle,
	"DYNLITA",      //kAnimStickLight,
	"DYNFUSEA",     //kAnimStickAim,
	"DYNDOWN",      //kAnimStickDown,
	"THROW",        //kAnimStickThrow,

	"DYNPRE2",      //kAnimBundleUp,
	"DYNIDLE2",     //kAnimBundleIdle,
	"DYNLITB",      //kAnimBundleLight,
	"DYNFUSEB",     //kAnimBundleAim,
	"DYNDOWN2",     //kAnimBundleDown,
	"THROW",        //kAnimBundleThrow,

	"FLARUP",
	"FLARIDLE",
	"FLARFIRE",
//	"FLARLOAD",
	"FLARDOWN",

	"SHOTUP",		//kAnimShotUp,
	"SHOTI1",       //kAnimShotIdle1,
	"SHOTI2",       //kAnimShotIdle2,
	"SHOTI3",       //kAnimShotIdle3,
	"SHOTF1",       //kAnimShotFireL,
	"SHOTF2",       //kAnimShotFireR,
	"SHOTL1",       //kAnimShotReload,
	"SHOTDOWN",     //kAnimShotDown,

	"TOMUP",        //kAnimTommyUp,
	"TOMIDLE",      //kAnimTommyIdle,
	"TOMFIRE1",     //kAnimTommyFire1,
 	"TOMFIRE2",     //kAnimTommyFire2,
 	"TOMFIRE3",     //kAnimTommyFire3,
	"TOMFIRE4",     //kAnimTommyFire4,
	"TOMDOWN",      //kAnimTommyDown,

	"SGUNUP",		//kAnimSpearUp,
	"SGUNIDL1",		//kAnimSpearIdle1,
	"SGUNIDL2",		//kAnimSpearIdle2,
	"SGUNFIRE",		//kAnimSpearFire,
	"SGUNDOWN",		//kAnimSpearDown,

	"DARKUP",       //kAnimShadowUp,
	"DARKIDLE",     //kAnimShadowIdle,
	"DARKFIRE",     //kAnimShadowFire,
	"DARKDOWN",     //kAnimShadowDown,

	"BSTUP",		//kAnimBeastUp
	"BSTIDLE",		//kAnimBeastIdle,
	"BSTATAK1",     //kAnimBeastAttack1,
	"BSTATAK2",     //kAnimBeastAttack2,
	"BSTATAK3",     //kAnimBeastAttack3,
	"BSTATAK4",     //kAnimBeastAttack4,
	"BSTDOWN",

	"VDUP",			//kAnimDollUp
	"VDIDLE1",		//kAnimDollIdleStill	(PLAYER IS STILL)
	"VDIDLE2",		//kAnimDollIdleMoving	(PLAYER IS WALKING/RUNNING)
	"VDFIRE1",		//kAnimDollStab1
	"VDFIRE2",		//kAnimDollStab2
	"VDFIRE3",		//kAnimDollStab3
	"VDFIRE4",		//kAnimDollStab4
	"VDFIRE5",		//kAnimDollStab5
	"VDDOWN", 		//kAnimDollDown
};


static Anim *weaponAnim[kAnimEnd];


void WeaponInit( void )
{
	// get pointers to all the animation resources
	for (int i = 0; i < kAnimEnd; i++)
	{
		RESHANDLE hAnim = gSysRes.Lookup(weaponAnimNames[i], ".QAV");
		if (hAnim == NULL)
		{
			dprintf("Could not load %s.QAV\n", weaponAnimNames[i]);
			ThrowError("Missing Weapon QAV file", ES_ERROR);
		}
		weaponAnim[i] = (Anim *)gSysRes.Lock(hAnim);
	}

	dprintf("Preload weapon animation tiles\n");
	for (i = 0; i < kAnimEnd; i++)
	{
		if (weaponAnim[i] != NULL )
			weaponAnim[i]->Preload();
	}
}


void WeaponDraw( PLAYER *pPlayer, int shade, int x, int y )
{
	dassert(pPlayer != NULL);
	Anim *anim = pPlayer->pWeaponAnim;
	ulong t;

	if ( anim == NULL )
		return;

	if (pPlayer->weaponTimer == 0)	// playing idle animation?
		t = gGameClock % anim->duration;
	else
		t = anim->duration - pPlayer->weaponTimer;

	anim->origin.x = x;
	anim->origin.y = y;

	int nPowerRemaining = powerupCheck( pPlayer, kItemLtdInvisibility - kItemBase );
	int nOrient = kOrientScale;

	if ( nPowerRemaining >= (kTimerRate * 8) || nPowerRemaining && ( gGameClock & 32 ) )
	{
		nOrient |= kOrientTranslucent;
		shade = -128;
	}

	anim->Play(0, t, shade, nOrient );
}


void WeaponStartAnimation( PLAYER *pPlayer, int weaponIndex )
{
	pPlayer->pWeaponAnim = weaponAnim[weaponIndex];
	pPlayer->weaponTimer = pPlayer->pWeaponAnim->duration;
}

#define kAimMaxDist 	M2X(100)
#define kAimMaxAngle	kAngle10
#define kAimMaxSlope	8 << 14			// z units per xy unit

static int WeaponGetAimVector( PLAYER *pPlayer, long *pdx, long *pdy, long *pdz )
{
	dassert( (pPlayer != NULL) && (pdx != NULL) && (pdy != NULL) && (pdz != NULL) );
	SPRITE *pSprite = pPlayer->sprite;

	int x = pSprite->x;
	int y = pSprite->y;
	int z = pSprite->z - pPlayer->weaponAboveZ;

	*pdx = Cos(pSprite->ang) >> 16;
	*pdy = Sin(pSprite->ang) >> 16;
	*pdz = pPlayer->slope;

	int nAimSprite = -1;

	if ( gAutoAim )
	{
		int closest = 0x7FFFFFFF;
		for (short nTarget = headspritestat[kStatDude]; nTarget >= 0; nTarget = nextspritestat[nTarget] )
		{
			SPRITE *pTarget = &sprite[nTarget];

			// don't target yourself!
			if ( pTarget == pSprite )
				continue;

			int dx, dy, dz;

			dx = pTarget->x - x;
			dy = pTarget->y - y;
			dz = pTarget->z - z;

			int dist = qdist(dx, dy);
			if ( dist > closest || dist > kAimMaxDist )
				continue;

			int ang = getangle(dx, dy);
			if ( qabs(((ang - pSprite->ang + kAngle180) & kAngleMask) - kAngle180) > kAimMaxAngle )
				continue;

			int nSlope = divscale(dz, dist, 14);
			if ( qabs(nSlope - pPlayer->slope) > kAimMaxSlope )
				continue;

			if ( cansee(x, y, z, pSprite->sectnum, pTarget->x, pTarget->y, pTarget->z, pTarget->sectnum) )
			{
				closest = dist;

				*pdx = Cos(ang) >> 16;
				*pdy = Sin(ang) >> 16;
				*pdz = nSlope;
				nAimSprite = nTarget;
			}
		}
	}
	return nAimSprite;
}


void WeaponLower( PLAYER *pPlayer, INPUT *pInput )
{
	dprintf("WeaponLower()\n");
	dassert( pPlayer != 0 );

	int nWeapon = pPlayer->weapon;
	int nState = pPlayer->weaponState[nWeapon];

	switch( nWeapon )
	{
		case kVoodooDoll:
			WeaponStartAnimation(pPlayer, kAnimDollDown);
			break;

		case kPitchfork:
			WeaponStartAnimation(pPlayer, kAnimForkDown);
			break;

		case kFlareGun:
			WeaponStartAnimation(pPlayer, kAnimFlareDown);
			break;

		case kSprayCan:
			switch( nState )
			{
				case kCanBicIdle:
					// switch without putting down lighter
					if ( pInput->newWeapon == kDynamite )
					{
						pPlayer->weapon = kDynamite;
						pPlayer->weaponState[kDynamite] = kTNTBicIdle;
						pPlayer->pWeaponAnim = weaponAnim[kAnimBicIdle];
						pInput->newWeapon = 0;
						return;
					}
			        WeaponStartAnimation(pPlayer, kAnimBicDown);
					break;

				default:
				    WeaponStartAnimation(pPlayer, kAnimSprayDown);
					pPlayer->weaponState[kSprayCan] = kCanBicIdle;
					return;

			}
			break;

		case kDynamite:
			switch( nState )
			{
				case kTNTBicIdle:
					if ( pInput->newWeapon == kSprayCan )
					{
						pPlayer->weapon = kSprayCan;
						pPlayer->weaponState[kSprayCan] = kCanBicIdle;
						pPlayer->pWeaponAnim = weaponAnim[kAnimBicIdle];
						pInput->newWeapon = 0;
						return;
					}
			        WeaponStartAnimation(pPlayer, kAnimBicDown);
					break;

				default:
					WeaponStartAnimation(pPlayer, kAnimStickDown);
					pPlayer->weaponState[kDynamite] = kTNTBicIdle;
					return;
			}
			break;

		case kShotgun:
	        WeaponStartAnimation(pPlayer, kAnimShotDown);
			break;

		case kTommyGun:
	        WeaponStartAnimation(pPlayer, kAnimTommyDown);
			break;

		case kSpearGun:
	        WeaponStartAnimation(pPlayer, kAnimSpearDown);
			break;

		case kShadowGun:
	        WeaponStartAnimation(pPlayer, kAnimShadowDown);
			break;

		case kBeastHands:
	        WeaponStartAnimation(pPlayer, kAnimBeastDown);
			break;
	}

	if (pInput->newWeapon == pPlayer->weapon)
		pInput->newWeapon = 0;

	pPlayer->weapon = 0;
}


void WeaponRaise( PLAYER *pPlayer, INPUT *pInput )
{
	dprintf("WeaponRaise()\n");
	dassert( pPlayer != 0 );

	pPlayer->weapon = pInput->newWeapon;
	pInput->newWeapon = 0;

	switch( pPlayer->weapon )
	{
		case kVoodooDoll:
			WeaponStartAnimation(pPlayer, kAnimDollUp);
			pPlayer->weaponState[kVoodooDoll] = kDollIdleStill;
			break;

		case kPitchfork:
			WeaponStartAnimation(pPlayer, kAnimForkUp);
			pPlayer->weaponState[kPitchfork] = kForkIdle;
			break;

		case kFlareGun:
			WeaponStartAnimation(pPlayer, kAnimFlareUp);
			pPlayer->weaponState[kFlareGun] = kFlareIdle;
			break;

		case kSprayCan:
			WeaponStartAnimation(pPlayer, kAnimBicUp);
			pPlayer->weaponState[kSprayCan] = kCanBicOpen;
			break;

		case kDynamite:
			WeaponStartAnimation(pPlayer, kAnimBicUp);
			pPlayer->weaponState[kDynamite] = kTNTBicOpen;
			break;

		case kShotgun:
			WeaponStartAnimation(pPlayer, kAnimShotUp);
			switch (pPlayer->weaponState[kShotgun])
			{
				case kShotInitialize:
				case kShotIdle3:
//					if (pPlayer->ammoCount[ kAmmoShells - kAmmoBase ] > 0)
//					    pPlayer->weaponState[kShotgun] = kShotReload;
//					else
					    pPlayer->weaponState[kShotgun] = kShotIdle3;
					break;
				case kShotIdle2:
					if (gInfiniteAmmo || pPlayer->ammoCount[ kAmmoShells - kAmmoBase ] > 1)
					    pPlayer->weaponState[kShotgun] = kShotReload;
					break;
			}
			break;

		case kTommyGun:
			WeaponStartAnimation(pPlayer, kAnimTommyUp);
			pPlayer->weaponState[kTommyGun] = kTommyIdle;
			break;

		case kSpearGun:
			WeaponStartAnimation(pPlayer, kAnimSpearUp);
			if ((pPlayer->ammoCount[ kAmmoSpear - kAmmoBase ] > 0) || (pPlayer->ammoCount[ kAmmoHESpears - kAmmoBase ] > 0))
				pPlayer->weaponState[kSpearGun] = kSpearIdle1;
			else
				pPlayer->weaponState[kSpearGun] = kSpearIdle2;
			break;

		case kShadowGun:
			WeaponStartAnimation(pPlayer, kAnimShadowUp);
			pPlayer->weaponState[kShadowGun] = kShadowIdle;
			break;

		case kBeastHands:
			WeaponStartAnimation(pPlayer, kAnimBeastUp); // no up yet...
			pPlayer->weaponState[kBeastHands] = kBeastIdle;
			break;
	}
}


void WeaponUseVoodoo( PLAYER *pPlayer, INPUT *pInput )
{
	int nStabType = Random(4);
	dassert(nStabType < 4);

	int nSelfSprite = pPlayer->nSprite;
	int z = pPlayer->sprite->z - pPlayer->weaponAboveZ;

	long dx, dy, dz;
	int nAimSprite = WeaponGetAimVector( pPlayer, &dx, &dy, &dz );

	if ( nAimSprite == -1 )
	{
		actDamageSprite(nSelfSprite, pPlayer->nSprite, kDamageStab, 2 << 4);
		nStabType = kVoodooStabSelf;
		if (pPlayer == gMe)
			scrSetMessage("Ouch!");
	}
	else
	{
		SPRITE *pTarget = &sprite[nAimSprite];

		HITINFO hitInfo;
		hitInfo.hitsprite = (short)nAimSprite;
		hitInfo.hitx = sprite[nAimSprite].x;
		hitInfo.hity = sprite[nAimSprite].y;
		hitInfo.hitz = sprite[nAimSprite].z;

		switch( nStabType )
		{
			case kVoodooStabChest:
				actSpawnRicochet(pTarget->sectnum, pTarget->x, pTarget->y, pTarget->z,
					surfType[pTarget->picnum]);
				actDamageSprite(nSelfSprite, nAimSprite, kDamageStab, 2 << 4);
				break;

			case kVoodooStabShoulder:
				actSpawnRicochet(pTarget->sectnum, pTarget->x, pTarget->y, pTarget->z,
					surfType[pTarget->picnum]);
				actDamageSprite(nSelfSprite, nAimSprite, kDamageStab, 4 << 4);
				if ( pTarget->type >= kDudePlayer1 && pTarget->type <= kDudePlayer8 )
				{
					PLAYER *pPlayerHit = &gPlayer[pTarget->type - kDudePlayer1];
					WeaponLower( pPlayerHit, pInput );
				}
				break;
			case kVoodooStabGroin:
				actSpawnRicochet(pTarget->sectnum, pTarget->x, pTarget->y, pTarget->z,
					surfType[pTarget->picnum]);
				actDamageSprite(nSelfSprite, nAimSprite, kDamageStab, 12 << 4);
				break;
			case kVoodooStabEye:
				actSpawnRicochet(pTarget->sectnum, pTarget->x, pTarget->y, pTarget->z,
					surfType[pTarget->picnum]);
				actDamageSprite(nSelfSprite, nAimSprite, kDamageStab, 6 << 4);
				if ( pTarget->type >= kDudePlayer1 && pTarget->type <= kDudePlayer8 )
				{
					PLAYER *pPlayerHit = &gPlayer[pTarget->type - kDudePlayer1];
				   	if (pPlayerHit == gView)	// change to gMe later!
				   		scrDacRelEffect(-96, -96, -96);
				}
				break;
		}
	}
	WeaponStartAnimation(pPlayer, kAnimDollStab1 + nStabType);
	pPlayer->weaponState[kVoodooDoll] = kDollIdleStill;
}


#define kMaxForkDist	M2X(2.25)

void WeaponUsePitchfork( PLAYER *pPlayer )
{
	SPRITE *pSprite = pPlayer->sprite;

	long dx, dy, dz;
	WeaponGetAimVector( pPlayer, &dx, &dy, &dz );
	int z = pSprite->z - pPlayer->weaponAboveZ;

	for ( int i = 0; i < 4; i++ )
	{
		// dispersal modifiers here; accuracy in GetAimVector()
		int ddx = BiRandom(2000);
		int ddy = BiRandom(2000);
		int ddz = BiRandom(32000);

		actFireVector(pPlayer->nSprite, z, dx + ddx, dy + ddy, dz + ddz, kMaxForkDist,
			kDamageStab, 3);
	}
}


void WeaponFireShotgun( PLAYER *pPlayer, int nBarrels )
{
	dassert(nBarrels > 0 && nBarrels <= kShotgunBarrels);

	int nVectors = nBarrels * kVectorsPerBarrel;

	SPRITE *pSprite = pPlayer->sprite;

	long dx, dy, dz;
	WeaponGetAimVector( pPlayer, &dx, &dy, &dz );
	int z = pSprite->z - pPlayer->weaponAboveZ;

	for ( int i = 0; i < nVectors; i++ )
	{
		// dispersal modifiers here; accuracy in GetAimVector()
		int ddx = BiRandom(2000);
		int ddy = BiRandom(2000);
		int ddz = BiRandom(32000);

		actFireVector(pPlayer->nSprite, z, dx + ddx, dy + ddy, dz + ddz, 0,
			kDamageBullet, 4);
	}
}


void WeaponFireTommy( PLAYER *pPlayer, int nShots )
{
	SPRITE *pSprite = pPlayer->sprite;

	long dx, dy, dz;
	WeaponGetAimVector( pPlayer, &dx, &dy, &dz );
	int z = pSprite->z - pPlayer->weaponAboveZ;

	for ( int i = 0; i < nShots; i++ )
	{
		// dispersal modifiers here; accuracy in GetAimVector()
		int ddx = BiRandom(500);
		int ddy = BiRandom(500);
		int ddz = BiRandom(8000);

		actFireVector(pPlayer->nSprite, z, dx + ddx, dy + ddy, dz + ddz, 0,
			kDamageBullet, 4 );
	}
}


int WeaponCheckAmmo( PLAYER *pPlayer, int nWeapon, int *pAmmoIndex )
{
	dassert( pAmmoIndex != NULL);
//	dprintf("WeaponCheckAmmo: ");

	int nAmmoCount = 999;	// default ammo count
	int nAmmoIndex = 0;		// default ammo index

	if ( !gInfiniteAmmo )
		switch (nWeapon)
		{
			case kPitchfork:
				break;
			case kVoodooDoll:
				break;
			case kBeastHands:
				break;

			case kSprayCan:
				nAmmoIndex = kAmmoSprayCan - kAmmoBase;
				nAmmoCount = pPlayer->ammoCount[ nAmmoIndex ];
				break;

			case kDynamite:
				nAmmoIndex = kAmmoTNTStick - kAmmoBase;
				nAmmoCount = pPlayer->ammoCount[ nAmmoIndex ];
				break;

			case kShotgun:
				nAmmoIndex = kAmmoShells - kAmmoBase;
				nAmmoCount = pPlayer->ammoCount[ nAmmoIndex ];
				break;

			case kTommyGun:
				if ( pPlayer->ammoCount[kAmmoAPBullets - kAmmoBase] > 0 )
					nAmmoIndex = kAmmoAPBullets - kAmmoBase;
				else
					nAmmoIndex = kAmmoBullets - kAmmoBase;
				nAmmoCount = pPlayer->ammoCount[ nAmmoIndex ];
				break;

			case kFlareGun:
				if ( pPlayer->ammoCount[ kAmmoStarFlares - kAmmoBase ] > 0 )
					nAmmoIndex = kAmmoStarFlares - kAmmoBase;
				else if ( pPlayer->ammoCount[ kAmmoHEFlares - kAmmoBase ] > 0 )
					nAmmoIndex = kAmmoHEFlares - kAmmoBase;
				else
					nAmmoIndex = kAmmoFlares - kAmmoBase;
				nAmmoCount = pPlayer->ammoCount[ nAmmoIndex ];
				break;


			case kSpearGun:
				if (pPlayer->ammoCount[ kAmmoHESpears - kAmmoBase ] > 0 )
					nAmmoIndex = kAmmoHESpears - kAmmoBase;
				else
					nAmmoIndex = kAmmoSpear - kAmmoBase;
				nAmmoCount = pPlayer->ammoCount[ nAmmoIndex ];
				break;

			case kShadowGun:
			{
				nAmmoCount = pPlayer->xsprite->health;
				nAmmoIndex = 0;
				//nAmmoCount = pPlayer->ammoCount[ nAmmoIndex ];
				break;
			}

		}
	*pAmmoIndex = nAmmoIndex;
//	dprintf("nAmmoIndex=%i  nAmmoCount = %i\n", nAmmoIndex, nAmmoCount );
	return nAmmoCount;
}


uchar WeaponFindLoaded( PLAYER *pPlayer )
{
	int nAmmoIndex = 0;
	int nNewWeapon = kPitchfork;

	dprintf("WeaponFindLoaded: ");

	if ( pPlayer->hasWeapon[ kShadowGun ] && powerupCheck(pPlayer, kItemInvulnerability - kItemBase) )
		nNewWeapon = kShadowGun;
	else if ( pPlayer->hasWeapon[ kTommyGun ] && WeaponCheckAmmo( pPlayer, kTommyGun, &nAmmoIndex ) > 0 )
		nNewWeapon = kTommyGun;
	else if ( pPlayer->hasWeapon[ kShotgun ] && WeaponCheckAmmo( pPlayer, kShotgun, &nAmmoIndex ) > 0 )
		nNewWeapon = kShotgun;
	else if ( pPlayer->hasWeapon[ kFlareGun ] && WeaponCheckAmmo( pPlayer, kFlareGun, &nAmmoIndex ) > 0 )
		nNewWeapon = kFlareGun;
	else if ( pPlayer->hasWeapon[ kSpearGun ] && WeaponCheckAmmo( pPlayer, kSpearGun, &nAmmoIndex ) > 0 )
		nNewWeapon = kSpearGun;
	else if ( pPlayer->hasWeapon[ kSprayCan ] && WeaponCheckAmmo( pPlayer, kSprayCan, &nAmmoIndex ) > 0 )
		nNewWeapon = kSprayCan;
	else if ( pPlayer->hasWeapon[ kDynamite ] && WeaponCheckAmmo( pPlayer, kDynamite, &nAmmoIndex ) > 0 )
		nNewWeapon = kDynamite;
	else if ( pPlayer->hasWeapon[ kVoodooDoll ] && WeaponCheckAmmo( pPlayer, kVoodooDoll, &nAmmoIndex ) > 0 )
		nNewWeapon = kVoodooDoll;
	else if ( pPlayer->hasWeapon[ kShadowGun ] )
	{
		if ( pPlayer->xsprite->health > 75 )
			nNewWeapon = kShadowGun;
	}

	dprintf("nNewWeapon = %i\n", nNewWeapon);
	return (uchar)nNewWeapon;
}


/*
;[Weapon#]
;Name            = "WeaponNameMax24"
;SelectKey       = [ None | scancode ]
;UseByDefault    = [ TRUE | FALSE ]
;HasByDefault    = [ TRUE | FALSE ]
;RequiresWeapon  = [ None | 0..9 ]
;
;Type            = [ NonWeapon | Melee | Thrown | Projectile | Vector ]
;Type            = NonWeapon
;
;Type            = Melee
;MaxDistance     = [ Infinite | 0..max ]
;
;Type            = Thrown
;MaxDistance     = [ Infinite | 0..max ]
;
;Type            = Projectile
;MaxDistance     = [ Infinite | 0..max ]
;
;Type            = Vector (uses hitscan)
;MaxDistance     = [ Infinite | 0..max ]
;Richochet       = [ TRUE | FALSE ]
;UsesNightScope  = [ TRUE | FALSE ]
;
;Shots           = [ 0..max shots or projectiles ]
;Dispersion      = [ 0..100% ]
;Accuracy        = [ 0..100% ]
;
;AmmoStyle       = [ Infinite, Rechargable, Expendable, Health ]
;AmmoStyle       = Infinite
;
;AmmoStyle       = Rechargable
;RechargeRate    = [ 0 | ammo/second (?) ]
;MaxAmmo         = [ 0..max ]
;
;AmmoStyle       = Expendable
;MaxAmmo         = [ 0..max ]
;AmmoType        = [ 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 ]
;
;AmmoStyle       = Health
;HealthPerShot   = [ 0..usedpershot ]
;
;JamFrequency    = [ 0..100% ]
;JamSingleFire   = [ TRUE | FALSE ]
;JamRapidFire    = [ TRUE | FALSE ]
;
;[Ammo#]
;Name            = "AmmoNameMax24"
;Velocity        = [ ? ]
;DamageTypes     = [ Explosion, Burn, Bullet, Stab, Pummel, Cleave, Ecto]
;DamageRates     = [ 0,         0,    0,      0,    0,      0,      0 ]
*/

void WeaponUpdateState( PLAYER *pPlayer)
{
	SPRITE *pSprite		= pPlayer->sprite;
	XSPRITE *pXSprite	= pPlayer->xsprite;

	int nWeapon = pPlayer->weapon;
	int nState = pPlayer->weaponState[nWeapon];

	switch( nWeapon )
	{
		case kVoodooDoll:
	 		switch ( pXSprite->moveState )
			{
				case kMoveFall:
				case kMoveStill:
				case kMoveStand:
				case kMoveSwim:
				case kMoveFly:
				case kMoveHang:
					nState = pPlayer->weaponState[kVoodooDoll] = kDollIdleStill;
					pPlayer->pWeaponAnim = weaponAnim[kAnimDollIdleStill];
					break;

				case kMoveWalk:
					if ( pPlayer->swayHeight )
					{
						nState = pPlayer->weaponState[kVoodooDoll] = kDollIdleMoving;
						pPlayer->pWeaponAnim = weaponAnim[kAnimDollIdleMoving];
					}
					else
					{
						nState = pPlayer->weaponState[kVoodooDoll] = kDollIdleStill;
						pPlayer->pWeaponAnim = weaponAnim[kAnimDollIdleStill];
					}
					break;

				case kMoveLand:
					nState = pPlayer->weaponState[kVoodooDoll] = kDollIdleMoving;
					pPlayer->pWeaponAnim = weaponAnim[kAnimDollIdleMoving];
					break;

				default:
					pPlayer->pWeaponAnim = NULL;
					break;
			}
			break;

		case kPitchfork:
			pPlayer->pWeaponAnim = weaponAnim[kAnimForkIdle];
			break;

		case kFlareGun:
			if (nState == kFlareIdle)
				pPlayer->pWeaponAnim = weaponAnim[kAnimFlareIdle];
			break;

		case kSpearGun:
			switch ( nState )
			{
				case kSpearIdle1:
					pPlayer->pWeaponAnim = weaponAnim[kAnimSpearIdle1];
					break;

				case kSpearIdle2:
					pPlayer->pWeaponAnim = weaponAnim[kAnimSpearIdle2];
					break;
			}
			break;

		case kTommyGun:
		{
			switch ( nState )
			{
				case kTommyIdle: // is idle
					pPlayer->pWeaponAnim = weaponAnim[kAnimTommyIdle];
					break;
				case kTommyFire1: // was idle, so fire first
				case kTommyFire2: // fire second
				case kTommyFire3: // fire third
	 			case kTommyFire4: // fire fourth and reset state
					if ( gInfiniteAmmo || pPlayer->ammoCount[kAmmoAPBullets - kAmmoBase] > 0 || pPlayer->ammoCount[kAmmoBullets - kAmmoBase] > 0 )
					{
						WeaponStartAnimation(pPlayer, kAnimTommyFire1 + (nState - kTommyFire1) );
						WeaponFireTommy( pPlayer, 1 );
						if ( !gInfiniteAmmo )
						{
							if ( pPlayer->ammoCount[ kAmmoAPBullets - kAmmoBase ] > 0 )
								pPlayer->ammoCount[ kAmmoAPBullets - kAmmoBase ]--;
							else
								pPlayer->ammoCount[ kAmmoBullets - kAmmoBase ]--;
						}
						if ( nState == kTommyFire4 )
							pPlayer->weaponState[kTommyGun] = kTommyIdle;
						else
							pPlayer->weaponState[kTommyGun]++;
						return;
					}
			}
			break;
		}

		case kSprayCan:
		{
			long dx, dy, dz;
			WeaponGetAimVector( pPlayer, &dx, &dy, &dz );

			switch( nState )
			{
				case kCanBicOpen:
					WeaponStartAnimation(pPlayer, kAnimBicFlame);
					pPlayer->weaponState[kSprayCan] = kCanBicIdle;
					break;

				case kCanBicIdle:
					if ( pPlayer->ammoCount[ kAmmoSprayCan - kAmmoBase ] > 0 )
					{
						WeaponStartAnimation(pPlayer, kAnimSprayUp);
						pPlayer->weaponState[kSprayCan] = kCanIdle;
					}
					else
						pPlayer->pWeaponAnim = weaponAnim[kAnimBicIdle];
					break;

				case kCanIdle:
					pPlayer->pWeaponAnim = weaponAnim[kAnimSprayIdle];
					break;

				case kCanFire1: // was idle, so fire first
				case kCanFire2: // fire second
				case kCanFire3: // fire third and reset state
						WeaponStartAnimation(pPlayer, kAnimSprayFire1 + (nState - kCanFire1) );
						playerFireMissile(pPlayer, dx, dy, dz, kMissileSprayFlame);
						if ( !gInfiniteAmmo )
							pPlayer->ammoCount[ kAmmoSprayCan - kAmmoBase ] -= kFrameTicks;
						if ( nState == kCanFire3 )
						{
							if ( pPlayer->ammoCount[ kAmmoSprayCan - kAmmoBase ] <= 0 )
							{
								pPlayer->ammoCount[ kAmmoSprayCan - kAmmoBase ] = 0;
								// drop the can, and idle the lighter
							    WeaponStartAnimation(pPlayer, kAnimSprayDown);
								pPlayer->weaponState[kSprayCan] = kCanBicIdle;
							}
							else
								pPlayer->weaponState[kSprayCan] = kCanIdle;
						}
						else
							pPlayer->weaponState[kSprayCan]++;
						return;
			}
			break;
		}

		case kDynamite:
			switch( nState )
			{
				case kTNTBicOpen:
					WeaponStartAnimation(pPlayer, kAnimBicFlame);
					pPlayer->weaponState[nWeapon] = kTNTBicIdle;
					break;

				case kTNTBicIdle:
					if ( pPlayer->ammoCount[ kAmmoTNTStick - kAmmoBase ] > 0)
					{
						WeaponStartAnimation(pPlayer, kAnimStickUp);
						pPlayer->weaponState[kDynamite] = kTNTIdle;
					}
					else
						pPlayer->pWeaponAnim = weaponAnim[kAnimBicIdle];
					break;

				case kTNTIdle:
					pPlayer->pWeaponAnim = weaponAnim[kAnimStickIdle];
					break;
			}
			break;

		case kShotgun:
			switch ( nState )
			{
				case kShotIdle1:
					pPlayer->pWeaponAnim = weaponAnim[kAnimShotIdle1];
					break;

				case kShotIdle2:
					pPlayer->pWeaponAnim = weaponAnim[kAnimShotIdle2];
					break;

				case kShotIdle3:
					if (gInfiniteAmmo || pPlayer->ammoCount[ kAmmoShells - kAmmoBase ] > 0)
					    pPlayer->weaponState[kShotgun] = kShotReload;
					pPlayer->pWeaponAnim = weaponAnim[kAnimShotIdle3];
					break;
			}
			break;

		case kShadowGun:
			pPlayer->pWeaponAnim = weaponAnim[kAnimShadowIdle];
			break;

		case kBeastHands:
			pPlayer->pWeaponAnim = weaponAnim[kAnimBeastIdle];
			break;
	}
}

void WeaponProcess( PLAYER *pPlayer, INPUT *pInput )
{
	// decrement weapon ready timer
	pPlayer->weaponTimer -= kFrameTicks;

	if (pPlayer->weaponTimer <= 0)
		pPlayer->weaponTimer = 0;

	// return if weapon is busy
	if (pPlayer->weaponTimer != 0)
		return;

	pPlayer->pWeaponAnim = NULL;

	int nWeapon = pPlayer->weapon;

	// out of ammo check, need to find a better way to do this
	// pPlayer->weapon is set to kMaxWeapons in the fire processing below
	if ( pPlayer->weaponState[nWeapon] == kFindWeapon )
	{
		dprintf("Out of ammo: looking for a new weapon\n");
		pPlayer->weaponState[nWeapon] = 0;
		pInput->newWeapon = WeaponFindLoaded( pPlayer );
	}

	// weapon select
	if ( pInput->newWeapon )
	{
		if ( pPlayer->xsprite->health == 0 || !pPlayer->hasWeapon[ pInput->newWeapon ] )
		{
			pInput->newWeapon = 0;
			return;
		}

		if (pPlayer->weapon == 0) 	// raise the new weapon
			WeaponRaise(pPlayer, pInput);
		else
			WeaponLower(pPlayer, pInput);	// lower the old weapon
		return;
	}

	if ( (nWeapon = pPlayer->weapon) == 0 )
		return;		// no weapon drawn, do nothing

    // auto reload test
	switch ( nWeapon )
	{
		case kShotgun:
			if ( pPlayer->weaponState[nWeapon] == kShotReload )
			{
				if (!gInfiniteAmmo && pPlayer->ammoCount[ kAmmoShells - kAmmoBase ] == 0)
					pPlayer->weaponState[kShotgun] = kShotIdle3;
				else
				{
			    	WeaponStartAnimation(pPlayer, kAnimShotReload);
					if (gInfiniteAmmo || pPlayer->ammoCount[ kAmmoShells - kAmmoBase ] > 1)
						pPlayer->weaponState[kShotgun] = kShotIdle1;
					else
						pPlayer->weaponState[kShotgun] = kShotIdle2;
				}
				return;
			}
			break;
	}

	// fire key
	if ( pInput->buttonFlags.shoot )
	{
		int nAmmoIndex;
		int nAmmoCount = WeaponCheckAmmo(pPlayer, nWeapon, &nAmmoIndex);

		if ( nAmmoCount > 0 )
		{
			long dx, dy, dz;
			WeaponGetAimVector( pPlayer, &dx, &dy, &dz );

			// firing weapon (primary)
			switch ( nWeapon )
			{
				case kVoodooDoll:
					WeaponUseVoodoo( pPlayer, pInput );
					return;

				case kFlareGun:
					if ( gInfiniteAmmo || (pPlayer->ammoCount[ kAmmoFlares - kAmmoBase ] > 0)
					|| ( pPlayer->ammoCount[ kAmmoHEFlares - kAmmoBase ] > 0) || ( pPlayer->ammoCount[ kAmmoStarFlares - kAmmoBase ] > 0))
					{
						WeaponStartAnimation(pPlayer, kAnimFlareFire);
					    pPlayer->weaponState[kFlareGun] = kFlareIdle;
						playerFireMissile( pPlayer, dx, dy, dz, kMissileFlare);
		 				if ( !gInfiniteAmmo )
						{
							if ( pPlayer->ammoCount[ kAmmoStarFlares - kAmmoBase ] > 0 )
								pPlayer->ammoCount[ kAmmoStarFlares - kAmmoBase ] -= 1;
							else if ( pPlayer->ammoCount[ kAmmoHEFlares - kAmmoBase ] > 0 )
								pPlayer->ammoCount[ kAmmoHEFlares - kAmmoBase ] -= 1;
							else if ( pPlayer->ammoCount[ kAmmoFlares - kAmmoBase ] > 0 )
								pPlayer->ammoCount[ kAmmoFlares - kAmmoBase ] -= 1;
						}
					}
					return;

				case kPitchfork:
					WeaponStartAnimation(pPlayer, kAnimForkStab);
					pPlayer->weaponState[kPitchfork] = kForkIdle;
					WeaponUsePitchfork( pPlayer );
					return;

				case kTommyGun:
					if ( pPlayer->weaponState[kTommyGun] == kTommyIdle )
						pPlayer->weaponState[kTommyGun] = kTommyFire1;
					break; // fall through to WeaponUpdateState

			    case kSpearGun:
					if ( gInfiniteAmmo || (pPlayer->ammoCount[ kAmmoSpear - kAmmoBase ] > 0)
					|| ( pPlayer->ammoCount[ kAmmoHESpears - kAmmoBase ] > 0) )
					{
						if ( pPlayer->ammoCount[ kAmmoHESpears - kAmmoBase ] > 0 )
							pPlayer->ammoCount[ kAmmoHESpears - kAmmoBase ] -= 1;
						else
							pPlayer->ammoCount[ kAmmoSpear - kAmmoBase ] -= 1;
						playerFireMissile(pPlayer, dx, dy, dz, kMissileSpear);
						WeaponStartAnimation(pPlayer, kAnimSpearFire);
						pPlayer->weaponState[kSpearGun] = kSpearIdle1;
					}
					else
						pPlayer->weaponState[kSpearGun] = kSpearIdle2;
					return;

				case kSprayCan:
					if ( pPlayer->weaponState[kSprayCan] == kCanIdle )
					{
						if ( gInfiniteAmmo || (pPlayer->ammoCount[ kAmmoSprayCan - kAmmoBase ] > 0))
							pPlayer->weaponState[kSprayCan] = kCanFire1;
					}
					break; // fall through to WeaponUpdateState

				case kDynamite:
					if ( gInfiniteAmmo || (pPlayer->ammoCount[ kAmmoTNTStick - kAmmoBase ] > 0))
					{
						WeaponStartAnimation( pPlayer, kAnimStickThrow );
						playerFireThing(pPlayer, kThingTNTStick);

						if ( !gInfiniteAmmo )
							pPlayer->ammoCount[ kAmmoTNTStick - kAmmoBase ] -= 1;

						if ( pPlayer->ammoCount[ kAmmoTNTStick - kAmmoBase ] == 0 )
						{
							// drop empty hand, and idle the lighter
						    WeaponStartAnimation(pPlayer, kAnimBicIdle);
							pPlayer->weaponState[kDynamite] = kTNTBicIdle;
						}
					}
					return;

			    case kShotgun:
					switch ( pPlayer->weaponState[kShotgun] )
					{
						case kShotIdle1:
							WeaponStartAnimation(pPlayer, kAnimShotFireL);
							pPlayer->weaponState[kShotgun] = kShotIdle2;
							if ( !gInfiniteAmmo )
								pPlayer->ammoCount[ kAmmoShells - kAmmoBase ] -= 1;
							WeaponFireShotgun( pPlayer, 1 );
							return;

						case kShotIdle2:
							WeaponStartAnimation(pPlayer, kAnimShotFireR);
						    pPlayer->weaponState[kShotgun] = kShotReload;
							if ( !gInfiniteAmmo )
								pPlayer->ammoCount[ kAmmoShells - kAmmoBase ] -= 1;
							WeaponFireShotgun( pPlayer, 1 );
							return;
					}
					return;

				case kShadowGun:
					WeaponStartAnimation(pPlayer, kAnimShadowFire);
				    pPlayer->weaponState[kShadowGun] = kShadowIdle;
					playerFireMissile( pPlayer, dx, dy, dz, kMissileEctoSkull);
	 				if ( !gInfiniteAmmo )
					{
						actDamageSprite(pPlayer->nSprite, pPlayer->nSprite, kDamageSpirit, 10 << 4);
					}
					return;

				case kBeastHands:
					WeaponStartAnimation(pPlayer, kAnimBeastAttack1 + Random(4));
					pPlayer->weaponState[kBeastHands] = kBeastIdle;
					return;
			}
		}
		else
		{
			dprintf("Primary out of ammo: setting weapon state to kFindWeapon\n");
			pPlayer->weaponState[nWeapon] = kFindWeapon;
			return;
		}
	}

	// alternate fire key
	if ( pInput->buttonFlags.shoot2 )
	{
		int nAmmoIndex;
		int nAmmoCount = WeaponCheckAmmo(pPlayer, nWeapon, &nAmmoIndex);

		if ( nAmmoCount > 0 )
		{
			long dx, dy, dz;
			WeaponGetAimVector( pPlayer, &dx, &dy, &dz );

			// firing weapon (alternate)
			switch ( nWeapon )
			{
				case kSpearGun:
					if ( gInfiniteAmmo || (pPlayer->ammoCount[ kAmmoSpear - kAmmoBase ] > 0)
					|| ( pPlayer->ammoCount[ kAmmoHESpears - kAmmoBase ] > 0) )
					{
						playerFireMissile(pPlayer, dx, dy, dz, kMissileSpear);
						if ( pPlayer->ammoCount[ kAmmoHESpears - kAmmoBase ] > 0 )
							pPlayer->ammoCount[ kAmmoHESpears - kAmmoBase ] -= 1;	// CHANGE TO <= 6
						else
							pPlayer->ammoCount[ kAmmoSpear - kAmmoBase ] -= 1;		// CHANGE TO <= 6
						WeaponStartAnimation(pPlayer, kAnimSpearFire);
						pPlayer->weaponState[kSpearGun] = kSpearIdle1;
					}
					else
						pPlayer->weaponState[kSpearGun] = kSpearIdle2;
					return;

				case kPitchfork:
					playerFireMissile(pPlayer, dx, dy, dz, kMissileFireball);
					pPlayer->weaponTimer = 20;
					return;

				case kFlareGun:
					return;

				case kTommyGun:
					return;

				case kSprayCan:
					break;

				case kDynamite:
					break;

				case kShotgun:
				{
					switch ( pPlayer->weaponState[kShotgun] )
					{
						case kShotIdle1:
							WeaponStartAnimation(pPlayer, kAnimShotFireR);
							pPlayer->weaponState[kShotgun] = kShotReload;
							if ( !gInfiniteAmmo )
								pPlayer->ammoCount[ kAmmoShells - kAmmoBase ] -= 2;
							WeaponFireShotgun( pPlayer, 2 );
							return;
						case kShotIdle2:
							WeaponStartAnimation(pPlayer, kAnimShotFireR);
							pPlayer->weaponState[kShotgun] = kShotReload;
							if ( !gInfiniteAmmo )
								pPlayer->ammoCount[ kAmmoShells - kAmmoBase ] -= 1;
							WeaponFireShotgun( pPlayer, 1 );
							return;
					}
					break;
				}
			}
		}
		else
		{
			dprintf("Alternate out of ammo: setting weapon state to kFindWeapon\n");
			pPlayer->weaponState[nWeapon] = kFindWeapon;
			return;
		}
	}

	WeaponUpdateState(pPlayer);
}
