#include <string.h>
#include <dos.h>

#include "sound.h"
#include "debug4g.h"
#include "error.h"
#include "inifile.h"
#include "misc.h"
#include "globals.h"
#include "music.h"

#include <memcheck.h>

static BOOL sndActive = FALSE;

int gMusicCard, gMusicVolume;

RESHANDLE hSong;
BYTE *pSong = NULL;
int songSize;


/*---------------------------------------------------------------------
   Function: LoadMidi

   Loads the midi file from disk.
---------------------------------------------------------------------*/
void sndPlaySong( char *songName, BOOL loopflag )
{
	if ( pSong )
		sndStopSong();

	if ( strlen(songName) == 0 )
		return;

	hSong = gSysRes.Lookup(songName, ".MID");
	if (hSong == NULL)
	{
		dprintf("Could not load song %s\n", songName);
		return;
	}

	songSize = gSysRes.Size(hSong);

	pSong = (BYTE *)gSysRes.Lock(hSong);
	dpmiLockMemory(FP_OFF(pSong), songSize);

	MUSIC_PlaySong(pSong, loopflag);
}


BOOL sndIsSongPlaying( void )
{
	return (BOOL)MUSIC_SongPlaying();
}


void sndStopSong( void )
{
	if ( pSong )
	{
		MUSIC_StopSong();
		dpmiUnlockMemory(FP_OFF(pSong), songSize);
		gSysRes.Unlock(hSong);
		pSong = NULL;
	}
}


void InitMusicCard(void)
{
	int address = 0x220;

	gMusicCard = BloodINI.GetKeyInt( "Sound Setup", "MusicDevice", SoundBlaster);
	gMusicVolume = BloodINI.GetKeyInt( "Sound Setup", "MusicVolume", 255 );

	switch ( gMusicCard )
	{
		case GenMidi:
		case WaveBlaster:
		case SoundScape:
		case SoundCanvas:
			address = BloodINI.GetKeyHex( "Sound Setup", "MidiPort", 0x330);
			break;
	}

	int status = MUSIC_Init( gMusicCard, address );
	if ( status != MUSIC_Ok )
	{
		// FX_Shutdown();
		ThrowError(MUSIC_ErrorString( status ), ES_ERROR);
	}

	RESHANDLE hTimbres = gSysRes.Lookup("GMTIMBRE", ".TMB");
	if ( hTimbres != NULL )
		MUSIC_RegisterTimbreBank((BYTE *)gSysRes.Load(hTimbres));

	MUSIC_SetVolume(gMusicVolume);
}


/***********************************************************************
 * Remove sound/music drivers
 **********************************************************************/
void sndTerm(void)
{
	// prevent this from shutting down more than once
	if ( sndActive )
	{
		sndStopSong();

		dprintf("MUSIC_Shutdown()\n");
		MUSIC_Shutdown();

		sndActive = FALSE;
	}
}

/***********************************************************************
 * Initialize sound/music drivers
 **********************************************************************/
void sndInit(void)
{
	InitMusicCard();
	atexit(sndTerm);
	sndActive = TRUE;
}
