#include <stdlib.h>
#include <string.h>

#include "actor.h"
#include "debug4g.h"
#include "engine.h"
#include "trig.h"
#include "gameutil.h"
#include "misc.h"
#include "db.h"
#include "multi.h"
#include "names.h"
#include "screen.h"
#include "sectorfx.h"
#include "triggers.h"
#include "error.h"
#include "globals.h"
#include "seq.h"
#include "eventq.h"
#include "dude.h"
#include "ai.h"
#include "view.h"
#include "warp.h"
#include "tile.h"
#include "player.h"
#include "options.h"

#include <memcheck.h>

#define kMaxSpareSprites	50

static void FireballCallback( int /* type */, int nXIndex );
//static void FlareCallback( int /* type */, int nXIndex );


struct MissileType {
	short picnum;
	int velocity;
	int angleOfs;
	uchar xrepeat;
	uchar yrepeat;
	char  shade;
} missileInfo[] = {
	{ kAnmButcherKnife,	(M2X(14.0) << 4) / kTimerRate,	kAngle90, 	32, 32,   -8 },	// kMissileButcherKnife
	{ kAnmFlare,		(M2X(20.0) << 4) / kTimerRate,	0,        	32, 32, -128 },	// kMissileFlare
	{ kAnmSprayFlame,	(M2X(4.0) << 4) / kTimerRate,	0,        	24, 24, -128 },	// kMissileSprayFlame
	{ 0,				(M2X(16.0) << 4) / kTimerRate,	0,        	32, 32, -128 },	// kMissileFireball
	{ kAnmSpear,		(M2X(16.0) << 4) / kTimerRate,	kAngle270, 	64, 64,   -8 },	// kMissileSpear	// 18.0
	{ kAnmEctoSkull,	(M2X(16.0) << 4) / kTimerRate,	0,			32, 32,  -24 },	// kMissileEctoSkull
	{ 0,				(M2X(12.0) << 4) / kTimerRate,	0,        	8,   8,    0 },	// kMissileGreenPuke
	{ 0,				(M2X(12.0) << 4) / kTimerRate,	0,        	8,   8,    0 },	// kMissileRedPuke
};


static int dragTable[] =
{
	0,
	0x0C00,		// kDepthTread
	0x1A00,		// kDepthWade
	0x1A00,		// kDepthSwim
};

// miscellaneous effects
enum
{
	kSeqSprayFlame1	= 0,
	kSeqSprayFlame2,
	kSeqSkull,
	kSeqExplosion1,
	kSeqExplosion2,
	kSeqExplosion3,
	kSeqExplosion4,
	kSeqSplash1,
	kSeqSplash2,
	kSeqSplash3,
	kSeqRicochet1,
	kSeqGoreWing,
	kSeqGoreHead,
	kSeqBarrel,
	kSeqBloodPool,
	kSeqRespawn,
	kSeqFlareSmoke,
	kSeqSquib1,
	kSeqFluorescentLight,
	kSeqClearGlass,
	kSeqStainedGlass,
	kSeqWeb,
	kSeqBeam,
	kSeqVase1,
	kSeqVase2,
	kSeqZombieBones,
	kSeqSkullExplode,
	kSeqMetalGrate1,
	kSeqFireball,
	kSeqBoneBreak,
	kSeqEffectMax,
};

static Seq *effectSeq[kSeqEffectMax];


struct THINGINFO
{
	short	startHealth;
	short	mass;			// in KG
	char	clipdist;
	ushort	flags;
	int 	damageShift[kDamageMax];	// use to indicate resistance to damage types
};

static THINGINFO thingInfo[kThingMax - kThingBase] =
{
	{	// kThingTNTBarrel
		40,
		150,
		32,
		kAttrMove | kAttrFall,
		{
			kNoDamage,	// kDamagePummel
			0,          // kDamageFall
			0,          // kDamageBurn
			0,          // kDamageBullet
			1,			// kDamageStab
			0,          // kDamageExplode
			kNoDamage,	// kDamageGas
			kNoDamage,  // kDamageDrown
			kNoDamage,  // kDamageSpirit
			kNoDamage,	// kDamageVoodoo
		},
	},

	{	// kThingTNTProxArmed
		5,
		5,
		16,
		0,
		{
			kNoDamage,	// kDamagePummel
			kNoDamage,	// kDamageFall
			0,			// kDamageBurn
			0,			// kDamageBullet
			kNoDamage,	// kDamageStab
			0,			// kDamageExplode
			kNoDamage,	// kDamageGas
			kNoDamage,	// kDamageDrown
			kNoDamage,	// kDamageSpirit
			kNoDamage,	// kDamageVoodoo
		},
	},

	{	// kThingTNTRemArmed
		5,
		5,
		16,
		0,
		{
			kNoDamage,	// kDamagePummel
			kNoDamage,	// kDamageFall
			0,			// kDamageBurn
			0,			// kDamageBullet
			kNoDamage,	// kDamageStab
			0,			// kDamageExplode
			kNoDamage,	// kDamageGas
			kNoDamage,	// kDamageDrown
			kNoDamage,	// kDamageSpirit
			kNoDamage,	// kDamageVoodoo
		},
	},

	{ 	// kThingBlueVase
		1,
		20,
		32,
		kAttrMove | kAttrFall,
		{
			kNoDamage,	// kDamagePummel
			0,			// kDamageFall
			kNoDamage,	// kDamageBurn
			0,			// kDamageBullet
			0,			// kDamageStab
			0,			// kDamageExplode
			kNoDamage,	// kDamageGas
			kNoDamage,	// kDamageDrown
			kNoDamage,	// kDamageSpirit
			kNoDamage,	// kDamageVoodoo
		},
	},

	{	// kThingBrownVase
		1,
		150,
		32,
		kAttrMove | kAttrFall,
		{
			kNoDamage,	// kDamagePummel
			0,			// kDamageFall
			kNoDamage,	// kDamageBurn
			0,			// kDamageBullet
			0,			// kDamageStab
			0,			// kDamageExplode
			kNoDamage,	// kDamageGas
			kNoDamage,	// kDamageDrown
			kNoDamage,	// kDamageSpirit
			kNoDamage,	// kDamageVoodoo
		},
	},

	{	// kThingCrateFace
		10,
		0,
		0,
		0,
		{
			kNoDamage,	// kDamagePummel
			kNoDamage,	// kDamageFall
			kNoDamage,	// kDamageBurn
			kNoDamage,	// kDamageBullet
			kNoDamage,	// kDamageStab
			0,			// kDamageExplode
			kNoDamage,	// kDamageGas
			kNoDamage,	// kDamageDrown
			kNoDamage,	// kDamageSpirit
			kNoDamage,	// kDamageVoodoo
		},
	},

	{	// kThingClearGlass
		1,
		0,
		0,
		0,
		{
			0,			// kDamagePummel
			0,			// kDamageFall
			kNoDamage,	// kDamageBurn
			0,			// kDamageBullet
			0,			// kDamageStab
			0,			// kDamageExplode
			kNoDamage,	// kDamageGas
			kNoDamage,	// kDamageDrown
			kNoDamage,	// kDamageSpirit
			kNoDamage,	// kDamageVoodoo
		},
	},

	{	// kThingFluorescent
		1,
		0,
		0,
		0,
		{
			0,			// kDamagePummel
			kNoDamage,	// kDamageFall
			kNoDamage,	// kDamageBurn
			0,			// kDamageBullet
			0,			// kDamageStab
			0,			// kDamageExplode
			kNoDamage,	// kDamageGas
			kNoDamage,	// kDamageDrown
			kNoDamage,	// kDamageSpirit
			kNoDamage,	// kDamageVoodoo
		},
	},

	{	// kThingWallCrack
		8,
		0,
		0,
		0,
		{
			kNoDamage,	// kDamagePummel
			kNoDamage,	// kDamageFall
			kNoDamage,	// kDamageBurn
			kNoDamage,	// kDamageBullet
			kNoDamage,	// kDamageStab
			0,			// kDamageExplode
			kNoDamage,	// kDamageGas
			kNoDamage,	// kDamageDrown
			kNoDamage,	// kDamageSpirit
			kNoDamage,	// kDamageVoodoo
		},
	},

	{	// kThingWoodBeam
		8,
		0,
		0,
		0,
		{
			0,			// kDamagePummel
			kNoDamage,	// kDamageFall
			kNoDamage,	// kDamageBurn
			0,			// kDamageBullet
			0,			// kDamageStab
			0,			// kDamageExplode
			kNoDamage,	// kDamageGas
			kNoDamage,	// kDamageDrown
			kNoDamage,	// kDamageSpirit
			kNoDamage,	// kDamageVoodoo
		},
	},

	{	// kThingWeb
		4,
		0,
		0,
		0,
		{
			0,			// kDamagePummel
			kNoDamage,	// kDamageFall
			0,			// kDamageBurn
			1,			// kDamageBullet
			0,			// kDamageStab
			0,			// kDamageExplode
			kNoDamage,	// kDamageGas
			kNoDamage,	// kDamageDrown
			kNoDamage,	// kDamageSpirit
			kNoDamage,	// kDamageVoodoo
		},
	},

	{	// kThingMetalGrate1
		20,
		0,
		0,
		0,
		{
			3,			// kDamagePummel
			kNoDamage,	// kDamageFall
			kNoDamage,	// kDamageBurn
			2,			// kDamageBullet
			4,			// kDamageStab
			1,			// kDamageExplode
			kNoDamage,	// kDamageGas
			kNoDamage,	// kDamageDrown
			kNoDamage,	// kDamageSpirit
			kNoDamage,	// kDamageVoodoo
		},
	},

	{	// kThingTNTStick
		5,
		3,
		16,
		kAttrMove | kAttrFall,
		{
			0,			// kDamagePummel
			kNoDamage,	// kDamageFall
			0,			// kDamageBurn
			0,			// kDamageBullet
			0,			// kDamageStab
			0,			// kDamageExplode
			kNoDamage,	// kDamageGas
			kNoDamage,	// kDamageDrown
			kNoDamage,	// kDamageSpirit
			kNoDamage,	// kDamageVoodoo
		},
	},

	{	// kThingBoneClub
		5,
		6,
		16,
		kAttrMove | kAttrFall,
		{
			kNoDamage,	// kDamagePummel
			kNoDamage,	// kDamageFall
			kNoDamage,	// kDamageBurn
			kNoDamage,	// kDamageBullet
			kNoDamage,	// kDamageStab
			kNoDamage,	// kDamageExplode
			kNoDamage,	// kDamageGas
			kNoDamage,	// kDamageDrown
			kNoDamage,	// kDamageSpirit
			kNoDamage,	// kDamageVoodoo
		},
	},

	{	// kThingZombieBones
		8,
		3,
		16,
		kAttrMove | kAttrFall,
		{
			0,			// kDamagePummel
			0,			// kDamageFall
			kNoDamage,	// kDamageBurn
			0,			// kDamageBullet
			0,			// kDamageStab
			0,			// kDamageExplode
			kNoDamage,	// kDamageGas
			kNoDamage,	// kDamageDrown
			kNoDamage,	// kDamageSpirit
			kNoDamage,	// kDamageVoodoo
		},
	},

	{	// kThingWaterDrip
		0,				// startHealth
		1,				// mass
		1,         		// clipDist
		kAttrMove | kAttrFall,
		{
			kNoDamage,	// kDamagePummel
			kNoDamage,	// kDamageFall
			kNoDamage,	// kDamageBurn
			kNoDamage,	// kDamageBullet
			kNoDamage,	// kDamageStab
			kNoDamage,	// kDamageExplode
			kNoDamage,	// kDamageGas
			kNoDamage,	// kDamageDrown
			kNoDamage,	// kDamageSpirit
			kNoDamage,	// kDamageVoodoo
		},
	},

	{	// kThingBloodDrip
		0,				// startHealth
		1,				// mass
		1,         		// clipDist
		kAttrMove | kAttrFall,
		{
			kNoDamage,	// kDamagePummel
			kNoDamage,	// kDamageFall
			kNoDamage,	// kDamageBurn
			kNoDamage,	// kDamageBullet
			kNoDamage,	// kDamageStab
			kNoDamage,	// kDamageExplode
			kNoDamage,	// kDamageGas
			kNoDamage,	// kDamageDrown
			kNoDamage,	// kDamageSpirit
			kNoDamage,	// kDamageVoodoo
		},
	},

	{	// kThingBubble
		0,				// startHealth
		-1,				// mass
		1,         		// clipDist
		kAttrMove,
		{
			kNoDamage,	// kDamagePummel
			kNoDamage,	// kDamageFall
			kNoDamage,	// kDamageBurn
			kNoDamage,	// kDamageBullet
			kNoDamage,	// kDamageStab
			kNoDamage,	// kDamageExplode
			kNoDamage,	// kDamageGas
			kNoDamage,	// kDamageDrown
			kNoDamage,	// kDamageSpirit
			kNoDamage,	// kDamageVoodoo
		},
	},

	{	// kThingBubbles
		0,				// startHealth
		-1,				// mass
		1,         		// clipDist
		kAttrMove,
		{
			kNoDamage,	// kDamagePummel
			kNoDamage,	// kDamageFall
			kNoDamage,	// kDamageBurn
			kNoDamage,	// kDamageBullet
			kNoDamage,	// kDamageStab
			kNoDamage,	// kDamageExplode
			kNoDamage,	// kDamageGas
			kNoDamage,	// kDamageDrown
			kNoDamage,	// kDamageSpirit
			kNoDamage,	// kDamageVoodoo
		},
	},

	{	// kThingGibSmall
		0,				// startHealth
		2,				// mass
		4,         		// clipDist
		kAttrMove | kAttrFall,
		{
			kNoDamage,	// kDamagePummel
			kNoDamage,	// kDamageFall
			kNoDamage,	// kDamageBurn
			kNoDamage,	// kDamageBullet
			kNoDamage,	// kDamageStab
			kNoDamage,	// kDamageExplode
			kNoDamage,	// kDamageGas
			kNoDamage,	// kDamageDrown
			kNoDamage,	// kDamageSpirit
			kNoDamage,	// kDamageVoodoo
		},
	},
};


void actAllocateSpares( void )
{
/*
	dprintf("Creating spare sprites\n");
	for (int i = 0; i < kMaxSpareSprites; i++)
	{
		int nSprite = insertsprite( 0, kStatSpares );
		dassert(nSprite != -1);
		dbInsertXSprite(nSprite);
		sprite[nSprite].cstat |= kSpriteInvisible;
	}
*/
}


/*******************************************************************************
	FUNCTION:		actInit()

	DESCRIPTION:	Initialize the actor subsystem.  Locks all sequences, and
					preloads all tiles for sequences used in the map.

	NOTES:
*******************************************************************************/
void actInit( void )
{
	int nSprite;
	BOOL used[kDudeMax - kDudeBase];

	memset(used, FALSE, sizeof(used));

	// allocate sprites to use for effects
	actAllocateSpares();

	// see which dudes are present
	for (nSprite = headspritestat[kStatDude]; nSprite >= 0; nSprite = nextspritestat[nSprite])
	{
		SPRITE *pSprite = &sprite[nSprite];
		if ( pSprite->type < kDudeBase || pSprite->type >= kDudeMax )
		{
			dprintf("ERROR IN SPRITE %i\n", nSprite);
			dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);
		}

		used[pSprite->type - kDudeBase] = TRUE;
	}

	// preload all effects sequences
	memset(effectSeq, NULL, sizeof(effectSeq));
	for (int i = 0; i < kSeqEffectMax; i++)
	{
		RESHANDLE hSeq = gSysRes.Lookup(i, ".SEQ");
		if (hSeq == NULL)
		{
			dprintf("Missing sequence #%d\n", i);
			continue;
		}
		effectSeq[i] = (Seq *)gSysRes.Lock(hSeq);
	}

	// preload dude sequences
	for (i = 0; i < kDudeMax - kDudeBase; i++)
	{
		memset(dudeInfo[i].pSeq, NULL, sizeof(dudeInfo[i].pSeq));
		if ( dudeInfo[i].seqStartID > 0 )
		{
			for (int j = 0; j < kSeqDudeMax; j++)
			{
				RESHANDLE hSeq = gSysRes.Lookup(dudeInfo[i].seqStartID + j, ".SEQ");
				if (hSeq != NULL)
				{
					dudeInfo[i].pSeq[j] = (Seq *)gSysRes.Lock(hSeq);
				}
			}
		}
	}

	// preload dude sequences
	for (i = kModeHuman; i <= kModeBeast; i++)
	{
		memset(gPlayerTemplate[i].pSeq, NULL, sizeof(gPlayerTemplate[i].pSeq));
		if ( gPlayerTemplate[i].seqStartID > 0 )
		{
			for (int j = 0; j < kSeqDudeMax; j++)
			{
				RESHANDLE hSeq = gSysRes.Lookup(gPlayerTemplate[i].seqStartID + j, ".SEQ");
				if (hSeq != NULL)
				{
					gPlayerTemplate[i].pSeq[j] = (Seq *)gSysRes.Lock(hSeq);
				}
			}
		}
	}

	// preload tiles for effect sequences
	dprintf("Preload effect sequence tiles\n");
	for (i = 0; i < kSeqEffectMax; i++)
	{
		if ( effectSeq[i] != NULL )
			effectSeq[i]->Preload();
	}

	// preload tiles for dude sequences
	dprintf("Preload dude sequence tiles\n");
	for (i = 0; i < kDudeMax - kDudeBase; i++)
	{
		if ( used[i] )
		{
			// only preload art for idle sequence
			if ( dudeInfo[i].pSeq[0] != NULL )
				dudeInfo[i].pSeq[0]->Preload();
		}
	}

	// initialize all dudes
	for (nSprite = headspritestat[kStatDude]; nSprite >= 0; nSprite = nextspritestat[nSprite])
	{
		SPRITE *pSprite = &sprite[nSprite];
		int nXSprite = pSprite->extra;
		dassert(nXSprite > 0 && nXSprite < kMaxXSprites);
		XSPRITE *pXSprite = &xsprite[nXSprite];

		int dudeIndex = pSprite->type - kDudeBase;

		pSprite->cstat |= kSpriteBlocking | kSpriteHitscan;
		pSprite->clipdist = dudeInfo[dudeIndex].clipdist;
		pSprite->flags = kAttrMove | kAttrFall;
		pSprite->xvel = pSprite->yvel = pSprite->zvel = 0;
		pXSprite->health = dudeInfo[dudeIndex].startHealth << 4;

		if (dudeInfo[dudeIndex].pSeq[kSeqDudeIdle] != NULL)
		{
			seqSpawn(dudeInfo[dudeIndex].pSeq[kSeqDudeIdle], SS_SPRITE, nXSprite);
		}
	}

	// initialize all things
	for (nSprite = headspritestat[kStatThing]; nSprite >= 0; nSprite = nextspritestat[nSprite])
	{
		SPRITE *pSprite = &sprite[nSprite];
		int nXSprite = pSprite->extra;
		dassert(nXSprite > 0 && nXSprite < kMaxXSprites);
		XSPRITE *pXSprite = &xsprite[nXSprite];

		int thingIndex = pSprite->type - kThingBase;

		pSprite->cstat |= kSpriteBlocking; //  | kSpriteHitscan should be handled in BSTUB by hitbit
		pSprite->clipdist = thingInfo[thingIndex].clipdist;
		pSprite->flags = thingInfo[thingIndex].flags;
 		pSprite->xvel = pSprite->yvel = pSprite->zvel = 0;
 		pXSprite->health = thingInfo[thingIndex].startHealth << 4;
		pXSprite->state = 1;
	}

	aiInit();
}


static void ConcussSprite( int nSource, int nSprite, int x, int y, int z, int force )
{
	SPRITE *pSprite = &sprite[nSprite];
	int dx = pSprite->x - x;
	int dy = pSprite->y - y;

#define ORIGINS_ARE_CENTER_OF_MASS		TRUE

#if ORIGINS_ARE_CENTER_OF_MASS
	int dz = (pSprite->z - z) >> 4;
#else
	int zTop, zBot;
	GetSpriteExtents(pSprite, &zTop, &zBot);
	int dz = ((zTop + zBot) / 2 - z) >> 4;
#endif

	int dist2 = ClipLow(dx * dx + dy * dy + dz * dz, 32 << 4);
	int nTile = pSprite->picnum;
 	int area = tilesizx[nTile] * pSprite->xrepeat * tilesizy[nTile] * pSprite->yrepeat >> 12;

	force = divscale16(force, dist2);

	if ( pSprite->flags & kAttrMove )
	{
		int mass = 0;

		if ( pSprite->type >= kDudeBase && pSprite->type < kDudeMax )
			mass = dudeInfo[pSprite->type - kDudeBase].mass;
		else if ( pSprite->type >= kThingBase && pSprite->type < kThingMax )
			mass = thingInfo[pSprite->type - kThingBase].mass;
		else
			ThrowError("Unexpected type encountered in ConcussSprite()", ES_ERROR);

		dassert(mass != 0);

		int impulse = muldiv(force, area, qabs(mass));
		dx = mulscale16(impulse, dx);
		dy = mulscale16(impulse, dy);
		dz = mulscale16(impulse, dz);

		pSprite->xvel += dx;
		pSprite->yvel += dy;
		pSprite->zvel += dz;
	}

	actDamageSprite(nSource, nSprite, kDamageExplode, force);
}


/*******************************************************************************
	FUNCTION:		ReflectVector()

	DESCRIPTION:	Reflects a vector off a wall

	PARAMETERS:     nFraction is elasticity (0x10000 == perfectly elastic)
*******************************************************************************/
static void ReflectVector( short *dx, short *dy, int nWall, int nFraction )
{
	// calculate normal for wall
	int nx = -(wall[wall[nWall].point2].y - wall[nWall].y) >> 4;
	int ny = (wall[wall[nWall].point2].x - wall[nWall].x) >> 4;
	int dotProduct = *dx * nx + *dy * ny;

	int length2 = nx * nx + ny * ny;
	dassert(length2 > 0);

	int dot2 = dotProduct + mulscale16(dotProduct, nFraction);

	*dx -= muldiv(dot2, nx, length2);
	*dy -= muldiv(dot2, ny, length2);
}


static void DropPickupObject( int nActor, int nObject )
{
	dassert( nActor >= 0 && nActor < kMaxSprites && sprite[nActor].statnum < kMaxStatus );
	dassert( nObject >= kItemBase && nObject < kItemMax
		|| nObject >= kAmmoBase && nObject < kAmmoMax
		|| nObject >= kWeaponBase && nObject < kWeaponMax);

	// create a sprite for the dropped ammo
	SPRITE *pActor = &sprite[nActor];

	int nSprite = actSpawnSprite( pActor->sectnum, pActor->x, pActor->y, sector[pActor->sectnum].floorz, kStatItem, FALSE );
	SPRITE *pSprite = &sprite[ nSprite ];

	if ( nObject >= kItemBase && nObject < kItemMax )
	{
		int nItemIndex = nObject - kItemBase;

		pSprite->type = (short)nObject;
		pSprite->picnum = gItemData[nItemIndex].picnum;
		pSprite->shade = gItemData[nItemIndex].shade ;
		pSprite->xrepeat = gItemData[nItemIndex].xrepeat;
		pSprite->yrepeat = gItemData[nItemIndex].yrepeat;
		if (nObject >= kItemKey1 && nObject <= kItemKey7)
		{
			if ( gNetMode == kNetModeCoop )	// force permanent keys in Coop mode
			{
				dbInsertXSprite( nSprite );
				XSPRITE *pXSprite = &xsprite[ pSprite->extra ];
				pXSprite->respawn = kRespawnPermanent;
				pXSprite->respawnTime = 0;
			}
		}
	}
	else if ( nObject >= kAmmoBase && nObject < kAmmoMax )
	{
		int nAmmoIndex = nObject - kAmmoBase;

		pSprite->type = (short)nObject;
		pSprite->picnum = gAmmoData[nAmmoIndex].picnum;
		pSprite->shade = gAmmoData[nAmmoIndex].shade ;
		pSprite->xrepeat = gAmmoData[nAmmoIndex].xrepeat;
		pSprite->yrepeat = gAmmoData[nAmmoIndex].yrepeat;
	}
	else if ( nObject >= kWeaponBase && nObject < kWeaponMax )
	{
		int nWeaponIndex = nObject - kWeaponBase;

		pSprite->type = (short)nObject;
		pSprite->picnum = gWeaponData[nWeaponIndex].picnum;
		pSprite->shade = gWeaponData[nWeaponIndex].shade ;
		pSprite->xrepeat = gWeaponData[nWeaponIndex].xrepeat;
		pSprite->yrepeat = gWeaponData[nWeaponIndex].yrepeat;
	}
	else
		ThrowError("Unhandled nObject passed to DropPickupObject()", ES_ERROR);
}


BOOL actHealDude( XSPRITE *pXDude, int healValue, int maxHealthClip)
{
	dassert(pXDude != NULL);

	healValue <<= 4;	// fix this later in the calling code
	maxHealthClip <<= 4;
	if ( pXDude->health < maxHealthClip )
	{
		pXDude->health = ClipHigh(pXDude->health + healValue, maxHealthClip);
		dprintf("Health=%d\n", pXDude->health >> 4);
		return TRUE;
	}
	return FALSE;
}


void actKillSprite( int nSource, int nSprite, DAMAGE_TYPE damageType )
{
	dassert(nSprite >= 0 && nSprite < kMaxSprites);
	SPRITE *pSprite = &sprite[nSprite];
	SPRITE *pSource = &sprite[nSource];

	dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);
	int dudeIndex = pSprite->type - kDudeBase;
	DUDEINFO *pDudeInfo = &dudeInfo[dudeIndex];

	int nXSprite = pSprite->extra;
	dassert(nXSprite > 0);
	XSPRITE *pXSprite = &xsprite[pSprite->extra];
	pXSprite->moveState = kMoveStill;

	// handle first cerberus head death
	if (pSprite->type == kDudeCerberus)
	{
		seqSpawn(dudeInfo[dudeIndex].pSeq[kSeqDudeDeath1], SS_SPRITE, nXSprite);
		return;
	}

	changespritestat((short)nSprite, kStatThing);
	trTriggerSprite( nSprite, pXSprite, kCommandOff ); // trigger death message

	pSprite->flags |= kAttrMove | kAttrFall;

	if ( IsPlayerSprite(nSprite) )
	{
		PLAYER *pPlayer = &gPlayer[pSprite->type - kDudePlayer1];
		powerupClear( pPlayer );
		dprintf("health = %i\n",pXSprite->health);
		if (pXSprite->health == 0)
			pPlayer->deathTime = 0;

		if ( IsPlayerSprite(nSource) )
		{
			int nKilledIndex = pSprite->type - kDudePlayer1;
			int nKillerIndex = pSource->type - kDudePlayer1;
			PLAYER *pFragger = &gPlayer[nKillerIndex];
			if (nSource == nSprite) // fragged yourself, eh?
			{
				pPlayer->fragCount--;
				pPlayer->fragInfo[nKillerIndex]--;	// frags against self is negative
			}
			else
			{
				pFragger->fragCount++;
				pFragger->fragInfo[nKilledIndex]++;	// frags against others are positive
			}
		}
	}

	if ( pXSprite->key > 0 )
		DropPickupObject( nSprite, kItemKey1 + pXSprite->key - 1 );

	int deathType;
	switch (damageType)
	{
		case kDamageExplode:
			deathType = kSeqDudeDeath2;
			break;

		case kDamageBurn:
			deathType = kSeqDudeDeath3;
			break;

		default:
			deathType = kSeqDudeDeath1;
			break;
	}

	// are we missing this sequence?  if so, just delete it
	if ( dudeInfo[dudeIndex].pSeq[deathType] == NULL )
	{
		dprintf("sprite missing death sequence: deleted\n");
		seqKill(SS_SPRITE, nXSprite);	// make sure we remove any active sequence
		deletesprite((short)nSprite);
		return;
	}

	switch (pSprite->type)
	{
		case kDudeFatZombie:
			if ( damageType == kDamageBurn )
			{
				int thingIndex = kThingZombieBones - kThingBase;

				pSprite->type = kThingZombieBones;
				pSprite->clipdist = thingInfo[thingIndex].clipdist;
				pSprite->flags = thingInfo[thingIndex].flags;
				pSprite->xvel = pSprite->yvel = pSprite->zvel = 0;
				pXSprite->health = thingInfo[thingIndex].startHealth << 4;
			}
			seqSpawn(dudeInfo[dudeIndex].pSeq[deathType], SS_SPRITE, nXSprite);
			break;

		default:
			seqSpawn(dudeInfo[dudeIndex].pSeq[deathType], SS_SPRITE, nXSprite);
			break;
	}

	// drop any items or weapons
	if (pSprite->type == kDudeTommyCultist)
	{
		int nDropCheck = Random(100);

		// constants?  table?
		if (nDropCheck < 80)
			DropPickupObject( nSprite, kAmmoBullets );
		else if (nDropCheck < 95)
			DropPickupObject( nSprite, kAmmoBulletBox );
		else
			DropPickupObject( nSprite, kWeaponTommyGun );
	}
	else if (pSprite->type == kDudeShotgunCultist)
	{
		int nDropCheck = Random(100);
		if (nDropCheck < 40)
			DropPickupObject( nSprite, kAmmoShells );
		else if (nDropCheck < 75)
			DropPickupObject( nSprite, kAmmoShellBox );
		else
			DropPickupObject( nSprite, kWeaponShotgun );
	}

//		SpawnGibs( nSprite, pXSprite );
	// gib generator
	if ( damageType == kDamageExplode )
	{
		int angle, velocity = 120;

		for (int i = 0; i < kGibMax && pDudeInfo->gib[i].chance > 0; i++)
		{
			if ( Random(256) < pDudeInfo->gib[i].chance )
			{
				SPRITE *pGib = &sprite[actCloneSprite(pSprite)];

				angle = Random(kAngle360);
				pGib->type = kThingGibSmall;
				pGib->picnum = pDudeInfo->gib[i].tile;
				pGib->xvel += mulscale30(velocity, Cos(angle));
				pGib->yvel += mulscale30(velocity, Sin(angle));
				pGib->zvel -= 128;	// toss it in the air a bit
				pGib->cstat &= ~kSpriteBlocking & ~kSpriteHitscan;
				pGib->flags = kAttrMove | kAttrFall;
				pGib->pal = kPLUNormal;
			}
		}
	}
}


void actDamageSprite( int nSource, int nSprite, DAMAGE_TYPE nDamageType, int nDamage )
{
	dassert(nSprite >= 0 && nSprite < kMaxSprites);
	SPRITE *pSprite = &sprite[nSprite];

	int nXSprite = pSprite->extra;
	if (nXSprite < 0)
		return;

	dassert(nXSprite > 0 && nXSprite < kMaxXSprites);
	XSPRITE *pXSprite = &xsprite[nXSprite];

	if (pXSprite->health == 0)	// it's already toast
		return;

//	nDamage <<= 4;
	switch ( pSprite->statnum )
	{
		case kStatDude:
		{
			dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);

			// calculate and apply damage to the dude or player sprite
			int nShift = dudeInfo[pSprite->type - kDudeBase].damageShift[nDamageType];
			nDamage = (nShift >= kNoDamage) ? 0 : nDamage >> nShift;
			pXSprite->health = ClipLow(pXSprite->health - nDamage, 0);

			// process results and effects of damage
			if ( IsPlayerSprite(pSprite) )
				playerDamageSprite( &gPlayer[pSprite->type - kDudePlayer1], nDamage );
			else
				aiDamageSprite( pSprite, pXSprite, nSource, nDamageType, nDamage );

			// kill dudes lacking health to sustain life
			if (pXSprite->health == 0)
			{
				// prevent dudes from exploding from weak explosion damage
				if ( nDamageType == kDamageExplode && nDamage < (5 << 4) )
					nDamageType = kDamagePummel;
				actKillSprite(nSource, nSprite, nDamageType);
			}
			break;
		}

		case kStatThing:
		case kStatProximity:
		{
			dassert(pSprite->type >= kThingBase && pSprite->type < kThingMax);
			int thingIndex = pSprite->type - kThingBase;
			nDamage >>= thingInfo[thingIndex].damageShift[nDamageType];
			pXSprite->health = ClipLow(pXSprite->health - nDamage, 0);
			if (pXSprite->health == 0)
			{
				pSprite->owner = (short)nSource;
				trTriggerSprite(nSprite, pXSprite, kCommandOff);

				switch ( pSprite->type )
				{
					case kThingBlueVase:
						seqSpawn(effectSeq[kSeqVase1], SS_SPRITE, pSprite->extra);
						if (pXSprite->data1 > 0)
							DropPickupObject( nSprite, pXSprite->data1 );
						if (pXSprite->data2 > 0)
							DropPickupObject( nSprite, pXSprite->data2 );
						break;
					case kThingBrownVase:
						seqSpawn(effectSeq[kSeqVase2], SS_SPRITE, pSprite->extra);
						if (pXSprite->data1 > 0)
							DropPickupObject( nSprite, pXSprite->data1 );
						if (pXSprite->data2 > 0)
							DropPickupObject( nSprite, pXSprite->data2 );
						break;
					case kThingClearGlass:
						pSprite->yrepeat >>= 1;
						seqSpawn(effectSeq[kSeqClearGlass], SS_SPRITE, pSprite->extra);
						break;
					case kThingFluorescent:
						seqSpawn(effectSeq[kSeqFluorescentLight], SS_SPRITE, pSprite->extra);
						break;
					case kThingWoodBeam:
						seqSpawn(effectSeq[kSeqBeam], SS_SPRITE, pSprite->extra);
						break;
					case kThingWeb:
						seqSpawn(effectSeq[kSeqWeb], SS_SPRITE, pSprite->extra);
						break;
					case kThingMetalGrate1:
						seqSpawn(effectSeq[kSeqMetalGrate1], SS_SPRITE, pSprite->extra);
						break;
					case kThingZombieBones:
						dprintf("damaging zombie bones\n");
						if ( seqGetStatus(SS_SPRITE, nXSprite) < 0 )	// body finished burning
							seqSpawn(effectSeq[kSeqZombieBones], SS_SPRITE, pSprite->extra);
						break;
				}
			}
			break;
		}
	}
}


void actImpactMissile( int nSprite, int hitInfo )
{
	SPRITE *pMissile = &sprite[nSprite];
	int hitType = hitInfo & kHitTypeMask;
	int hitObject = hitInfo & kHitIndexMask;

	int nXSprite = pMissile->extra;

	switch (pMissile->type)
	{
		case kMissileFireball:
			actExplodeSprite( nSprite );
			break;

		case kMissileFlare:
			seqKill(SS_SPRITE, nXSprite);
			deletesprite((short)nSprite);
			if ( hitType == kHitSprite && sprite[hitObject].extra > 0 )
			{
				XSPRITE *pXObject = &xsprite[sprite[hitObject].extra];
				actAddBurnTime(pMissile->owner, pXObject, 8 * kTimerRate);
				actDamageSprite(pMissile->owner, hitObject, kDamageStab, 10 << 4);
			}
			break;

		case kMissileSprayFlame:
//			seqKill(SS_SPRITE, nXSprite);
//			deletesprite((short)nSprite);
			if ( hitType == kHitSprite && sprite[hitObject].extra > 0 )
			{
				XSPRITE *pXObject = &xsprite[sprite[hitObject].extra];
				actAddBurnTime( pMissile->owner, pXObject, kFrameTicks );
			}
			break;

		case kMissileEctoSkull:
			changespritestat((short)nSprite, kStatEffect);
			seqSpawn(effectSeq[kSeqSkullExplode], SS_SPRITE, pMissile->extra);
			if ( hitType == kHitSprite && sprite[hitObject].statnum == kStatDude )
			{
				actDamageSprite(pMissile->owner, hitObject, kDamageSpirit, 50 << 4);
				SPRITE *pDude = &sprite[pMissile->owner];
				XSPRITE *pXDude = &xsprite[pDude->extra];
				if ( pXDude->health > 0 )
					actHealDude(pXDude, 25);
			}
			break;

		default:
			seqKill(SS_SPRITE, nXSprite);
			deletesprite((short)nSprite);
			if (hitType == kHitSprite)
				actDamageSprite(pMissile->owner, hitObject, kDamagePummel, 5 << 4);
			break;
	}
}

#define kGroundFriction	6
#define kAirFriction	2
#define kMinZVel 		(6 << 4)


static int MoveThing( int nSprite, char cliptype )
{
	long dx, dy, dz, z;
	int rval;
	short nSector;
	dassert(nSprite >= 0 && nSprite < kMaxSprites);
	int zTop, zBot;
	SPRITE *pSprite = &sprite[nSprite];
	long ceilz, ceilhit, floorz, floorhit;

	GetSpriteExtents(pSprite, &zTop, &zBot);
	nSector = pSprite->sectnum;
	dassert(nSector >= 0 && nSector < kMaxSectors);

	dx = pSprite->xvel;
	dy = pSprite->yvel;
	dz = pSprite->zvel;

	if ( !(dx | dy | dz) && zBot == sector[nSector].floorz )
		return 0;

	z = pSprite->z;

	if (dx || dy)
	{
		short oldcstat = pSprite->cstat;
		pSprite->cstat &= ~kSpriteBlocking & ~kSpriteHitscan;

		rval = ClipMove(&pSprite->x, &pSprite->y, &z, &nSector,
			dx * kFrameTicks >> 4, dy * kFrameTicks >> 4, pSprite->clipdist << 2,
			z - zTop, zBot - z, cliptype);

 		pSprite->cstat = oldcstat;

		if ((nSector != pSprite->sectnum) && (nSector >= 0))
			changespritesect((short)nSprite, nSector);

		switch (rval & kHitTypeMask)
		{
			case kHitWall:
			{
				int nWall = rval & kHitIndexMask;
				ReflectVector(&pSprite->xvel, &pSprite->yvel, nWall, 0x4000);
				pSprite->zvel = (short)mulscale16(pSprite->zvel, 0x8000);
				break;
			}
		}
	}

	GetZRange(pSprite, &ceilz, &ceilhit, &floorz, &floorhit, cliptype);

	if ( dz || zBot < floorz )
	{
		pSprite->z += dz * kFrameTicks;
		if ( pSprite->flags & kAttrFall )
		{
			pSprite->z += kGravity * kFrameTicks * kFrameTicks / 2;
			pSprite->zvel += kGravity * kFrameTicks;
		}
	}


	// check for warping in linked sectors
	int nUpper = gUpperLink[nSector], nLower = gLowerLink[nSector];
	if ( nUpper >= 0 && pSprite->z < sprite[nUpper].z )
	{
		nLower = sprite[nUpper].owner;
		changespritesect((short)nSprite, sprite[nLower].sectnum);
		pSprite->x += sprite[nLower].x - sprite[nUpper].x;
		pSprite->y += sprite[nLower].y - sprite[nUpper].y;
		pSprite->z += sprite[nLower].z - sprite[nUpper].z;
		viewBackupSpriteLoc(nSprite, pSprite);	// prevent interpolation
		GetZRange(pSprite, &ceilz, &ceilhit, &floorz, &floorhit, cliptype);
	}
	else if ( nLower >= 0 && pSprite->z > sprite[nLower].z )
	{
		nUpper = sprite[nLower].owner;
		changespritesect((short)nSprite, sprite[nUpper].sectnum);
		pSprite->x += sprite[nUpper].x - sprite[nLower].x;
		pSprite->y += sprite[nUpper].y - sprite[nLower].y;
		pSprite->z += sprite[nUpper].z - sprite[nLower].z;
		viewBackupSpriteLoc(nSprite, pSprite);	// prevent interpolation
		GetZRange(pSprite, &ceilz, &ceilhit, &floorz, &floorhit, cliptype);
	}

	GetSpriteExtents(pSprite, &zTop, &zBot);

	// hit floor?
	if ( zBot > floorz )
	{
		pSprite->z += ClipHigh(floorz - zBot, 0);

		if (pSprite->type == kThingTNTStick)
			pSprite->picnum = kAnmTNTBurnOff;

		pSprite->xvel = (short)mulscale16(pSprite->xvel, 0x8000);
		pSprite->yvel = (short)mulscale16(pSprite->yvel, 0x8000);
		pSprite->zvel = (short)mulscale16(-pSprite->zvel, 0x2000);
		if ( qabs(pSprite->zvel) < kMinZVel)
			pSprite->zvel = 0;
		return kHitFloor | nSector;
	}

	// hit ceiling
	if ( zTop < ceilz && ((ceilhit & kHitTypeMask) != kHitSector || !(sector[nSector].ceilingstat & kSectorParallax)) )
	{
		pSprite->z += ClipLow(ceilz - zTop, 0);

		pSprite->xvel = (short)mulscale16(pSprite->xvel, 0x8000);
		pSprite->yvel = (short)mulscale16(pSprite->yvel, 0x8000);
		pSprite->zvel = (short)mulscale16(-pSprite->zvel, 0x2000);
		if ( qabs(pSprite->zvel) < kMinZVel)
			pSprite->zvel = 0;
		return kHitCeiling | nSector;
	}

	if ( zBot == floorz && !pSprite->zvel && (pSprite->xvel || pSprite->yvel) ) 		// sliding
	{
		int vel = qdist(pSprite->xvel, pSprite->yvel);
		int nDrag = ClipHigh(kFrameTicks * kGroundFriction, vel);

		if ( (floorhit & kHitTypeMask) == kHitSprite )
		{
			int nUnderSprite = floorhit & kHitIndexMask;
			if ( (sprite[nUnderSprite].cstat & kSpriteRMask) == kSpriteFace )
			{
				// push it off the face sprite
				pSprite->xvel += mulscale(kFrameTicks, pSprite->x - sprite[nUnderSprite].x, 6);
				pSprite->yvel += mulscale(kFrameTicks, pSprite->y - sprite[nUnderSprite].y, 6);
				return rval;
			}
		}

		dassert(vel != 0);
		nDrag = divscale16(nDrag, vel);
		pSprite->xvel -= mulscale16(nDrag, pSprite->xvel);
		pSprite->yvel -= mulscale16(nDrag, pSprite->yvel);
	}

	return rval;
}


static void MoveDude( int nSprite, char cliptype )
{
	long dx, dy, dz, z;
	int rval;
	short nSector;
	dassert(nSprite >= 0 && nSprite < kMaxSprites);
	int zTop, zBot;
	SPRITE *pSprite = &sprite[nSprite];
	XSPRITE *pXSprite = &xsprite[pSprite->extra];
	long ceilz, ceilhit, floorz, floorhit;
	dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];

	GetSpriteExtents(pSprite, &zTop, &zBot);
	nSector = pSprite->sectnum;
	dassert(nSector >= 0 && nSector < kMaxSectors);

	dx = pSprite->xvel;
	dy = pSprite->yvel;
	dz = pSprite->zvel;

	if ( !(dx | dy | dz) && zBot == sector[nSector].floorz )
		return;

	z = pSprite->z;

	if (dx || dy)
	{
		short oldcstat = pSprite->cstat;
		pSprite->cstat &= ~kSpriteBlocking & ~kSpriteHitscan;

		rval = ClipMove(&pSprite->x, &pSprite->y, &z, &nSector,
			dx * kFrameTicks >> 4, dy * kFrameTicks >> 4, pSprite->clipdist << 2,
			z - zTop, (zBot - z) / 2, cliptype);

 		pSprite->cstat = oldcstat;

		if ((nSector != pSprite->sectnum) && (nSector >= 0))
			changespritesect((short)nSprite, nSector);

		switch (rval & kHitTypeMask)
		{
			case kHitWall:
			{
				int nWall = rval & kHitIndexMask;
				ReflectVector(&pSprite->xvel, &pSprite->yvel, nWall, 0x4000);
				pSprite->zvel = (short)mulscale16(pSprite->zvel, 0x8000);
				break;
			}
		}
	}

	GetZRange(pSprite, &ceilz, &ceilhit, &floorz, &floorhit, cliptype);

	if ( dz || zBot < floorz )
	{
		pSprite->z += dz * kFrameTicks;
		if ( pSprite->flags & kAttrFall )
		{
			pSprite->z += kGravity * kFrameTicks * kFrameTicks / 2;
			pSprite->zvel += kGravity * kFrameTicks;
		}
	}

	GetSpriteExtents(pSprite, &zTop, &zBot);

	// hit floor?
	if ( zBot > floorz )
	{
		pSprite->z += ClipHigh(floorz - zBot, 0);

		pSprite->xvel = (short)mulscale16(pSprite->xvel, 0x8000);
		pSprite->yvel = (short)mulscale16(pSprite->yvel, 0x8000);
		pSprite->zvel = (short)mulscale16(-pSprite->zvel, 0x2000);
		if ( qabs(pSprite->zvel) < kMinZVel)
			pSprite->zvel = 0;
	}

	// hit ceiling
	if ( zTop < ceilz && ((ceilhit & kHitTypeMask) != kHitSector || !(sector[nSector].ceilingstat & kSectorParallax)) )
	{
		pSprite->z += ClipLow(ceilz - zTop, 0);

		pSprite->xvel = (short)mulscale16(pSprite->xvel, 0x8000);
		pSprite->yvel = (short)mulscale16(pSprite->yvel, 0x8000);
		pSprite->zvel = (short)mulscale16(-pSprite->zvel, 0x2000);
		if ( qabs(pSprite->zvel) < kMinZVel)
			pSprite->zvel = 0;
	}

	// sliding
	if ( zBot == floorz && !pSprite->zvel && (pSprite->xvel || pSprite->yvel) )
	{
		int vel = qdist(pSprite->xvel, pSprite->yvel);
		int nDrag = ClipHigh(kFrameTicks * kGroundFriction, vel);

		if ( (floorhit & kHitTypeMask) == kHitSprite )
		{
			int nUnderSprite = floorhit & kHitIndexMask;
			if ( (sprite[nUnderSprite].cstat & kSpriteRMask) == kSpriteFace )
			{
				// push it off the face sprite
				pSprite->xvel += mulscale(kFrameTicks, pSprite->x - sprite[nUnderSprite].x, 6);
				pSprite->yvel += mulscale(kFrameTicks, pSprite->y - sprite[nUnderSprite].y, 6);
				return;
			}
		}

		dassert(vel != 0);
		nDrag = divscale16(nDrag, vel);
		pSprite->xvel -= mulscale16(nDrag, pSprite->xvel);
		pSprite->yvel -= mulscale16(nDrag, pSprite->yvel);
	}
	else if ( pSprite->xvel || pSprite->yvel )	// air resistance
	{
		int vel = qdist(pSprite->xvel, pSprite->yvel);
		int nDrag = ClipHigh(kFrameTicks * kAirFriction, vel);

		dassert(vel != 0);
		nDrag = divscale16(nDrag, vel);
		pSprite->xvel -= mulscale16(nDrag, pSprite->xvel);
		pSprite->yvel -= mulscale16(nDrag, pSprite->yvel);
	}
}


// missiles are self-propelled and are unaffected by gravity
static int MoveMissile( int nSprite )
{
	dassert(nSprite >= 0 && nSprite < kMaxSprites);
	SPRITE *pSprite = &sprite[nSprite];

	return movesprite((short)nSprite, pSprite->xvel, pSprite->yvel, pSprite->zvel,
		4 << 8, 4 << 8, 1, kFrameTicks);
}


void actExplodeSprite( int nSprite )
{
	SPRITE *pSprite = &sprite[nSprite];
	int nXSprite = pSprite->extra;
	dassert(nXSprite > 0 && nXSprite < kMaxXSprites);

	// already exploding?
	if (pSprite->statnum == kStatExplosion)
		return;

	switch ( pSprite->type )
	{
		case kThingTNTStick:
			if (pSprite->zvel)
				seqSpawn(effectSeq[kSeqExplosion4], SS_SPRITE, nXSprite);
			else
				seqSpawn(effectSeq[kSeqExplosion2], SS_SPRITE, nXSprite);
			break;

		case kThingTNTBarrel:
			{
				// spawn an explosion effect
				int nEffect = actSpawnSprite( pSprite->sectnum, pSprite->x, pSprite->y, pSprite->z, kStatExplosion, TRUE );
				sprite[nEffect].owner = pSprite->owner;	// set owner for frag/targeting

				// place barrel on the respawn list or just delete it
				if ( actCheckRespawn( nSprite ) )
				{
					XSPRITE *pXSprite = &xsprite[nXSprite];
					pXSprite->state = 0;
					pXSprite->health = thingInfo[kThingTNTBarrel - kThingBase].startHealth << 4;
				}
				else
					deletesprite( (short)nSprite );

				// reset locals to point at the effect, not the barrel
				nSprite = nEffect;
				pSprite = &sprite[nEffect];
				nXSprite = pSprite->extra;
				seqSpawn(effectSeq[kSeqExplosion1], SS_SPRITE, nXSprite);
			}
			break;

		default:
			seqSpawn(effectSeq[kSeqExplosion4], SS_SPRITE, nXSprite);
			break;
	}
	pSprite->xvel = pSprite->yvel = pSprite->zvel = 0;
	changespritestat((short)nSprite, kStatExplosion);
	pSprite->flags &= ~kAttrMove;
}


void actProcessSprites(void)
{
	int nSprite, nNext;
	int nDude, nNextDude;

	// process proximity triggered sprites
	for (nSprite = headspritestat[kStatProximity]; nSprite >= 0; nSprite = nNext)
	{
		nNext = nextspritestat[nSprite];
		SPRITE *pSprite = &sprite[nSprite];
		int nXSprite = pSprite->extra;
		dassert(nXSprite > 0 && nXSprite < kMaxXSprites);
		XSPRITE *pXSprite = &xsprite[nXSprite];

		for (nDude = headspritestat[kStatDude]; nDude >= 0; nDude = nNextDude)
		{
			nNextDude = nextspritestat[nDude];
			if ( CheckProximity(&sprite[nDude], pSprite->x, pSprite->y, pSprite->z, pSprite->sectnum, 64) )
				trTriggerSprite(nSprite, pXSprite, kCommandSpriteProximity);
		}
	}

	// process things for effects
	for (nSprite = headspritestat[kStatThing]; nSprite >= 0; nSprite = nNext)
	{
		nNext = nextspritestat[nSprite];
		SPRITE *pSprite = &sprite[nSprite];

		if ( pSprite->extra > 0 )
		{
			XSPRITE *pXSprite = &xsprite[pSprite->extra];
			if ( actGetBurnTime(pXSprite) > 0 )
			{
				pXSprite->burnTime = ClipLow(pXSprite->burnTime - kFrameTicks, 0);
				actDamageSprite( pXSprite->burnSource, nSprite, kDamageBurn, 2 * kFrameTicks );
			}
		}
 	}

	// process things for movement
	for (nSprite = headspritestat[kStatThing]; nSprite >= 0; nSprite = nNext)
	{
		SPRITE *pSprite = &sprite[nSprite];
		nNext = nextspritestat[nSprite];

		if ( pSprite->flags & kAttrMove )
		{
			viewBackupSpriteLoc(nSprite, pSprite);

			int hitInfo = MoveThing(nSprite, 1);
			if (hitInfo != 0)
			{
				int nXSprite = pSprite->extra;
				if (nXSprite > 0)
				{
					XSPRITE *pXSprite = &xsprite[nXSprite];
					if ( pXSprite->triggerProximity )	// proximity bomb?
						trTriggerSprite(nSprite, pXSprite, kCommandSpriteProximity);
					switch( pSprite->type )
					{
						case kThingBoneClub:
							seqSpawn(effectSeq[kSeqBoneBreak], SS_SPRITE, nXSprite);
							if ( (hitInfo & kHitTypeMask) == kHitSprite )
								actDamageSprite( pSprite->owner, hitInfo & kHitIndexMask, kDamagePummel, 12 );
							break;
					}
				}
			}
		}
 	}

	// process missile sprites
	for (nSprite = headspritestat[kStatMissile]; nSprite >= 0; nSprite = nNext)
	{
		SPRITE *pSprite = &sprite[nSprite];
		nNext = nextspritestat[nSprite];
		int hitInfo = MoveMissile( nSprite );

		viewBackupSpriteLoc(nSprite, pSprite);

		// process impacts
		if (hitInfo != 0)
			actImpactMissile( nSprite, hitInfo );
	}

	// process explosions
	for (nSprite = headspritestat[kStatExplosion]; nSprite >= 0; nSprite = nextspritestat[nSprite])
	{
		SPRITE *pSprite = &sprite[nSprite];
		int x = pSprite->x, y = pSprite->y, z = pSprite->z, nSector = pSprite->sectnum;
		int nAffected;
		int radius = tilesizx[pSprite->picnum] * pSprite->xrepeat >> 6;

		for (nAffected=headspritestat[kStatDude]; nAffected >= 0; nAffected = nNext)
		{
			nNext = nextspritestat[nAffected];
			if ( CheckProximity(&sprite[nAffected], x, y, z, nSector, radius) )
				ConcussSprite(pSprite->owner, nAffected, x, y, z, kFrameTicks * 256);
		}

		for (nAffected=headspritestat[kStatThing]; nAffected >= 0; nAffected = nNext)
		{
			nNext = nextspritestat[nAffected];
			if ( CheckProximity(&sprite[nAffected], x, y, z, nSector, radius) )
				ConcussSprite(pSprite->owner, nAffected, x, y, z, kFrameTicks * 256);
		}

		for (nAffected = headspritestat[kStatProximity]; nAffected >= 0; nAffected = nNext)
		{
			nNext = nextspritestat[nAffected];
			if ( CheckProximity(&sprite[nAffected], x, y, z, nSector, radius) )
			{
				ConcussSprite(pSprite->owner, nAffected, x, y, z, kFrameTicks * 256);
				// actExplodeSprite(nAffected);
			}
		}
	}

	// process dudes for effects
	for (nSprite = headspritestat[kStatDude]; nSprite >= 0; nSprite = nNext)
	{
		nNext = nextspritestat[nSprite];
		SPRITE *pSprite = &sprite[nSprite];

		if ( pSprite->extra > 0 )
		{
			XSPRITE *pXSprite = &xsprite[pSprite->extra];
			if ( actGetBurnTime(pXSprite) > 0 )
			{
				pXSprite->burnTime = ClipLow(pXSprite->burnTime - kFrameTicks, 0);
				actDamageSprite( pXSprite->burnSource, nSprite, kDamageBurn, 2 * kFrameTicks );
			}
		}
	}

	// process dudes for movement
	for (nSprite = headspritestat[kStatDude]; nSprite >= 0; nSprite = nNext)
	{
		SPRITE *pSprite = &sprite[nSprite];
		nNext = nextspritestat[nSprite];
		int nSector = pSprite->sectnum;

		viewBackupSpriteLoc(nSprite, pSprite);

		// special sector processing
		if ( sector[nSector].extra > 0 )
		{
			int nXSector = sector[ nSector ]. extra;
			dassert(nXSector > 0 && nXSector < kMaxXSectors);
			dassert(xsector[nXSector].reference == nSector);

			XSECTOR *pXSector = &xsector[nXSector];
			if ( pXSector->panVel && (pXSector->panAlways || pXSector->busy))
			{
				int windDrag = 0x0200; // 16:16 fixed point fraction
				int panVel = pXSector->panVel;
				int panAngle = pXSector->panAngle;

	 			if ( !pXSector->panAlways && pXSector->busy )
	 				panVel = mulscale16(panVel, pXSector->busy);

				if (sector[nSector].floorstat & kSectorRelAlign)
				{
					panAngle += GetWallAngle(sector[nSector].wallptr) + kAngle90;
					panAngle &= kAngleMask;
				}

				if (pXSector->wind)
				{
			        int windX = mulscale30(panVel, Cos(panAngle)) - pSprite->xvel;
			        int windY = mulscale30(panVel, Sin(panAngle)) - pSprite->yvel;
				    pSprite->xvel += mulscale16(kFrameTicks * windX, windDrag);
				    pSprite->yvel += mulscale16(kFrameTicks * windY, windDrag);
				}
			}

			// handle water dragging
			if ( pXSector->depth > 0 )
			{
				if ( pSprite->z >= sector[nSector].floorz )
				{
					int pushDrag  = dragTable[pXSector->depth];
					int panVel   = 0;
					int panAngle = pXSector->panAngle;

					if (pXSector->panAlways || pXSector->state || pXSector->busy)
					{
						panVel = pXSector->panVel;
						if ( !pXSector->panAlways && pXSector->busy )
							panVel = mulscale16(panVel, pXSector->busy);
					}

					if (sector[nSector].floorstat & kSectorRelAlign)
					{
						panAngle += GetWallAngle(sector[nSector].wallptr) + kAngle90;
						panAngle &= kAngleMask;
					}
				    int pushX = mulscale30(panVel, Cos(panAngle)) - pSprite->xvel;
				    int pushY = mulscale30(panVel, Sin(panAngle)) - pSprite->yvel;
					int nDrag = ClipHigh(kFrameTicks * pushDrag, 0x10000);
					pSprite->xvel += mulscale16(pushX, nDrag);
					pSprite->yvel += mulscale16(pushY, nDrag);
				}
			} else if ( pXSector->underwater )	// handle underwater dragging
			{
				int pushDrag  = dragTable[pXSector->depth];
				int panVel   = 0;
				int panAngle = pXSector->panAngle;

				if (pXSector->panAlways || pXSector->state || pXSector->busy)
				{
					panVel = pXSector->panVel;
					if ( !pXSector->panAlways && pXSector->busy )
						panVel = mulscale16(panVel, pXSector->busy);
				}

				if (sector[nSector].floorstat & kSectorRelAlign)
				{
					panAngle += GetWallAngle(sector[nSector].wallptr) + kAngle90;
					panAngle &= kAngleMask;
				}
				int pushX = mulscale30(panVel, Cos(panAngle)) - pSprite->xvel;
				int pushY = mulscale30(panVel, Sin(panAngle)) - pSprite->yvel;
				pSprite->xvel += mulscale16(kFrameTicks * pushX, pushDrag);
				pSprite->yvel += mulscale16(kFrameTicks * pushY, pushDrag);
			}
		}

		if (pSprite->type < kDudePlayer1 || pSprite->type > kDudePlayer8)
			MoveDude(nSprite, 0);
 	}

	aiProcessDudes();
}


/***********************************************************************
 * actSpawnSprite()
 *
 * Spawns a new sprite at the specified world coordinates.
 **********************************************************************/
int actSpawnSprite( short nSector, int x, int y, int z, short nStatus, BOOL bAddXSprite )
{
	int nSprite = insertsprite( nSector, nStatus );
	if (nSprite >= 0)
		sprite[nSprite].extra = -1;
	else
	{
		dprintf("Out of sprites -- reclaiming sprite from purge list\n");
		nSprite = headspritestat[kStatPurge];
		dassert(nSprite >= 0);
		changespritesect((short)nSprite, nSector);
		changespritestat((short)nSprite, nStatus);
	}
	setsprite( (short)nSprite, x, y, z );

	SPRITE *pSprite = &sprite[nSprite];

	pSprite->type = 0;

	viewBackupSpriteLoc(nSprite, pSprite);

	// optionally create an xsprite
	if ( bAddXSprite && pSprite->extra == -1 )
		dbInsertXSprite(nSprite);

	return nSprite;
}


int actCloneSprite( SPRITE *pSourceSprite )
{
	int nSprite = insertsprite( pSourceSprite->sectnum, pSourceSprite->statnum);
	if ( nSprite < 0 )
	{
		dprintf("Out of sprites -- reclaiming sprite from purge list\n");
		nSprite = headspritestat[kStatPurge];
		dassert(nSprite >= 0);
		changespritesect((short)nSprite, pSourceSprite->sectnum);
		changespritestat((short)nSprite, pSourceSprite->statnum);
	}
	SPRITE *pSprite = &sprite[nSprite];
	*pSprite = *pSourceSprite;
	viewBackupSpriteLoc(nSprite, pSprite);

	// don't copy xsprite
	pSprite->extra = -1;
	return nSprite;
}


void actFireThing( int nActor, int z, int nSlope, int thingType )
{
	dassert( thingType >= kThingBase && thingType < kThingMax );

	SPRITE *pActor = &sprite[nActor];

	int nThing = actSpawnSprite(pActor->sectnum,
		pActor->x + mulscale30(pActor->clipdist << 2, Cos(pActor->ang)),
		pActor->y + mulscale30(pActor->clipdist << 2, Sin(pActor->ang)),
		z, kStatThing, TRUE );

	SPRITE *pThing = &sprite[nThing];
	pThing->type = (short)thingType;
	int thingIndex = thingType - kThingBase;

	int nXThing = pThing->extra;
	dassert(nXThing > 0 && nXThing < kMaxXSprites);
	XSPRITE *pXThing = &xsprite[nXThing];

	switch (thingType)
	{
		case kThingTNTStick:
		{
			pThing->clipdist = thingInfo[thingIndex].clipdist;
			pThing->flags = thingInfo[thingIndex].flags;
			pXThing->health = thingInfo[thingIndex].startHealth << 4;

			pThing->owner = (short)nActor;
			pThing->shade = -32;
			pThing->pal = 0;
			pThing->xrepeat = 32;
			pThing->yrepeat = 32;
			pThing->picnum = kAnmTNTStick;
			pThing->ang = pActor->ang;

			int velocity = (M2X(7.0) << 4) / kTimerRate;
			pThing->xvel = (short)mulscale30(velocity, Cos(pActor->ang));
			pThing->yvel = (short)mulscale30(velocity, Sin(pActor->ang));
			pThing->zvel = (short)mulscale(velocity, nSlope, 18);
			pThing->xvel += pActor->xvel / 2;
			pThing->yvel += pActor->yvel / 2;
			pThing->zvel += pActor->zvel / 2;
			pThing->zvel -= 384;

			show2dsprite[nThing >> 3] |= (uchar)(1 << (nThing & 7));
			evPost(nThing, SS_SPRITE, 2 * kTimerRate);		// 2 second burn off
			break;
		}
		case kThingBoneClub:
		{
			pThing->clipdist = thingInfo[thingIndex].clipdist;
			pThing->flags = thingInfo[thingIndex].flags;
			pXThing->health = thingInfo[thingIndex].startHealth << 4;

			pThing->owner = (short)nActor;
			pThing->shade = -32;
			pThing->pal = 0;
			pThing->xrepeat = 32;
			pThing->yrepeat = 32;
			pThing->picnum = kAnmBoneClub;
			pThing->ang = pActor->ang;

			int velocity = (M2X(12.0) << 4) / kTimerRate;
			pThing->xvel = (short)mulscale30(velocity, Cos(pActor->ang));
			pThing->yvel = (short)mulscale30(velocity, Sin(pActor->ang));
			pThing->zvel = (short)mulscale(velocity, nSlope, 18);
			pThing->xvel += pActor->xvel / 2;
			pThing->yvel += pActor->yvel / 2;
			pThing->zvel += pActor->zvel / 2;
			pThing->zvel -= 256;

			show2dsprite[nThing >> 3] |= (uchar)(1 << (nThing & 7));
			break;
		}
	}
}


void actFireMissile( int nActor, int z, int dx, int dy, int dz, int missileType )
{
	dassert( missileType >= kMissileBase && missileType < kMissileMax );

	SPRITE *pActor = &sprite[nActor];

	int nSprite = actSpawnSprite(pActor->sectnum,
		pActor->x + mulscale30(pActor->clipdist << 1, Cos(pActor->ang)),
		pActor->y + mulscale30(pActor->clipdist << 1, Sin(pActor->ang)),
		z, kStatMissile, TRUE );

	SPRITE *pSprite = &sprite[nSprite];
	pSprite->type = (short)missileType;

	MissileType *pMissType = &missileInfo[ missileType - kMissileBase ];
	int velocity = pMissType->velocity;

	pSprite->shade = pMissType->shade;
	pSprite->pal = 0;
	pSprite->clipdist = 16;
	pSprite->flags = kAttrMove;
	pSprite->xrepeat = pMissType->xrepeat;
	pSprite->yrepeat = pMissType->yrepeat;
	pSprite->picnum = pMissType->picnum;
	pSprite->ang = (short)((pActor->ang +
		missileInfo[missileType - kMissileBase].angleOfs) & kAngleMask);
	pSprite->xvel = (short)mulscale(velocity, dx, 14);
	pSprite->yvel = (short)mulscale(velocity, dy, 14);
	pSprite->zvel = (short)mulscale(velocity, dz, 18);
	pSprite->owner = (short)nActor;
	show2dsprite[nSprite >> 3] |= (uchar)(1 << (nSprite & 7));

	int nXSprite = pSprite->extra;
	dassert(nXSprite > 0 && nXSprite < kMaxXSprites);
	XSPRITE *pXSprite = &xsprite[nXSprite];

	switch( missileType )
	{
		case kMissileButcherKnife:
			pSprite->cstat |= kSpriteWall;
			break;
		case kMissileSpear:
			pSprite->cstat |= kSpriteWall;
			break;
		case kMissileEctoSkull:
			seqSpawn(effectSeq[kSeqSkull], SS_SPRITE, nXSprite);
 			break;
		case kMissileFireball:
			seqSpawn(effectSeq[kSeqFireball], SS_SPRITE, nXSprite, &FireballCallback);
 			break;
		case kMissileSprayFlame:
			if ( Random(2) )
				seqSpawn(effectSeq[kSeqSprayFlame1], SS_SPRITE, nXSprite);
			else
				seqSpawn(effectSeq[kSeqSprayFlame2], SS_SPRITE, nXSprite);
			pSprite->xvel += (pActor->xvel / 2) + BiRandom(50);
			pSprite->yvel += (pActor->yvel / 2) + BiRandom(50);
			pSprite->zvel += (pActor->zvel / 2) + BiRandom(50);
			break;
		case kMissileFlare:
			//seqSpawn(effectSeq[kSeqFlare], SS_SPRITE, nXSprite, &FlareCallback);
			//pSprite->pal = kPLURed;
			break;
	}
}


/***********************************************************************
 * actSpawnEffect()
 *
 * Spawns an effect sprite at the specified world coordinates.
 **********************************************************************/
void actSpawnEffect( short nSector, int x, int y, int z, int nEffect )
{
//	show2dsprite[nSprite >> 3] &= ~(uchar)(1 << (nSprite & 7)); // don't show this

	int nSprite = actSpawnSprite(nSector, x, y, z, kStatEffect, TRUE);
	SPRITE *pSprite = &sprite[nSprite];
	int nXSprite = pSprite->extra;
	dassert(nXSprite > 0 && nXSprite < kMaxXSprites);

	switch( nEffect )
	{
		case ET_Ricochet1:
			if ( Random(2) )
				pSprite->cstat |= kSpriteFlipX;
			seqSpawn(effectSeq[kSeqRicochet1], SS_SPRITE, nXSprite);
			break;

		case ET_Squib1:
			if ( Random(2) )
				pSprite->cstat |= kSpriteFlipX;
			seqSpawn(effectSeq[kSeqSquib1], SS_SPRITE, nXSprite);
			break;

		case ET_SmokeTrail:
			if ( Random(2) )
				pSprite->cstat |= kSpriteFlipX;
			if ( Random(2) )
				pSprite->cstat |= kSpriteFlipY;
			seqSpawn(effectSeq[kSeqFlareSmoke], SS_SPRITE, nXSprite);
			break;

		default:
			// don't have a sequence yet, so just delete it for now
			deletesprite((short)nSprite);
			break;
	}
}


BOOL actCheckRespawn( int nSprite )
{
	SPRITE *pSprite = &sprite[nSprite];

	if ( pSprite->extra > 0 )
	{
		XSPRITE *pXSprite = &xsprite[pSprite->extra];

		if (pXSprite->respawn == kRespawnPermanent)
			return TRUE;

		// if (sprite respawn is forced OR (sprite has optional respawning set
		// AND (sprite is a dude AND global dude respawning is set)
		// OR (sprite is a thing AND global thing respawning is set)))
		if (pXSprite->respawn == kRespawnAlways || (pXSprite->respawn == kRespawnOptional
		&& (pSprite->type >= kDudeBase && pSprite->type < kDudeMax && gRespawnEnemies)
		|| ((pSprite->type < kDudeBase || pSprite->type > kDudeMax) && gRespawnItems)))
		{
			if  (pXSprite->respawnTime > 0)
			{
				dprintf("Respawn statnum: %d\n",pSprite->statnum);
				pSprite->owner = pSprite->statnum; // store the sprite's status list for respawning
				changespritestat( (short)nSprite, kStatRespawn );
				pSprite->cstat &= ~kSpriteBlocking & ~kSpriteHitscan;
				pSprite->cstat |= kSpriteInvisible;
			 	evPost(nSprite, SS_SPRITE, pXSprite->respawnTime * kTimerRate / 10, kCommandRespawn);
				return TRUE;
			}
		}
	}
	return FALSE;	// indicate sprite will not respawn, and should be deleted, exploded, etc.
}


void actRespawnDude( int nSprite )
{
	SPRITE *pSprite = &sprite[nSprite];
	XSPRITE *pXSprite = &xsprite[pSprite->extra];

	// probably need more dude-specific initialization
	pXSprite->moveState = 0;
	pXSprite->aiState = 0;
	pXSprite->health = 0;
	pXSprite->target = 0;
	pXSprite->targetX = 0;
	pXSprite->targetY = 0;
	pXSprite->targetZ = 0;
	pXSprite->key = 0;	// respawned dudes don't drop additional permanent keys
}


void actRespawnSprite( int nSprite )
{
	SPRITE *pSprite = &sprite[nSprite];

	changespritestat( (short)nSprite, pSprite->owner ); // restore the sprite's status list
	pSprite->owner = -1;
	pSprite->cstat &= ~kSpriteInvisible;

	if ( pSprite->extra > 0)
	{
		XSPRITE *pXSprite = &xsprite[pSprite->extra];
		pXSprite->isTriggered = 0;

		if (pSprite->type >= kDudeBase && pSprite->type < kDudeMax)
			actRespawnDude( nSprite );
	}

	// spawn the respawn special effect
	int nRespawn = actSpawnSprite( pSprite->sectnum, pSprite->x, pSprite->y, pSprite->z, kStatDefault, TRUE );
	seqSpawn(effectSeq[kSeqRespawn], SS_SPRITE, sprite[nRespawn].extra);
}



// this function should probably be replace by a table
void actSpawnRicochet( short nSector, int x, int y, int z, int nSurface )
{
	switch ( nSurface )
	{
		case kSurfStone:
		case kSurfMetal:
			actSpawnEffect( nSector, x, y, z, ET_Ricochet1);
			break;

		case kSurfFlesh:
			actSpawnEffect( nSector, x, y, z, ET_Squib1);
			break;
	}
}


void actFireVector( int nActor, int z, int dx, int dy, int dz, int maxDist,
	DAMAGE_TYPE damageType, int damageValue )
{
	SPRITE *pActor = &sprite[nActor];
	HITINFO hitInfo;

	int hitCode = HitScan( pActor, z, dx, dy, dz, &hitInfo );

	if ( maxDist == 0 || qdist(hitInfo.hitx - pActor->x, hitInfo.hity - pActor->y) < maxDist )
	{
		switch ( hitCode )
		{
			case SS_CEILING:
				if ( !(sector[hitInfo.hitsect].ceilingstat & kSectorParallax) )
					actSpawnRicochet(hitInfo.hitsect, hitInfo.hitx, hitInfo.hity,
						hitInfo.hitz, surfType[sector[hitInfo.hitsect].ceilingpicnum]);
				break;

			case SS_FLOOR:
				if ( !(sector[hitInfo.hitsect].floorstat & kSectorParallax) )
					actSpawnRicochet(hitInfo.hitsect, hitInfo.hitx, hitInfo.hity,
						hitInfo.hitz, surfType[sector[hitInfo.hitsect].floorpicnum]);
				break;

			case SS_WALL:
			{
				int nWall = hitInfo.hitwall;
				dassert( nWall >= 0 && nWall < kMaxWalls );
				WALL *pWall = &wall[nWall];

				int nx, ny;
				GetWallNormal(nWall, &nx, &ny);

				// this needs check for which part of wall (upper, masked, or lower)
				actSpawnRicochet(hitInfo.hitsect, hitInfo.hitx + (nx >> 14),
					hitInfo.hity + (ny >> 14), hitInfo.hitz, surfType[wall[nWall].picnum]);

				int nXWall = wall[nWall].extra;
				if ( nXWall > 0 )
				{
					XWALL *pXWall = &xwall[nXWall];
					if ( pXWall->triggerImpact )
						trTriggerWall( nWall, pXWall, kCommandWallImpact );
				}
				break;
			}

			case SS_SPRITE:
			{
				int nSprite = hitInfo.hitsprite;
				dassert( nSprite >= 0 && nSprite < kMaxSprites);
				SPRITE *pSprite = &sprite[nSprite];

				actSpawnRicochet(pSprite->sectnum, hitInfo.hitx, hitInfo.hity, hitInfo.hitz,
					surfType[pSprite->picnum]);

				actDamageSprite(nActor, nSprite, damageType, damageValue << 4);

				int nXSprite = pSprite->extra;
				if ( nXSprite > 0 )
				{
					XSPRITE *pXSprite = &xsprite[nXSprite];
					if ( pXSprite->triggerImpact )
						trTriggerSprite( nSprite, pXSprite, kCommandSpriteImpact );
				}
				break;
			}
		}
	}
}

static void FireballCallback( int /* type */, int nXIndex )
{
	XSPRITE *pXSprite = &xsprite[nXIndex];
	int nSprite = pXSprite->reference;
	SPRITE *pSprite = &sprite[nSprite];

	actSpawnEffect( pSprite->sectnum, pSprite->x, pSprite->y, pSprite->z, ET_SmokeTrail );
}

//static void FlareCallback( int /* type */, int nXIndex )
//{
//	XSPRITE *pXSprite = &xsprite[nXIndex];
//	int nSprite = pXSprite->reference;
//	SPRITE *pSprite = &sprite[nSprite];
//
//	actSpawnEffect( pSprite->sectnum, pSprite->x, pSprite->y, pSprite->z, ET_SmokeTrail );
//}


