#include "gfx.h"
#include "helix.h"
#include "debug4g.h"
#include "misc.h"

#include <memcheck.h>


static int clipX0 = 0, clipY0 = 0, clipX1 = 320, clipY1 = 200;
Rect clipRect(0, 0, 320, 200);

static BYTE *fontTable = (BYTE *)0xFFA6E;



void gfxDrawBitmap( QBITMAP *qbm, int x, int y )
{
	dassert(qbm != NULL);

	Point p(x,y);
	Rect dest(x, y, x + qbm->cols, y + qbm->rows);
	dest &= clipRect;

	if ( !dest )
		return;

	Rect source(dest);
	source -= p;

	switch ( qbm->bitModel )
	{
		case BM_RAW:
			Video.BlitM2V(qbm, source.x0, source.y0, source.x1, source.y1, 0, dest.x0, dest.y0);
			break;

		case BM_TRAW:
			Video.BlitMT2V(qbm, source.x0, source.y0, source.x1, source.y1, 0, dest.x0, dest.y0);
			break;
	}
}


void gfxPixel( int x, int y )
{
	if ( clipRect.contains(x, y) )
		Video.SetPixel(0, x, y);
}


void gfxHLine( int y, int x0, int x1 )
{
	if ( y < clipRect.y0 || y >= clipRect.y1 )
		return;

	x0 = ClipLow(x0, clipRect.x0);
	x1 = ClipHigh(x1, clipRect.x1 - 1);

	if ( x0 <= x1 )
		Video.HLine(0, y, x0, x1);
}


void gfxVLine( int x, int y0, int y1 )
{
	if ( x < clipRect.x0 || x >= clipRect.x1 )
		return;

	y0 = ClipLow(y0, clipRect.y0);
	y1 = ClipHigh(y1, clipRect.y1 - 1);

	if ( y0 <= y1 )
		Video.VLine(0, x, y0, y1);
}


void gfxHLineROP( int y, int x0, int x1 )
{
	if ( y < clipRect.y0 || y >= clipRect.y1 )
		return;

	x0 = ClipLow(x0, clipRect.x0);
	x1 = ClipHigh(x1, clipRect.x1 - 1);

	if ( x0 <= x1 )
		Video.HLineROP(0, y, x0, x1);
}


void gfxVLineROP( int x, int y0, int y1 )
{
	if ( x < clipRect.x0 || x >= clipRect.x1 )
		return;

	y0 = ClipLow(y0, clipRect.y0);
	y1 = ClipHigh(y1, clipRect.y1 - 1);

	if ( y0 <= y1 )
		Video.VLineROP(0, x, y0, y1);
}


void gfxFillBox( int x0, int y0, int x1, int y1 )
{
	Rect r(x0, y0, x1, y1);
	r &= clipRect;

	if ( r.isValid() )
		Video.FillBox(0, r.x0, r.y0, r.x1, r.y1);
}


void gfxSetClip( int x0, int y0, int x1, int y1 )
{
	clipRect.x0 = x0;
	clipRect.y0 = y0;
	clipRect.x1 = x1;
	clipRect.y1 = y1;

	clipX0 = x0 << 8;
	clipY0 = y0 << 8;
	clipX1 = (x1 << 8) - 1;
	clipY1 = (y1 << 8) - 1;
}


#define TOP 	8
#define BOTTOM	4
#define LEFT	2
#define RIGHT	1



/*******************************************************************************
	FUNCTION:		gfxDrawLine()

	DESCRIPTION:	Draw and clip a line to the screen

	PARAMETERS:		screen coordinates in 24:8 format
*******************************************************************************/
void gfxDrawLine(int x0, int y0, int x1, int y1, int nColor)
{
	int code0, code1;

	while (1)
	{
		code0 = code1 = 0;

		if (y0 < clipY0) code0 |= TOP;
		else if (y0 > clipY1) code0 |= BOTTOM;

		if (x0 < clipX0) code0 |= LEFT;
		else if (x0 > clipX1) code0 |= RIGHT;

		if (y1 < clipY0) code1 |= TOP;
		else if (y1 > clipY1) code1 |= BOTTOM;

		if (x1 < clipX0) code1 |= LEFT;
		else if (x1 > clipX1) code1 |= RIGHT;

		// trivial rejection
		if (code0 & code1) return;

		// clip
		if (code0 != code1)
		{
			// clip lines
			if (code0 & TOP)
			{
				x0 += muldiv(x1 - x0, clipY0 - y0, y1 - y0);
				y0 = clipY0;
				continue;
			}

			if (code0 & BOTTOM)
			{
				x0 += muldiv(x1 - x0, clipY1 - y0, y1 - y0);
				y0 = clipY1;
				continue;
			}

			if (code0 & LEFT)
			{
				y0 += muldiv(y1 - y0, clipX0 - x0, x1 - x0);
				x0 = clipX0;
				continue;
			}

			if (code0 & RIGHT)
			{
				y0 += muldiv(y1 - y0, clipX1 - x0, x1 - x0);
				x0 = clipX1;
				continue;
			}

			if (code1 & TOP)
			{
				x1 += muldiv(x1 - x0, clipY0 - y1, y1 - y0);
				y1 = clipY0;
				continue;
			}

			if (code1 & BOTTOM)
			{
				x1 += muldiv(x1 - x0, clipY1 - y1, y1 - y0);
				y1 = clipY1;
				continue;
			}

			if (code1 & LEFT)
			{
				y1 += muldiv(y1 - y0, clipX0 - x1, x1 - x0);
				x1 = clipX0;
				continue;
			}

			if (code1 & RIGHT)
			{
				y1 += muldiv(y1 - y0, clipX1 - x1, x1 - x0);
				x1 = clipX1;
				continue;
			}
		}

		if (y0 == y1)
		{
			Video.SetColor(nColor);
			if (x0 < x1)
				Video.HLine(0, y0 >> 8, x0 >> 8, x1 >> 8);
			else
				Video.HLine(0, y0 >> 8, x1 >> 8, x0 >> 8);
			return;
		}

		if (x0 == x1)
		{
			Video.SetColor(nColor);
			if (y0 < y1)
				Video.VLine(0, x0 >> 8, y0 >> 8, y1 >> 8);
			else
				Video.VLine(0, x0 >> 8, y1 >> 8, y0 >> 8);
			return;
		}

		Video.SetColor(nColor);
		Video.Line(0, x0 >> 8, y0 >> 8, x1 >> 8, y1 >> 8);
		return;
	}
}


void gfxDrawPixel( int x, int y, int nColor )
{
	if ( clipRect.contains(x, y) )
	{
		Video.SetColor(nColor);
		Video.SetPixel(0, x, y);
	}
}


static void printChar( int x, int y, char c )
{
	for (int row = 0; row < 8; row++)
	{
		BYTE mask = 0x80;
		for (int col = 0; col < 8; col++, mask >>= 1)
		{
			if ( fontTable[(c << 3) + row] & mask )
				Video.SetPixel(0, x + col, y + row);
		}
	}
}


void gfxDrawText( int x, int y, int nForeColor, char *string )
{
	Video.SetColor(nForeColor);
	for (char *p = string; *p; p++)
	{
		Rect dest(x, y, x + 8, y + 8);
		if ( clipRect.contains(dest) )
			printChar(x, y, *p);
		x += 8;
	}
}


void gfxDrawLabel( int x, int y, int nForeColor, char *string )
{
	Video.SetColor(nForeColor);
	for (char *p = string; *p; p++)
	{
		Rect dest(x, y, x + 8, y + 8);
		if ( clipRect.contains(dest) )
		{
			if ( *p == '&' )
				gfxHLine(y + 8, x, x + 6);
			else
				printChar(x, y, *p);
		}

		if ( *p != '&' )
			x += 8;
	}
}

