#include "weapon.h"
#include "globals.h"
#include "qav.h"
#include "debug4g.h"
#include "actor.h"
#include "db.h"
#include "actor.h"
#include "gameutil.h"
#include "trig.h"
#include "view.h"
#include "options.h"
#include "names.h"
#include "triggers.h"
#include "error.h"
#include "misc.h"
#include "screen.h"
#include "tile.h"

#define kShotgunBarrels		2
#define kVectorsPerBarrel	16
#define kMaxShotgunVectors	(kShotgunBarrels * kVectorsPerBarrel)

enum {
	kVoodooStabChest	= 0,
	kVoodooStabShoulder,
	kVoodooStabEye,
	kVoodooStabGroin,
	kVoodooStabSelf,
};

// Weapon States
enum {
	kFindWeapon			= -1,	// flag to find a loaded weapon

	kForkInitialize		= 0,
	kForkRaise,
	kForkIdle,
	kForkAttack,
	kForkLower,

	kTNTInitialize		= 0,
	kTNTBicOpen,
	kTNTBicIdle,
	kTNTRaise,
	kTNTIdle,
	kTNTLower,

	kCanInitialize		= 0,
	kCanBicOpen,
	kCanBicIdle,
	kCanIdle,
	kCanLower,
	kCanFire,

	kShotInitialize		= 0,	// indicates first-time reload required
	kShotLoaded,
	kShotRaise,
	kShotIdle1,
	kShotIdle2,
 	kShotIdle3,
	kShotFire1,
	kShotFire2,
	kShotReload,
	kShotLower,

	kFlareInitialize	= 0,
	kFlareRaise,
	kFlareIdle,
	kFlareFire,
	kFlareReload,
	kFlareLower,

	kTommyInitialize	= 0,
	kTommyRaise,
	kTommyIdle,
	kTommyFire1,
	kTommyLower,

	kSpearInitialize	= 0,
	kSpearRaise,
	kSpearIdle1,
	kSpearIdle2,
	kSpearFire,
	kSpearReload,
	kSpearLower,

	kShadowInitialize	= 0,
	kShadowRaise,
	kShadowIdle,
	kShadowFire,
	kShadowLower,

	kBeastInitialize	= 0,
	kBeastRaise,
	kBeastIdle,
	kBeastAttack,
	kBeastLower,

	kDollInitialize	= 0,
	kDollRaise,
	kDollIdle,
	kDollStab1,
	kDollStab2,
	kDollStab3,
	kDollStab4,
	kDollStab5,
	kDollLower
};

enum {
	kQAVForkUp = 0,
	kQAVForkIdle,
	kQAVForkStab,
	kQAVForkDown,

	kQAVBicUp,			// raise lighter and open
	kQAVBicFlame,		// light lighter
	kQAVBicIdle,		// burning lighter
	kQAVBicDown,		// lower burning lighter
	kQAVBicDown_NoFlame,	// lower unlit lighter

	kQAVSprayUp,
	kQAVSprayIdle,
	kQAVSprayFire,
	kQAVSprayDown,

	kQAVStickUp,
	kQAVStickIdle,
	kQAVStickLight,
	kQAVStickAim,
	kQAVStickDown,
	kQAVStickThrow,

	kQAVBundleUp,
	kQAVBundleIdle,
	kQAVBundleLight,
	kQAVBundleAim,
	kQAVBundleDown,
	kQAVBundleThrow,

	kQAVFlareUp,
	kQAVFlareIdle,
	kQAVFlareFire,
//	kQAVFlareReload,
	kQAVFlareDown,

	kQAVShotUp,
	kQAVShotIdle1,
	kQAVShotIdle2,
	kQAVShotIdle3,
	kQAVShotFireL,
	kQAVShotFireR,
	kQAVShotReload,
	kQAVShotDown,

	kQAVTommyUp,
	kQAVTommyIdle,
	kQAVTommyFire1,
 	kQAVTommyDown,

	kQAVSpearUp,
	kQAVSpearIdle1,
	kQAVSpearIdle2,
	kQAVSpearFire,
	kQAVSpearDown,

	kQAVShadowUp,
	kQAVShadowIdle,
	kQAVShadowFire,
	kQAVShadowDown,

	kQAVBeastUp,
	kQAVBeastIdle,
	kQAVBeastAttack1,
	kQAVBeastAttack2,
 	kQAVBeastAttack3,
 	kQAVBeastAttack4,
	kQAVBeastDown,

	kQAVDollUp,
	kQAVDollIdleStill,
	kQAVDollIdleMoving,
	kQAVDollStab1,
	kQAVDollStab2,
	kQAVDollStab3,
	kQAVDollStab4,
	kQAVDollStab5,
	kQAVDollDown,

	kQAVEnd
};

static char * weaponQAVNames[kQAVEnd] =
{
	"FORKUP",  		//kQAVForkUp = 0,
	"FORKIDLE",     //kQAVForkIdle,
	"PFORK",        //kQAVForkStab,
	"FORKDOWN",     //kQAVForkDown,

	"LITEOPEN",     //kQAVBicUp,
	"LITEFLAM",     //kQAVBicFlame,
	"LITEIDLE",     //kQAVBicIdle,
	"LITECLO2",     //kQAVBicDown,
	"LITECLOS",     //kQAVBicDown_NoFlame,

	"CANPREF",      //kQAVSprayUp,
	"CANIDLE",      //kQAVSprayIdle,
	"CANFIRE",		//kQAVSprayFire1,
	"CANDOWN",      //kQAVSprayDown,

	"DYNPRE",       //kQAVStickUp,
	"DYNIDLE1",     //kQAVStickIdle,
	"DYNLITA",      //kQAVStickLight,
	"DYNFUSEA",     //kQAVStickAim,
	"DYNDOWN",      //kQAVStickDown,
	"THROW",        //kQAVStickThrow,

	"DYNPRE2",      //kQAVBundleUp,
	"DYNIDLE2",     //kQAVBundleIdle,
	"DYNLITB",      //kQAVBundleLight,
	"DYNFUSEB",     //kQAVBundleAim,
	"DYNDOWN2",     //kQAVBundleDown,
	"THROW",        //kQAVBundleThrow,

	"FLARUP",
	"FLARIDLE",
	"FLARFIRE",
//	"FLARLOAD",
	"FLARDOWN",

	"SHOTUP",		//kQAVShotUp,
	"SHOTI1",       //kQAVShotIdle1,
	"SHOTI2",       //kQAVShotIdle2,
	"SHOTI3",       //kQAVShotIdle3,
	"SHOTF1",       //kQAVShotFireL,
	"SHOTF2",       //kQAVShotFireR,
	"SHOTL1",       //kQAVShotReload,
	"SHOTDOWN",     //kQAVShotDown,

	"TOMUP",        //kQAVTommyUp,
	"TOMIDLE",      //kQAVTommyIdle,
	"TOMFIRE",		//kQAVTommyFire1,
	"TOMDOWN",      //kQAVTommyDown,

	"SGUNUP",		//kQAVSpearUp,
	"SGUNIDL1",		//kQAVSpearIdle1,
	"SGUNIDL2",		//kQAVSpearIdle2,
	"SGUNFIRE",		//kQAVSpearFire,
	"SGUNDOWN",		//kQAVSpearDown,

	"DARKUP",       //kQAVShadowUp,
	"DARKIDLE",     //kQAVShadowIdle,
	"DARKFIRE",     //kQAVShadowFire,
	"DARKDOWN",     //kQAVShadowDown,

	"BSTUP",		//kQAVBeastUp
	"BSTIDLE",		//kQAVBeastIdle,
	"BSTATAK1",     //kQAVBeastAttack1,
	"BSTATAK2",     //kQAVBeastAttack2,
	"BSTATAK3",     //kQAVBeastAttack3,
	"BSTATAK4",     //kQAVBeastAttack4,
	"BSTDOWN",

	"VDUP",			//kQAVDollUp
	"VDIDLE1",		//kQAVDollIdleStill	(PLAYER IS STILL)
	"VDIDLE2",		//kQAVDollIdleMoving	(PLAYER IS WALKING/RUNNING)
	"VDFIRE1",		//kQAVDollStab1
	"VDFIRE2",		//kQAVDollStab2
	"VDFIRE3",		//kQAVDollStab3
	"VDFIRE4",		//kQAVDollStab4
	"VDFIRE5",		//kQAVDollStab5
	"VDDOWN", 		//kQAVDollDown
};


static QAV *weaponQAV[kQAVEnd];


static BOOL CheckAmmo( PLAYER *pPlayer, int nWeapon )
{
	if ( gInfiniteAmmo )
		return TRUE;

	int nAmmo = pPlayer->weaponAmmo[nWeapon];
	if ( nAmmo == kAmmoNone )
		return TRUE;

	return pPlayer->ammoCount[nAmmo] > 0;
}


static void UseAmmo( PLAYER *pPlayer, int nWeapon, int nCount )
{
	if ( gInfiniteAmmo )
		return;

	int nAmmo = pPlayer->weaponAmmo[nWeapon];
	if ( nAmmo == kAmmoNone )
		return;

	pPlayer->ammoCount[nAmmo] = ClipLow(pPlayer->ammoCount[nAmmo] - nCount, 0);
}


void WeaponInit( void )
{
	// get pointers to all the QAV resources
	for (int i = 0; i < kQAVEnd; i++)
	{
		RESHANDLE hQAV = gSysRes.Lookup(weaponQAVNames[i], ".QAV");
		if (hQAV == NULL)
		{
			dprintf("Could not load %s.QAV\n", weaponQAVNames[i]);
			ThrowError("Missing Weapon QAV file", ES_ERROR);
		}
		weaponQAV[i] = (QAV *)gSysRes.Lock(hQAV);
	}

	dprintf("Preload weapon QAV tiles\n");
	for (i = 0; i < kQAVEnd; i++)
	{
		if (weaponQAV[i] != NULL )
			weaponQAV[i]->Preload();
	}

	for (i = 0; i < kMaxPlayers; i++)
	{
		gPlayer[i].weaponAmmo[kWeaponPitchfork] = kAmmoNone;
		gPlayer[i].weaponAmmo[kWeaponSprayCan] = kAmmoSprayCan;
		gPlayer[i].weaponAmmo[kWeaponTNT] = kAmmoTNTStick;
		gPlayer[i].weaponAmmo[kWeaponShotgun] = kAmmoShell;
		gPlayer[i].weaponAmmo[kWeaponTommy] = kAmmoBullet;
		gPlayer[i].weaponAmmo[kWeaponFlare] = kAmmoFlare;
		gPlayer[i].weaponAmmo[kWeaponVoodoo] = kAmmoVoodoo;
		gPlayer[i].weaponAmmo[kWeaponSpear] = kAmmoSpear;
		gPlayer[i].weaponAmmo[kWeaponShadow] = kAmmoNone;
	}
}


void WeaponDraw( PLAYER *pPlayer, int shade, int x, int y )
{
	dassert(pPlayer != NULL);
	QAV *pQAV = pPlayer->pWeaponQAV;
	ulong t;

	if ( pQAV == NULL )
		return;

	if (pPlayer->weaponTimer == 0)	// playing idle QAV?
		t = gGameClock % pQAV->duration;
	else
		t = pQAV->duration - pPlayer->weaponTimer;

	pQAV->origin.x = x;
	pQAV->origin.y = y;

	int nPowerRemaining = powerupCheck( pPlayer, kItemLtdInvisibility - kItemBase );
	int nFlags = kQFrameScale;

	if ( nPowerRemaining >= (kTimerRate * 8) || nPowerRemaining && ( gGameClock & 32 ) )
	{
		nFlags |= kQFrameTranslucent;
		shade = -128;
	}

	pQAV->Draw(t, shade, kQFrameScale);
}


void WeaponPlay( PLAYER *pPlayer )
{
	dassert(pPlayer != NULL);
	QAV *pQAV = pPlayer->pWeaponQAV;

	if ( pQAV == NULL )
		return;

	ulong t = pQAV->duration - pPlayer->weaponTimer;

	pQAV->Play(t - kFrameTicks, t, pPlayer->weaponCallback, pPlayer);
}


static void StartQAV( PLAYER *pPlayer, int weaponIndex, QAVCALLBACK callback = NULL, BOOL fLoop = FALSE)
{
	pPlayer->pWeaponQAV = weaponQAV[weaponIndex];
	pPlayer->weaponTimer = pPlayer->pWeaponQAV->duration;
	pPlayer->weaponCallback = callback;
	pPlayer->fLoopQAV = fLoop;
}


#define kAimMaxDist 	M2X(100)
#define kAimMaxAngle	kAngle10
#define kAimMaxSlope	8 << 14			// z units per xy unit

static int WeaponGetAimVector( PLAYER *pPlayer, long *pdx, long *pdy, long *pdz )
{
	dassert( (pPlayer != NULL) && (pdx != NULL) && (pdy != NULL) && (pdz != NULL) );
	SPRITE *pSprite = pPlayer->sprite;

	int x = pSprite->x;
	int y = pSprite->y;
	int z = pSprite->z - pPlayer->weaponAboveZ;

	*pdx = Cos(pSprite->ang) >> 16;
	*pdy = Sin(pSprite->ang) >> 16;
	*pdz = pPlayer->slope;

	int nAimSprite = -1;

	if ( gAutoAim )
	{
		int closest = 0x7FFFFFFF;
		for (short nTarget = headspritestat[kStatDude]; nTarget >= 0; nTarget = nextspritestat[nTarget] )
		{
			SPRITE *pTarget = &sprite[nTarget];

			// don't target yourself!
			if ( pTarget == pSprite )
				continue;

			int dx, dy, dz;

			dx = pTarget->x - x;
			dy = pTarget->y - y;
			dz = pTarget->z - z;

			int dist = qdist(dx, dy);
			if ( dist > closest || dist > kAimMaxDist )
				continue;

			int ang = getangle(dx, dy);
			if ( qabs(((ang - pSprite->ang + kAngle180) & kAngleMask) - kAngle180) > kAimMaxAngle )
				continue;

			int nSlope = divscale(dz, dist, 14);
			if ( qabs(nSlope - pPlayer->slope) > kAimMaxSlope )
				continue;

			if ( cansee(x, y, z, pSprite->sectnum, pTarget->x, pTarget->y, pTarget->z, pTarget->sectnum) )
			{
				closest = dist;

				*pdx = Cos(ang) >> 16;
				*pdy = Sin(ang) >> 16;
				*pdz = nSlope;
				nAimSprite = nTarget;
			}
		}
	}
	return nAimSprite;
}


void WeaponLower( PLAYER *pPlayer, INPUT *pInput )
{
	dprintf("WeaponLower()\n");
	dassert( pPlayer != 0 );

	int nWeapon = pPlayer->weapon;
	int nState = pPlayer->weaponState[nWeapon];

	switch( nWeapon )
	{
		case kWeaponVoodoo:
			StartQAV(pPlayer, kQAVDollDown);
			break;

		case kWeaponPitchfork:
			StartQAV(pPlayer, kQAVForkDown);
			break;

		case kWeaponFlare:
			StartQAV(pPlayer, kQAVFlareDown);
			break;

		case kWeaponSprayCan:
			switch( nState )
			{
				case kCanBicIdle:
					// switch without putting down lighter
					if ( pInput->newWeapon == kWeaponTNT )
					{
						pPlayer->weapon = kWeaponTNT;
						pPlayer->weaponState[kWeaponTNT] = kTNTBicIdle;
						pPlayer->pWeaponQAV = weaponQAV[kQAVBicIdle];
						pInput->newWeapon = 0;
						return;
					}
			        StartQAV(pPlayer, kQAVBicDown);
					break;

				default:
				    StartQAV(pPlayer, kQAVSprayDown);
					pPlayer->weaponState[kWeaponSprayCan] = kCanBicIdle;
					return;

			}
			break;

		case kWeaponTNT:
			switch( nState )
			{
				case kTNTBicIdle:
					if ( pInput->newWeapon == kWeaponSprayCan )
					{
						pPlayer->weapon = kWeaponSprayCan;
						pPlayer->weaponState[kWeaponSprayCan] = kCanBicIdle;
						pPlayer->pWeaponQAV = weaponQAV[kQAVBicIdle];
						pInput->newWeapon = 0;
						return;
					}
			        StartQAV(pPlayer, kQAVBicDown);
					break;

				default:
					StartQAV(pPlayer, kQAVStickDown);
					pPlayer->weaponState[kWeaponTNT] = kTNTBicIdle;
					return;
			}
			break;

		case kWeaponShotgun:
	        StartQAV(pPlayer, kQAVShotDown);
			break;

		case kWeaponTommy:
	        StartQAV(pPlayer, kQAVTommyDown);
			break;

		case kWeaponSpear:
	        StartQAV(pPlayer, kQAVSpearDown);
			break;

		case kWeaponShadow:
	        StartQAV(pPlayer, kQAVShadowDown);
			break;

		case kWeaponBeast:
	        StartQAV(pPlayer, kQAVBeastDown);
			break;
	}

	if (pInput->newWeapon == pPlayer->weapon)
		pInput->newWeapon = 0;

	pPlayer->weapon = 0;
}


void WeaponRaise( PLAYER *pPlayer, INPUT *pInput )
{
	dprintf("WeaponRaise()\n");
	dassert( pPlayer != 0 );

	pPlayer->weapon = pInput->newWeapon;
	pInput->newWeapon = 0;

	switch( pPlayer->weapon )
	{
		case kWeaponVoodoo:
			StartQAV(pPlayer, kQAVDollUp);
			pPlayer->weaponState[kWeaponVoodoo] = kDollIdle;
			break;

		case kWeaponPitchfork:
			StartQAV(pPlayer, kQAVForkUp);
			pPlayer->weaponState[kWeaponPitchfork] = kForkIdle;
			break;

		case kWeaponFlare:
			StartQAV(pPlayer, kQAVFlareUp);
			pPlayer->weaponState[kWeaponFlare] = kFlareIdle;
			break;

		case kWeaponSprayCan:
			StartQAV(pPlayer, kQAVBicUp);
			pPlayer->weaponState[kWeaponSprayCan] = kCanBicOpen;
			break;

		case kWeaponTNT:
			StartQAV(pPlayer, kQAVBicUp);
			pPlayer->weaponState[kWeaponTNT] = kTNTBicOpen;
			break;

		case kWeaponShotgun:
			StartQAV(pPlayer, kQAVShotUp);
			switch (pPlayer->weaponState[kWeaponShotgun])
			{
				case kShotInitialize:
				case kShotIdle3:
//					if (pPlayer->ammoCount[ kAmmoShells - kAmmoBase ] > 0)
//					    pPlayer->weaponState[kWeaponShotgun] = kShotReload;
//					else
					    pPlayer->weaponState[kWeaponShotgun] = kShotIdle3;
					break;
				case kShotIdle2:
					if (gInfiniteAmmo || pPlayer->ammoCount[kAmmoShell] > 1)
					    pPlayer->weaponState[kWeaponShotgun] = kShotReload;
					break;
			}
			break;

		case kWeaponTommy:
			StartQAV(pPlayer, kQAVTommyUp);
			pPlayer->weaponState[kWeaponTommy] = kTommyIdle;
			break;

		case kWeaponSpear:
			StartQAV(pPlayer, kQAVSpearUp);
			if ( pPlayer->ammoCount[kAmmoSpear] > 0 || pPlayer->ammoCount[kAmmoSpearXP] > 0 )
				pPlayer->weaponState[kWeaponSpear] = kSpearIdle1;
			else
				pPlayer->weaponState[kWeaponSpear] = kSpearIdle2;
			break;

		case kWeaponShadow:
			StartQAV(pPlayer, kQAVShadowUp);
			pPlayer->weaponState[kWeaponShadow] = kShadowIdle;
			break;

		case kWeaponBeast:
			StartQAV(pPlayer, kQAVBeastUp); // no up yet...
			pPlayer->weaponState[kWeaponBeast] = kBeastIdle;
			break;
	}
}


static void FireVoodoo( int nStabType, PLAYER *pPlayer )
{
	SPRITE *pTarget = NULL;
	int nSelfSprite = pPlayer->nSprite;
	int z = pPlayer->sprite->z - pPlayer->weaponAboveZ;

	long dx, dy, dz;
	int nAimSprite = WeaponGetAimVector( pPlayer, &dx, &dy, &dz );

	if ( nAimSprite >= 0 )
		pTarget = &sprite[nAimSprite];

	switch( nStabType )
	{
		case kVoodooStabSelf:
			actDamageSprite(nSelfSprite, pPlayer->nSprite, kDamageStab, 2 << 4);
			if (pPlayer == gMe)
				scrSetMessage("Ouch!");
			break;

		case kVoodooStabChest:
			actSpawnRicochet(pTarget->sectnum, pTarget->x, pTarget->y, pTarget->z,
				surfType[pTarget->picnum]);
			actDamageSprite(nSelfSprite, nAimSprite, kDamageStab, 2 << 4);
			break;

		case kVoodooStabShoulder:
			actSpawnRicochet(pTarget->sectnum, pTarget->x, pTarget->y, pTarget->z,
				surfType[pTarget->picnum]);
			actDamageSprite(nSelfSprite, nAimSprite, kDamageStab, 4 << 4);
			if ( pTarget->type >= kDudePlayer1 && pTarget->type <= kDudePlayer8 )
			{
				PLAYER *pPlayerHit = &gPlayer[pTarget->type - kDudePlayer1];

				// yikes!! This is very bad -- wrong pInput!
//					WeaponLower( pPlayerHit, pInput );
			}
			break;

		case kVoodooStabGroin:
			actSpawnRicochet(pTarget->sectnum, pTarget->x, pTarget->y, pTarget->z,
				surfType[pTarget->picnum]);
			actDamageSprite(nSelfSprite, nAimSprite, kDamageStab, 12 << 4);
			break;
		case kVoodooStabEye:
			actSpawnRicochet(pTarget->sectnum, pTarget->x, pTarget->y, pTarget->z,
				surfType[pTarget->picnum]);
			actDamageSprite(nSelfSprite, nAimSprite, kDamageStab, 6 << 4);
			if ( pTarget->type >= kDudePlayer1 && pTarget->type <= kDudePlayer8 )
			{
				PLAYER *pPlayerHit = &gPlayer[pTarget->type - kDudePlayer1];
				if (pPlayerHit == gView)	// change to gMe later!
				   	scrDacRelEffect(-96, -96, -96);
			}
			break;
	}
	pPlayer->weaponState[kWeaponVoodoo] = kDollIdle;
}


static void FireShotgun( PLAYER *pPlayer, int nBarrels )
{
	dassert(nBarrels > 0 && nBarrels <= kShotgunBarrels);

	int nVectors = nBarrels * kVectorsPerBarrel;

	SPRITE *pSprite = pPlayer->sprite;

	long dx, dy, dz;
	WeaponGetAimVector( pPlayer, &dx, &dy, &dz );
	int z = pSprite->z - pPlayer->weaponAboveZ;

	for ( int i = 0; i < nVectors; i++ )
	{
		// dispersal modifiers here; accuracy in GetAimVector()
		int ddx = BiRandom(2000);
		int ddy = BiRandom(2000);
		int ddz = BiRandom(32000);

		actFireVector(pPlayer->nSprite, z, dx + ddx, dy + ddy, dz + ddz, 0,
			kDamageBullet, 4);
	}
}


static void FireSpray( int /* nTrigger */, PLAYER *pPlayer )
{
	long dx, dy, dz;
	WeaponGetAimVector( pPlayer, &dx, &dy, &dz );

	playerFireMissile(pPlayer, dx, dy, dz, kMissileSprayFlame);
	UseAmmo(pPlayer, kWeaponSprayCan, kFrameTicks);
}


static void FireTommy( int /* nTrigger */, PLAYER *pPlayer )
{
	SPRITE *pSprite = pPlayer->sprite;

	long dx, dy, dz;
	WeaponGetAimVector( pPlayer, &dx, &dy, &dz );
	int z = pSprite->z - pPlayer->weaponAboveZ;

	UseAmmo(pPlayer, kWeaponTommy, 1);

	// dispersal modifiers here; accuracy in GetAimVector()
	int ddx = BiRandom(500);
	int ddy = BiRandom(500);
	int ddz = BiRandom(8000);

	actFireVector(pPlayer->nSprite, z, dx + ddx, dy + ddy, dz + ddz, 0,
		kDamageBullet, 4 );
}


#define kMaxForkDist	M2X(2.25)

static void FirePitchfork( int /* nTrigger */, PLAYER *pPlayer )
{
	SPRITE *pSprite = pPlayer->sprite;

	long dx, dy, dz;
	WeaponGetAimVector( pPlayer, &dx, &dy, &dz );
	int z = pSprite->z - pPlayer->weaponAboveZ;

	for ( int i = 0; i < 4; i++ )
	{
		// dispersal modifiers here; accuracy in GetAimVector()
		int ddx = BiRandom(2000);
		int ddy = BiRandom(2000);
		int ddz = BiRandom(32000);

		actFireVector(pPlayer->nSprite, z, dx + ddx, dy + ddy, dz + ddz, kMaxForkDist,
			kDamageStab, 3);
	}
}


uchar WeaponFindLoaded( PLAYER *pPlayer )
{
	int nAmmoIndex = 0;
	int nNewWeapon = kWeaponPitchfork;

	dprintf("WeaponFindLoaded: ");

	if ( pPlayer->hasWeapon[ kWeaponShadow ] && powerupCheck(pPlayer, kItemInvulnerability - kItemBase) )
		nNewWeapon = kWeaponShadow;
	else if ( pPlayer->hasWeapon[kWeaponTommy] && CheckAmmo(pPlayer, kWeaponTommy) )
		nNewWeapon = kWeaponTommy;
	else if ( pPlayer->hasWeapon[kWeaponShotgun] && CheckAmmo( pPlayer, kWeaponShotgun) )
		nNewWeapon = kWeaponShotgun;
	else if ( pPlayer->hasWeapon[kWeaponFlare] && CheckAmmo(pPlayer, kWeaponFlare) )
		nNewWeapon = kWeaponFlare;
	else if ( pPlayer->hasWeapon[kWeaponSpear] && CheckAmmo(pPlayer, kWeaponSpear) )
		nNewWeapon = kWeaponSpear;
	else if ( pPlayer->hasWeapon[kWeaponSprayCan] && CheckAmmo(pPlayer, kWeaponSprayCan) )
		nNewWeapon = kWeaponSprayCan;
	else if ( pPlayer->hasWeapon[kWeaponTNT] && CheckAmmo(pPlayer, kWeaponTNT) )
		nNewWeapon = kWeaponTNT;
	else if ( pPlayer->hasWeapon[kWeaponVoodoo] && CheckAmmo(pPlayer, kWeaponVoodoo) )
		nNewWeapon = kWeaponVoodoo;
	else if ( pPlayer->hasWeapon[kWeaponShadow] )
	{
		if ( pPlayer->xsprite->health > 75 )
			nNewWeapon = kWeaponShadow;
	}

	dprintf("nNewWeapon = %i\n", nNewWeapon);
	return (uchar)nNewWeapon;
}


/*
;[Weapon#]
;Name            = "WeaponNameMax24"
;SelectKey       = [ None | scancode ]
;UseByDefault    = [ TRUE | FALSE ]
;HasByDefault    = [ TRUE | FALSE ]
;RequiresWeapon  = [ None | 0..9 ]
;
;Type            = [ NonWeapon | Melee | Thrown | Projectile | Vector ]
;Type            = NonWeapon
;
;Type            = Melee
;MaxDistance     = [ Infinite | 0..max ]
;
;Type            = Thrown
;MaxDistance     = [ Infinite | 0..max ]
;
;Type            = Projectile
;MaxDistance     = [ Infinite | 0..max ]
;
;Type            = Vector (uses hitscan)
;MaxDistance     = [ Infinite | 0..max ]
;Richochet       = [ TRUE | FALSE ]
;UsesNightScope  = [ TRUE | FALSE ]
;
;Shots           = [ 0..max shots or projectiles ]
;Dispersion      = [ 0..100% ]
;Accuracy        = [ 0..100% ]
;
;AmmoStyle       = [ Infinite, Rechargable, Expendable, Health ]
;AmmoStyle       = Infinite
;
;AmmoStyle       = Rechargable
;RechargeRate    = [ 0 | ammo/second (?) ]
;MaxAmmo         = [ 0..max ]
;
;AmmoStyle       = Expendable
;MaxAmmo         = [ 0..max ]
;AmmoType        = [ 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 ]
;
;AmmoStyle       = Health
;HealthPerShot   = [ 0..usedpershot ]
;
;JamFrequency    = [ 0..100% ]
;JamSingleFire   = [ TRUE | FALSE ]
;JamRapidFire    = [ TRUE | FALSE ]
;
;[Ammo#]
;Name            = "AmmoNameMax24"
;Velocity        = [ ? ]
;DamageTypes     = [ Explosion, Burn, Bullet, Stab, Pummel, Cleave, Ecto]
;DamageRates     = [ 0,         0,    0,      0,    0,      0,      0 ]
*/

void WeaponUpdateState( PLAYER *pPlayer)
{
	SPRITE *pSprite		= pPlayer->sprite;
	XSPRITE *pXSprite	= pPlayer->xsprite;

	int nWeapon = pPlayer->weapon;
	int nState = pPlayer->weaponState[nWeapon];

	switch( nWeapon )
	{
		case kWeaponVoodoo:
	 		switch ( pXSprite->moveState )
			{
				case kMoveWalk:
					if ( pPlayer->swayHeight )
						pPlayer->pWeaponQAV = weaponQAV[kQAVDollIdleMoving];
					else
						pPlayer->pWeaponQAV = weaponQAV[kQAVDollIdleStill];
					break;

				case kMoveLand:
					pPlayer->pWeaponQAV = weaponQAV[kQAVDollIdleMoving];
					break;

				default:
					pPlayer->pWeaponQAV = weaponQAV[kQAVDollIdleStill];
					break;
			}
			break;

		case kWeaponPitchfork:
			pPlayer->pWeaponQAV = weaponQAV[kQAVForkIdle];
			break;

		case kWeaponFlare:
			if (nState == kFlareIdle)
				pPlayer->pWeaponQAV = weaponQAV[kQAVFlareIdle];
			break;

		case kWeaponSpear:
			switch ( nState )
			{
				case kSpearIdle1:
					pPlayer->pWeaponQAV = weaponQAV[kQAVSpearIdle1];
					break;

				case kSpearIdle2:
					pPlayer->pWeaponQAV = weaponQAV[kQAVSpearIdle2];
					break;
			}
			break;

		case kWeaponTommy:
			pPlayer->pWeaponQAV = weaponQAV[kQAVTommyIdle];
			pPlayer->weaponState[kWeaponTommy] = kTommyIdle;
			break;

		case kWeaponSprayCan:
		{
			switch( nState )
			{
				case kCanBicOpen:
					StartQAV(pPlayer, kQAVBicFlame);
					pPlayer->weaponState[kWeaponSprayCan] = kCanBicIdle;
					break;

				case kCanBicIdle:
					if ( CheckAmmo(pPlayer, kWeaponSprayCan) )
					{
						StartQAV(pPlayer, kQAVSprayUp);
						pPlayer->weaponState[kWeaponSprayCan] = kCanIdle;
					}
					else
						pPlayer->pWeaponQAV = weaponQAV[kQAVBicIdle];
					break;

				case kCanIdle:
					pPlayer->pWeaponQAV = weaponQAV[kQAVSprayIdle];
					break;

				case kCanFire:
					if ( CheckAmmo(pPlayer, kWeaponSprayCan) )
					{
						pPlayer->pWeaponQAV = weaponQAV[kQAVSprayIdle];
						pPlayer->weaponState[kWeaponSprayCan] = kCanIdle;
					}
					else
					{
						StartQAV(pPlayer, kQAVSprayDown);
						pPlayer->weaponState[kWeaponSprayCan] = kCanBicIdle;
					}
					break;
			}
			break;
		}

		case kWeaponTNT:
			switch( nState )
			{
				case kTNTBicOpen:
					StartQAV(pPlayer, kQAVBicFlame);
					pPlayer->weaponState[nWeapon] = kTNTBicIdle;
					break;

				case kTNTBicIdle:
					if ( pPlayer->ammoCount[kAmmoTNTStick] > 0)
					{
						StartQAV(pPlayer, kQAVStickUp);
						pPlayer->weaponState[kWeaponTNT] = kTNTIdle;
					}
					else
						pPlayer->pWeaponQAV = weaponQAV[kQAVBicIdle];
					break;

				case kTNTIdle:
					pPlayer->pWeaponQAV = weaponQAV[kQAVStickIdle];
					break;
			}
			break;

		case kWeaponShotgun:
			switch ( nState )
			{
				case kShotIdle1:
					pPlayer->pWeaponQAV = weaponQAV[kQAVShotIdle1];
					break;

				case kShotIdle2:
					pPlayer->pWeaponQAV = weaponQAV[kQAVShotIdle2];
					break;

				case kShotIdle3:
					if (gInfiniteAmmo || pPlayer->ammoCount[kAmmoShell] > 0)
					    pPlayer->weaponState[kWeaponShotgun] = kShotReload;
					pPlayer->pWeaponQAV = weaponQAV[kQAVShotIdle3];
					break;
			}
			break;

		case kWeaponShadow:
			pPlayer->pWeaponQAV = weaponQAV[kQAVShadowIdle];
			break;

		case kWeaponBeast:
			pPlayer->pWeaponQAV = weaponQAV[kQAVBeastIdle];
			break;
	}
}


void WeaponProcess( PLAYER *pPlayer, INPUT *pInput )
{
	WeaponPlay(pPlayer);

	// decrement weapon ready timer
	pPlayer->weaponTimer -= kFrameTicks;

	if ( pPlayer->fLoopQAV )
	{
		if ( pInput->buttonFlags.shoot && pPlayer->ammoCount[pPlayer->weaponAmmo[pPlayer->weapon]] )
		{
			while ( pPlayer->weaponTimer <= 0 )
				pPlayer->weaponTimer += pPlayer->pWeaponQAV->duration;
			return;
		}
		pPlayer->weaponTimer = 0;
		pPlayer->fLoopQAV = FALSE;
	}
	else
		pPlayer->weaponTimer = ClipLow(pPlayer->weaponTimer, 0);

	// return if weapon is busy
	if (pPlayer->weaponTimer > 0)
		return;

	pPlayer->pWeaponQAV = NULL;

	int nWeapon = pPlayer->weapon;

	// pPlayer->weapon is set to kMaxWeapons in the fire processing below
	if ( pPlayer->weaponState[nWeapon] == kFindWeapon )
	{
		dprintf("Out of ammo: looking for a new weapon\n");
		pPlayer->weaponState[nWeapon] = 0;
		pInput->newWeapon = WeaponFindLoaded( pPlayer );
	}

	// weapon select
	if ( pInput->newWeapon )
	{
		if ( pPlayer->xsprite->health == 0 || !pPlayer->hasWeapon[ pInput->newWeapon ] )
		{
			pInput->newWeapon = 0;
			return;
		}

		if (pPlayer->weapon == 0) 	// raise the new weapon
			WeaponRaise(pPlayer, pInput);
		else
			WeaponLower(pPlayer, pInput);	// lower the old weapon
		return;
	}

	if ( (nWeapon = pPlayer->weapon) == 0 )
		return;		// no weapon drawn, do nothing

    // auto reload test
	switch ( nWeapon )
	{
		case kWeaponShotgun:
			if ( pPlayer->weaponState[nWeapon] == kShotReload )
			{
				if ( !CheckAmmo(pPlayer, kWeaponShotgun) )
					pPlayer->weaponState[kWeaponShotgun] = kShotIdle3;
				else
				{
			    	StartQAV(pPlayer, kQAVShotReload);
					if (gInfiniteAmmo || pPlayer->ammoCount[kAmmoShell] > 1)
						pPlayer->weaponState[kWeaponShotgun] = kShotIdle1;
					else
						pPlayer->weaponState[kWeaponShotgun] = kShotIdle2;
				}
				return;
			}
			break;
	}

	// fire key
	if ( pInput->buttonFlags.shoot )
	{
		if ( CheckAmmo(pPlayer, nWeapon) )
		{
			long dx, dy, dz;
			int nAimSprite = WeaponGetAimVector( pPlayer, &dx, &dy, &dz );

			// firing weapon (primary)
			switch ( nWeapon )
			{
				case kWeaponPitchfork:
					StartQAV(pPlayer, kQAVForkStab, (QAVCALLBACK)FirePitchfork, FALSE);
					return;

				case kWeaponSprayCan:
					switch ( pPlayer->weaponState[kWeaponSprayCan] )
					{
						case kCanIdle:
							StartQAV(pPlayer, kQAVSprayFire, (QAVCALLBACK)FireSpray, TRUE);
							pPlayer->weaponState[kWeaponSprayCan] = kCanFire;
							return;
					}
					break;

				case kWeaponTNT:
					if ( gInfiniteAmmo || pPlayer->ammoCount[kAmmoTNTStick] > 0 )
					{
						StartQAV( pPlayer, kQAVStickThrow );
						playerFireThing(pPlayer, kThingTNTStick);

						if ( !gInfiniteAmmo )
							pPlayer->ammoCount[kAmmoTNTStick] -= 1;

						if ( pPlayer->ammoCount[kAmmoTNTStick] == 0 )
						{
							// drop empty hand, and idle the lighter
						    StartQAV(pPlayer, kQAVBicIdle);
							pPlayer->weaponState[kWeaponTNT] = kTNTBicIdle;
						}
					}
					return;

			    case kWeaponShotgun:
					switch ( pPlayer->weaponState[kWeaponShotgun] )
					{
						case kShotIdle1:
							StartQAV(pPlayer, kQAVShotFireL);
							pPlayer->weaponState[kWeaponShotgun] = kShotIdle2;
							if ( !gInfiniteAmmo )
								pPlayer->ammoCount[kAmmoShell] -= 1;
							FireShotgun( pPlayer, 1 );
							return;

						case kShotIdle2:
							StartQAV(pPlayer, kQAVShotFireR);
						    pPlayer->weaponState[kWeaponShotgun] = kShotReload;
							if ( !gInfiniteAmmo )
								pPlayer->ammoCount[kAmmoShell] -= 1;
							FireShotgun( pPlayer, 1 );
							return;
					}
					return;

				case kWeaponTommy:
					StartQAV(pPlayer, kQAVTommyFire1, (QAVCALLBACK)FireTommy, TRUE);
					return;

				case kWeaponFlare:
					StartQAV(pPlayer, kQAVFlareFire);
					pPlayer->weaponState[kWeaponFlare] = kFlareIdle;
					playerFireMissile( pPlayer, dx, dy, dz, kMissileFlare);
					UseAmmo(pPlayer, kWeaponFlare, 1);
					return;

				case kWeaponVoodoo:
				{
					int nStabType = Random(4);
					if ( nAimSprite == -1 )
						nStabType = kVoodooStabSelf;

					StartQAV(pPlayer, kQAVDollStab1 + nStabType, (QAVCALLBACK)FireVoodoo);
					return;
				}

			    case kWeaponSpear:
						StartQAV(pPlayer, kQAVSpearFire);
						playerFireMissile(pPlayer, dx, dy, dz, kMissileSpear);
						UseAmmo(pPlayer, kWeaponSpear, 1);
						pPlayer->weaponState[kWeaponSpear] = kSpearIdle1;
						return;

				case kWeaponShadow:
					StartQAV(pPlayer, kQAVShadowFire);
				    pPlayer->weaponState[kWeaponShadow] = kShadowIdle;
					playerFireMissile( pPlayer, dx, dy, dz, kMissileEctoSkull);
	 				if ( !gInfiniteAmmo )
					{
						actDamageSprite(pPlayer->nSprite, pPlayer->nSprite, kDamageSpirit, 10 << 4);
					}
					return;

				case kWeaponBeast:
					StartQAV(pPlayer, kQAVBeastAttack1 + Random(4));
					pPlayer->weaponState[kWeaponBeast] = kBeastIdle;
					return;
			}
		}
		else
		{
			dprintf("Primary out of ammo: setting weapon state to kFindWeapon\n");
			pPlayer->weaponState[nWeapon] = kFindWeapon;
			return;
		}
	}

	// alternate fire key
	if ( pInput->buttonFlags.shoot2 )
	{
		if ( CheckAmmo(pPlayer, nWeapon) )
		{
			long dx, dy, dz;
			WeaponGetAimVector( pPlayer, &dx, &dy, &dz );

			// firing weapon (alternate)
			switch ( nWeapon )
			{
				case kWeaponPitchfork:
					playerFireMissile(pPlayer, dx, dy, dz, kMissileFireball);
					pPlayer->weaponTimer = 20;
					return;

				case kWeaponSprayCan:
					break;

				case kWeaponTNT:
					break;

				case kWeaponShotgun:
					switch ( pPlayer->weaponState[kWeaponShotgun] )
					{
						case kShotIdle1:
							StartQAV(pPlayer, kQAVShotFireR);
							pPlayer->weaponState[kWeaponShotgun] = kShotReload;
							UseAmmo(pPlayer, kWeaponShotgun, 2);
							FireShotgun(pPlayer, 2);
							return;
						case kShotIdle2:
							StartQAV(pPlayer, kQAVShotFireR);
							pPlayer->weaponState[kWeaponShotgun] = kShotReload;
							UseAmmo(pPlayer, kWeaponShotgun, 1);
							FireShotgun(pPlayer, 1);
							return;
					}
					break;

				case kWeaponTommy:
					return;

				case kWeaponFlare:
					return;

				case kWeaponSpear:
					playerFireMissile(pPlayer, dx, dy, dz, kMissileSpear);
					UseAmmo(pPlayer, kWeaponSpear, 1);
					StartQAV(pPlayer, kQAVSpearFire);
					pPlayer->weaponState[kWeaponSpear] = kSpearIdle1;
					return;

			}
		}
		else
		{
			dprintf("Alternate out of ammo: setting weapon state to kFindWeapon\n");
			pPlayer->weaponState[nWeapon] = kFindWeapon;
			return;
		}
	}

	WeaponUpdateState(pPlayer);
}
