/*******************************************************************************
	FILE:			GRABFONT.CPP

	DESCRIPTION:	Grab a font from a graphic file

	AUTHOR:			Peter M. Freese
	CREATED:		12-11-95
	COPYRIGHT:		Copyright (c) 1995 Q Studios Corporation
*******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <helix.h>
#include <string.h>
#include <dos.h>
#include <io.h>
#include <fcntl.h>
#include <conio.h>

#include "typedefs.h"
#include "key.h"
#include "misc.h"
#include "debug4g.h"
#include "getopt.h"
#include "pcx.h"
#include "lbm.h"
#include "gfx.h"

enum FILE_TYPE
{
	FT_NONE,
	FT_PAL,
	FT_PCX,
	FT_LBM,
};

static struct
{
	char *ext;
	FILE_TYPE fileType;
} ftTable[] =
{
	{ ".LBM", FT_LBM },
	{ ".PAL", FT_PAL },
	{ ".PCX", FT_PCX },
	{ ".PCC", FT_PCX },
};


char srcName[_MAX_PATH] = "";

BOOL ignore[256];
BYTE gridColor, baselineColor, tcolor;
int gridX[256], gridY[256];
int baselineY[256];
int gridRows, gridCols;
int startChar = 0;
ushort fontType = FT_COLOR;

PALETTE palette;
QBITMAP *pBitmap = NULL;
QFONT *pFont = NULL;



/***********************************************************************
 * ShowUsage
 *
 * Display command-line parameter usage, then exit.
 **********************************************************************/
void ShowUsage(void)
{
	printf("Syntax:  GRABFONT [options] from\n");
	printf("-?       This help\n");
	exit(0);
}


void QuitMessage(char * fmt, ...)
{
	char msg[80];
	va_list argptr;
	va_start( argptr, fmt );
	vsprintf( msg, fmt, argptr );
	va_end(argptr);
	printf(msg);
	exit(1);
}


FILE_TYPE GetFileType( char *filename )
{
	char ext[_MAX_EXT];
	_splitpath(filename, NULL, NULL, NULL, ext);

	for ( int i = 0; i < LENGTH(ftTable); i++ )
	{
		if ( stricmp(ext, ftTable[i].ext) == 0 )
			return ftTable[i].fileType;
	}

	return FT_NONE;
}


void LoadSource( void )
{
	int nResult;
	int srcWidth = 0, srcHeight = 0;

	FILE_TYPE fileType = GetFileType(srcName);

	if ( fileType == FT_NONE )
		QuitMessage("Don't know how to handle file %s", srcName);

	// determine the size of the source image
	switch ( fileType )
	{
		case FT_LBM:
			nResult = ReadLBM(srcName, NULL, &srcWidth, &srcHeight, NULL);
			break;

		case FT_PCX:
			nResult = ReadPCX(srcName, NULL, &srcWidth, &srcHeight, NULL);
			break;
	}

	if (nResult != 0)
		QuitMessage("Problem reading %s: error=%d", srcName, nResult);

	pBitmap = (QBITMAP *)malloc(sizeof(QBITMAP) + srcWidth * srcHeight);
	dassert(pBitmap != NULL);

	BYTE *pSourceBits = pBitmap->data;

	// decode the image
	switch ( fileType )
	{
		case FT_LBM:
			nResult = ReadLBM(srcName, palette, &srcWidth, &srcHeight, &pSourceBits);
			break;

		case FT_PCX:
			nResult = ReadPCX(srcName, palette, &srcWidth, &srcHeight, &pSourceBits);
			break;
	}


	pBitmap->bitModel = BM_RAW;
	pBitmap->tcolor = 0;
	pBitmap->cols = (short)srcWidth;
	pBitmap->rows = (short)srcHeight;
	pBitmap->stride = (short)srcWidth;
}


void ShrinkRect( Rect *rect )
{
	BYTE *pBits;
	int i;

	// top edge
	while (rect->y0 < rect->y1)
	{
		pBits = &pBitmap->data[rect->y0 * pBitmap->stride + rect->x0];
		for (i = rect->x0; i < rect->x1; i++, pBits++)
		{
			if ( !ignore[*pBits] )
				break;
		}
		if ( i < rect->x1 )
			break;

		rect->y0++;
	}

	// bottom edge
	while (rect->y0 < rect->y1)
	{
		pBits = &pBitmap->data[(rect->y1 - 1) * pBitmap->stride + rect->x0];
		for (i = rect->x0; i < rect->x1; i++, pBits++)
		{
			if ( !ignore[*pBits] )
				break;
		}
		if ( i < rect->x1 )
			break;

		rect->y1--;
	}

	// left edge
	while (rect->x0 < rect->x1)
	{
		pBits = &pBitmap->data[rect->y0 * pBitmap->stride + rect->x0];
		for (i = rect->y0; i < rect->y1; i++, pBits += pBitmap->stride)
		{
			if ( !ignore[*pBits] )
				break;
		}
		if ( i < rect->y1 )
			break;

		rect->x0++;
	}

	// right edge
	while (rect->x0 < rect->x1)
	{
		pBits = &pBitmap->data[rect->y0 * pBitmap->stride + rect->x1 - 1];
		for (i = rect->y0; i < rect->y1; i++, pBits += pBitmap->stride)
		{
			if ( !ignore[*pBits] )
				break;
		}
		if ( i < rect->y1 )
			break;

		rect->x1--;
	}
}

int CopyCharacter( BYTE *pDest, BYTE *pSource, const Rect &r )
{
	int x, y;

	BYTE *pDestStart = pDest;

	switch ( fontType )
	{
		case FT_MONO:
		{
			BYTE mask = 0x80;
			int sourceInc = pBitmap->stride - r.cols();
			for (y = r.y0; y < r.y1; y++)
			{
				if ( (mask <<= 1) == 0 )	// shifted bit out?
					mask = 0x01;
				else
					pDest -= r.cols();

				for (x = r.x0; x < r.x1; x++, pDest++, pSource++)
				{
					if ( !ignore[*pSource] )
						*pDest |= mask;
				}

				pSource += sourceInc;
			}
			return pDest - pDestStart;
		}

		case FT_COLOR:
		{
			int sourceInc = pBitmap->stride - r.cols();
			for (y = r.y0; y < r.y1; y++)
			{
				for (x = r.x0; x < r.x1; x++, pDest++, pSource++)
				{
					if ( !ignore[*pSource] )
						*pDest = *pSource;
					else
						*pDest = tcolor;
				}

				pSource += sourceInc;
			}
			return pDest - pDestStart;
		}
	}

	return 0;
}


void RipFont( void )
{
	int i, j;
	BYTE *p;

	// get color information from upper left corner
	gridColor = pBitmap->data[0];
	baselineColor = pBitmap->data[1];
	tcolor = pBitmap->data[2];

	memset(ignore, FALSE, sizeof(ignore));

	// read ignore colors until we find a repeat (background)
	for ( i = 1; !ignore[pBitmap->data[i]]; i++ )
		ignore[pBitmap->data[i]] = TRUE;

	// allocate a font large enough to hold the entire source bitmap
	pFont = (QFONT *)malloc(sizeof(QFONT) + pBitmap->rows * pBitmap->cols);
	dassert(pFont != NULL);

	memset(pFont, 0, sizeof(QFONT));

	memcpy(pFont->signature, kFontSig, sizeof(pFont->signature));
	pFont->version = kFontVersion;
	pFont->type = fontType;
	pFont->charSpace = 1;
	pFont->tcolor = tcolor;

	// find upper left of grid
	p = &pBitmap->data[pBitmap->stride];	// start of line 1
	int x = 0, y = 1;
	while ( *p != gridColor )
	{
		p++;
		if ( ++x == pBitmap->stride )
		{
			x = 0;
			y++;
		}
	}

	gridX[0] = x;
	gridY[0] = y;

	// find right edge of grid columns
	gridCols = 0;
	x = gridX[0] + 1;
	y = gridY[0] + 1;
	for ( p = &pBitmap->data[y * pBitmap->stride + x]; x < pBitmap->cols; x++, p++ )
	{
		if ( *p == gridColor )
			gridX[++gridCols] = x;
	}

	// find bottom edge and baseline of grid rows
	gridRows = 0;
	x = gridX[0] + 1;
	y = gridY[0] + 1;
	for ( p = &pBitmap->data[y * pBitmap->stride + x]; y < pBitmap->rows; y++, p += pBitmap->stride )
	{
		if ( *p == baselineColor )
			baselineY[gridRows] = y;

		if ( *p == gridColor )
			gridY[++gridRows] = y;
	}

	int nChar = startChar;
	int nOffset = 0;
	CHARINFO *pInfo = pFont->info;
	for ( i = 0; i < gridRows; i++ )
	{
		for ( j = 0; j < gridCols; j++, nChar++, pInfo++ )
		{
			Rect r(gridX[j] + 1, gridY[i] + 1, gridX[j + 1], gridY[i + 1]);
			ShrinkRect(&r);

			// skip if cell is blank
			if ( r.isEmpty() )
				continue;

			if ( r.cols() > pFont->width )
				pFont->width = (uchar)r.cols();		// max character width
			if ( r.rows() > pFont->height )
				pFont->height = (uchar)r.rows();	// max character height

			if ( baselineY[i] - r.y0 > pFont->baseline )
				pFont->baseline = (uchar)(baselineY[i] - r.y0);

			pInfo->offset = nOffset;
			pInfo->cols = (uchar)r.cols();
			pInfo->rows = (uchar)r.rows();
			pInfo->hspace = (uchar)pInfo->cols;
			pInfo->voffset = (schar)(r.y0 - baselineY[i]);

			int nSize = CopyCharacter(&pFont->data[nOffset],
				&pBitmap->data[r.y0 * pBitmap->stride + r.x0], r);
			pFont->totalSize += nSize;
			nOffset += nSize;
		}
	}

	char fontName[_MAX_PATH];
	strcpy(fontName, srcName);
	ChangeExtension(fontName, ".QFN");

	if ( !FileSave(fontName, pFont, sizeof(QFONT) + pFont->totalSize - 1) )
		QuitMessage("Unable to save file %s", fontName);
}


/***********************************************************************
 * Process command line arguments
 **********************************************************************/
void ParseOptions( void )
{
	enum {
		kSwitchHelp,
		kSwitchRLE,
		kSwitchTRaw,
		kSwitchRaw,
	};
	static SWITCH switches[] = {
		{ "?", kSwitchHelp, FALSE },
	};
	int r;
	while ( (r = GetOptions(switches)) != GO_EOF ) {
		switch (r)
		{
			case GO_INVALID:
				QuitMessage("Invalid argument: %s", OptArgument);

			case GO_FULL:
				if ( srcName[0] == '\0' )
					strcpy(srcName, OptArgument);
				else
					QuitMessage("Unexpected argument: %s", OptArgument);
				break;

			case kSwitchHelp:
				ShowUsage();
				break;
		}
	}
}


void main( int argc )
{
	if (argc < 2)
		ShowUsage();

	ParseOptions();

	if ( srcName[0] == '\0' )
		ShowUsage();

	AddExtension(srcName, ".PCX");

	LoadSource();

	RipFont();
}
