#include "typedefs.h"
#include "engine.h"
#include "gameutil.h"
#include "globals.h"
#include "debug4g.h"
#include "palette.h"
#include "misc.h"


/***********************************************************************
 * AreSectorsNeighbors()
 *
 * Determines if two sectors adjoin
 **********************************************************************/
BOOL AreSectorsNeighbors(int sect1, int sect2)
{
	int i, nWall, num1, num2;

	dassert(sect1 >= 0 && sect1 < kMaxSectors);
	dassert(sect2 >= 0 && sect2 < kMaxSectors);

	num1 = sector[sect1].wallnum;
	num2 = sector[sect2].wallnum;

	//Traverse walls of sector with fewest walls for speed
	if (num1 < num2)
	{
		nWall = sector[sect1].wallptr;
		for( i = 0; i < num1; i++, nWall++ )
			if ( wall[nWall].nextsector == sect2 )
				return TRUE;
	}
	else
	{
		nWall = sector[sect2].wallptr;
		for( i = 0; i < num2; i++, nWall++ )
			if ( wall[nWall].nextsector == sect1 )
				return TRUE;
	}
	return FALSE;
}


/*******************************************************************************
	FUNCTION:		FindSector()

	DESCRIPTION:	This function works like Build's updatesector() function
					except it takes into account z position, which makes it
					give correct values in areas where sectors overlap.

	PARAMETERS:		You should supplies a starting search sector in the nSector
	variable.

	RETURNS:		TRUE if point found and updates nSector, FALSE otherwise.
*******************************************************************************/
BOOL FindSector(int x, int y, int z, short *nSector)
{
	WALL *pWall;
	long ceilz, floorz;

	if ( inside(x, y, *nSector) )
	{
		getzsofslope(*nSector, x, y, &ceilz, &floorz);
		if (z >= ceilz && z <= floorz)
			return TRUE;
	}

	pWall = &wall[sector[*nSector].wallptr];
	for (int i = sector[*nSector].wallnum; i > 0; i--, pWall++)
	{
		short j = pWall->nextsector;
		if ( j >= 0 && inside(x, y, j) )
		{
			getzsofslope(j, x, y, &ceilz, &floorz);
			if (z >= ceilz && z <= floorz)
			{
				*nSector = j;
				return TRUE;
			}
		}
	}

	for (i = 0; i < numsectors; i++)
	{
		if ( inside(x, y, (short)i) )
		{
			getzsofslope((short)i, x, y, &ceilz, &floorz);
			if (z >= ceilz && z <= floorz)
			{
				*nSector = (short)i;
				return TRUE;
			}
		}
	}

	return FALSE;
}




#define FRAMES 64
void CalcFrameRate( void )
{
	static long ticks[FRAMES], index = 0;

	if (gFrameClock != ticks[index])
	{
		gFrameRate = FRAMES * kTimerRate / (gFrameClock - ticks[index]);
		ticks[index] = gFrameClock;
	}
	index = (index + 1) & (FRAMES - 1);
}


/*******************************************************************************
	FUNCTION:		CheckProximity()

	DESCRIPTION:	Test to see if a point is within a given range to a sprite

	RETURNS:        TRUE if the point is in range
*******************************************************************************/
BOOL CheckProximity( SPRITE *pSprite, int x, int y, int z, int nSector, int dist )
{
	dassert(pSprite != NULL);

	int dx = qabs(x - pSprite->x) >> 4;
	if (dx < dist)
	{
		int dy = qabs(y - pSprite->y) >> 4;
		if (dy < dist)
		{
			int dz = qabs(z - pSprite->z) >> 8;
			if ( dz < dist && qdist(dx, dy) < dist )
			{
				if ( cansee(pSprite->x, pSprite->y,  pSprite->z, pSprite->sectnum,
					x, y, z, (short)nSector) )
					return TRUE;
			}
		}
	}
	return FALSE;
}


/*******************************************************************************
	FUNCTION:		GetWallAngle()

	DESCRIPTION:	Get the angle for a wall vector

	RETURNS:		Angle in the range 0 - kMax360

	NOTES:			Add kAngle90 to get the wall Normal
*******************************************************************************/
int GetWallAngle( int nWall )
{
	int dx = wall[wall[nWall].point2].x - wall[nWall].x;
	int dy = wall[wall[nWall].point2].y - wall[nWall].y;
	return getangle( dx, dy );
}


void GetWallNormal( int nWall, int *nx, int *ny )
{
	int dx, dy;
	dx = -(wall[wall[nWall].point2].y - wall[nWall].y) >> 4;
	dy = (wall[wall[nWall].point2].x - wall[nWall].x) >> 4;
	int length = ksqrt(dx * dx + dy * dy);
	dassert(length != 0);
	*nx = divscale16(dx, length);
	*ny = divscale16(dy, length);
}


/*******************************************************************************
	FUNCTION:		HitScan()

	DESCRIPTION:	Returns the object hit, prioritized in sprite/wall/object
					order.

	PARAMETERS:

	RETURNS:		SS_SPRITE, SS_WALL, SS_FLOOR, SS_CEILING, or -1

	NOTES:			To simplify return object handling: SS_SPRITE is returned
					for flat and floor sprites. SS_WALL is returned for masked
					and one-way walls. In addition to the standard return
					value, HitScan fills the specified hitInfo structure with
					relevant hit data.
*******************************************************************************/
int HitScan( SPRITE *pSprite, int startZ, int dx, int dy, int dz, HITINFO *hitInfo )
{
	dassert( pSprite != NULL );
	dassert( hitInfo != NULL );

	hitInfo->hitsect = -1;
	hitInfo->hitwall = -1;
	hitInfo->hitsprite = -1;

	// offset the starting point for the vector so we don't hit the source!
	int x = pSprite->x; // + mulscale30(16, Cos(pSprite->ang));
	int y = pSprite->y; // + mulscale30(16, Sin(pSprite->ang));

	short oldcstat = pSprite->cstat;
	pSprite->cstat &= ~kSpriteHitscan;

	hitscan(x, y, startZ, pSprite->sectnum, dx, dy, dz << 4,
		&hitInfo->hitsect, &hitInfo->hitwall, &hitInfo->hitsprite,
		&hitInfo->hitx, &hitInfo->hity, &hitInfo->hitz);

	pSprite->cstat = oldcstat;

	if (hitInfo->hitsprite >= kMaxSprites || hitInfo->hitwall >= kMaxWalls || hitInfo->hitsect >= kMaxSectors)
	{
		dprintf("hitscan failed!\n");
		dprintf("hitscan(%i, %i, %i, %i, %i, %i, %i ...)\n", x, y, startZ, pSprite->sectnum, dx, dy, dz);
		dprintf("  hitsect   = %i\n", hitInfo->hitsect);
		dprintf("  hitwall   = %i\n", hitInfo->hitwall);
		dprintf("  hitsprite = %i\n", hitInfo->hitsprite);
		dprintf("  hitx      = %i\n", hitInfo->hitx);
		dprintf("  hity      = %i\n", hitInfo->hity);
		dprintf("  hitz      = %i\n", hitInfo->hitz);
		return -1;
	}

	if (hitInfo->hitsprite >= 0)
		return SS_SPRITE;
	if (hitInfo->hitwall >= 0)
	{
		short nNextSector = wall[hitInfo->hitwall].nextsector;

		if ( nNextSector == -1 )	// single-sided wall
			return SS_WALL;
		else						// double-sided wall, check if we hit a masked wall
		{
			long ceilz, floorz;

			if ( sector[nNextSector].ceilingstat & kSectorSloped || sector[nNextSector].floorstat & kSectorSloped )
				getzsofslope(nNextSector, hitInfo->hitx, hitInfo->hity, &ceilz, &floorz);
			else
			{
				ceilz = sector[nNextSector].ceilingz;
				floorz = sector[nNextSector].floorz;
			}

			if ( hitInfo->hitz > ceilz && hitInfo->hitz < floorz )
				return SS_MASKED;
			else
				return SS_WALL;
		}

	}
	if (hitInfo->hitsect >= 0)
		return (hitInfo->hitz > startZ) ? SS_FLOOR : SS_CEILING;

	return -1;
}



/*******************************************************************************
	FUNCTION:		GetZRange()

	DESCRIPTION:	Cover function for Ken's getzrange

	PARAMETERS:

	RETURNS:

	NOTES:
*******************************************************************************/
void GetZRange( SPRITE *pSprite, long *ceilZ, long *ceilHit, long *floorZ, long *floorHit, char cliptype )
{
	dassert(pSprite != NULL);

	short oldcstat = pSprite->cstat;
	pSprite->cstat &= ~kSpriteBlocking & ~kSpriteHitscan;
	getzrange(pSprite->x, pSprite->y, pSprite->z, pSprite->sectnum,
		ceilZ, ceilHit, floorZ, floorHit, pSprite->clipdist << 2, cliptype);
	pSprite->cstat = oldcstat;
}


unsigned ClipMove( long *px, long *py, long *pz, short *pnSector, long dx, long dy,
	int wallDist, int ceilDist, int floorDist, char clipType )
{
//	return clipmove(px, py, pz, pnSector, dx << 14, dy << 14,
//		wallDist, ceilDist, floorDist, clipType);
	long x = *px;
	long y = *py;
	long z = *pz;
	short nSector = *pnSector;

	unsigned ccode = clipmove(px, py, pz, pnSector, dx << 14, dy << 14,
		wallDist, ceilDist, floorDist, clipType);

	// force temporary fix to ken's inside() bug
	if ( *pnSector == -1)
	{
		// return to last known good location
		*px = x;
		*py = y;
		*pz = z;
		*pnSector = nSector;
	}
	return ccode;
}


#if 0
//if mode == 0 then AB is a line segment
//if mode == 1 then AB is a ray
int lintersect( int xa, int ya, int za, int xb, int yb, int zb, int xc, int yc,
	int xd, int yd, int *x, int *y, int *z)
{
	// bounding box test
	if ( xa < xc && xa < xd && xb < xc && xb < xd ) return FALSE;
	if ( xa > xc && xa > xd && xb > xc && xb > xd ) return FALSE;
	if ( ya < yc && ya < yd && yb < yc && yb < yd ) return FALSE;
	if ( ya > yc && ya > yd && yb > yc && yb > yd ) return FALSE;

	int den = (xb - xa) * (yd - yc) - (yb - ya) * (xd - xc);
	if ( den == 0 )
		return 0;	// lines are parallel

	int snum = (ya - yc) * (xb - xa) - (xa - xc) * (yb - ya);
	if ( (den < 0 && snum > 0) || (den > 0 && snum < 0))
		return 0;	// before CD

	if ( qabs(snum) >= qabs(den) )
		return 0;	// after CD

	int rnum = (ya - yc) * (xd - xc) - (xa - xc) * (yd - yc);
	if ( (den < 0 && rnum > 0) || (den > 0 && rnum < 0))
		return 0;	// before AB

	if ( qabs(rnum) >= qabs(den) )
		return 0;	// after AB

	int r = divscale16(rnum, den);
	*x = xa + mulscale16(xb - xa, r);
	*y = ya + mulscale16(yb - ya, r);
	*z = za + mulscale16(zb - za, r);
	return 1;
}
#endif

#if 0
int rintersect( int x, int y, int z, int vx, int vy, int vz, int xc,
	int yc, int xd, int yd, int *xint, int *yint, int *zint)
{
	int lx = xd - xc;
	int ly = yd - yc;
	int qx = x - xc;
	int qy = y - yc;

	int den = vx * ly - vy * lx;
	if ( den <= 0 )
		return 0;	// lines are parallel or wrong side of line

	int snum = qy * vx - qx * vy;
	if ( snum < 0 || snum >= den )
		return 0;	// not within CD

	int rnum = qy * lx - qx * ly;
	if ( rnum < 0 )
		return 0;	// before AB

	int r = divscale16(rnum, den);
	*xint = x + mulscale16(vx, r);
	*yint = y + mulscale16(vy, r);
	*zint = z + mulscale16(vz, r);
	return 1;
}
#endif


