#include <stdio.h>

#include "engine.h"
#include "ai.h"
#include "db.h"
#include "trig.h"
#include "misc.h"
#include "actor.h"
#include "player.h"
#include "globals.h"
#include "gameutil.h"
#include "multi.h"
#include "dude.h"
#include "debug4g.h"
#include "seq.h"

/****************************************************************************
** LOCAL CONSTANTS
****************************************************************************/

#define kAIThinkRate		4	// how often high level AI is sampled (in frames)
#define kAIThinkMask		(kAIThinkRate - 1)
#define kAIThinkTime		(kAIThinkRate * kFrameTicks)

#define kAxeZombMeleeDist	M2X(2.0)	//M2X(1.6)
#define kFatZombMeleeDist	M2X(2.0)	//M2X(1.6)
#define kGargoyleMeleeDist	M2X(2.0)	//M2X(1.6)
#define kHoundMeleeDist		M2X(1.6)
#define kCultistTFireDist	M2X(100)
#define kCultistSFireDist	M2X(10)
#define kCultistThrowDist1	M2X(16)
#define kCultistThrowDist2	M2X(10)

#define kGargoyleThrowDist1	M2X(14)	// used for bone
#define kGargoyleThrowDist2	M2X(10)

#define kGargoyleBlastDist1	M2X(20)	// used for paralyzing blast
#define kGargoyleBlastDist2	M2X(14)

#define kHoundBlastDist1		M2X(8)	// used for fire blast
#define kHoundBlastDist2		M2X(4)

#define kTentacleActivateDist	M2X(5)	// activates and stays on until target reaches deactivate distance?
#define kTentacleDeactivateDist	M2X(9)
#define kTentacleMeleeDist		M2X(2)

#define kPodActivateDist		M2X(8)
#define kPodDeactivateDist		M2X(14)
#define kPodFireDist1			M2X(12)
#define kPodFireDist2			M2X(8)

#define kGillBeastMeleeDist		M2X(1.6)
#define kGillBeastSummonDist1	M2X(16)
#define kGillBeastSummonDist2	M2X(12)

#define kEelMeleeDist		M2X(1)
#define kRatMeleeDist		M2X(1)
#define kHandMeleeDist		M2X(1)

#define kCerberusMeleeDist	M2X(2)
#define kCerberusBlastDist1	M2X(14)	// used for fireball
#define kCerberusBlastDist2	M2X(10)

#define kPhantasmMeleeDist	M2X(1.6)
#define kPhantasmThrowDist1	M2X(16)
#define kPhantasmThrowDist2	M2X(12)


typedef void (*THINK)( SPRITE *sprite, XSPRITE *xsprite);
typedef void (*MOVE)( SPRITE *sprite, XSPRITE *xsprite);


struct STATE
{
	int				seqId;
	SEQCALLBACK		seqCallback;
	int				ticks;
	MOVE			move;
	THINK			think;
	STATE			*next;
};


static void CultistTFireCallback( int /* type */, int nXIndex );
static void CultistSFireCallback( int /* type */, int nXIndex );
static void CultistThrowCallback( int /* type */, int nXIndex );
static void AxeZombAttackCallback( int /* type */, int nXIndex );
static void FatZombAttackCallback( int /* type */, int nXIndex );
static void GargoyleAttackCallback( int /* type */, int nXIndex );
static void HoundAttackCallback( int /* type */, int nXIndex );

static void DoMoveState( SPRITE *pSprite, XSPRITE *pXSprite );
static void DoDodgeState( SPRITE *pSprite, XSPRITE *pXSprite );
static void FindTarget( SPRITE *pSprite, XSPRITE *pXSprite );

static void CultistSearchThink( SPRITE *pSprite, XSPRITE *pXSprite );
static void CultistGotoThink( SPRITE *pSprite, XSPRITE *pXSprite );
static void CultistChaseThink( SPRITE *pSprite, XSPRITE *pXSprite );

static void FatZombieSearchThink( SPRITE *pSprite, XSPRITE *pXSprite );
static void FatZombieChaseThink( SPRITE *pSprite, XSPRITE *pXSprite );

static void AxeZombieSearchThink( SPRITE *pSprite, XSPRITE *pXSprite );
static void AxeZombieChaseThink( SPRITE *pSprite, XSPRITE *pXSprite );

static STATE genIdle = { kSeqDudeIdle, NULL, 0, NULL, NULL, NULL };

static STATE cultistIdle	= { kSeqDudeIdle, NULL, 0, NULL, FindTarget, NULL };
static STATE cultistChase	= { kSeqCultistWalk, NULL, 0, DoMoveState, CultistChaseThink, NULL };
static STATE cultistDodge	= { kSeqCultistWalk, NULL, 90, DoDodgeState, NULL, &cultistChase };
static STATE cultistThrow	= { kSeqCultistAttack2, CultistThrowCallback, 120, NULL, NULL, &cultistDodge };
static STATE cultistGoto	= { kSeqCultistWalk, NULL, 3600, DoMoveState, CultistGotoThink, &cultistIdle };
static STATE cultistSearch	= { kSeqCultistWalk, NULL, 3600, DoMoveState, CultistSearchThink, &cultistIdle };
static STATE cultistSFire	= { kSeqCultistAttack1, CultistSFireCallback, 60, NULL, NULL, &cultistChase };
static STATE cultistTFire	= { kSeqCultistAttack1, CultistTFireCallback, 0, NULL, CultistChaseThink, &cultistTFire };

static STATE fatZombieIdle		= { kSeqDudeIdle, NULL, 0, NULL, FindTarget, NULL };
static STATE fatZombieChase		= { kSeqFatZombWalk, NULL, 0, DoMoveState, FatZombieChaseThink, NULL };
static STATE fatZombieHack		= { kSeqFatZombAttack, FatZombAttackCallback, 120, NULL, NULL, &fatZombieChase };
static STATE fatZombieSearch	= { kSeqFatZombWalk, NULL, 3600, DoMoveState, FatZombieSearchThink, &fatZombieIdle };

static STATE axeZombieIdle		= { kSeqDudeIdle, NULL, 0, NULL, FindTarget, NULL };
static STATE axeZombieChase		= { kSeqAxeZombWalk, NULL, 0, DoMoveState, AxeZombieChaseThink, NULL };
static STATE axeZombieHack		= { kSeqAxeZombAttack, AxeZombAttackCallback, 120, NULL, NULL, &axeZombieChase };
static STATE axeZombieSearch	= { kSeqAxeZombWalk, NULL, 3600, DoMoveState, AxeZombieSearchThink, &axeZombieIdle };

STATE axeZombAttack	= { kSeqAxeZombAttack, AxeZombAttackCallback, 0, NULL, NULL, NULL };
STATE fatZombAttack	= { kSeqFatZombAttack, FatZombAttackCallback, 0, NULL, NULL, NULL };
STATE gargoyleAttack	= { kSeqGargoyleAttack, GargoyleAttackCallback, 0, NULL, NULL, NULL };
STATE houndAttack	= { kSeqHoundAttack1, HoundAttackCallback, 0, NULL, NULL, NULL };

/****************************************************************************
** GLOBALS
****************************************************************************/


/****************************************************************************
** LOCALS
****************************************************************************/


static void CultistTFireCallback( int /* type */, int nXIndex )
{
	XSPRITE *pXSprite = &xsprite[nXIndex];
	int nSprite = pXSprite->reference;
	SPRITE *pSprite = &sprite[nSprite];

	int dx = Cos(pSprite->ang) >> 16;
	int dy = Sin(pSprite->ang) >> 16;
	int dz = 0;

	// dispersal modifiers here
	dx += BiRandom(2000);
	dy += BiRandom(2000);
	dz += BiRandom(2000 << 4);

	actFireVector(nSprite, pSprite->z, dx, dy, dz, 0,
		kDamageBullet, 2 );
}


static void CultistSFireCallback( int /* type */, int nXIndex )
{
	XSPRITE *pXSprite = &xsprite[nXIndex];
	int nSprite = pXSprite->reference;
	SPRITE *pSprite = &sprite[nSprite];

	int dx = Cos(pSprite->ang) >> 16;
	int dy = Sin(pSprite->ang) >> 16;
	int dz = 0;

	// aim modifiers
	dx += BiRandom(4000);
	dy += BiRandom(4000);
	dz += BiRandom(4000 << 4);

	for ( int i = 0; i < 8; i++ )
	{
		actFireVector(nSprite, pSprite->z, dx + BiRandom(2000), dy + BiRandom(2000),
			dz + BiRandom(2000 << 4), 0, kDamageBullet, 4 );
	}
}


static void CultistThrowCallback( int /* type */, int nXIndex )
{
	XSPRITE *pXSprite = &xsprite[nXIndex];
	int nSprite = pXSprite->reference;
	SPRITE *pSprite = &sprite[nSprite];

	actFireThing(nSprite, pSprite->z, 0, kThingTNTStick);
}



static void AxeZombAttackCallback( int /* type */, int nXIndex )
{
	XSPRITE *pXSprite = &xsprite[nXIndex];
	int nSprite = pXSprite->reference;
	SPRITE *pSprite = &sprite[nSprite];

	int dx = Cos(pSprite->ang) >> 16;
	int dy = Sin(pSprite->ang) >> 16;
	int dz = 0;

	actFireVector(nSprite, pSprite->z, dx, dy, dz, kAxeZombMeleeDist,
		kDamageStab, 20 );
}


static void FatZombAttackCallback( int /* type */, int nXIndex )
{
	XSPRITE *pXSprite = &xsprite[nXIndex];
	int nSprite = pXSprite->reference;
	SPRITE *pSprite = &sprite[nSprite];

	int dx = Cos(pSprite->ang) >> 16;
	int dy = Sin(pSprite->ang) >> 16;
	int dz = 0;

	actFireVector(nSprite, pSprite->z, dx, dy, dz, kFatZombMeleeDist,
		kDamageStab, 10 );
}


static void GargoyleAttackCallback( int /* type */, int nXIndex )
{
	XSPRITE *pXSprite = &xsprite[nXIndex];
	int nSprite = pXSprite->reference;
	SPRITE *pSprite = &sprite[nSprite];

	int dx = Cos(pSprite->ang) >> 16;
	int dy = Sin(pSprite->ang) >> 16;
	int dz = 0;

	actFireVector(nSprite, pSprite->z, dx, dy, dz, kGargoyleMeleeDist,
		kDamageStab, 10 );
}


static void HoundAttackCallback( int /* type */, int nXIndex )
{
	XSPRITE *pXSprite = &xsprite[nXIndex];
	int nSprite = pXSprite->reference;
	SPRITE *pSprite = &sprite[nSprite];

	int dx = Cos(pSprite->ang) >> 16;
	int dy = Sin(pSprite->ang) >> 16;
	int dz = 0;

	actFireVector(nSprite, pSprite->z, dx, dy, dz, kHoundMeleeDist,
		kDamageStab, 10 );
}


static void NewState( SPRITE *pSprite, XSPRITE *pXSprite, STATE *pState )
{
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];
	pXSprite->stateTimer = pState->ticks;
	pXSprite->aiState = pState;
	if ( pDudeInfo->pSeq[pState->seqId] == NULL )
	{
		dprintf("NULL sequence, dudeType = %d, seqId = %d\n", pSprite->type, pState->seqId);
		return;
	}
	seqSpawn(pDudeInfo->pSeq[pState->seqId], SS_SPRITE, pSprite->extra, pState->seqCallback);
}


BOOL CanMove( SPRITE *pSprite, int nTarget, int ang, int dist )
{
	long x, y, z;
	short nSector;
	int zTop, zBot;
	int r;

	GetSpriteExtents(pSprite, &zTop, &zBot);

	x = pSprite->x;
	y = pSprite->y;
	z = pSprite->z;
	nSector = pSprite->sectnum;

	int dx = mulscale30(dist, Cos(ang));
	int dy = mulscale30(dist, Sin(ang));

	HITINFO hitInfo;
	r = HitScan(pSprite, pSprite->z, dx, dy, 0, &hitInfo);
	int hitDist = qdist(pSprite->x - hitInfo.hitx, pSprite->y - hitInfo.hity);
	if ( hitDist > dist )
		r = 0;

//	r = ClipMove(&x, &y, &z, &nSector, dx, dy, pSprite->clipdist << 2,
//		z - zTop, (zBot - z) / 2, 0);

	if ( r != 0 )
	{
		// okay to be blocked by target
		if ( (r & 0xC000) == 0xC000 )
		{
			if ( (r & 0x3FFF) == nTarget )
			{
//				dprintf("Dude blocked by sprite %d\n", r & 0x3FFF);
				return TRUE;
			}
		}
		return FALSE;
	}

	updatesector(pSprite->x + dx, pSprite->y + dy, &nSector);

//	long ceilZ, ceilHit, floorZ, floorHit;
//	getzrange( x, y, z, nSector, &ceilZ, &ceilHit, &floorZ, &floorHit,
//		pSprite->clipdist << 2, 0);
	long floorZ = sector[nSector].floorz;

	// this should go into the dude table and be time relative
	if ( floorZ - zBot - z > M2Z(1.5) )
		return FALSE;

	return TRUE;
}


static void ChooseDirection( SPRITE *pSprite, XSPRITE *pXSprite, int ang )
{
	dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];

	int dang = ((kAngle180 + ang - pSprite->ang) & kAngleMask) - kAngle180;

	long sin = Sin(pSprite->ang);
	long cos = Cos(pSprite->ang);

	// find vel and svel relative to current angle
	long vel = dmulscale30(pSprite->xvel, cos, pSprite->yvel, sin);
	long svel = dmulscale30(pSprite->xvel, sin, -pSprite->yvel, cos);

	int avoidDist = vel * kAIThinkTime;
	int turnTo = kAngle60;
	if (dang < 0 )
		turnTo = -turnTo;

	// clear movement toward target?
	if ( CanMove(pSprite, pXSprite->target, pSprite->ang + dang, avoidDist) )
		pXSprite->goalAng = (pSprite->ang + dang) & kAngleMask;
	// clear movement partially toward target?
	else if ( CanMove(pSprite, pXSprite->target, pSprite->ang + dang / 2, avoidDist) )
		pXSprite->goalAng = (pSprite->ang + dang / 2) & kAngleMask;
	// try turning in target direction
	else if ( CanMove(pSprite, pXSprite->target, pSprite->ang + turnTo, avoidDist) )
		pXSprite->goalAng = (pSprite->ang + turnTo) & kAngleMask;
	// clear movement straight?
	else if ( CanMove(pSprite, pXSprite->target, pSprite->ang, avoidDist) )
		pXSprite->goalAng = pSprite->ang;
	// try turning away
	else if ( CanMove(pSprite, pXSprite->target, pSprite->ang - turnTo, avoidDist) )
		pXSprite->goalAng = (pSprite->ang - turnTo) & kAngleMask;
	else
	// just turn around
		pXSprite->goalAng = (pSprite->ang + kAngle180) & kAngleMask;

	// choose dodge direction
	pXSprite->dodgeDir = (Random(2) == 0) ? 1 : -1;
	if ( !CanMove(pSprite, pXSprite->target, pSprite->ang + kAngle90 * pXSprite->dodgeDir,
		pDudeInfo->sideSpeed * 90) )
	{
		pXSprite->dodgeDir = -pXSprite->dodgeDir;
		if ( !CanMove(pSprite, pXSprite->target, pSprite->ang + kAngle90 * pXSprite->dodgeDir,
			pDudeInfo->sideSpeed * 90) )
			pXSprite->dodgeDir = 0;
	}

//	pSprite->zvel = (short)(dz >> 8);
}


static void DoMoveState( SPRITE *pSprite, XSPRITE *pXSprite )
{
	dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];

	pXSprite->moveState = kMoveWalk;

	int dang = ((kAngle180 + pXSprite->goalAng - pSprite->ang) & kAngleMask) - kAngle180;
	int maxTurn = pDudeInfo->angSpeed * kFrameTicks >> 4;

	pSprite->ang = (short)((pSprite->ang + ClipRange(dang, -maxTurn, maxTurn)) & kAngleMask);
	int accel = pDudeInfo->frontAccel * kFrameTicks;

	// don't move forward if trying to turn around
	if ( qabs(dang) > kAngle60 )
		return;

	long sin = Sin(pSprite->ang);
	long cos = Cos(pSprite->ang);

	// find vel and svel relative to current angle
	long vel = dmulscale30(pSprite->xvel, cos, pSprite->yvel, sin);
	long svel = dmulscale30(pSprite->xvel, sin, -pSprite->yvel, cos);

	// acceleration
	if ( accel > 0 )
	{
		if ( vel < pDudeInfo->frontSpeed )
			vel = ClipHigh(vel + accel, pDudeInfo->frontSpeed);
	}
	else
	{
		if ( vel > -pDudeInfo->backSpeed )
			vel = ClipLow(vel + accel, pDudeInfo->backSpeed);
	}

	// reconstruct x and y velocities
	pSprite->xvel = (short)dmulscale30(vel, cos, svel, sin);
	pSprite->yvel = (short)dmulscale30(vel, sin, -svel, cos);
}


static void DoDodgeState( SPRITE *pSprite, XSPRITE *pXSprite )
{
	dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];

	pXSprite->moveState = kMoveWalk;

	int dang = ((kAngle180 + pXSprite->goalAng - pSprite->ang) & kAngleMask) - kAngle180;
	int maxTurn = pDudeInfo->angSpeed * kFrameTicks >> 4;
	pSprite->ang = (short)((pSprite->ang + ClipRange(dang, -maxTurn, maxTurn)) & kAngleMask);

	if ( pXSprite->dodgeDir == 0 )
		return;

	int accel = pDudeInfo->sideAccel * kFrameTicks;
	long sin = Sin(pSprite->ang);
	long cos = Cos(pSprite->ang);

	// find vel and svel relative to current angle
	long vel = dmulscale30(pSprite->xvel, cos, pSprite->yvel, sin);
	long svel = dmulscale30(pSprite->xvel, sin, -pSprite->yvel, cos);

	if ( pXSprite->dodgeDir > 0 )
	{
		if ( svel < pDudeInfo->sideSpeed )
			svel = ClipHigh(svel + accel, pDudeInfo->sideSpeed);
	}
	else
	{
		if ( svel > -pDudeInfo->sideSpeed )
			svel = ClipLow(svel - accel, -pDudeInfo->sideSpeed);
	}

	// reconstruct x and y velocities
	pSprite->xvel = (short)dmulscale30(vel, cos, svel, sin);
	pSprite->yvel = (short)dmulscale30(vel, sin, -svel, cos);
}


void ActivateDude( SPRITE *pSprite, XSPRITE *pXSprite, STATE *state )
{
	dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];

	pXSprite->state = 1;

	ChooseDirection(pSprite, pXSprite, pSprite->ang);

	switch ( pSprite->type )
	{
		case kDudeBrownSpider:
		case kDudeRedSpider:
		case kDudeBlackSpider:
			pSprite->cstat &= ~kSpriteFlipY;
			break;

		case kDudeTommyCultist:
		case kDudeShotgunCultist:
			NewState(pSprite, pXSprite, state);
			break;

		case kDudeAxeZombie:
		case kDudeEarthZombie:
			NewState(pSprite, pXSprite, state);
			break;

		case kDudeFatZombie:
			NewState(pSprite, pXSprite, state);
			break;

		case kDudeFleshStatue:
			seqSpawn(pDudeInfo->pSeq[kSeqFleshStatueTurn], SS_SPRITE, pSprite->extra);
			pSprite->type = kDudeFleshGargoyle;
//			pXSprite->aiState = kAIMorph;
			break;

		case kDudeStoneStatue:
			seqSpawn(pDudeInfo->pSeq[kSeqStoneStatueTurn], SS_SPRITE, pSprite->extra);
			pSprite->type = kDudeStoneGargoyle;
//			pXSprite->aiState = kAIMorph;
			break;

	}
}


void aiSetTarget( SPRITE *pSprite, XSPRITE *pXSprite, int nTarget )
{
	SPRITE *pTarget = &sprite[nTarget];

	if (pTarget->type < kDudeBase || pTarget->type >= kDudeMax)
		return;

	if ( pXSprite->target == nTarget )
		return;

	dassert(pTarget->type >= kDudeBase && pTarget->type < kDudeMax);
	DUDEINFO *pTargetInfo = &dudeInfo[pTarget->type - kDudeBase];
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];

	switch ( pSprite->type )
	{
		case kDudeTommyCultist:
		case kDudeShotgunCultist:
			if (pTarget->type == kDudeTommyCultist || pTarget->type == kDudeShotgunCultist)
			{
				ActivateDude(pSprite, pXSprite, &cultistDodge);
				return;
			}
			break;
	}

	pXSprite->target = nTarget;
	pXSprite->targetX = pTarget->x;
	pXSprite->targetY = pTarget->y;
	pXSprite->targetZ = pTarget->z - (pTargetInfo->eyeHeight * pTarget->yrepeat << 2);

	switch ( pSprite->type )
	{
		case kDudeTommyCultist:
		case kDudeShotgunCultist:
			ActivateDude(pSprite, pXSprite, &cultistDodge);
			break;

		case kDudeFatZombie:
			ActivateDude(pSprite, pXSprite, &fatZombieChase);
			break;

		case kDudeAxeZombie:
			ActivateDude(pSprite, pXSprite, &axeZombieChase);
			break;

		default:
			break;
	}
}


static void FindTarget( SPRITE *pSprite, XSPRITE *pXSprite )
{
	dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];

	for (int i = 0; i < numplayers; i++)
	{
		PLAYER *pPlayer = &gPlayer[i];
		if ( powerupCheck( pPlayer, kItemLtdInvisibility - kItemBase ) > 0 )
			continue;

		int x = pPlayer->sprite->x;
		int y = pPlayer->sprite->y;
		int z = pPlayer->sprite->z;
		short nSector = pPlayer->sprite->sectnum;

		int dx = x - pSprite->x;
		int dy = y - pSprite->y;

		int dist = qdist(dx, dy);

		if ( dist <= pDudeInfo->seeDist || dist <= pDudeInfo->hearDist )
		{
			int nAngle = getangle(dx, dy);
			int losAngle = ((kAngle180 + nAngle - pSprite->ang) & kAngleMask) - kAngle180;
			int eyeAboveZ = pDudeInfo->eyeHeight * pSprite->yrepeat << 2;

			// is there a line of sight to the player?
			if ( cansee(x, y, z, nSector, pSprite->x, pSprite->y, pSprite->z - eyeAboveZ,
				pSprite->sectnum) )
			{
				// is the player visible?
				if ( dist < pDudeInfo->seeDist && qabs(losAngle) <= pDudeInfo->periphery )
				{
					aiSetTarget(pSprite, pXSprite, pPlayer->nSprite);
					return;
				}

				if ( dist < pDudeInfo->hearDist )
				{
					// we may want to make hearing a function of sensitivity, rather than distance
					pXSprite->target = -1;
					pXSprite->targetX = x;
					pXSprite->targetY = y;
					pXSprite->targetZ = z;

					// PETER: make it work with all dudes
					switch ( pSprite->type )
					{
						case kDudeTommyCultist:
						case kDudeShotgunCultist:
							ActivateDude(pSprite, pXSprite, &cultistGoto);
							break;

						case kDudeFatZombie:
							ActivateDude(pSprite, pXSprite, &fatZombieChase);
							break;

						case kDudeAxeZombie:
							ActivateDude(pSprite, pXSprite, &axeZombieChase);
							break;
					}
					return;
				}
			}
		}
	}
}


static BOOL TargetNearExplosion( SPRITE *pTarget )
{
	for (short nSprite = headspritesect[pTarget->sectnum]; nSprite >= 0; nSprite = nextspritesect[nSprite])
	{
		// check for TNT sticks or explosions in the same sector as the target
		if (sprite[nSprite].type == kThingTNTStick || sprite[nSprite].statnum == kStatExplosion)
			return TRUE; // indicate danger
	}
	return FALSE;
}


static void CultistSearchThink( SPRITE *pSprite, XSPRITE *pXSprite )
{
	ChooseDirection(pSprite, pXSprite, pSprite->ang);
	FindTarget(pSprite, pXSprite);
}


static void CultistGotoThink( SPRITE *pSprite, XSPRITE *pXSprite )
{
	int dx, dy, dist;

	dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];

	SPRITE *pTarget = &sprite[pXSprite->target];
	XSPRITE *pXTarget = &xsprite[pTarget->extra];

	dx = pXSprite->targetX - pSprite->x;
	dy = pXSprite->targetY - pSprite->y;

	ChooseDirection(pSprite, pXSprite, getangle(dx, dy));

	dist = qdist(dx, dy);

	// if reached target, change to search mode
	if ( dist < M2X(2.0) )
		NewState(pSprite, pXSprite, &cultistSearch);

	// check target
	dx = pTarget->x - pSprite->x;
	dy = pTarget->y - pSprite->y;

	if ( dist <= pDudeInfo->seeDist )
	{
		int nAngle = getangle(dx, dy);
		int losAngle = ((kAngle180 + nAngle - pSprite->ang) & kAngleMask) - kAngle180;
		int eyeAboveZ = pDudeInfo->eyeHeight * pSprite->yrepeat << 2;

		// is there a line of sight to the target?
		if ( cansee(pTarget->x, pTarget->y, pTarget->z, pTarget->sectnum,
			pSprite->x, pSprite->y, pSprite->z - eyeAboveZ, pSprite->sectnum) )
		{
			// is the target visible?
			if ( dist < pDudeInfo->seeDist && qabs(losAngle) <= pDudeInfo->periphery )
			{
				dassert(pTarget->type >= kDudeBase && pTarget->type < kDudeMax);
				DUDEINFO *pTargetInfo = &dudeInfo[pTarget->type - kDudeBase];

				pXSprite->targetX = pTarget->x;
				pXSprite->targetY = pTarget->y;
				pXSprite->targetZ = pTarget->z - (pTargetInfo->eyeHeight * pTarget->yrepeat << 2);

				// check to see if we can attack
				switch ( pSprite->type )
				{
					case kDudeTommyCultist:
						if ( dist < kCultistThrowDist1 && dist > kCultistThrowDist2 && qabs(losAngle) < kAngle15
						&& !TargetNearExplosion(pTarget) && (pTarget->flags & kAttrFall)
						&& !( IsPlayerSprite(pXSprite->target) && gPlayer[pTarget->type - kDudePlayer1].run ))
							NewState(pSprite, pXSprite, &cultistThrow);
						else if ( dist < kCultistTFireDist && qabs(losAngle) < kAngle5 )
							NewState(pSprite, pXSprite, &cultistTFire);
						break;

					case kDudeShotgunCultist:
						if ( dist < kCultistThrowDist1 && dist > kCultistThrowDist2 && qabs(losAngle) < kAngle15
						&& !TargetNearExplosion(pTarget) && (pTarget->flags & kAttrFall)
						&& !( IsPlayerSprite(pXSprite->target) && gPlayer[pTarget->type - kDudePlayer1].run ))
							NewState(pSprite, pXSprite, &cultistThrow);
						else if ( dist < kCultistSFireDist && qabs(losAngle) < kAngle5 )
							NewState(pSprite, pXSprite, &cultistSFire);
						break;
				}
			}
		}
	}
}


static void CultistChaseThink( SPRITE *pSprite, XSPRITE *pXSprite )
{
	int dx, dy, dist;

	dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];

	SPRITE *pTarget = &sprite[pXSprite->target];
	XSPRITE *pXTarget = &xsprite[pTarget->extra];

	dx = pXSprite->targetX - pSprite->x;
	dy = pXSprite->targetY - pSprite->y;

	ChooseDirection(pSprite, pXSprite, getangle(dx, dy));

	if ( pXTarget->health == 0 )
	{
		// target is dead
		NewState(pSprite, pXSprite, &cultistSearch);
		return;
	}

	if ( IsPlayerSprite( pTarget ) )
	{
		PLAYER *pPlayer = &gPlayer[ pTarget->type - kDudePlayer1 ];
		if ( powerupCheck( pPlayer, kItemLtdInvisibility - kItemBase ) > 0 )
		{
			NewState(pSprite, pXSprite, &cultistSearch);
			return;
		}
	}

	// check target
	dx = pTarget->x - pSprite->x;
	dy = pTarget->y - pSprite->y;

	dist = qdist(dx, dy);

	// if reached target, change to search mode
//	if ( pXSprite->aiState == kAIGoto && dist < M2X(2.0) )
//		pXSprite->aiState = kAISearch;

	if ( dist <= pDudeInfo->seeDist )
	{
		int nAngle = getangle(dx, dy);
		int losAngle = ((kAngle180 + nAngle - pSprite->ang) & kAngleMask) - kAngle180;
		int eyeAboveZ = pDudeInfo->eyeHeight * pSprite->yrepeat << 2;

		// is there a line of sight to the target?
		if ( cansee(pTarget->x, pTarget->y, pTarget->z, pTarget->sectnum,
			pSprite->x, pSprite->y, pSprite->z - eyeAboveZ, pSprite->sectnum) )
		{
			// is the target visible?
			if ( dist < pDudeInfo->seeDist && qabs(losAngle) <= pDudeInfo->periphery )
			{
				dassert(pTarget->type >= kDudeBase && pTarget->type < kDudeMax);
				DUDEINFO *pTargetInfo = &dudeInfo[pTarget->type - kDudeBase];

				pXSprite->targetX = pTarget->x;
				pXSprite->targetY = pTarget->y;
				pXSprite->targetZ = pTarget->z - (pTargetInfo->eyeHeight * pTarget->yrepeat << 2);

				// check to see if we can attack
				switch ( pSprite->type )
				{
					case kDudeTommyCultist:
						if ( dist < kCultistThrowDist1 && dist > kCultistThrowDist2 && qabs(losAngle) < kAngle15
						&& !TargetNearExplosion(pTarget) && (pTarget->flags & kAttrFall)
						&& !( IsPlayerSprite(pXSprite->target) && gPlayer[pTarget->type - kDudePlayer1].run ))
							NewState(pSprite, pXSprite, &cultistThrow);
						else if ( dist < kCultistTFireDist && qabs(losAngle) < kAngle15 )
							NewState(pSprite, pXSprite, &cultistTFire);
						break;

					case kDudeShotgunCultist:
						if ( dist < kCultistThrowDist1 && dist > kCultistThrowDist2 && qabs(losAngle) < kAngle15
						&& !TargetNearExplosion(pTarget) && (pTarget->flags & kAttrFall)
						&& !( IsPlayerSprite(pXSprite->target) && gPlayer[pTarget->type - kDudePlayer1].run ))
							NewState(pSprite, pXSprite, &cultistThrow);
						else if ( dist < kCultistSFireDist && qabs(losAngle) < kAngle15 )
							NewState(pSprite, pXSprite, &cultistSFire);
						break;
				}
				return;
			}
		}
	}
	NewState(pSprite, pXSprite, &cultistGoto);
}


static void FatZombieSearchThink( SPRITE *pSprite, XSPRITE *pXSprite )
{
	ChooseDirection(pSprite, pXSprite, pSprite->ang);
	FindTarget(pSprite, pXSprite);
}


static void FatZombieChaseThink( SPRITE *pSprite, XSPRITE *pXSprite )
{
	int dx, dy, dist;

	dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];

	SPRITE *pTarget = &sprite[pXSprite->target];
	XSPRITE *pXTarget = &xsprite[pTarget->extra];

	dx = pXSprite->targetX - pSprite->x;
	dy = pXSprite->targetY - pSprite->y;

	ChooseDirection(pSprite, pXSprite, getangle(dx, dy));

	if ( pXTarget->health == 0 )
	{
		// target is dead
		NewState(pSprite, pXSprite, &fatZombieSearch);
		return;
	}

	if ( IsPlayerSprite( pTarget ) )
	{
		PLAYER *pPlayer = &gPlayer[ pTarget->type - kDudePlayer1 ];
		if ( powerupCheck( pPlayer, kItemLtdInvisibility - kItemBase ) > 0 )
		{
			NewState(pSprite, pXSprite, &fatZombieSearch);
			return;
		}
	}

	// check target
	dx = pTarget->x - pSprite->x;
	dy = pTarget->y - pSprite->y;

	dist = qdist(dx, dy);

	if ( dist <= pDudeInfo->seeDist )
	{
		int nAngle = getangle(dx, dy);
		int losAngle = ((kAngle180 + nAngle - pSprite->ang) & kAngleMask) - kAngle180;
		int eyeAboveZ = pDudeInfo->eyeHeight * pSprite->yrepeat << 2;

		// is there a line of sight to the target?
		if ( cansee(pTarget->x, pTarget->y, pTarget->z, pTarget->sectnum,
			pSprite->x, pSprite->y, pSprite->z - eyeAboveZ, pSprite->sectnum) )
		{
			// is the target visible?
			if ( dist < pDudeInfo->seeDist && qabs(losAngle) <= pDudeInfo->periphery )
			{
				dassert(pTarget->type >= kDudeBase && pTarget->type < kDudeMax);
				DUDEINFO *pTargetInfo = &dudeInfo[pTarget->type - kDudeBase];

				pXSprite->targetX = pTarget->x;
				pXSprite->targetY = pTarget->y;
				pXSprite->targetZ = pTarget->z - (pTargetInfo->eyeHeight * pTarget->yrepeat << 2);

				// check to see if we can attack
				if ( dist < kFatZombMeleeDist && qabs(losAngle) < kAngle15 )
					NewState(pSprite, pXSprite, &fatZombieHack);
				return;
			}
		}
	}
	NewState(pSprite, pXSprite, &fatZombieChase);
}


static void AxeZombieSearchThink( SPRITE *pSprite, XSPRITE *pXSprite )
{
	ChooseDirection(pSprite, pXSprite, pSprite->ang);
	FindTarget(pSprite, pXSprite);
}


static void AxeZombieChaseThink( SPRITE *pSprite, XSPRITE *pXSprite )
{
	int dx, dy, dist;

	dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];

	SPRITE *pTarget = &sprite[pXSprite->target];
	XSPRITE *pXTarget = &xsprite[pTarget->extra];

	dx = pXSprite->targetX - pSprite->x;
	dy = pXSprite->targetY - pSprite->y;

	ChooseDirection(pSprite, pXSprite, getangle(dx, dy));

	if ( pXTarget->health == 0 )
	{
		// target is dead
		NewState(pSprite, pXSprite, &axeZombieSearch);
		return;
	}

	if ( IsPlayerSprite( pTarget ) )
	{
		PLAYER *pPlayer = &gPlayer[ pTarget->type - kDudePlayer1 ];
		if ( powerupCheck( pPlayer, kItemLtdInvisibility - kItemBase ) > 0 )
		{
			NewState(pSprite, pXSprite, &axeZombieSearch);
			return;
		}
	}

	// check target
	dx = pTarget->x - pSprite->x;
	dy = pTarget->y - pSprite->y;

	dist = qdist(dx, dy);

	if ( dist <= pDudeInfo->seeDist )
	{
		int nAngle = getangle(dx, dy);
		int losAngle = ((kAngle180 + nAngle - pSprite->ang) & kAngleMask) - kAngle180;
		int eyeAboveZ = pDudeInfo->eyeHeight * pSprite->yrepeat << 2;

		// is there a line of sight to the target?
		if ( cansee(pTarget->x, pTarget->y, pTarget->z, pTarget->sectnum,
			pSprite->x, pSprite->y, pSprite->z - eyeAboveZ, pSprite->sectnum) )
		{
			// is the target visible?
			if ( dist < pDudeInfo->seeDist && qabs(losAngle) <= pDudeInfo->periphery )
			{
				dassert(pTarget->type >= kDudeBase && pTarget->type < kDudeMax);
				DUDEINFO *pTargetInfo = &dudeInfo[pTarget->type - kDudeBase];

				pXSprite->targetX = pTarget->x;
				pXSprite->targetY = pTarget->y;
				pXSprite->targetZ = pTarget->z - (pTargetInfo->eyeHeight * pTarget->yrepeat << 2);

				// check to see if we can attack
				if ( dist < kAxeZombMeleeDist && qabs(losAngle) < kAngle15 )
					NewState(pSprite, pXSprite, &axeZombieHack);
				return;
			}
		}
	}
	NewState(pSprite, pXSprite, &axeZombieChase);
}


/*
void CheckTarget( SPRITE *pSprite, XSPRITE *pXSprite )
{
	dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];

	SPRITE *pTarget = &sprite[pXSprite->target];
	XSPRITE *pXTarget = &xsprite[pTarget->extra];

	if ( pXTarget->health == 0 )
	{
		// target is dead
		pXSprite->aiState = kAISearch;
		return;
	}

	int dx = pTarget->x - pSprite->x;
	int dy = pTarget->y - pSprite->y;

	int dist = qdist(dx, dy);

	if ( dist <= pDudeInfo->seeDist )
	{
		int nAngle = getangle(dx, dy);
		int losAngle = ((kAngle180 + nAngle - pSprite->ang) & kAngleMask) - kAngle180;
		int eyeAboveZ = pDudeInfo->eyeHeight * pSprite->yrepeat << 2;

		// is there a line of sight to the target?
		if ( cansee(pTarget->x, pTarget->y, pTarget->z, pTarget->sectnum,
			pSprite->x, pSprite->y, pSprite->z - eyeAboveZ, pSprite->sectnum) )
		{
			// is the target visible?
			if ( dist < pDudeInfo->seeDist && qabs(losAngle) <= pDudeInfo->periphery )
			{
				dassert(pTarget->type >= kDudeBase && pTarget->type < kDudeMax);
				DUDEINFO *pTargetInfo = &dudeInfo[pTarget->type - kDudeBase];

				pXSprite->targetX = pTarget->x;
				pXSprite->targetY = pTarget->y;
				pXSprite->targetZ = pTarget->z - (pTargetInfo->eyeHeight * pTarget->yrepeat << 2);

				if ( pXSprite->weaponTimer > 0 )
					return;

				// check to see if we can attack
				switch ( pSprite->type )
				{
					case kDudeTommyCultist:
						if ( dist < kCultistThrowDist1 && dist > kCultistThrowDist2 && qabs(losAngle) < kAngle15 )
						{
							seqSpawn(pDudeInfo->pSeq[kSeqCultistAttack2], SS_SPRITE,
								pSprite->extra, CultistThrowAttackCallback);
							pXSprite->aiState = kAIAttack;
						}
						else if ( dist < kCultistFireDist && qabs(losAngle) < kAngle15 )
						{
							seqSpawn(pDudeInfo->pSeq[kSeqCultistAttack1], SS_SPRITE,
								pSprite->extra, TommyCultistAttackCallback);
							pXSprite->aiState = kAIAttack;
						}
						break;

					case kDudeShotgunCultist:
						if ( dist < kCultistThrowDist1 && dist > kCultistThrowDist2 && qabs(losAngle) < kAngle15 )
						{
							seqSpawn(pDudeInfo->pSeq[kSeqCultistAttack2], SS_SPRITE,
								pSprite->extra, CultistThrowAttackCallback);
							pXSprite->aiState = kAIAttack;
						}
						else if ( dist < kCultistFireDist && qabs(losAngle) < kAngle15 )
						{
							seqSpawn(pDudeInfo->pSeq[kSeqCultistAttack1], SS_SPRITE,
								pSprite->extra, ShotgunCultistAttackCallback);
							pXSprite->aiState = kAIAttack;
						}
						break;

					case kDudeAxeZomb:
						if ( dist < kAxeZombMeleeDist && qabs(losAngle) < kAngle15 )
						{
							seqSpawn(pDudeInfo->pSeq[kSeqAxeZombAttack], SS_SPRITE,
								pSprite->extra, AxeZombAttackCallback);
							pXSprite->aiState = kAIAttack;
						}
						break;

					case kDudeFatZomb:
						if ( dist < kFatZombMeleeDist && qabs(losAngle) < kAngle15 )
						{
							seqSpawn(pDudeInfo->pSeq[kSeqFatZombAttack], SS_SPRITE,
								pSprite->extra, FatZombAttackCallback);
							pXSprite->aiState = kAIAttack;
						}
						break;

					case kDudeFleshGargoyle:
					case kDudeStoneGargoyle:
						if ( dist < kGargoyleMeleeDist && qabs(losAngle) < kAngle15 )
						{
							seqSpawn(pDudeInfo->pSeq[kSeqGargoyleAttack], SS_SPRITE,
								pSprite->extra, GargoyleAttackCallback);
							pXSprite->aiState = kAIAttack;
						}
						break;

					case kDudeHound:
						if ( dist < kHoundMeleeDist && qabs(losAngle) < kAngle15 )
						{
							seqSpawn(pDudeInfo->pSeq[kSeqHellAttack1], SS_SPRITE,
								pSprite->extra, HoundAttackCallback);
							pXSprite->aiState = kAIAttack;
						}
						break;
				}

				return;
			}
		}
	}

	pXSprite->aiState = kAIGoto;
}
*/


void aiProcessDudes( void )
{
	// process active sprites
	for (short nSprite = headspritestat[kStatDude]; nSprite >= 0; nSprite = nextspritestat[nSprite])
	{
		SPRITE *pSprite = &sprite[nSprite];
		int nXSprite = pSprite->extra;
		XSPRITE *pXSprite = &xsprite[nXSprite];

		// don't manipulate players!
		if ( pSprite->type >= kDudePlayer1 && pSprite->type <= kDudePlayer8 )
			continue;

		pXSprite->stateTimer = ClipLow(pXSprite->stateTimer - kFrameTicks, 0);

		if ( pXSprite->aiState->move )
			pXSprite->aiState->move(pSprite, pXSprite);

		if ( pXSprite->aiState->think && (gFrame & kAIThinkMask) == (nSprite & kAIThinkMask) )
			pXSprite->aiState->think(pSprite, pXSprite);

		if ( pXSprite->stateTimer == 0 && pXSprite->aiState->next != NULL )
		{
			if ( pXSprite->aiState->ticks > 0 )
				NewState(pSprite, pXSprite, pXSprite->aiState->next);
			else if ( seqGetStatus(SS_SPRITE, nXSprite) < 0 )
				NewState(pSprite, pXSprite, pXSprite->aiState->next);
		}

/*
		// special processing for converting dude types
		switch ( pSprite->type )
		{
			case kDudeCerberus:
				if ( pXSprite->health <= 0 && seqGetStatus(SS_SPRITE, nXSprite) < 0 )	// head #1 finished dying?
				{
					pXSprite->health = dudeInfo[kDudeCerberus2 - kDudeBase].startHealth << 4;
					pSprite->type = kDudeCerberus2;		// change his type
//						ActivateDude(pSprite, pXSprite);	// reactivate him
				}
				break;
		}
*/
	}
}


/*******************************************************************************
	FUNCTION:		aiInit()

	DESCRIPTION:

	PARAMETERS:		void

	RETURNS:		void

	NOTES:
*******************************************************************************/
void aiInit( void )
{
	for (short nSprite = headspritestat[kStatDude]; nSprite >= 0; nSprite = nextspritestat[nSprite])
	{
		SPRITE *pDude = &sprite[nSprite];
		XSPRITE *pXDude = &xsprite[pDude->extra];

		switch ( pDude->type )
		{
			case kDudeTommyCultist:
			case kDudeShotgunCultist:
				NewState(pDude, pXDude, &cultistIdle);
				break;

			case kDudeFatZombie:
				NewState(pDude, pXDude, &fatZombieIdle);
				break;

			case kDudeAxeZombie:
				NewState(pDude, pXDude, &axeZombieIdle);
				break;

			default:
				NewState(pDude, pXDude, &genIdle);
		}

		pXDude->target		= -1;
		pXDude->targetX		= 0;
		pXDude->targetY		= 0;
		pXDude->targetZ		= 0;

		pXDude->stateTimer	= 0;

		switch ( pDude->type )
		{
			case kDudeBrownSpider:
			case kDudeRedSpider:
			case kDudeBlackSpider:
				if ( pDude->cstat & kSpriteFlipY )
					pDude->flags &= ~kAttrFall;
				break;

			case kDudeBat:
			case kDudeFleshGargoyle:
			case kDudeStoneGargoyle:
				pDude->flags &= ~kAttrFall;
				break;
		}
	}
}
