#include "weapon.h"
#include "globals.h"
#include "qav.h"
#include "debug4g.h"
#include "actor.h"
#include "db.h"
#include "actor.h"
#include "gameutil.h"
#include "trig.h"
#include "view.h"
#include "options.h"
#include "names.h"
#include "triggers.h"
#include "error.h"
#include "misc.h"
#include "screen.h"
#include "tile.h"

#define kShotgunBarrels		2
#define kVectorsPerBarrel	16
#define kMaxShotgunVectors	(kShotgunBarrels * kVectorsPerBarrel)

enum {
	kVoodooStabChest	= 1,
	kVoodooStabShoulder,
	kVoodooStabEye,
	kVoodooStabGroin,
	kVoodooStabSelf,
};

// Weapon States
enum {
	kFindWeapon			= -1,	// flag to find a loaded weapon

	kForkIdle			= 0,

	kBicOpen			= 0,
	kBicIdle,
	kBicSwitch,
	kBicNext,

	kCanIdle			= kBicNext,
	kCanFire,

	kTNTStickIdle		= kBicNext,
	kTNTStickFuse1,
	kTNTStickFuse2,
	kTNTStickFuse3,
	kTNTBundleIdle,
	kTNTBundleFuse1,
	kTNTBundleFuse2,
	kTNTBundleFuse3,

	kShotInitialize		= 0,	// indicates first-time reload required
	kShotLoaded,
	kShotRaise,
	kShotIdle1,
	kShotIdle2,
 	kShotIdle3,
	kShotFire1,
	kShotFire2,
	kShotReload,
	kShotLower,

	kFlareInitialize	= 0,
	kFlareRaise,
	kFlareIdle,
	kFlareFire,
	kFlareReload,
	kFlareLower,

	kTommyIdle			= 0,

	kSpearInitialize	= 0,
	kSpearRaise,
	kSpearIdle1,
	kSpearIdle2,
	kSpearFire,
	kSpearReload,
	kSpearLower,

	kShadowInitialize	= 0,
	kShadowRaise,
	kShadowIdle,
	kShadowFire,
	kShadowLower,

	kBeastInitialize	= 0,
	kBeastRaise,
	kBeastIdle,
	kBeastAttack,
	kBeastLower,

	kDollInitialize	= 0,
	kDollRaise,
	kDollIdle,
	kDollStab1,
	kDollStab2,
	kDollStab3,
	kDollStab4,
	kDollStab5,
	kDollLower,
};

enum {
	kQAVForkUp = 0,
	kQAVForkIdle,
	kQAVForkStab,
	kQAVForkDown,

	kQAVBicUp,				// raise lighter and open
	kQAVBicFlame,			// light lighter
	kQAVBicIdle,			// burning lighter
	kQAVBicDown,			// lower and close burning lighter

	kQAVSprayUp,
	kQAVSprayIdle,
	kQAVSprayFire,
	kQAVSprayDown,

	kQAVStickUp,			// raise just stick
	kQAVStickDown,			// lower just stick
	kQAVStickUp2,			// raise lighter and stick
	kQAVStickDown2,			// lower lighter and stick
	kQAVStickIdle,
	kQAVStickFuse,
	kQAVStickThrow,

	kQAVBundleUp,			// raise just bundle
	kQAVBundleDown,			// lower just bundle
	kQAVBundleUp2,			// raise lighter and bundle
	kQAVBundleDown2,		// lower lighter and bundle
	kQAVBundleIdle,
	kQAVBundleFuse,
	kQAVBundleThrow,

	kQAVTNTExplode,

	kQAVFlareUp,
	kQAVFlareIdle,
	kQAVFlareFire,
//	kQAVFlareReload,
	kQAVFlareDown,

	kQAVShotUp,
	kQAVShotIdle1,
	kQAVShotIdle2,
	kQAVShotIdle3,
	kQAVShotFireL,
	kQAVShotFireR,
	kQAVShotReload,
	kQAVShotDown,

	kQAVTommyUp,
	kQAVTommyIdle,
	kQAVTommyFire1,
 	kQAVTommyDown,

	kQAVSpearUp,
	kQAVSpearIdle1,
	kQAVSpearIdle2,
	kQAVSpearFire,
	kQAVSpearDown,

	kQAVShadowUp,
	kQAVShadowIdle,
	kQAVShadowFire,
	kQAVShadowDown,

	kQAVBeastUp,
	kQAVBeastIdle,
	kQAVBeastAttack1,
	kQAVBeastAttack2,
 	kQAVBeastAttack3,
 	kQAVBeastAttack4,
	kQAVBeastDown,

	kQAVDollUp,
	kQAVDollIdleStill,
	kQAVDollIdleMoving,
	kQAVDollStab1,
	kQAVDollStab2,
	kQAVDollStab3,
	kQAVDollStab4,
	kQAVDollStab5,
	kQAVDollDown,

	kQAVEnd
};

static char * weaponQAVNames[kQAVEnd] =
{
	"FORKUP",  		//kQAVForkUp = 0,
	"FORKIDLE",     //kQAVForkIdle,
	"PFORK",        //kQAVForkStab,
	"FORKDOWN",     //kQAVForkDown,

	"LITEOPEN",     //kQAVBicUp,
	"LITEFLAM",     //kQAVBicFlame,
	"LITEIDLE",     //kQAVBicIdle,
	"LITECLO2",     //kQAVBicDown,

	"CANPREF",      //kQAVSprayUp,
	"CANIDLE",      //kQAVSprayIdle,
	"CANFIRE",		//kQAVSprayFire1,
	"CANDOWN",      //kQAVSprayDown,

	"DYNUP",		//kQAVStickUp,
	"DYNDOWN",		//kQAVStickDown,
	"DYNUP2",		//kQAVStickUp2,
	"DYNDOWN2",		//kQAVStickDown2,
	"DYNIDLE",		//kQAVStickIdle,
	"DYNFUSE",		//kQAVStickFuse,
	"DYNTHRO",		//kQAVStickThrow,

	"BUNUP",		//kQAVBundleUp,
	"BUNDOWN",		//kQAVBundleDown,
	"BUNUP2",		//kQAVBundleUp2,
	"BUNDOWN2",		//kQAVBundleDown2,
	"BUNIDLE",		//kQAVBundleIdle,
	"BUNFUSE",		//kQAVBundleFuse,
	"BUNTHRO",		//kQAVBundleThrow,

	"DYNEXPLO",		//kQAVTNTExplode,

	"FLARUP",
	"FLARIDLE",
	"FLARFIR2",
//	"FLARLOAD",
	"FLARDOWN",

	"SHOTUP",		//kQAVShotUp,
	"SHOTI1",       //kQAVShotIdle1,
	"SHOTI2",       //kQAVShotIdle2,
	"SHOTI3",       //kQAVShotIdle3,
	"SHOTF1",       //kQAVShotFireL,
	"SHOTF2",       //kQAVShotFireR,
	"SHOTL1",       //kQAVShotReload,
	"SHOTDOWN",     //kQAVShotDown,

	"TOMUP",        //kQAVTommyUp,
	"TOMIDLE",      //kQAVTommyIdle,
	"TOMFIRE",		//kQAVTommyFire1,
	"TOMDOWN",      //kQAVTommyDown,

	"SGUNUP",		//kQAVSpearUp,
	"SGUNIDL1",		//kQAVSpearIdle1,
	"SGUNIDL2",		//kQAVSpearIdle2,
	"SGUNFIRE",		//kQAVSpearFire,
	"SGUNDOWN",		//kQAVSpearDown,

	"DARKUP",       //kQAVShadowUp,
	"DARKIDLE",     //kQAVShadowIdle,
	"DARKFIRE",     //kQAVShadowFire,
	"DARKDOWN",     //kQAVShadowDown,

	"BSTUP",		//kQAVBeastUp
	"BSTIDLE",		//kQAVBeastIdle,
	"BSTATAK1",     //kQAVBeastAttack1,
	"BSTATAK2",     //kQAVBeastAttack2,
	"BSTATAK3",     //kQAVBeastAttack3,
	"BSTATAK4",     //kQAVBeastAttack4,
	"BSTDOWN",

	"VDUP",			//kQAVDollUp
	"VDIDLE1",		//kQAVDollIdleStill	(PLAYER IS STILL)
	"VDIDLE2",		//kQAVDollIdleMoving	(PLAYER IS WALKING/RUNNING)
	"VDFIRE1",		//kQAVDollStab1
	"VDFIRE2",		//kQAVDollStab2
	"VDFIRE3",		//kQAVDollStab3
	"VDFIRE4",		//kQAVDollStab4
	"VDFIRE5",		//kQAVDollStab5
	"VDDOWN", 		//kQAVDollDown
};


static QAV *weaponQAV[kQAVEnd];


static BOOL CheckAmmo( PLAYER *pPlayer, int nWeapon )
{
	if ( gInfiniteAmmo )
		return TRUE;

	int nAmmo = pPlayer->weaponAmmo[nWeapon];
	if ( nAmmo == kAmmoNone )
		return TRUE;

	return pPlayer->ammoCount[nAmmo] > 0;
}


static void UseAmmo( PLAYER *pPlayer, int nWeapon, int nCount )
{
	if ( gInfiniteAmmo )
		return;

	int nAmmo = pPlayer->weaponAmmo[nWeapon];
	if ( nAmmo == kAmmoNone )
		return;

	pPlayer->ammoCount[nAmmo] = ClipLow(pPlayer->ammoCount[nAmmo] - nCount, 0);
}


void WeaponInit( void )
{
	// get pointers to all the QAV resources
	for (int i = 0; i < kQAVEnd; i++)
	{
		RESHANDLE hQAV = gSysRes.Lookup(weaponQAVNames[i], ".QAV");
		if (hQAV == NULL)
		{
			dprintf("Could not load %s.QAV\n", weaponQAVNames[i]);
			ThrowError("Missing Weapon QAV file", ES_ERROR);
		}
		weaponQAV[i] = (QAV *)gSysRes.Lock(hQAV);
	}

	dprintf("Preload weapon QAV tiles\n");
	for (i = 0; i < kQAVEnd; i++)
	{
		if (weaponQAV[i] != NULL )
			weaponQAV[i]->Preload();
	}

	for (i = 0; i < kMaxPlayers; i++)
	{
		gPlayer[i].weaponAmmo[kWeaponPitchfork] = kAmmoNone;
		gPlayer[i].weaponAmmo[kWeaponSprayCan] = kAmmoSprayCan;
		gPlayer[i].weaponAmmo[kWeaponTNT] = kAmmoNone;
		gPlayer[i].weaponAmmo[kWeaponShotgun] = kAmmoShell;
		gPlayer[i].weaponAmmo[kWeaponTommy] = kAmmoBullet;
		gPlayer[i].weaponAmmo[kWeaponFlare] = kAmmoFlare;
		gPlayer[i].weaponAmmo[kWeaponVoodoo] = kAmmoVoodoo;
		gPlayer[i].weaponAmmo[kWeaponSpear] = kAmmoSpear;
		gPlayer[i].weaponAmmo[kWeaponShadow] = kAmmoNone;
	}
}


void WeaponDraw( PLAYER *pPlayer, int shade, int x, int y )
{
	dassert(pPlayer != NULL);
	QAV *pQAV = pPlayer->pWeaponQAV;
	ulong t;

	if ( pQAV == NULL )
		return;

	if (pPlayer->weaponTimer == 0)	// playing idle QAV?
		t = gGameClock % pQAV->duration;
	else
		t = pQAV->duration - pPlayer->weaponTimer;

	pQAV->origin.x = x;
	pQAV->origin.y = y;

	int nPowerRemaining = powerupCheck( pPlayer, kItemLtdInvisibility - kItemBase );
	int nFlags = kQFrameScale;

	if ( nPowerRemaining >= (kTimerRate * 8) || nPowerRemaining && ( gGameClock & 32 ) )
	{
		nFlags |= kQFrameTranslucent;
		shade = -128;
	}

	pQAV->Draw(t, shade, kQFrameScale);
}


void WeaponPlay( PLAYER *pPlayer )
{
	dassert(pPlayer != NULL);
	QAV *pQAV = pPlayer->pWeaponQAV;

	if ( pQAV == NULL )
		return;

	ulong t = pQAV->duration - pPlayer->weaponTimer;

	pQAV->Play(t - kFrameTicks, t, pPlayer->weaponCallback, pPlayer);
}


static void StartQAV( PLAYER *pPlayer, int weaponIndex, QAVCALLBACK callback = NULL, BOOL fLoop = FALSE)
{
	pPlayer->pWeaponQAV = weaponQAV[weaponIndex];
	pPlayer->weaponTimer = pPlayer->pWeaponQAV->duration;
	pPlayer->weaponCallback = callback;
	pPlayer->fLoopQAV = fLoop;
}


#define kAimMaxDist 	M2X(100)
#define kAimMaxAngle	kAngle10
#define kAimMaxSlope	8 << 10			//

#define kTrackRate		0xD000			// 0 tracks instantly, 0x10000 takes forever

static void UpdateAimVector( PLAYER *pPlayer )
{
	dassert(pPlayer != NULL);
	SPRITE *pSprite = pPlayer->sprite;

	int x = pSprite->x;
	int y = pSprite->y;
	int z = pSprite->z - pPlayer->weaponAboveZ;

	VECTOR aim;

	aim.dx = Cos(pSprite->ang) >> 16;
	aim.dy = Sin(pSprite->ang) >> 16;
	aim.dz = pPlayer->slope;

	int nAimSprite = -1;

	if ( gAutoAim )
	{
		int closest = 0x7FFFFFFF;
		for (short nTarget = headspritestat[kStatDude]; nTarget >= 0; nTarget = nextspritestat[nTarget] )
		{
			SPRITE *pTarget = &sprite[nTarget];

			// don't target yourself!
			if ( pTarget == pSprite )
				continue;

			int dx, dy, dz;

			dx = pTarget->x - x;
			dy = pTarget->y - y;
			dz = pTarget->z - z;

			int dist = qdist(dx, dy);
			if ( dist > closest || dist > kAimMaxDist )
				continue;

			int ang = getangle(dx, dy);
			if ( qabs(((ang - pSprite->ang + kAngle180) & kAngleMask) - kAngle180) > kAimMaxAngle )
				continue;

			int nSlope = divscale(dz, dist << 4, 14);
			if ( qabs(nSlope - pPlayer->slope) > kAimMaxSlope )
				continue;

			if ( cansee(x, y, z, pSprite->sectnum, pTarget->x, pTarget->y, pTarget->z, pTarget->sectnum) )
			{
				closest = dist;

				aim.dx = Cos(ang) >> 16;
				aim.dy = Sin(ang) >> 16;
				aim.dz = nSlope;
				nAimSprite = nTarget;
			}
		}
	}

	VECTOR relAim = aim;
	RotateVector(&relAim.dx, &relAim.dy, -pSprite->ang);

	pPlayer->relAim.dx = dmulscale16r(pPlayer->relAim.dx, kTrackRate, relAim.dx, 0x10000 - kTrackRate);
	pPlayer->relAim.dy = dmulscale16r(pPlayer->relAim.dy, kTrackRate, relAim.dy, 0x10000 - kTrackRate);
	pPlayer->relAim.dz = dmulscale16r(pPlayer->relAim.dz, kTrackRate, relAim.dz, 0x10000 - kTrackRate);

	pPlayer->aim = pPlayer->relAim;
	RotateVector(&pPlayer->aim.dx, &pPlayer->aim.dy, pSprite->ang);

	pPlayer->aimSprite = nAimSprite;
}


void WeaponLower( PLAYER *pPlayer, INPUT *pInput )
{
	dprintf("WeaponLower()\n");
	dassert( pPlayer != 0 );

	int nWeapon = pPlayer->weapon;
	int nState = pPlayer->weaponState[nWeapon];

	switch( nWeapon )
	{
		case kWeaponPitchfork:
			StartQAV(pPlayer, kQAVForkDown);
			break;

		case kWeaponSprayCan:
			switch( nState )
			{
				case kBicIdle:
			        StartQAV(pPlayer, kQAVBicDown);
					break;

				case kBicSwitch:
					pPlayer->weapon = pInput->newWeapon;
					pPlayer->weaponState[pPlayer->weapon] = kBicIdle;
					pInput->newWeapon = 0;
					return;

				default:
				    StartQAV(pPlayer, kQAVSprayDown);
					if ( pInput->newWeapon == kWeaponTNT )
						pPlayer->weaponState[kWeaponSprayCan] = kBicSwitch;
					else
						pPlayer->weaponState[kWeaponSprayCan] = kBicIdle;
					return;

			}
			break;

		case kWeaponTNT:
		{
			switch( nState )
			{
				case kBicIdle:
			        StartQAV(pPlayer, kQAVBicDown);
					break;

				case kBicSwitch:
					pPlayer->weapon = pInput->newWeapon;
					pPlayer->weaponState[pPlayer->weapon] = kBicIdle;
					pInput->newWeapon = 0;
					return;

				case kTNTStickIdle:
					if ( pInput->newWeapon == kWeaponSprayCan )
					{
						StartQAV(pPlayer, kQAVStickDown);
						pPlayer->weaponState[kWeaponTNT] = kBicSwitch;
						return;
					}
					StartQAV(pPlayer, kQAVStickDown2);
					break;

				case kTNTBundleIdle:
					if ( pInput->newWeapon == kWeaponSprayCan )
					{
						StartQAV(pPlayer, kQAVBundleDown);
						pPlayer->weaponState[kWeaponTNT] = kBicSwitch;
						return;
					}
					StartQAV(pPlayer, kQAVBundleDown2);
					break;
			}
			break;
		}

		case kWeaponShotgun:
	        StartQAV(pPlayer, kQAVShotDown);
			break;

		case kWeaponTommy:
	        StartQAV(pPlayer, kQAVTommyDown);
			break;

		case kWeaponFlare:
			StartQAV(pPlayer, kQAVFlareDown);
			break;

		case kWeaponVoodoo:
			StartQAV(pPlayer, kQAVDollDown);
			break;

		case kWeaponSpear:
	        StartQAV(pPlayer, kQAVSpearDown);
			break;

		case kWeaponShadow:
	        StartQAV(pPlayer, kQAVShadowDown);
			break;

		case kWeaponBeast:
	        StartQAV(pPlayer, kQAVBeastDown);
			break;
	}

	if (pInput->newWeapon == pPlayer->weapon)
		pInput->newWeapon = 0;

	pPlayer->weapon = 0;
}


void WeaponRaise( PLAYER *pPlayer, INPUT *pInput )
{
	dprintf("WeaponRaise()\n");
	dassert( pPlayer != 0 );

	pPlayer->weapon = pInput->newWeapon;
	pInput->newWeapon = 0;

	switch( pPlayer->weapon )
	{
		case kWeaponVoodoo:
			StartQAV(pPlayer, kQAVDollUp);
			pPlayer->weaponState[kWeaponVoodoo] = kDollIdle;
			break;

		case kWeaponPitchfork:
			StartQAV(pPlayer, kQAVForkUp);
			pPlayer->weaponState[kWeaponPitchfork] = kForkIdle;
			break;

		case kWeaponFlare:
			StartQAV(pPlayer, kQAVFlareUp);
			pPlayer->weaponState[kWeaponFlare] = kFlareIdle;
			break;

		case kWeaponSprayCan:
			StartQAV(pPlayer, kQAVBicUp);
			pPlayer->weaponState[kWeaponSprayCan] = kBicOpen;
			break;

		case kWeaponTNT:
			StartQAV(pPlayer, kQAVBicUp);
			pPlayer->weaponState[kWeaponTNT] = kBicOpen;
			break;

		case kWeaponShotgun:
			StartQAV(pPlayer, kQAVShotUp);
			switch (pPlayer->weaponState[kWeaponShotgun])
			{
				case kShotInitialize:
				case kShotIdle3:
//					if (pPlayer->ammoCount[ kAmmoShells - kAmmoBase ] > 0)
//					    pPlayer->weaponState[kWeaponShotgun] = kShotReload;
//					else
					    pPlayer->weaponState[kWeaponShotgun] = kShotIdle3;
					break;
				case kShotIdle2:
					if (gInfiniteAmmo || pPlayer->ammoCount[kAmmoShell] > 1)
					    pPlayer->weaponState[kWeaponShotgun] = kShotReload;
					break;
			}
			break;

		case kWeaponTommy:
			StartQAV(pPlayer, kQAVTommyUp);
			pPlayer->weaponState[kWeaponTommy] = kTommyIdle;
			break;

		case kWeaponSpear:
			StartQAV(pPlayer, kQAVSpearUp);
			if ( pPlayer->ammoCount[kAmmoSpear] > 0 || pPlayer->ammoCount[kAmmoSpearXP] > 0 )
				pPlayer->weaponState[kWeaponSpear] = kSpearIdle1;
			else
				pPlayer->weaponState[kWeaponSpear] = kSpearIdle2;
			break;

		case kWeaponShadow:
			StartQAV(pPlayer, kQAVShadowUp);
			pPlayer->weaponState[kWeaponShadow] = kShadowIdle;
			break;

		case kWeaponBeast:
			StartQAV(pPlayer, kQAVBeastUp);
			pPlayer->weaponState[kWeaponBeast] = kBeastIdle;
			break;
	}
}


#define kMaxForkDist	M2X(2.25)

static void FirePitchfork( int /* nTrigger */, PLAYER *pPlayer )
{
	SPRITE *pSprite = pPlayer->sprite;

	int z = pSprite->z - pPlayer->weaponAboveZ;

	for ( int i = 0; i < 4; i++ )
	{
		// dispersal modifiers
		int ddx = BiRandom(2000);
		int ddy = BiRandom(2000);
		int ddz = BiRandom(2000);

		actFireVector(pPlayer->nSprite, z,
			pPlayer->aim.dx + ddx, pPlayer->aim.dy + ddy, pPlayer->aim.dz + ddz,
			kMaxForkDist, kDamageStab, 3);
	}
}


static void FireSpray( int /* nTrigger */, PLAYER *pPlayer )
{
	playerFireMissile(pPlayer, pPlayer->aim.dx, pPlayer->aim.dy, pPlayer->aim.dz,
		kMissileSprayFlame);
	UseAmmo(pPlayer, kWeaponSprayCan, kFrameTicks);
}


#define kMaxThrowTime 	(2 * kTimerRate)	// how long to hold fire to get farthest throw
#define kMinThrowVel	((M2X(2.0) << 4) / kTimerRate)
#define kMaxThrowVel	((M2X(10.0) << 4) / kTimerRate)

static void FireStick( int /* nTrigger */, PLAYER *pPlayer )
{
	pPlayer->throwTime = ClipHigh(pPlayer->throwTime, kMaxThrowTime);
	int velocity = muldiv(pPlayer->throwTime, kMaxThrowVel - kMinThrowVel, kMaxThrowTime) + kMinThrowVel;
	int nThing = playerFireThing(pPlayer, kThingTNTStick, velocity);
	evPost(nThing, SS_SPRITE, pPlayer->fuseTime);
	UseAmmo(pPlayer, kWeaponTNT, 1);
}


static void ExplodeStick( int /* nTrigger */, PLAYER *pPlayer )
{
	int nThing = playerFireThing(pPlayer, kThingTNTStick, 0);
	evPost(nThing, SS_SPRITE, 0);	// trigger it immediately
	UseAmmo(pPlayer, kWeaponTNT, 1);
	StartQAV(pPlayer, kQAVTNTExplode);
	pPlayer->weapon = 0;	// drop the hands

}


static void FireBundle( int /* nTrigger */, PLAYER *pPlayer )
{
	pPlayer->throwTime = ClipHigh(pPlayer->throwTime, kMaxThrowTime);
	int velocity = muldiv(pPlayer->throwTime, kMaxThrowVel - kMinThrowVel, kMaxThrowTime) + kMinThrowVel;
	int nThing = playerFireThing(pPlayer, kThingTNTStick, velocity);
	evPost(nThing, SS_SPRITE, pPlayer->fuseTime);
	UseAmmo(pPlayer, kWeaponTNT, 7);
}


static void ExplodeBundle( int /* nTrigger */, PLAYER *pPlayer )
{
	int nThing = playerFireThing(pPlayer, kThingTNTBundle, 0);
	evPost(nThing, SS_SPRITE, 0);	// trigger it immediately
	UseAmmo(pPlayer, kWeaponTNT, 7);
	StartQAV(pPlayer, kQAVTNTExplode);
	pPlayer->weapon = 0;	// drop the hands

}


static void FireShotgun( PLAYER *pPlayer, int nBarrels )
{
	dassert(nBarrels > 0 && nBarrels <= kShotgunBarrels);

	int nVectors = nBarrels * kVectorsPerBarrel;

	SPRITE *pSprite = pPlayer->sprite;

	int z = pSprite->z - pPlayer->weaponAboveZ;

	for ( int i = 0; i < nVectors; i++ )
	{
		// dispersal modifiers
		int ddx = BiRandom(2000);
		int ddy = BiRandom(2000);
		int ddz = BiRandom(2000);

		actFireVector(pPlayer->nSprite, z,
			pPlayer->aim.dx + ddx, pPlayer->aim.dy + ddy, pPlayer->aim.dz + ddz,
			0, kDamageBullet, 4);
	}
}


static void FireTommy( int /* nTrigger */, PLAYER *pPlayer )
{
	SPRITE *pSprite = pPlayer->sprite;

	int z = pSprite->z - pPlayer->weaponAboveZ;

	UseAmmo(pPlayer, kWeaponTommy, 1);

	// dispersal modifiers
	int ddx = BiRandom(500);
	int ddy = BiRandom(500);
	int ddz = BiRandom(500);

	actFireVector(pPlayer->nSprite, z,
		pPlayer->aim.dx + ddx, pPlayer->aim.dy + ddy, pPlayer->aim.dz + ddz,
		0, kDamageBullet, 4 );
}


static void FireFlare( int /* nTrigger */, PLAYER *pPlayer )
{
	playerFireMissile(pPlayer, pPlayer->aim.dx, pPlayer->aim.dy, pPlayer->aim.dz,
		kMissileFlare);
	UseAmmo(pPlayer, kWeaponFlare, 1);
}


static void FireVoodoo( int nStabType, PLAYER *pPlayer )
{
	SPRITE *pTarget = NULL;
	int nSelfSprite = pPlayer->nSprite;
	int z = pPlayer->sprite->z - pPlayer->weaponAboveZ;

	if ( nStabType == kVoodooStabSelf )
	{
		actDamageSprite(nSelfSprite, pPlayer->nSprite, kDamageStab, 2 << 4);
		if (pPlayer == gMe)
			scrSetMessage("Ouch!");
		return;
	}

	dassert(pPlayer->aimSprite >= 0);
	pTarget = &sprite[pPlayer->aimSprite];

	switch( nStabType )
	{
		case kVoodooStabChest:
			actSpawnRicochet(pTarget->sectnum, pTarget->x, pTarget->y, pTarget->z,
				surfType[pTarget->picnum]);
			actDamageSprite(nSelfSprite, pPlayer->aimSprite, kDamageStab, 2 << 4);
			break;

		case kVoodooStabShoulder:
			actSpawnRicochet(pTarget->sectnum, pTarget->x, pTarget->y, pTarget->z,
				surfType[pTarget->picnum]);
			actDamageSprite(nSelfSprite, pPlayer->aimSprite, kDamageStab, 4 << 4);
			if ( pTarget->type >= kDudePlayer1 && pTarget->type <= kDudePlayer8 )
			{
				int nPlayerHit = pTarget->type - kDudePlayer1;
				WeaponLower(&gPlayer[nPlayerHit], &gPlayerInput[nPlayerHit]);
			}
			break;

		case kVoodooStabGroin:
			actSpawnRicochet(pTarget->sectnum, pTarget->x, pTarget->y, pTarget->z,
				surfType[pTarget->picnum]);
			actDamageSprite(nSelfSprite, pPlayer->aimSprite, kDamageStab, 12 << 4);
			break;

		case kVoodooStabEye:
			actSpawnRicochet(pTarget->sectnum, pTarget->x, pTarget->y, pTarget->z,
				surfType[pTarget->picnum]);
			actDamageSprite(nSelfSprite, pPlayer->aimSprite, kDamageStab, 6 << 4);
			if ( pTarget->type >= kDudePlayer1 && pTarget->type <= kDudePlayer8 )
			{
				PLAYER *pPlayerHit = &gPlayer[pTarget->type - kDudePlayer1];
				if (pPlayerHit == gView)	// change to gMe later!
				   	scrDacRelEffect(-96, -96, -96);
			}
			break;
	}
}


static void FireSpear( int /* nTrigger */, PLAYER *pPlayer )
{
	playerFireMissile(pPlayer, pPlayer->aim.dx, pPlayer->aim.dy, pPlayer->aim.dz,
		kMissileSpear);
	UseAmmo(pPlayer, kWeaponSpear, 1);
}


static void FireShadow( int /* nTrigger */, PLAYER *pPlayer )
{
	playerFireMissile(pPlayer, pPlayer->aim.dx, pPlayer->aim.dy, pPlayer->aim.dz,
		kMissileEctoSkull);

	if ( !gInfiniteAmmo )
		actDamageSprite(pPlayer->nSprite, pPlayer->nSprite, kDamageSpirit, 10 << 4);
}


uchar WeaponFindLoaded( PLAYER *pPlayer )
{
	int nAmmoIndex = 0;
	int nNewWeapon = kWeaponPitchfork;

	dprintf("WeaponFindLoaded: ");

	if ( pPlayer->hasWeapon[ kWeaponShadow ] && powerupCheck(pPlayer, kItemInvulnerability - kItemBase) )
		nNewWeapon = kWeaponShadow;
	else if ( pPlayer->hasWeapon[kWeaponTommy] && CheckAmmo(pPlayer, kWeaponTommy) )
		nNewWeapon = kWeaponTommy;
	else if ( pPlayer->hasWeapon[kWeaponShotgun] && CheckAmmo( pPlayer, kWeaponShotgun) )
		nNewWeapon = kWeaponShotgun;
	else if ( pPlayer->hasWeapon[kWeaponFlare] && CheckAmmo(pPlayer, kWeaponFlare) )
		nNewWeapon = kWeaponFlare;
	else if ( pPlayer->hasWeapon[kWeaponSpear] && CheckAmmo(pPlayer, kWeaponSpear) )
		nNewWeapon = kWeaponSpear;
	else if ( pPlayer->hasWeapon[kWeaponSprayCan] && CheckAmmo(pPlayer, kWeaponSprayCan) )
		nNewWeapon = kWeaponSprayCan;
	else if ( pPlayer->hasWeapon[kWeaponTNT] && CheckAmmo(pPlayer, kWeaponTNT) )
		nNewWeapon = kWeaponTNT;
	else if ( pPlayer->hasWeapon[kWeaponVoodoo] && CheckAmmo(pPlayer, kWeaponVoodoo) )
		nNewWeapon = kWeaponVoodoo;
	else if ( pPlayer->hasWeapon[kWeaponShadow] )
	{
		if ( pPlayer->xsprite->health > 75 )
			nNewWeapon = kWeaponShadow;
	}

	dprintf("nNewWeapon = %i\n", nNewWeapon);
	return (uchar)nNewWeapon;
}


/*
;[Weapon#]
;Name            = "WeaponNameMax24"
;SelectKey       = [ None | scancode ]
;UseByDefault    = [ TRUE | FALSE ]
;HasByDefault    = [ TRUE | FALSE ]
;RequiresWeapon  = [ None | 0..9 ]
;
;Type            = [ NonWeapon | Melee | Thrown | Projectile | Vector ]
;Type            = NonWeapon
;
;Type            = Melee
;MaxDistance     = [ Infinite | 0..max ]
;
;Type            = Thrown
;MaxDistance     = [ Infinite | 0..max ]
;
;Type            = Projectile
;MaxDistance     = [ Infinite | 0..max ]
;
;Type            = Vector (uses hitscan)
;MaxDistance     = [ Infinite | 0..max ]
;Richochet       = [ TRUE | FALSE ]
;UsesNightScope  = [ TRUE | FALSE ]
;
;Shots           = [ 0..max shots or projectiles ]
;Dispersion      = [ 0..100% ]
;Accuracy        = [ 0..100% ]
;
;AmmoStyle       = [ Infinite, Rechargable, Expendable, Health ]
;AmmoStyle       = Infinite
;
;AmmoStyle       = Rechargable
;RechargeRate    = [ 0 | ammo/second (?) ]
;MaxAmmo         = [ 0..max ]
;
;AmmoStyle       = Expendable
;MaxAmmo         = [ 0..max ]
;AmmoType        = [ 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 ]
;
;AmmoStyle       = Health
;HealthPerShot   = [ 0..usedpershot ]
;
;JamFrequency    = [ 0..100% ]
;JamSingleFire   = [ TRUE | FALSE ]
;JamRapidFire    = [ TRUE | FALSE ]
;
;[Ammo#]
;Name            = "AmmoNameMax24"
;Velocity        = [ ? ]
;DamageTypes     = [ Explosion, Burn, Bullet, Stab, Pummel, Cleave, Ecto]
;DamageRates     = [ 0,         0,    0,      0,    0,      0,      0 ]
*/

void WeaponUpdateState( PLAYER *pPlayer)
{
	SPRITE *pSprite		= pPlayer->sprite;
	XSPRITE *pXSprite	= pPlayer->xsprite;

	int nWeapon = pPlayer->weapon;
	int nState = pPlayer->weaponState[nWeapon];

	switch( nWeapon )
	{
		case kWeaponPitchfork:
			pPlayer->pWeaponQAV = weaponQAV[kQAVForkIdle];
			break;

		case kWeaponSprayCan:
		{
			switch( nState )
			{
				case kBicOpen:
					StartQAV(pPlayer, kQAVBicFlame);
					pPlayer->weaponState[kWeaponSprayCan] = kBicIdle;
					break;

				case kBicIdle:
					if ( CheckAmmo(pPlayer, kWeaponSprayCan) )
					{
						StartQAV(pPlayer, kQAVSprayUp);
						pPlayer->weaponState[kWeaponSprayCan] = kCanIdle;
					}
					else
						pPlayer->pWeaponQAV = weaponQAV[kQAVBicIdle];
					break;

				case kCanIdle:
					pPlayer->pWeaponQAV = weaponQAV[kQAVSprayIdle];
					break;

				case kCanFire:
					if ( CheckAmmo(pPlayer, kWeaponSprayCan) )
					{
						pPlayer->pWeaponQAV = weaponQAV[kQAVSprayIdle];
						pPlayer->weaponState[kWeaponSprayCan] = kCanIdle;
					}
					else
					{
						StartQAV(pPlayer, kQAVSprayDown);
						pPlayer->weaponState[kWeaponSprayCan] = kBicIdle;
					}
					break;
			}
			break;
		}

		case kWeaponTNT:
			switch( nState )
			{
				case kBicOpen:
					StartQAV(pPlayer, kQAVBicFlame);
					pPlayer->weaponState[nWeapon] = kBicIdle;
					break;

				case kBicIdle:
					if ( pPlayer->ammoCount[kAmmoTNTStick] > 0)
					{
						StartQAV(pPlayer, kQAVStickUp);
						pPlayer->weaponState[kWeaponTNT] = kTNTStickIdle;
					}
					else
						pPlayer->pWeaponQAV = weaponQAV[kQAVBicIdle];
					break;

				case kTNTStickIdle:
					pPlayer->pWeaponQAV = weaponQAV[kQAVStickIdle];
					break;

				case kTNTBundleIdle:
					pPlayer->pWeaponQAV = weaponQAV[kQAVBundleIdle];
					break;
			}
			break;

		case kWeaponShotgun:
			switch ( nState )
			{
				case kShotIdle1:
					pPlayer->pWeaponQAV = weaponQAV[kQAVShotIdle1];
					break;

				case kShotIdle2:
					pPlayer->pWeaponQAV = weaponQAV[kQAVShotIdle2];
					break;

				case kShotIdle3:
					if (gInfiniteAmmo || pPlayer->ammoCount[kAmmoShell] > 0)
					    pPlayer->weaponState[kWeaponShotgun] = kShotReload;
					pPlayer->pWeaponQAV = weaponQAV[kQAVShotIdle3];
					break;
			}
			break;

		case kWeaponTommy:
			pPlayer->pWeaponQAV = weaponQAV[kQAVTommyIdle];
			break;

		case kWeaponFlare:
			if (nState == kFlareIdle)
				pPlayer->pWeaponQAV = weaponQAV[kQAVFlareIdle];
			break;

		case kWeaponVoodoo:
	 		switch ( pXSprite->moveState )
			{
				case kMoveWalk:
					if ( pPlayer->swayHeight )
						pPlayer->pWeaponQAV = weaponQAV[kQAVDollIdleMoving];
					else
						pPlayer->pWeaponQAV = weaponQAV[kQAVDollIdleStill];
					break;

				case kMoveLand:
					pPlayer->pWeaponQAV = weaponQAV[kQAVDollIdleMoving];
					break;

				default:
					pPlayer->pWeaponQAV = weaponQAV[kQAVDollIdleStill];
					break;
			}
			break;

		case kWeaponSpear:
			switch ( nState )
			{
				case kSpearIdle1:
					pPlayer->pWeaponQAV = weaponQAV[kQAVSpearIdle1];
					break;

				case kSpearIdle2:
					pPlayer->pWeaponQAV = weaponQAV[kQAVSpearIdle2];
					break;
			}
			break;

		case kWeaponShadow:
			pPlayer->pWeaponQAV = weaponQAV[kQAVShadowIdle];
			break;

		case kWeaponBeast:
			pPlayer->pWeaponQAV = weaponQAV[kQAVBeastIdle];
			break;
	}
}


void WeaponProcess( PLAYER *pPlayer, INPUT *pInput )
{
	WeaponPlay(pPlayer);

	// decrement weapon ready timer
	pPlayer->weaponTimer -= kFrameTicks;

	if ( pPlayer->fLoopQAV )
	{
		if ( pInput->buttonFlags.shoot && pPlayer->ammoCount[pPlayer->weaponAmmo[pPlayer->weapon]] )
		{
			while ( pPlayer->weaponTimer <= 0 )
				pPlayer->weaponTimer += pPlayer->pWeaponQAV->duration;
			UpdateAimVector(pPlayer);
			return;
		}
		pPlayer->weaponTimer = 0;
		pPlayer->fLoopQAV = FALSE;
	}
	else
		pPlayer->weaponTimer = ClipLow(pPlayer->weaponTimer, 0);

	// special processing for TNT
	if ( pPlayer->weapon == kWeaponTNT )
	{
		switch ( pPlayer->weaponState[kWeaponTNT] )
		{
			case kTNTStickFuse1:
				// player released fire key?
				if ( !pInput->buttonFlags.shoot )
					pPlayer->weaponState[kWeaponTNT] = kTNTStickFuse2;
				break;

			case kTNTStickFuse2:
				// throwing?
				if ( pInput->buttonFlags.shoot )
				{
					pPlayer->weaponState[kWeaponTNT] = kTNTStickFuse3;
					pPlayer->throwTime = gFrameClock;
				}
				break;

			case kTNTStickFuse3:
				// releasing?
				if ( !pInput->buttonFlags.shoot )
				{
					pPlayer->fuseTime = pPlayer->weaponTimer;
					pPlayer->throwTime = gFrameClock - pPlayer->throwTime;
					StartQAV(pPlayer, kQAVStickThrow, (QAVCALLBACK)FireStick);
					pPlayer->weaponState[kWeaponTNT] = kBicIdle;
					return;
				}
				break;

			case kTNTBundleFuse1:
				// player released fire key?
				if ( !pInput->buttonFlags.shoot )
					pPlayer->weaponState[kWeaponTNT] = kTNTBundleFuse2;
				break;

			case kTNTBundleFuse2:
				// throwing?
				if ( pInput->buttonFlags.shoot )
				{
					pPlayer->weaponState[kWeaponTNT] = kTNTBundleFuse3;
					pPlayer->throwTime = gFrameClock;
				}
				break;

			case kTNTBundleFuse3:
				// releasing?
				if ( !pInput->buttonFlags.shoot )
				{
					pPlayer->fuseTime = pPlayer->weaponTimer;
					pPlayer->throwTime = gFrameClock - pPlayer->throwTime;
					StartQAV(pPlayer, kQAVBundleThrow, (QAVCALLBACK)FireBundle);
					pPlayer->weaponState[kWeaponTNT] = kBicIdle;
					return;
				}
				break;

		}
	}

	// return if weapon is busy
	if (pPlayer->weaponTimer > 0)
		return;

	pPlayer->pWeaponQAV = NULL;

	int nWeapon = pPlayer->weapon;

	// pPlayer->weapon is set to kMaxWeapons in the fire processing below
	if ( pPlayer->weaponState[nWeapon] == kFindWeapon )
	{
		dprintf("Out of ammo: looking for a new weapon\n");
		pPlayer->weaponState[nWeapon] = 0;
		pInput->newWeapon = WeaponFindLoaded( pPlayer );
	}

	// weapon select
	if ( pInput->newWeapon )
	{
		if ( pPlayer->xsprite->health == 0 || !pPlayer->hasWeapon[ pInput->newWeapon ] )
		{
			pInput->newWeapon = 0;
			return;
		}

		if (pPlayer->weapon == 0) 	// raise the new weapon
			WeaponRaise(pPlayer, pInput);
		else
			WeaponLower(pPlayer, pInput);	// lower the old weapon
		return;
	}

	if ( (nWeapon = pPlayer->weapon) == 0 )
		return;		// no weapon drawn, do nothing

    // auto reload test
	switch ( nWeapon )
	{
		case kWeaponShotgun:
			if ( pPlayer->weaponState[nWeapon] == kShotReload )
			{
				if ( !CheckAmmo(pPlayer, kWeaponShotgun) )
					pPlayer->weaponState[kWeaponShotgun] = kShotIdle3;
				else
				{
			    	StartQAV(pPlayer, kQAVShotReload);
					if (gInfiniteAmmo || pPlayer->ammoCount[kAmmoShell] > 1)
						pPlayer->weaponState[kWeaponShotgun] = kShotIdle1;
					else
						pPlayer->weaponState[kWeaponShotgun] = kShotIdle2;
				}
				return;
			}
			break;
	}

	UpdateAimVector(pPlayer);

	// fire key
	if ( pInput->buttonFlags.shoot )
	{
		if ( CheckAmmo(pPlayer, nWeapon) )
		{
			// firing weapon (primary)
			switch ( nWeapon )
			{
				case kWeaponPitchfork:
					StartQAV(pPlayer, kQAVForkStab, (QAVCALLBACK)FirePitchfork, FALSE);
					return;

				case kWeaponSprayCan:
					if ( pPlayer->weaponState[kWeaponSprayCan] == kCanIdle )
					{
						StartQAV(pPlayer, kQAVSprayFire, (QAVCALLBACK)FireSpray, TRUE);
						pPlayer->weaponState[kWeaponSprayCan] = kCanFire;
						return;
					}
					break;

				case kWeaponTNT:
					switch ( pPlayer->weaponState[kWeaponTNT] )
					{
						case kTNTStickIdle:
							StartQAV(pPlayer, kQAVStickFuse, (QAVCALLBACK)ExplodeStick);
							pPlayer->weaponState[kWeaponTNT] = kTNTStickFuse1;
							return;

						case kTNTBundleIdle:
							StartQAV(pPlayer, kQAVBundleFuse, (QAVCALLBACK)ExplodeBundle);
							pPlayer->weaponState[kWeaponTNT] = kTNTBundleFuse1;
							return;
					}
					break;

			    case kWeaponShotgun:
					switch ( pPlayer->weaponState[kWeaponShotgun] )
					{
						case kShotIdle1:
							StartQAV(pPlayer, kQAVShotFireL);
							pPlayer->weaponState[kWeaponShotgun] = kShotIdle2;
							if ( !gInfiniteAmmo )
								pPlayer->ammoCount[kAmmoShell] -= 1;
							FireShotgun( pPlayer, 1 );
							return;

						case kShotIdle2:
							StartQAV(pPlayer, kQAVShotFireR);
						    pPlayer->weaponState[kWeaponShotgun] = kShotReload;
							if ( !gInfiniteAmmo )
								pPlayer->ammoCount[kAmmoShell] -= 1;
							FireShotgun( pPlayer, 1 );
							return;
					}
					return;

				case kWeaponTommy:
					StartQAV(pPlayer, kQAVTommyFire1, (QAVCALLBACK)FireTommy, TRUE);
					return;

				case kWeaponFlare:
					StartQAV(pPlayer, kQAVFlareFire, (QAVCALLBACK)FireFlare);
					return;

				case kWeaponVoodoo:
				{
					int nStabType = Random(4);
					if ( pPlayer->aimSprite == -1 )
						nStabType = kVoodooStabSelf;

					StartQAV(pPlayer, kQAVDollStab1 + nStabType - 1, (QAVCALLBACK)FireVoodoo);
					return;
				}

			    case kWeaponSpear:
					StartQAV(pPlayer, kQAVSpearFire, (QAVCALLBACK)FireSpear);
					return;

				case kWeaponShadow:
					StartQAV(pPlayer, kQAVShadowFire, (QAVCALLBACK)FireShadow);
					return;

				case kWeaponBeast:
					StartQAV(pPlayer, kQAVBeastAttack1 + Random(4));
					pPlayer->weaponState[kWeaponBeast] = kBeastIdle;
					return;
			}
		}
		else
		{
			dprintf("Primary out of ammo: setting weapon state to kFindWeapon\n");
			pPlayer->weaponState[nWeapon] = kFindWeapon;
			return;
		}
	}

	// alternate fire key
	if ( pInput->buttonFlags.shoot2 )
	{
		if ( CheckAmmo(pPlayer, nWeapon) )
		{
			// firing weapon (alternate)
			switch ( nWeapon )
			{
				case kWeaponPitchfork:
					playerFireMissile(pPlayer,
						pPlayer->aim.dx, pPlayer->aim.dy, pPlayer->aim.dz,
						kMissileFireball);
					pPlayer->weaponTimer = 20;
					return;

				case kWeaponSprayCan:
					if ( pPlayer->weaponState[kWeaponSprayCan] == kCanIdle )
					{
						StartQAV(pPlayer, kQAVSprayFire, (QAVCALLBACK)FireSpray, TRUE);
						pPlayer->weaponState[kWeaponSprayCan] = kCanFire;
						return;
					}
					break;

				case kWeaponTNT:
					break;

				case kWeaponShotgun:
					switch ( pPlayer->weaponState[kWeaponShotgun] )
					{
						case kShotIdle1:
							StartQAV(pPlayer, kQAVShotFireR);
							pPlayer->weaponState[kWeaponShotgun] = kShotReload;
							UseAmmo(pPlayer, kWeaponShotgun, 2);
							FireShotgun(pPlayer, 2);
							return;
						case kShotIdle2:
							StartQAV(pPlayer, kQAVShotFireR);
							pPlayer->weaponState[kWeaponShotgun] = kShotReload;
							UseAmmo(pPlayer, kWeaponShotgun, 1);
							FireShotgun(pPlayer, 1);
							return;
					}
					break;

				case kWeaponTommy:
					StartQAV(pPlayer, kQAVTommyFire1, (QAVCALLBACK)FireTommy, TRUE);
					return;

				case kWeaponFlare:
					StartQAV(pPlayer, kQAVFlareFire, (QAVCALLBACK)FireFlare);
					return;

				case kWeaponVoodoo:
				{
					int nStabType = Random(4);
					if ( pPlayer->aimSprite == -1 )
						nStabType = kVoodooStabSelf;

					StartQAV(pPlayer, kQAVDollStab1 + nStabType, (QAVCALLBACK)FireVoodoo);
					return;
				}

			    case kWeaponSpear:
					StartQAV(pPlayer, kQAVSpearFire, (QAVCALLBACK)FireSpear);
					return;

				case kWeaponShadow:
					StartQAV(pPlayer, kQAVShadowFire, (QAVCALLBACK)FireShadow);
					return;
			}
		}
		else
		{
			dprintf("Alternate out of ammo: setting weapon state to kFindWeapon\n");
			pPlayer->weaponState[nWeapon] = kFindWeapon;
			return;
		}
	}

	WeaponUpdateState(pPlayer);
}
