/*******************************************************************************
	FILE:			ANIMEDIT.CPP

	DESCRIPTION:	Allows editing of 2D animation sequences

	AUTHOR:			Peter M. Freese
	CREATED:		02-26-95
	COPYRIGHT:		Copyright (c) 1995 Q Studios Corporation
*******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <helix.h>
#include <string.h>
#include <io.h>
#include <fcntl.h>

#include "typedefs.h"
#include "engine.h"
#include "key.h"
#include "misc.h"
#include "gameutil.h"
#include "gfx.h"
#include "bstub.h"
#include "globals.h"
#include "debug4g.h"
#include "error.h"
#include "gui.h"
#include "tile.h"
#include "anim.h"
#include "screen.h"
#include "textio.h"
#include "inifile.h"
#include "options.h"
#include "timer.h"
#include "mouse.h"

#include <memcheck.h>

#define kQAVSig		"QAV\x1A"
#define kQAVVersion	0x0100
#define kMaxFrames	1024

#define kAttrTitle      (kColorGreen * 16 + kColorWhite)


struct EditAnim : public Anim
{
	void 	ClipFrameTile( FRAME *f );
	void	SetOrigin( int x, int y );
	void 	DrawOrigin( int nColor );
	void 	HighlightFrame( FRAME *f );
};


char gAnimName[128];

int editFrames, edit;
FRAME *editFrame[kMaxFrames];

int soundFrames, sound;
FRAME *soundFrame[kMaxFrames];

int triggerFrames, trigger;
FRAME *triggerFrame[kMaxFrames];

int step = 5;	// 1/24 second
int lastTile = 2052;
BOOL looping = TRUE;
char buffer[256];
EditAnim *gAnim = NULL;
static BOOL isModified;
EHF prevErrorHandler;


void faketimerhandler( void )
{ }


/***********************************************************************
 * EditorErrorHandler()
 *
 * Terminate from error condition, displaying a message in text mode.
 *
 **********************************************************************/
ErrorResult EditorErrorHandler( const Error& error )
{
	uninitengine();
	keyRemove();
	setvmode(gOldDisplayMode);

	// chain to the default error handler
	return prevErrorHandler(error);
};


/***********************************************************************
 * Ensure that the tile location isn't off the screen
 **********************************************************************/
void EditAnim::ClipFrameTile( FRAME *f )
{
	int xOffset, yOffset;

	if ( f->flags & kQFrameCorner )
		xOffset = yOffset = 0;
	else
	{
		xOffset = picanm[f->id].xcenter + tilesizx[f->id] / 2 - origin.x;
		yOffset = picanm[f->id].ycenter + tilesizy[f->id] / 2 - origin.y;
	}

	if (f->x < xOffset - tilesizx[f->id] + 1)
		f->x = xOffset - tilesizx[f->id] + 1;

	if (f->x > 319 + xOffset)
		f->x = 319 + xOffset;

	if (f->y < yOffset - tilesizy[f->id] + 1)
		f->y = yOffset - tilesizy[f->id] + 1;

	if (f->y > 199 + yOffset)
		f->y = 199 + yOffset;
}


void EditAnim::SetOrigin( int x, int y )
{
	if (keystatus[KEY_LCTRL] || keystatus[KEY_RCTRL])
	{
		x = ClipRange(x, 0, 319);
		y = ClipRange(y, 0, 199);
		int dx = x - origin.x;
		int dy = y - origin.y;

		for (int i = 0; i < frames; i++)
		{
			frame[i].x -= dx;
			frame[i].y -= dy;
		}
		origin.x = x;
		origin.y = y;
	}
	else
	{
		origin.x = ClipRange(x, 0, 319);
		origin.y = ClipRange(y, 0, 199);

		// ensure that no frame are offscreen
		for (int i = 0; i < frames; i++)
			ClipFrameTile(&frame[i]);
	}
}


void EditAnim::DrawOrigin( int nColor )
{
	Video.SetColor(nColor);
	gfxHLine(origin.y, origin.x - 6, origin.x - 1);
	gfxHLine(origin.y, origin.x + 1, origin.x + 6);
	gfxVLine(origin.x, origin.y - 5, origin.y - 1);
	gfxVLine(origin.x, origin.y + 1, origin.y + 5);
}


void EditAnim::HighlightFrame( FRAME *f )
{
	int xOffset, yOffset;

	if ( f->flags & kQFrameCorner )
		xOffset = 0, yOffset = 0;
	else
	{
		xOffset = picanm[f->id].xcenter + tilesizx[f->id] / 2 - origin.x;
		yOffset = picanm[f->id].ycenter + tilesizy[f->id] / 2 - origin.y;
	}

	int x0 = origin.x + f->x - xOffset - 1;
	int y0 = origin.y + f->y - yOffset - 1;
	int x1 = x0 + tilesizx[f->id] + 1;
	int y1 = y0 + tilesizy[f->id] + 1;

	Video.SetColor(gStdColor[8]);
	gfxHLine(y0, x0, x1);
	gfxHLine(y1, x0, x1);
	gfxVLine(x0, y0, y1);
	gfxVLine(x1, y0, y1);
}


void SaveAnim( void )
{
	char filename[128];
	int hFile;

	strcpy(filename, gAnimName);
	ChangeExtension(filename, ".QAV");

	hFile = open(filename, O_CREAT | O_WRONLY | O_BINARY | O_TRUNC, S_IWUSR);
	if ( hFile == -1 )
		ThrowError("Error creating QAV file", ES_ERROR);

	memcpy(gAnim->signature, kQAVSig, sizeof(gAnim->signature));
	gAnim->version = kQAVVersion;

	if ( !FileWrite(hFile, gAnim, sizeof(Anim) + gAnim->frames * sizeof(FRAME)) )
		goto WriteError;

	close(hFile);
	isModified = FALSE;

	scrSetMessage("Animation saved.");
	return;

WriteError:
	close(hFile);
	ThrowError("Error writing QAV file", ES_ERROR);
}


BOOL LoadAnim( void )
{
	char filename[128];
	int hFile;
	int nSize;

	strcpy(filename, gAnimName);
	ChangeExtension(filename, ".QAV");

	hFile = open(filename, O_RDONLY | O_BINARY);
	if ( hFile == -1 )
		return FALSE;

	nSize = filelength(hFile);

	if ( !FileRead(hFile, gAnim, nSize) )
		goto ReadError;

	if (memcmp(gAnim->signature, kQAVSig, sizeof(gAnim->signature)) != 0)
	{
		close(hFile);
		ThrowError("QAV file corrupted", ES_ERROR);
	}

/*******************************************************************************
I'm ignoring version information for now.  We'll probably have to support
more versions as we make enhancements to the animation editor.
*******************************************************************************/

	close(hFile);
	isModified = FALSE;
	return TRUE;

ReadError:
	close(hFile);
	ThrowError("Error reading QAV file", ES_ERROR);
	return FALSE;
}


BOOL SaveAs( void )
{
	if ( GetStringBox("Save As", gAnimName) )
	{
		// load it if it seems valid
		if (strlen(gAnimName) > 0)
		{
			SaveAnim();
			return TRUE;
		}
	}
	return FALSE;
}


BOOL LoadAs( void )
{
	if ( GetStringBox("Load Anim", gAnimName) )
	{
		// load it if it seems valid
		if (strlen(gAnimName) > 0)
		{
			LoadAnim();
			return TRUE;
		}

	}
	return FALSE;
}


/*******************************************************************************
** ChangeAcknowledge()
**
** Returns TRUE if the YES/NO selected, or FALSE if aborted or escape pressed.
*******************************************************************************/
BOOL ChangeAcknowledge(void)
{
	// create the dialog
	Window dialog(59, 80, 202, 46, "Save Changes?");

	TextButton *pbYes    = new TextButton( 4, 4, 60,  20, "&Yes", mrYes );
	TextButton *pbNo     = new TextButton( 68, 4, 60, 20, "&No", mrNo );
	TextButton *pbCancel = new TextButton( 132, 4, 60, 20, "&Cancel", mrCancel );
	dialog.Insert(pbYes);
	dialog.Insert(pbNo);
 	dialog.Insert(pbCancel);

	ShowModal(&dialog);
	switch ( dialog.endState )
	{
		case mrCancel:
			return FALSE;

		case mrOk:
		case mrYes:
			return SaveAs();
	}

	return TRUE;
}

/***********************************************************************
 * GetEditFrames
 **********************************************************************/
void GetEditFrames( ulong t )
{
	int i;
	editFrames = 0;
	gAnim->duration = 0;

	for (i = 0; i < gAnim->frames; i++)
	{
		if (gAnim->frame[i].timeStart < t + step && gAnim->frame[i].timeStop > t)
			editFrame[editFrames++] = &gAnim->frame[i];

		if (gAnim->frame[i].timeStop > gAnim->duration)
			gAnim->duration = gAnim->frame[i].timeStop;
	}

	// set the current instruction to the last one found at this time index
	edit = editFrames - 1;
}


/***********************************************************************
 * GetSoundFrames
 **********************************************************************/
void GetSoundFrames( ulong t )
{
	int i;
	soundFrames = 0;

	for (i = 0; i < gAnim->frames; i++)
	{
		if (gAnim->frame[i].type == kQFrameSound )
		{
			if (gAnim->frame[i].timeStart < t + step && gAnim->frame[i].timeStop > t)
				soundFrame[soundFrames++] = &gAnim->frame[i];
		}
	}

	// set the current instruction to the last one found at this time index
	sound = soundFrames - 1;
}


/***********************************************************************
 * GetTriggerFrames
 **********************************************************************/
void GetTriggerFrames( ulong t )
{
	int i;
	triggerFrames = 0;

	for (i = 0; i < gAnim->frames; i++)
	{
		if (gAnim->frame[i].type == kQFrameTrigger )
		{
			if (gAnim->frame[i].timeStart < t + step && gAnim->frame[i].timeStop > t)
				triggerFrame[triggerFrames++] = &gAnim->frame[i];
		}
	}

	// set the current instruction to the last one found at this time index
	trigger = triggerFrames - 1;
}


void PlayIt( void )
{
	ulong timeBase = gGameClock;
	ulong t = 0;
	int shift;

	while (1)
	{
		if ( keyGet() == KEY_ESC )
			break;

		if ((t >> 16) >= gAnim->duration)
		{
			if (!looping)
				break;

			t -= (gAnim->duration << 16);
			continue;
		}

		clearview(0);

		gAnim->Play(t >> 16, 0, 0);

		sprintf(buffer, "%2i:%03i", (t >> 16) / kTimerRate, (t >> 16) % kTimerRate);
		printext256(270, 190, gStdColor[15], -1, buffer, 1);

		if (vidoption != 1)
			WaitVSync();

		scrNextPage();

		gFrameTicks = gGameClock - gFrameClock;
		gFrameClock += gFrameTicks;

		shift = 16;
		if (keystatus[KEY_LCTRL])
			shift = 15;
		if (keystatus[KEY_LSHIFT])
			shift = 14;

		t += gFrameTicks << shift;
	}
}


void GetFrameTables( ulong t )
{
	GetEditFrames(t);
	GetSoundFrames(t);
	GetTriggerFrames(t);
}


void DrawEditFrames( void )
{
	int soundY = 0;
	int triggerY = 0;

	for (int i = 0; i < editFrames; i++)
	{
		switch ( editFrame[i]->type )
		{
			case kQFrameTile:
				gAnim->DrawFrame(editFrame[i], 0, 0);
				break;

			case kQFrameSound:
			{
				// show the sound info
				FRAME *f = editFrame[i];
				sprintf( buffer, "SND: %04i VOL: %04i PRI: %04i", f->soundID, f->volume, f->priority );
				printext256(200, soundY, gStdColor[15], -1, buffer, 1);
				soundY += 8;
				break;
			}
			case kQFrameTrigger:
			{
				// show the trigger info
				FRAME *f = editFrame[i];
				sprintf( buffer, "TRIGGER: %+5i", f->trigger );
				printext256(0, triggerY, gStdColor[15], -1, buffer, 1);
				triggerY += 8;
				break;
			}
			default:
				ThrowError("Invalid frame type in QAV file", ES_ERROR);
				break; // code should not return from ThrowError
		}
		if (!keystatus[KEY_O] && i == edit && IsBlinkOn() )
			gAnim->HighlightFrame(editFrame[i]);
	}

	gAnim->DrawOrigin(gStdColor[keystatus[KEY_O] ? 15 : 8]);
}


void EditLoop( void )
{
	BYTE key, shift, alt, ctrl, pad5;
	int i;
	int t = 0;
	int nbuttons;
	static int obuttons = 0;
	char buffer[256];

	GetFrameTables(t);

	while (1)
	{
		gFrameTicks = gGameClock - gFrameClock;
		gFrameClock += gFrameTicks;
		UpdateBlinkClock(gFrameTicks);

		clearview(0);

		DrawEditFrames();

		sprintf(buffer,"%2i:%03i [%3i]", t / kTimerRate, t % kTimerRate, step);
		printext256(270, 190, gStdColor[15], -1, buffer, 1);

		if (keystatus[KEY_O])
		{
			sprintf(buffer,"%3i,%3i", gAnim->origin.x, gAnim->origin.y);
			printext256(0, 190, gStdColor[15], -1, buffer, 1);
		}
		else if (edit >= 0)
		{
			sprintf(buffer,"%+4i,%+4i  %2i:%03i-%2i:%03i  Shade:%+3i Pal:%+2i",
				editFrame[edit]->x,
				editFrame[edit]->y,
				editFrame[edit]->timeStart / kTimerRate, editFrame[edit]->timeStart % kTimerRate,
				editFrame[edit]->timeStop / kTimerRate, editFrame[edit]->timeStop % kTimerRate,
				editFrame[edit]->shade,
				editFrame[edit]->pal
			);
			printext256(0, 190, gStdColor[15], -1, buffer, 1);
		}

		scrDisplayMessage(gStdColor[kColorWhite]);

		if (vidoption != 1)
			WaitVSync();

		scrNextPage();

		key = keyGet();
		shift = keystatus[KEY_LSHIFT] | keystatus[KEY_RSHIFT];
		ctrl = keystatus[KEY_LCTRL] | keystatus[KEY_RCTRL];
		alt = keystatus[KEY_LALT] | keystatus[KEY_RALT];
		pad5 = keystatus[KEY_PAD5];

		switch (key)
		{
			case KEY_ESC:
				keystatus[KEY_ESC] = 0;
				if (!isModified || ChangeAcknowledge())
				{
					free(gAnim);
					gAnim = NULL;
					return;
				}
				break;

			case KEY_SPACE:
				PlayIt();
				break;

			case KEY_F2:
				if (keystatus[KEY_LCTRL] || keystatus[KEY_RCTRL] || strlen(gAnimName) == 0)
					SaveAs();
				else
					SaveAnim();
				break;

			case KEY_F3:
				if (!isModified || ChangeAcknowledge())
				{
					LoadAs();
					t = 0;
					GetFrameTables(t);
				}
				break;

			case KEY_2:
				// adjust stop time
				if ( edit >= 0 && keystatus[KEY_PERIOD] )
				{
					editFrame[edit]->timeStop += step;
					isModified = TRUE;
				}

				// adjust start time
				if ( edit >= 0 && keystatus[KEY_COMMA] )
				{
					editFrame[edit]->timeStart = ClipHigh(editFrame[edit]->timeStart + step, editFrame[edit]->timeStop - step);
					t = ClipRange(t, editFrame[edit]->timeStart, editFrame[edit]->timeStop - step);
					isModified = TRUE;
				}

				if ( !(keystatus[KEY_COMMA] || keystatus[KEY_PERIOD]) )
					t += step;

				GetFrameTables(t);
				break;

			case KEY_1:
				// adjust start time
				if ( edit >= 0 && keystatus[KEY_COMMA] )
				{
					editFrame[edit]->timeStart = ClipLow(editFrame[edit]->timeStart - step, 0);
					isModified = TRUE;
				}

				// adjust stop time
				if ( edit >= 0 && keystatus[KEY_PERIOD] )
				{
					editFrame[edit]->timeStop = ClipLow(editFrame[edit]->timeStop - step, editFrame[edit]->timeStart + step);
					t = ClipRange(t, editFrame[edit]->timeStart, editFrame[edit]->timeStop - step);
					isModified = TRUE;
				}

				if ( !(keystatus[KEY_COMMA] || keystatus[KEY_PERIOD]) )
					t = ClipLow(t - step, 0);

				GetFrameTables(t);
				break;

			case KEY_INSERT:
			{
				// find first frame starting after t
				for (i = 0; i < gAnim->frames; i++)
				{
					if (gAnim->frame[i].timeStart > t)
						break;
				}

				if (i < gAnim->frames)
					memmove(&gAnim->frame[i+1], &gAnim->frame[i], (gAnim->frames - i) * sizeof(FRAME));

				gAnim->frames++;

				memset(&gAnim->frame[i], 0, sizeof(FRAME));
				gAnim->frame[i].timeStart = t;
				gAnim->frame[i].timeStop = t + step;
				gAnim->frame[i].type = kQFrameTile;
				gAnim->frame[i].flags = 0;
				gAnim->frame[i].x = 0;
				gAnim->frame[i].y = 0;

				lastTile = gAnim->frame[i].id = tilePick((short)lastTile, SS_ALL);

				GetFrameTables(t);
				isModified = TRUE;
				break;
			}

			case KEY_DELETE:
				// ***
				--gAnim->frames;
				if (editFrame[edit] < &gAnim->frame[gAnim->frames])
					memmove(editFrame[edit], editFrame[edit] + 1, (&gAnim->frame[gAnim->frames] - editFrame[edit]) * sizeof(FRAME));
				GetFrameTables(t);
				isModified = TRUE;
				break;

			case KEY_P:
				if (edit >= 0 && editFrame[edit]->type == kQFrameTile)
				{
					// set the palette for the highlighted tile frame
					if ( alt )
						editFrame[edit]->pal = GetNumberBox("Palookup", editFrame[edit]->pal, editFrame[edit]->pal );
					else
						editFrame[edit]->pal = IncRotate(editFrame[edit]->pal, kMaxPLU);
					isModified = TRUE;
				}
				break;

			case KEY_R:
				if (edit >= 0 && editFrame[edit]->type == kQFrameTile)
				{
					editFrame[edit]->flags ^= kQFrameCorner;
					isModified = TRUE;
				}
				break;

			case KEY_T:
				if (edit >= 0 && editFrame[edit]->type == kQFrameTile)
				{
					// set translucency flags for the highlighted tile frame
					int flags = editFrame[edit]->flags;
					int level = 0;

					if ( flags & kQFrameTranslucent )
					{
						level = 1;
						if ( flags & kQFrameTranslucentR )
							level = 2;
					}
					level = IncRotate(level, 3);

					switch ( level )
					{
						case 0:
							flags &= ~kQFrameTranslucent & ~kQFrameTranslucentR;
							break;

						case 1:
							flags &= ~kQFrameTranslucentR;
							flags |= kQFrameTranslucent;
							break;

						case 2:
							flags |= kQFrameTranslucent | kQFrameTranslucentR;
							break;
					}
					editFrame[edit]->flags = flags;
					isModified = TRUE;
				}
				break;

			case KEY_V:
				if (edit >= 0 && editFrame[edit]->type == kQFrameTile)
				{
					// pick a tile for the tile frame
					lastTile = editFrame[edit]->id = tilePick((short)editFrame[edit]->id, SS_ALL);
					isModified = TRUE;
				}
				break;

			case KEY_X:
				if (edit >= 0 && editFrame[edit]->type == kQFrameTile)
				{
					editFrame[edit]->flags ^= kQFrameXFlip;
					isModified = TRUE;
				}
				break;

			case KEY_Y:
				if (edit >= 0 && editFrame[edit]->type == kQFrameTile)
				{
					editFrame[edit]->flags ^= kQFrameYFlip;
					isModified = TRUE;
				}
				break;

			case KEY_Z:
				if (edit >= 0 && editFrame[edit]->type == kQFrameTile)
				{
					if ( alt )
					{
						// input frame zoom factor through gui dialog box
						break;
					}

					if ( keystatus[KEY_MINUS] )
					{
						// decrement frame zoom factor by n
						break;
					}

					if ( keystatus[KEY_PLUS] )
					{
						// increment frame zoom factor by n
						break;
					}
				}
				break;

			case KEY_PADPLUS:
				if (edit >= 0 && editFrame[edit]->type == kQFrameTile)
				{
					editFrame[edit]->shade = ClipLow( editFrame[edit]->shade - 1, -128 );
					isModified = TRUE;
				}
				break;

			case KEY_PADMINUS:
				if (edit >= 0 && editFrame[edit]->type == kQFrameTile)
				{
					editFrame[edit]->shade = ClipHigh( editFrame[edit]->shade + 1, 127 );
					isModified = TRUE;
				}
				break;

			case KEY_PAD0:
				if (edit >= 0 && editFrame[edit]->type == kQFrameTile)
				{
					editFrame[edit]->shade = 0;
					isModified = TRUE;
				}
				break;

			case KEY_TAB:
				SetBlinkOn();
				if (editFrames > 1)
				{
					if ( keystatus[KEY_LSHIFT] || keystatus[KEY_RSHIFT] )
						edit = DecRotate(edit, editFrames);
					else
						edit = IncRotate(edit, editFrames);
				}
				break;

			case KEY_UP:
				SetBlinkOff();
				if ( keystatus[KEY_O] )
				{
					gAnim->SetOrigin(gAnim->origin.x, gAnim->origin.y - 1);
					isModified = TRUE;
					break;
				}

				if (edit >= 0 && editFrame[edit]->type == kQFrameTile)
				{
					editFrame[edit]->y--;
					isModified = TRUE;
				}
				break;

			case KEY_DOWN:
				SetBlinkOff();
				if ( keystatus[KEY_O] )
				{
					gAnim->SetOrigin(gAnim->origin.x, gAnim->origin.y + 1);
					isModified = TRUE;
					break;
				}

				if (edit >= 0 && editFrame[edit]->type == kQFrameTile)
				{
					editFrame[edit]->y++;
					isModified = TRUE;
				}
				break;

			case KEY_LEFT:
				SetBlinkOff();
				if ( keystatus[KEY_O] )
				{
					gAnim->SetOrigin(gAnim->origin.x - 1, gAnim->origin.y);
					isModified = TRUE;
					break;
				}

				if (edit >= 0 && editFrame[edit]->type == kQFrameTile)
				{
					editFrame[edit]->x--;
 					isModified = TRUE;
				}
				break;

			case KEY_RIGHT:
				SetBlinkOff();
				if ( keystatus[KEY_O] )
				{
					gAnim->SetOrigin(gAnim->origin.x + 1, gAnim->origin.y);
					isModified = TRUE;
					break;
				}

				if (edit >= 0 && editFrame[edit]->type == kQFrameTile)
				{
					editFrame[edit]->x++;
					isModified = TRUE;
				}
				break;
		}

		Mouse::Read(gFrameTicks);

		// which buttons just got pressed
		nbuttons = (short)(~obuttons & Mouse::buttons);
		obuttons = Mouse::buttons;

		if ( (nbuttons & 2) && editFrames > 1)
		{
			edit = IncRotate(edit, editFrames);
			SetBlinkOn();
		}

		if ( Mouse::buttons & 1 )
		{
			if ( keystatus[KEY_O] )
			{
				gAnim->SetOrigin(gAnim->origin.x + Mouse::dX2, gAnim->origin.y + Mouse::dY2);
				isModified = TRUE;
			}
			else
			{
				SetBlinkOff();

				if (edit >= 0)
				{
					editFrame[edit]->x += Mouse::dX2;
					editFrame[edit]->y += Mouse::dY2;
					isModified = TRUE;
				}
			}
		}

		// make sure x and y don't cause the tile to go off screen
		if (edit >= 0)
			gAnim->ClipFrameTile(editFrame[edit]);
	}
}


void main( int argc, char *argv[] )
{
	char title[256];

	gOldDisplayMode = getvmode();

	sprintf(title, "Animation Editor [%s] -- DO NOT DISTRIBUTE", gBuildDate);
	tioInit();
	tioCenterString(0, 0, tioScreenCols - 1, title, kAttrTitle);
	tioCenterString(tioScreenRows - 1, 0, tioScreenCols - 1,
		"Copyright (c) 1994, 1995 Q Studios Corporation", kAttrTitle);

	tioWindow(1, 0, tioScreenRows - 2, tioScreenCols);

	if ( _grow_handles(kRequiredFiles) < kRequiredFiles )
		ThrowError("Not enough file handles available", ES_ERROR);

	gGamma = BloodINI.GetKeyInt("View", "Gamma", 0);

	tioPrint("Initializing heap and resource system");
	Resource::heap = new QHeap(dpmiDetermineMaxRealAlloc());

	tioPrint("Initializing resource archive");
	gSysRes.Init("BLOOD.RFF", "*.*");
	gGuiRes.Init("GUI.RFF", NULL);

	tioPrint("Initializing mouse");
	if ( !initmouse() )
		tioPrint("Mouse not detected");

	// install our error handler
	prevErrorHandler = errSetHandler(EditorErrorHandler);

	InitEngine();

	Mouse::SetRange(xdim, ydim);

	tioPrint("Loading tiles");
	if (tileInit() == 0)
		ThrowError("ART files not found", ES_ERROR);

	tioPrint("Initializing screen");
	scrInit();

	tioPrint("Installing keyboard handler");
	keyInstall();

	scrCreateStdColors();

	tioPrint("Installing timer");
	timerRegisterClient(ClockStrobe, kTimerRate);
	timerInstall();

	tioPrint("Engaging graphics subsystem...");

	scrSetGameMode();

	scrSetGamma(gGamma);
	scrSetDac(0);

	strcpy(gAnimName, "");

	isModified = FALSE;

	gAnim = (EditAnim *)malloc(sizeof(EditAnim) + kMaxFrames * sizeof(FRAME));
	dassert(gAnim != NULL);

	if ( argc > 1 )
	{
 		strcpy(gAnimName, argv[1]);
		LoadAnim();
	}
	else
	{
		gAnim->frames = 0;
		gAnim->duration = 0;
		gAnim->origin.x = 160;
		gAnim->origin.y = 100;
	}

	EditLoop();

	setvmode(gOldDisplayMode);

	dprintf("Removing timer\n");
	timerRemove();

 	dprintf("uninitengine()\n");
	uninitengine();

	dprintf("All subsystems shut down.  Processing exit functions\n");

	errSetHandler(prevErrorHandler);
}



