#include "weapon.h"
#include "globals.h"
#include "qav.h"
#include "debug4g.h"
#include "actor.h"
#include "db.h"
#include "actor.h"
#include "gameutil.h"
#include "trig.h"
#include "view.h"
#include "options.h"
#include "names.h"
#include "triggers.h"
#include "error.h"
#include "misc.h"
#include "screen.h"
#include "tile.h"

#define kShotgunBarrels		2
#define kVectorsPerBarrel	16
#define kMaxShotgunVectors	(kShotgunBarrels * kVectorsPerBarrel)

enum {
	kVoodooStabChest	= 0,
	kVoodooStabShoulder,
	kVoodooStabEye,
	kVoodooStabGroin,
	kVoodooStabSelf,
};

// Weapon States
enum {
	kFindWeapon			= -1,	// flag to find a loaded weapon

	kForkInitialize		= 0,
	kForkRaise,
	kForkIdle,
	kForkAttack,
	kForkLower,

	kTNTInitialize		= 0,
	kTNTBicOpen,
	kTNTBicIdle,
	kTNTRaise,
	kTNTIdle,
	kTNTLower,

	kCanInitialize		= 0,
	kCanBicOpen,
	kCanBicIdle,
	kCanIdle,
	kCanLower,
	kCanFire1,
	kCanFire2,
	kCanFire3,

	kShotInitialize		= 0,	// indicates first-time reload required
	kShotLoaded,
	kShotRaise,
	kShotIdle1,
	kShotIdle2,
 	kShotIdle3,
	kShotFire1,
	kShotFire2,
	kShotReload,
	kShotLower,

	kFlareInitialize	= 0,
	kFlareRaise,
	kFlareIdle,
	kFlareFire,
	kFlareReload,
	kFlareLower,

	kTommyInitialize	= 0,
	kTommyRaise,
	kTommyIdle,
	kTommyFire1,
	kTommyFire2,
	kTommyFire3,
	kTommyFire4,
	kTommyLower,

	kSpearInitialize	= 0,
	kSpearRaise,
	kSpearIdle1,
	kSpearIdle2,
	kSpearFire,
	kSpearReload,
	kSpearLower,

	kShadowInitialize	= 0,
	kShadowRaise,
	kShadowIdle,
	kShadowFire,
	kShadowLower,

	kBeastInitialize	= 0,
	kBeastRaise,
	kBeastIdle,
	kBeastAttack,
	kBeastLower,

	kDollInitialize	= 0,
	kDollRaise,
	kDollIdleStill,
	kDollIdleMoving,
	kDollStab1,
	kDollStab2,
	kDollStab3,
	kDollStab4,
	kDollStab5,
	kDollLower
};

enum {
	kQAVForkUp = 0,
	kQAVForkIdle,
	kQAVForkStab,
	kQAVForkDown,

	kQAVBicUp,			// raise lighter and open
	kQAVBicFlame,		// light lighter
	kQAVBicIdle,		// burning lighter
	kQAVBicDown,		// lower burning lighter
	kQAVBicDown_NoFlame,	// lower unlit lighter

	kQAVSprayUp,
	kQAVSprayIdle,
	kQAVSprayFire1,
	kQAVSprayFire2,
	kQAVSprayFire3,
	kQAVSprayDown,

	kQAVStickUp,
	kQAVStickIdle,
	kQAVStickLight,
	kQAVStickAim,
	kQAVStickDown,
	kQAVStickThrow,

	kQAVBundleUp,
	kQAVBundleIdle,
	kQAVBundleLight,
	kQAVBundleAim,
	kQAVBundleDown,
	kQAVBundleThrow,

	kQAVFlareUp,
	kQAVFlareIdle,
	kQAVFlareFire,
//	kQAVFlareReload,
	kQAVFlareDown,

	kQAVShotUp,
	kQAVShotIdle1,
	kQAVShotIdle2,
	kQAVShotIdle3,
	kQAVShotFireL,
	kQAVShotFireR,
	kQAVShotReload,
	kQAVShotDown,

	kQAVTommyUp,
	kQAVTommyIdle,
	kQAVTommyFire1,
	kQAVTommyFire2,
 	kQAVTommyFire3,
 	kQAVTommyFire4,
 	kQAVTommyDown,

	kQAVSpearUp,
	kQAVSpearIdle1,
	kQAVSpearIdle2,
	kQAVSpearFire,
	kQAVSpearDown,

	kQAVShadowUp,
	kQAVShadowIdle,
	kQAVShadowFire,
	kQAVShadowDown,

	kQAVBeastUp,
	kQAVBeastIdle,
	kQAVBeastAttack1,
	kQAVBeastAttack2,
 	kQAVBeastAttack3,
 	kQAVBeastAttack4,
	kQAVBeastDown,

	kQAVDollUp,
	kQAVDollIdleStill,
	kQAVDollIdleMoving,
	kQAVDollStab1,
	kQAVDollStab2,
	kQAVDollStab3,
	kQAVDollStab4,
	kQAVDollStab5,
	kQAVDollDown,

	kQAVEnd
};

static char * weaponQAVNames[kQAVEnd] =
{
	"FORKUP",  		//kQAVForkUp = 0,
	"FORKIDLE",     //kQAVForkIdle,
	"PFORK",        //kQAVForkStab,
	"FORKDOWN",     //kQAVForkDown,

	"LITEOPEN",     //kQAVBicUp,
	"LITEFLAM",     //kQAVBicFlame,
	"LITEIDLE",     //kQAVBicIdle,
	"LITECLO2",     //kQAVBicDown,
	"LITECLOS",     //kQAVBicDown_NoFlame,

	"CANPREF",      //kQAVSprayUp,
	"CANIDLE",      //kQAVSprayIdle,
	"CANFIRE1",     //kQAVSprayFire1,
	"CANFIRE2",     //kQAVSprayFire2,
	"CANFIRE3",     //kQAVSprayFire3,
	"CANDOWN",      //kQAVSprayDown,

	"DYNPRE",       //kQAVStickUp,
	"DYNIDLE1",     //kQAVStickIdle,
	"DYNLITA",      //kQAVStickLight,
	"DYNFUSEA",     //kQAVStickAim,
	"DYNDOWN",      //kQAVStickDown,
	"THROW",        //kQAVStickThrow,

	"DYNPRE2",      //kQAVBundleUp,
	"DYNIDLE2",     //kQAVBundleIdle,
	"DYNLITB",      //kQAVBundleLight,
	"DYNFUSEB",     //kQAVBundleAim,
	"DYNDOWN2",     //kQAVBundleDown,
	"THROW",        //kQAVBundleThrow,

	"FLARUP",
	"FLARIDLE",
	"FLARFIRE",
//	"FLARLOAD",
	"FLARDOWN",

	"SHOTUP",		//kQAVShotUp,
	"SHOTI1",       //kQAVShotIdle1,
	"SHOTI2",       //kQAVShotIdle2,
	"SHOTI3",       //kQAVShotIdle3,
	"SHOTF1",       //kQAVShotFireL,
	"SHOTF2",       //kQAVShotFireR,
	"SHOTL1",       //kQAVShotReload,
	"SHOTDOWN",     //kQAVShotDown,

	"TOMUP",        //kQAVTommyUp,
	"TOMIDLE",      //kQAVTommyIdle,
	"TOMFIRE1",     //kQAVTommyFire1,
 	"TOMFIRE2",     //kQAVTommyFire2,
 	"TOMFIRE3",     //kQAVTommyFire3,
	"TOMFIRE4",     //kQAVTommyFire4,
	"TOMDOWN",      //kQAVTommyDown,

	"SGUNUP",		//kQAVSpearUp,
	"SGUNIDL1",		//kQAVSpearIdle1,
	"SGUNIDL2",		//kQAVSpearIdle2,
	"SGUNFIRE",		//kQAVSpearFire,
	"SGUNDOWN",		//kQAVSpearDown,

	"DARKUP",       //kQAVShadowUp,
	"DARKIDLE",     //kQAVShadowIdle,
	"DARKFIRE",     //kQAVShadowFire,
	"DARKDOWN",     //kQAVShadowDown,

	"BSTUP",		//kQAVBeastUp
	"BSTIDLE",		//kQAVBeastIdle,
	"BSTATAK1",     //kQAVBeastAttack1,
	"BSTATAK2",     //kQAVBeastAttack2,
	"BSTATAK3",     //kQAVBeastAttack3,
	"BSTATAK4",     //kQAVBeastAttack4,
	"BSTDOWN",

	"VDUP",			//kQAVDollUp
	"VDIDLE1",		//kQAVDollIdleStill	(PLAYER IS STILL)
	"VDIDLE2",		//kQAVDollIdleMoving	(PLAYER IS WALKING/RUNNING)
	"VDFIRE1",		//kQAVDollStab1
	"VDFIRE2",		//kQAVDollStab2
	"VDFIRE3",		//kQAVDollStab3
	"VDFIRE4",		//kQAVDollStab4
	"VDFIRE5",		//kQAVDollStab5
	"VDDOWN", 		//kQAVDollDown
};


static QAV *weaponQAV[kQAVEnd];


void WeaponInit( void )
{
	// get pointers to all the QAV resources
	for (int i = 0; i < kQAVEnd; i++)
	{
		RESHANDLE hQAV = gSysRes.Lookup(weaponQAVNames[i], ".QAV");
		if (hQAV == NULL)
		{
			dprintf("Could not load %s.QAV\n", weaponQAVNames[i]);
			ThrowError("Missing Weapon QAV file", ES_ERROR);
		}
		weaponQAV[i] = (QAV *)gSysRes.Lock(hQAV);
	}

	dprintf("Preload weapon QAV tiles\n");
	for (i = 0; i < kQAVEnd; i++)
	{
		if (weaponQAV[i] != NULL )
			weaponQAV[i]->Preload();
	}

	// Nick: enumerate the ammo types and use them instead of these calculations
	for (i = 0; i < kMaxPlayers; i++)
	{
		gPlayer[i].weaponAmmo[kPitchfork] = 0;
		gPlayer[i].weaponAmmo[kSprayCan] = kAmmoSprayCan - kAmmoBase;
		gPlayer[i].weaponAmmo[kDynamite] = kAmmoTNTStick - kAmmoBase;
		gPlayer[i].weaponAmmo[kShotgun] = kAmmoShells - kAmmoBase;
		gPlayer[i].weaponAmmo[kTommyGun] = kAmmoBullets - kAmmoBase;
		gPlayer[i].weaponAmmo[kFlareGun] = kAmmoFlares - kAmmoBase;
		gPlayer[i].weaponAmmo[kVoodooDoll] = 0;
		gPlayer[i].weaponAmmo[kSpearGun] = kAmmoSpear - kAmmoBase;
		gPlayer[i].weaponAmmo[kShadowGun] = 0;
	}
}


void WeaponDraw( PLAYER *pPlayer, int shade, int x, int y )
{
	dassert(pPlayer != NULL);
	QAV *pQAV = pPlayer->pWeaponQAV;
	ulong t;

	if ( pQAV == NULL )
		return;

	if (pPlayer->weaponTimer == 0)	// playing idle QAV?
		t = gGameClock % pQAV->duration;
	else
		t = pQAV->duration - pPlayer->weaponTimer;

	pQAV->origin.x = x;
	pQAV->origin.y = y;

	int nPowerRemaining = powerupCheck( pPlayer, kItemLtdInvisibility - kItemBase );
	int nOrient = kOrientScale;

	if ( nPowerRemaining >= (kTimerRate * 8) || nPowerRemaining && ( gGameClock & 32 ) )
	{
		nOrient |= kOrientTranslucent;
		shade = -128;
	}

	pQAV->Draw(t, shade, kQFrameScale);
}


void WeaponPlay( PLAYER *pPlayer )
{
	(void)pPlayer;
	// This will get called by main game loop and should play QAV
	// and allow callbacks to handle weapon firing
}


void WeaponStartQAV( PLAYER *pPlayer, int weaponIndex )
{
	pPlayer->pWeaponQAV = weaponQAV[weaponIndex];
	pPlayer->weaponTimer = pPlayer->pWeaponQAV->duration;
}

#define kAimMaxDist 	M2X(100)
#define kAimMaxAngle	kAngle10
#define kAimMaxSlope	8 << 14			// z units per xy unit

static int WeaponGetAimVector( PLAYER *pPlayer, long *pdx, long *pdy, long *pdz )
{
	dassert( (pPlayer != NULL) && (pdx != NULL) && (pdy != NULL) && (pdz != NULL) );
	SPRITE *pSprite = pPlayer->sprite;

	int x = pSprite->x;
	int y = pSprite->y;
	int z = pSprite->z - pPlayer->weaponAboveZ;

	*pdx = Cos(pSprite->ang) >> 16;
	*pdy = Sin(pSprite->ang) >> 16;
	*pdz = pPlayer->slope;

	int nAimSprite = -1;

	if ( gAutoAim )
	{
		int closest = 0x7FFFFFFF;
		for (short nTarget = headspritestat[kStatDude]; nTarget >= 0; nTarget = nextspritestat[nTarget] )
		{
			SPRITE *pTarget = &sprite[nTarget];

			// don't target yourself!
			if ( pTarget == pSprite )
				continue;

			int dx, dy, dz;

			dx = pTarget->x - x;
			dy = pTarget->y - y;
			dz = pTarget->z - z;

			int dist = qdist(dx, dy);
			if ( dist > closest || dist > kAimMaxDist )
				continue;

			int ang = getangle(dx, dy);
			if ( qabs(((ang - pSprite->ang + kAngle180) & kAngleMask) - kAngle180) > kAimMaxAngle )
				continue;

			int nSlope = divscale(dz, dist, 14);
			if ( qabs(nSlope - pPlayer->slope) > kAimMaxSlope )
				continue;

			if ( cansee(x, y, z, pSprite->sectnum, pTarget->x, pTarget->y, pTarget->z, pTarget->sectnum) )
			{
				closest = dist;

				*pdx = Cos(ang) >> 16;
				*pdy = Sin(ang) >> 16;
				*pdz = nSlope;
				nAimSprite = nTarget;
			}
		}
	}
	return nAimSprite;
}


void WeaponLower( PLAYER *pPlayer, INPUT *pInput )
{
	dprintf("WeaponLower()\n");
	dassert( pPlayer != 0 );

	int nWeapon = pPlayer->weapon;
	int nState = pPlayer->weaponState[nWeapon];

	switch( nWeapon )
	{
		case kVoodooDoll:
			WeaponStartQAV(pPlayer, kQAVDollDown);
			break;

		case kPitchfork:
			WeaponStartQAV(pPlayer, kQAVForkDown);
			break;

		case kFlareGun:
			WeaponStartQAV(pPlayer, kQAVFlareDown);
			break;

		case kSprayCan:
			switch( nState )
			{
				case kCanBicIdle:
					// switch without putting down lighter
					if ( pInput->newWeapon == kDynamite )
					{
						pPlayer->weapon = kDynamite;
						pPlayer->weaponState[kDynamite] = kTNTBicIdle;
						pPlayer->pWeaponQAV = weaponQAV[kQAVBicIdle];
						pInput->newWeapon = 0;
						return;
					}
			        WeaponStartQAV(pPlayer, kQAVBicDown);
					break;

				default:
				    WeaponStartQAV(pPlayer, kQAVSprayDown);
					pPlayer->weaponState[kSprayCan] = kCanBicIdle;
					return;

			}
			break;

		case kDynamite:
			switch( nState )
			{
				case kTNTBicIdle:
					if ( pInput->newWeapon == kSprayCan )
					{
						pPlayer->weapon = kSprayCan;
						pPlayer->weaponState[kSprayCan] = kCanBicIdle;
						pPlayer->pWeaponQAV = weaponQAV[kQAVBicIdle];
						pInput->newWeapon = 0;
						return;
					}
			        WeaponStartQAV(pPlayer, kQAVBicDown);
					break;

				default:
					WeaponStartQAV(pPlayer, kQAVStickDown);
					pPlayer->weaponState[kDynamite] = kTNTBicIdle;
					return;
			}
			break;

		case kShotgun:
	        WeaponStartQAV(pPlayer, kQAVShotDown);
			break;

		case kTommyGun:
	        WeaponStartQAV(pPlayer, kQAVTommyDown);
			break;

		case kSpearGun:
	        WeaponStartQAV(pPlayer, kQAVSpearDown);
			break;

		case kShadowGun:
	        WeaponStartQAV(pPlayer, kQAVShadowDown);
			break;

		case kBeastHands:
	        WeaponStartQAV(pPlayer, kQAVBeastDown);
			break;
	}

	if (pInput->newWeapon == pPlayer->weapon)
		pInput->newWeapon = 0;

	pPlayer->weapon = 0;
}


void WeaponRaise( PLAYER *pPlayer, INPUT *pInput )
{
	dprintf("WeaponRaise()\n");
	dassert( pPlayer != 0 );

	pPlayer->weapon = pInput->newWeapon;
	pInput->newWeapon = 0;

	switch( pPlayer->weapon )
	{
		case kVoodooDoll:
			WeaponStartQAV(pPlayer, kQAVDollUp);
			pPlayer->weaponState[kVoodooDoll] = kDollIdleStill;
			break;

		case kPitchfork:
			WeaponStartQAV(pPlayer, kQAVForkUp);
			pPlayer->weaponState[kPitchfork] = kForkIdle;
			break;

		case kFlareGun:
			WeaponStartQAV(pPlayer, kQAVFlareUp);
			pPlayer->weaponState[kFlareGun] = kFlareIdle;
			break;

		case kSprayCan:
			WeaponStartQAV(pPlayer, kQAVBicUp);
			pPlayer->weaponState[kSprayCan] = kCanBicOpen;
			break;

		case kDynamite:
			WeaponStartQAV(pPlayer, kQAVBicUp);
			pPlayer->weaponState[kDynamite] = kTNTBicOpen;
			break;

		case kShotgun:
			WeaponStartQAV(pPlayer, kQAVShotUp);
			switch (pPlayer->weaponState[kShotgun])
			{
				case kShotInitialize:
				case kShotIdle3:
//					if (pPlayer->ammoCount[ kAmmoShells - kAmmoBase ] > 0)
//					    pPlayer->weaponState[kShotgun] = kShotReload;
//					else
					    pPlayer->weaponState[kShotgun] = kShotIdle3;
					break;
				case kShotIdle2:
					if (gInfiniteAmmo || pPlayer->ammoCount[ kAmmoShells - kAmmoBase ] > 1)
					    pPlayer->weaponState[kShotgun] = kShotReload;
					break;
			}
			break;

		case kTommyGun:
			WeaponStartQAV(pPlayer, kQAVTommyUp);
			pPlayer->weaponState[kTommyGun] = kTommyIdle;
			break;

		case kSpearGun:
			WeaponStartQAV(pPlayer, kQAVSpearUp);
			if ((pPlayer->ammoCount[ kAmmoSpear - kAmmoBase ] > 0) || (pPlayer->ammoCount[ kAmmoHESpears - kAmmoBase ] > 0))
				pPlayer->weaponState[kSpearGun] = kSpearIdle1;
			else
				pPlayer->weaponState[kSpearGun] = kSpearIdle2;
			break;

		case kShadowGun:
			WeaponStartQAV(pPlayer, kQAVShadowUp);
			pPlayer->weaponState[kShadowGun] = kShadowIdle;
			break;

		case kBeastHands:
			WeaponStartQAV(pPlayer, kQAVBeastUp); // no up yet...
			pPlayer->weaponState[kBeastHands] = kBeastIdle;
			break;
	}
}


void WeaponUseVoodoo( PLAYER *pPlayer, INPUT *pInput )
{
	int nStabType = Random(4);
	dassert(nStabType < 4);

	int nSelfSprite = pPlayer->nSprite;
	int z = pPlayer->sprite->z - pPlayer->weaponAboveZ;

	long dx, dy, dz;
	int nAimSprite = WeaponGetAimVector( pPlayer, &dx, &dy, &dz );

	if ( nAimSprite == -1 )
	{
		actDamageSprite(nSelfSprite, pPlayer->nSprite, kDamageStab, 2 << 4);
		nStabType = kVoodooStabSelf;
		if (pPlayer == gMe)
			scrSetMessage("Ouch!");
	}
	else
	{
		SPRITE *pTarget = &sprite[nAimSprite];

		HITINFO hitInfo;
		hitInfo.hitsprite = (short)nAimSprite;
		hitInfo.hitx = sprite[nAimSprite].x;
		hitInfo.hity = sprite[nAimSprite].y;
		hitInfo.hitz = sprite[nAimSprite].z;

		switch( nStabType )
		{
			case kVoodooStabChest:
				actSpawnRicochet(pTarget->sectnum, pTarget->x, pTarget->y, pTarget->z,
					surfType[pTarget->picnum]);
				actDamageSprite(nSelfSprite, nAimSprite, kDamageStab, 2 << 4);
				break;

			case kVoodooStabShoulder:
				actSpawnRicochet(pTarget->sectnum, pTarget->x, pTarget->y, pTarget->z,
					surfType[pTarget->picnum]);
				actDamageSprite(nSelfSprite, nAimSprite, kDamageStab, 4 << 4);
				if ( pTarget->type >= kDudePlayer1 && pTarget->type <= kDudePlayer8 )
				{
					PLAYER *pPlayerHit = &gPlayer[pTarget->type - kDudePlayer1];
					WeaponLower( pPlayerHit, pInput );
				}
				break;
			case kVoodooStabGroin:
				actSpawnRicochet(pTarget->sectnum, pTarget->x, pTarget->y, pTarget->z,
					surfType[pTarget->picnum]);
				actDamageSprite(nSelfSprite, nAimSprite, kDamageStab, 12 << 4);
				break;
			case kVoodooStabEye:
				actSpawnRicochet(pTarget->sectnum, pTarget->x, pTarget->y, pTarget->z,
					surfType[pTarget->picnum]);
				actDamageSprite(nSelfSprite, nAimSprite, kDamageStab, 6 << 4);
				if ( pTarget->type >= kDudePlayer1 && pTarget->type <= kDudePlayer8 )
				{
					PLAYER *pPlayerHit = &gPlayer[pTarget->type - kDudePlayer1];
				   	if (pPlayerHit == gView)	// change to gMe later!
				   		scrDacRelEffect(-96, -96, -96);
				}
				break;
		}
	}
	WeaponStartQAV(pPlayer, kQAVDollStab1 + nStabType);
	pPlayer->weaponState[kVoodooDoll] = kDollIdleStill;
}


#define kMaxForkDist	M2X(2.25)

void WeaponUsePitchfork( PLAYER *pPlayer )
{
	SPRITE *pSprite = pPlayer->sprite;

	long dx, dy, dz;
	WeaponGetAimVector( pPlayer, &dx, &dy, &dz );
	int z = pSprite->z - pPlayer->weaponAboveZ;

	for ( int i = 0; i < 4; i++ )
	{
		// dispersal modifiers here; accuracy in GetAimVector()
		int ddx = BiRandom(2000);
		int ddy = BiRandom(2000);
		int ddz = BiRandom(32000);

		actFireVector(pPlayer->nSprite, z, dx + ddx, dy + ddy, dz + ddz, kMaxForkDist,
			kDamageStab, 3);
	}
}


void WeaponFireShotgun( PLAYER *pPlayer, int nBarrels )
{
	dassert(nBarrels > 0 && nBarrels <= kShotgunBarrels);

	int nVectors = nBarrels * kVectorsPerBarrel;

	SPRITE *pSprite = pPlayer->sprite;

	long dx, dy, dz;
	WeaponGetAimVector( pPlayer, &dx, &dy, &dz );
	int z = pSprite->z - pPlayer->weaponAboveZ;

	for ( int i = 0; i < nVectors; i++ )
	{
		// dispersal modifiers here; accuracy in GetAimVector()
		int ddx = BiRandom(2000);
		int ddy = BiRandom(2000);
		int ddz = BiRandom(32000);

		actFireVector(pPlayer->nSprite, z, dx + ddx, dy + ddy, dz + ddz, 0,
			kDamageBullet, 4);
	}
}


void WeaponFireTommy( PLAYER *pPlayer, int nShots )
{
	SPRITE *pSprite = pPlayer->sprite;

	long dx, dy, dz;
	WeaponGetAimVector( pPlayer, &dx, &dy, &dz );
	int z = pSprite->z - pPlayer->weaponAboveZ;

	for ( int i = 0; i < nShots; i++ )
	{
		// dispersal modifiers here; accuracy in GetAimVector()
		int ddx = BiRandom(500);
		int ddy = BiRandom(500);
		int ddz = BiRandom(8000);

		actFireVector(pPlayer->nSprite, z, dx + ddx, dy + ddy, dz + ddz, 0,
			kDamageBullet, 4 );
	}
}


int WeaponCheckAmmo( PLAYER *pPlayer, int nWeapon, int *pAmmoIndex )
{
	dassert( pAmmoIndex != NULL);
//	dprintf("WeaponCheckAmmo: ");

	int nAmmoCount = 999;	// default ammo count
	int nAmmoIndex = 0;		// default ammo index

	if ( !gInfiniteAmmo )
		switch (nWeapon)
		{
			case kPitchfork:
				break;
			case kVoodooDoll:
				break;
			case kBeastHands:
				break;

			case kSprayCan:
			case kDynamite:
			case kShotgun:
			case kTommyGun:
			case kFlareGun:
			case kSpearGun:
				nAmmoCount = pPlayer->ammoCount[pPlayer->weaponAmmo[nWeapon]];
				break;

			case kShadowGun:
			{
				nAmmoCount = pPlayer->xsprite->health;
				nAmmoIndex = 0;
				//nAmmoCount = pPlayer->ammoCount[ nAmmoIndex ];
				break;
			}

		}
	*pAmmoIndex = nAmmoIndex;
//	dprintf("nAmmoIndex=%i  nAmmoCount = %i\n", nAmmoIndex, nAmmoCount );
	return nAmmoCount;
}


uchar WeaponFindLoaded( PLAYER *pPlayer )
{
	int nAmmoIndex = 0;
	int nNewWeapon = kPitchfork;

	dprintf("WeaponFindLoaded: ");

	if ( pPlayer->hasWeapon[ kShadowGun ] && powerupCheck(pPlayer, kItemInvulnerability - kItemBase) )
		nNewWeapon = kShadowGun;
	else if ( pPlayer->hasWeapon[ kTommyGun ] && WeaponCheckAmmo( pPlayer, kTommyGun, &nAmmoIndex ) > 0 )
		nNewWeapon = kTommyGun;
	else if ( pPlayer->hasWeapon[ kShotgun ] && WeaponCheckAmmo( pPlayer, kShotgun, &nAmmoIndex ) > 0 )
		nNewWeapon = kShotgun;
	else if ( pPlayer->hasWeapon[ kFlareGun ] && WeaponCheckAmmo( pPlayer, kFlareGun, &nAmmoIndex ) > 0 )
		nNewWeapon = kFlareGun;
	else if ( pPlayer->hasWeapon[ kSpearGun ] && WeaponCheckAmmo( pPlayer, kSpearGun, &nAmmoIndex ) > 0 )
		nNewWeapon = kSpearGun;
	else if ( pPlayer->hasWeapon[ kSprayCan ] && WeaponCheckAmmo( pPlayer, kSprayCan, &nAmmoIndex ) > 0 )
		nNewWeapon = kSprayCan;
	else if ( pPlayer->hasWeapon[ kDynamite ] && WeaponCheckAmmo( pPlayer, kDynamite, &nAmmoIndex ) > 0 )
		nNewWeapon = kDynamite;
	else if ( pPlayer->hasWeapon[ kVoodooDoll ] && WeaponCheckAmmo( pPlayer, kVoodooDoll, &nAmmoIndex ) > 0 )
		nNewWeapon = kVoodooDoll;
	else if ( pPlayer->hasWeapon[ kShadowGun ] )
	{
		if ( pPlayer->xsprite->health > 75 )
			nNewWeapon = kShadowGun;
	}

	dprintf("nNewWeapon = %i\n", nNewWeapon);
	return (uchar)nNewWeapon;
}


/*
;[Weapon#]
;Name            = "WeaponNameMax24"
;SelectKey       = [ None | scancode ]
;UseByDefault    = [ TRUE | FALSE ]
;HasByDefault    = [ TRUE | FALSE ]
;RequiresWeapon  = [ None | 0..9 ]
;
;Type            = [ NonWeapon | Melee | Thrown | Projectile | Vector ]
;Type            = NonWeapon
;
;Type            = Melee
;MaxDistance     = [ Infinite | 0..max ]
;
;Type            = Thrown
;MaxDistance     = [ Infinite | 0..max ]
;
;Type            = Projectile
;MaxDistance     = [ Infinite | 0..max ]
;
;Type            = Vector (uses hitscan)
;MaxDistance     = [ Infinite | 0..max ]
;Richochet       = [ TRUE | FALSE ]
;UsesNightScope  = [ TRUE | FALSE ]
;
;Shots           = [ 0..max shots or projectiles ]
;Dispersion      = [ 0..100% ]
;Accuracy        = [ 0..100% ]
;
;AmmoStyle       = [ Infinite, Rechargable, Expendable, Health ]
;AmmoStyle       = Infinite
;
;AmmoStyle       = Rechargable
;RechargeRate    = [ 0 | ammo/second (?) ]
;MaxAmmo         = [ 0..max ]
;
;AmmoStyle       = Expendable
;MaxAmmo         = [ 0..max ]
;AmmoType        = [ 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 ]
;
;AmmoStyle       = Health
;HealthPerShot   = [ 0..usedpershot ]
;
;JamFrequency    = [ 0..100% ]
;JamSingleFire   = [ TRUE | FALSE ]
;JamRapidFire    = [ TRUE | FALSE ]
;
;[Ammo#]
;Name            = "AmmoNameMax24"
;Velocity        = [ ? ]
;DamageTypes     = [ Explosion, Burn, Bullet, Stab, Pummel, Cleave, Ecto]
;DamageRates     = [ 0,         0,    0,      0,    0,      0,      0 ]
*/

void WeaponUpdateState( PLAYER *pPlayer)
{
	SPRITE *pSprite		= pPlayer->sprite;
	XSPRITE *pXSprite	= pPlayer->xsprite;

	int nWeapon = pPlayer->weapon;
	int nState = pPlayer->weaponState[nWeapon];

	switch( nWeapon )
	{
		case kVoodooDoll:
	 		switch ( pXSprite->moveState )
			{
				case kMoveFall:
				case kMoveStill:
				case kMoveStand:
				case kMoveSwim:
				case kMoveFly:
				case kMoveHang:
					nState = pPlayer->weaponState[kVoodooDoll] = kDollIdleStill;
					pPlayer->pWeaponQAV = weaponQAV[kQAVDollIdleStill];
					break;

				case kMoveWalk:
					if ( pPlayer->swayHeight )
					{
						nState = pPlayer->weaponState[kVoodooDoll] = kDollIdleMoving;
						pPlayer->pWeaponQAV = weaponQAV[kQAVDollIdleMoving];
					}
					else
					{
						nState = pPlayer->weaponState[kVoodooDoll] = kDollIdleStill;
						pPlayer->pWeaponQAV = weaponQAV[kQAVDollIdleStill];
					}
					break;

				case kMoveLand:
					nState = pPlayer->weaponState[kVoodooDoll] = kDollIdleMoving;
					pPlayer->pWeaponQAV = weaponQAV[kQAVDollIdleMoving];
					break;

				default:
					pPlayer->pWeaponQAV = NULL;
					break;
			}
			break;

		case kPitchfork:
			pPlayer->pWeaponQAV = weaponQAV[kQAVForkIdle];
			break;

		case kFlareGun:
			if (nState == kFlareIdle)
				pPlayer->pWeaponQAV = weaponQAV[kQAVFlareIdle];
			break;

		case kSpearGun:
			switch ( nState )
			{
				case kSpearIdle1:
					pPlayer->pWeaponQAV = weaponQAV[kQAVSpearIdle1];
					break;

				case kSpearIdle2:
					pPlayer->pWeaponQAV = weaponQAV[kQAVSpearIdle2];
					break;
			}
			break;

		case kTommyGun:
		{
			switch ( nState )
			{
				case kTommyIdle: // is idle
					pPlayer->pWeaponQAV = weaponQAV[kQAVTommyIdle];
					break;
				case kTommyFire1: // was idle, so fire first
				case kTommyFire2: // fire second
				case kTommyFire3: // fire third
	 			case kTommyFire4: // fire fourth and reset state
					if ( gInfiniteAmmo || pPlayer->ammoCount[pPlayer->weaponAmmo[kTommyGun]] > 0 )
					{
						WeaponStartQAV(pPlayer, kQAVTommyFire1 + (nState - kTommyFire1) );
						WeaponFireTommy( pPlayer, 1 );
						if ( !gInfiniteAmmo )
						{
							if ( --pPlayer->ammoCount[pPlayer->weaponAmmo[kTommyGun]] == 0 )
							{
								if ( pPlayer->ammoCount[kAmmoAPBullets - kAmmoBase] )
									pPlayer->weaponAmmo[kTommyGun] = kAmmoAPBullets - kAmmoBase;
								else if ( pPlayer->ammoCount[kAmmoBullets - kAmmoBase] )
									pPlayer->weaponAmmo[kTommyGun] = kAmmoBullets - kAmmoBase;
							}
						}
						if ( nState == kTommyFire4 )
							pPlayer->weaponState[kTommyGun] = kTommyIdle;
						else
							pPlayer->weaponState[kTommyGun]++;
						return;
					}
			}
			break;
		}

		case kSprayCan:
		{
			long dx, dy, dz;
			WeaponGetAimVector( pPlayer, &dx, &dy, &dz );

			switch( nState )
			{
				case kCanBicOpen:
					WeaponStartQAV(pPlayer, kQAVBicFlame);
					pPlayer->weaponState[kSprayCan] = kCanBicIdle;
					break;

				case kCanBicIdle:
					if ( pPlayer->ammoCount[pPlayer->weaponAmmo[kSprayCan]] > 0 )
					{
						WeaponStartQAV(pPlayer, kQAVSprayUp);
						pPlayer->weaponState[kSprayCan] = kCanIdle;
					}
					else
						pPlayer->pWeaponQAV = weaponQAV[kQAVBicIdle];
					break;

				case kCanIdle:
					pPlayer->pWeaponQAV = weaponQAV[kQAVSprayIdle];
					break;

				case kCanFire1: // was idle, so fire first
				case kCanFire2: // fire second
				case kCanFire3: // fire third and reset state
						WeaponStartQAV(pPlayer, kQAVSprayFire1 + (nState - kCanFire1) );
						playerFireMissile(pPlayer, dx, dy, dz, kMissileSprayFlame);
						if ( !gInfiniteAmmo )
							pPlayer->ammoCount[ kAmmoSprayCan - kAmmoBase ] -= kFrameTicks;
						if ( nState == kCanFire3 )
						{
							if ( pPlayer->ammoCount[ kAmmoSprayCan - kAmmoBase ] <= 0 )
							{
								pPlayer->ammoCount[ kAmmoSprayCan - kAmmoBase ] = 0;
								// drop the can, and idle the lighter
							    WeaponStartQAV(pPlayer, kQAVSprayDown);
								pPlayer->weaponState[kSprayCan] = kCanBicIdle;
							}
							else
								pPlayer->weaponState[kSprayCan] = kCanIdle;
						}
						else
							pPlayer->weaponState[kSprayCan]++;
						return;
			}
			break;
		}

		case kDynamite:
			switch( nState )
			{
				case kTNTBicOpen:
					WeaponStartQAV(pPlayer, kQAVBicFlame);
					pPlayer->weaponState[nWeapon] = kTNTBicIdle;
					break;

				case kTNTBicIdle:
					if ( pPlayer->ammoCount[ kAmmoTNTStick - kAmmoBase ] > 0)
					{
						WeaponStartQAV(pPlayer, kQAVStickUp);
						pPlayer->weaponState[kDynamite] = kTNTIdle;
					}
					else
						pPlayer->pWeaponQAV = weaponQAV[kQAVBicIdle];
					break;

				case kTNTIdle:
					pPlayer->pWeaponQAV = weaponQAV[kQAVStickIdle];
					break;
			}
			break;

		case kShotgun:
			switch ( nState )
			{
				case kShotIdle1:
					pPlayer->pWeaponQAV = weaponQAV[kQAVShotIdle1];
					break;

				case kShotIdle2:
					pPlayer->pWeaponQAV = weaponQAV[kQAVShotIdle2];
					break;

				case kShotIdle3:
					if (gInfiniteAmmo || pPlayer->ammoCount[ kAmmoShells - kAmmoBase ] > 0)
					    pPlayer->weaponState[kShotgun] = kShotReload;
					pPlayer->pWeaponQAV = weaponQAV[kQAVShotIdle3];
					break;
			}
			break;

		case kShadowGun:
			pPlayer->pWeaponQAV = weaponQAV[kQAVShadowIdle];
			break;

		case kBeastHands:
			pPlayer->pWeaponQAV = weaponQAV[kQAVBeastIdle];
			break;
	}
}

void WeaponProcess( PLAYER *pPlayer, INPUT *pInput )
{
	// decrement weapon ready timer
	pPlayer->weaponTimer -= kFrameTicks;

	if (pPlayer->weaponTimer <= 0)
		pPlayer->weaponTimer = 0;

	// return if weapon is busy
	if (pPlayer->weaponTimer != 0)
		return;

	pPlayer->pWeaponQAV = NULL;

	int nWeapon = pPlayer->weapon;

	// out of ammo check, need to find a better way to do this
	// pPlayer->weapon is set to kMaxWeapons in the fire processing below
	if ( pPlayer->weaponState[nWeapon] == kFindWeapon )
	{
		dprintf("Out of ammo: looking for a new weapon\n");
		pPlayer->weaponState[nWeapon] = 0;
		pInput->newWeapon = WeaponFindLoaded( pPlayer );
	}

	// weapon select
	if ( pInput->newWeapon )
	{
		if ( pPlayer->xsprite->health == 0 || !pPlayer->hasWeapon[ pInput->newWeapon ] )
		{
			pInput->newWeapon = 0;
			return;
		}

		if (pPlayer->weapon == 0) 	// raise the new weapon
			WeaponRaise(pPlayer, pInput);
		else
			WeaponLower(pPlayer, pInput);	// lower the old weapon
		return;
	}

	if ( (nWeapon = pPlayer->weapon) == 0 )
		return;		// no weapon drawn, do nothing

    // auto reload test
	switch ( nWeapon )
	{
		case kShotgun:
			if ( pPlayer->weaponState[nWeapon] == kShotReload )
			{
				if (!gInfiniteAmmo && pPlayer->ammoCount[ kAmmoShells - kAmmoBase ] == 0)
					pPlayer->weaponState[kShotgun] = kShotIdle3;
				else
				{
			    	WeaponStartQAV(pPlayer, kQAVShotReload);
					if (gInfiniteAmmo || pPlayer->ammoCount[ kAmmoShells - kAmmoBase ] > 1)
						pPlayer->weaponState[kShotgun] = kShotIdle1;
					else
						pPlayer->weaponState[kShotgun] = kShotIdle2;
				}
				return;
			}
			break;
	}

	// fire key
	if ( pInput->buttonFlags.shoot )
	{
		int nAmmoIndex;
		int nAmmoCount = WeaponCheckAmmo(pPlayer, nWeapon, &nAmmoIndex);

		if ( nAmmoCount > 0 )
		{
			long dx, dy, dz;
			WeaponGetAimVector( pPlayer, &dx, &dy, &dz );

			// firing weapon (primary)
			switch ( nWeapon )
			{
				case kVoodooDoll:
					WeaponUseVoodoo( pPlayer, pInput );
					return;

				case kFlareGun:
					if ( gInfiniteAmmo || (pPlayer->ammoCount[ kAmmoFlares - kAmmoBase ] > 0)
					|| ( pPlayer->ammoCount[ kAmmoHEFlares - kAmmoBase ] > 0) || ( pPlayer->ammoCount[ kAmmoStarFlares - kAmmoBase ] > 0))
					{
						WeaponStartQAV(pPlayer, kQAVFlareFire);
					    pPlayer->weaponState[kFlareGun] = kFlareIdle;
						playerFireMissile( pPlayer, dx, dy, dz, kMissileFlare);
		 				if ( !gInfiniteAmmo )
						{
							if ( pPlayer->ammoCount[ kAmmoStarFlares - kAmmoBase ] > 0 )
								pPlayer->ammoCount[ kAmmoStarFlares - kAmmoBase ] -= 1;
							else if ( pPlayer->ammoCount[ kAmmoHEFlares - kAmmoBase ] > 0 )
								pPlayer->ammoCount[ kAmmoHEFlares - kAmmoBase ] -= 1;
							else if ( pPlayer->ammoCount[ kAmmoFlares - kAmmoBase ] > 0 )
								pPlayer->ammoCount[ kAmmoFlares - kAmmoBase ] -= 1;
						}
					}
					return;

				case kPitchfork:
					WeaponStartQAV(pPlayer, kQAVForkStab);
					pPlayer->weaponState[kPitchfork] = kForkIdle;
					WeaponUsePitchfork( pPlayer );
					return;

				case kTommyGun:
					if ( pPlayer->weaponState[kTommyGun] == kTommyIdle )
						pPlayer->weaponState[kTommyGun] = kTommyFire1;
					break; // fall through to WeaponUpdateState

			    case kSpearGun:
					if ( gInfiniteAmmo || (pPlayer->ammoCount[ kAmmoSpear - kAmmoBase ] > 0)
					|| ( pPlayer->ammoCount[ kAmmoHESpears - kAmmoBase ] > 0) )
					{
						if ( pPlayer->ammoCount[ kAmmoHESpears - kAmmoBase ] > 0 )
							pPlayer->ammoCount[ kAmmoHESpears - kAmmoBase ] -= 1;
						else
							pPlayer->ammoCount[ kAmmoSpear - kAmmoBase ] -= 1;
						playerFireMissile(pPlayer, dx, dy, dz, kMissileSpear);
						WeaponStartQAV(pPlayer, kQAVSpearFire);
						pPlayer->weaponState[kSpearGun] = kSpearIdle1;
					}
					else
						pPlayer->weaponState[kSpearGun] = kSpearIdle2;
					return;

				case kSprayCan:
					if ( pPlayer->weaponState[kSprayCan] == kCanIdle )
					{
						if ( gInfiniteAmmo || (pPlayer->ammoCount[ kAmmoSprayCan - kAmmoBase ] > 0))
							pPlayer->weaponState[kSprayCan] = kCanFire1;
					}
					break; // fall through to WeaponUpdateState

				case kDynamite:
					if ( gInfiniteAmmo || (pPlayer->ammoCount[ kAmmoTNTStick - kAmmoBase ] > 0))
					{
						WeaponStartQAV( pPlayer, kQAVStickThrow );
						playerFireThing(pPlayer, kThingTNTStick);

						if ( !gInfiniteAmmo )
							pPlayer->ammoCount[ kAmmoTNTStick - kAmmoBase ] -= 1;

						if ( pPlayer->ammoCount[ kAmmoTNTStick - kAmmoBase ] == 0 )
						{
							// drop empty hand, and idle the lighter
						    WeaponStartQAV(pPlayer, kQAVBicIdle);
							pPlayer->weaponState[kDynamite] = kTNTBicIdle;
						}
					}
					return;

			    case kShotgun:
					switch ( pPlayer->weaponState[kShotgun] )
					{
						case kShotIdle1:
							WeaponStartQAV(pPlayer, kQAVShotFireL);
							pPlayer->weaponState[kShotgun] = kShotIdle2;
							if ( !gInfiniteAmmo )
								pPlayer->ammoCount[ kAmmoShells - kAmmoBase ] -= 1;
							WeaponFireShotgun( pPlayer, 1 );
							return;

						case kShotIdle2:
							WeaponStartQAV(pPlayer, kQAVShotFireR);
						    pPlayer->weaponState[kShotgun] = kShotReload;
							if ( !gInfiniteAmmo )
								pPlayer->ammoCount[ kAmmoShells - kAmmoBase ] -= 1;
							WeaponFireShotgun( pPlayer, 1 );
							return;
					}
					return;

				case kShadowGun:
					WeaponStartQAV(pPlayer, kQAVShadowFire);
				    pPlayer->weaponState[kShadowGun] = kShadowIdle;
					playerFireMissile( pPlayer, dx, dy, dz, kMissileEctoSkull);
	 				if ( !gInfiniteAmmo )
					{
						actDamageSprite(pPlayer->nSprite, pPlayer->nSprite, kDamageSpirit, 10 << 4);
					}
					return;

				case kBeastHands:
					WeaponStartQAV(pPlayer, kQAVBeastAttack1 + Random(4));
					pPlayer->weaponState[kBeastHands] = kBeastIdle;
					return;
			}
		}
		else
		{
			dprintf("Primary out of ammo: setting weapon state to kFindWeapon\n");
			pPlayer->weaponState[nWeapon] = kFindWeapon;
			return;
		}
	}

	// alternate fire key
	if ( pInput->buttonFlags.shoot2 )
	{
		int nAmmoIndex;
		int nAmmoCount = WeaponCheckAmmo(pPlayer, nWeapon, &nAmmoIndex);

		if ( nAmmoCount > 0 )
		{
			long dx, dy, dz;
			WeaponGetAimVector( pPlayer, &dx, &dy, &dz );

			// firing weapon (alternate)
			switch ( nWeapon )
			{
				case kSpearGun:
					if ( gInfiniteAmmo || (pPlayer->ammoCount[ kAmmoSpear - kAmmoBase ] > 0)
					|| ( pPlayer->ammoCount[ kAmmoHESpears - kAmmoBase ] > 0) )
					{
						playerFireMissile(pPlayer, dx, dy, dz, kMissileSpear);
						if ( pPlayer->ammoCount[ kAmmoHESpears - kAmmoBase ] > 0 )
							pPlayer->ammoCount[ kAmmoHESpears - kAmmoBase ] -= 1;	// CHANGE TO <= 6
						else
							pPlayer->ammoCount[ kAmmoSpear - kAmmoBase ] -= 1;		// CHANGE TO <= 6
						WeaponStartQAV(pPlayer, kQAVSpearFire);
						pPlayer->weaponState[kSpearGun] = kSpearIdle1;
					}
					else
						pPlayer->weaponState[kSpearGun] = kSpearIdle2;
					return;

				case kPitchfork:
					playerFireMissile(pPlayer, dx, dy, dz, kMissileFireball);
					pPlayer->weaponTimer = 20;
					return;

				case kFlareGun:
					return;

				case kTommyGun:
					return;

				case kSprayCan:
					break;

				case kDynamite:
					break;

				case kShotgun:
				{
					switch ( pPlayer->weaponState[kShotgun] )
					{
						case kShotIdle1:
							WeaponStartQAV(pPlayer, kQAVShotFireR);
							pPlayer->weaponState[kShotgun] = kShotReload;
							if ( !gInfiniteAmmo )
								pPlayer->ammoCount[ kAmmoShells - kAmmoBase ] -= 2;
							WeaponFireShotgun( pPlayer, 2 );
							return;
						case kShotIdle2:
							WeaponStartQAV(pPlayer, kQAVShotFireR);
							pPlayer->weaponState[kShotgun] = kShotReload;
							if ( !gInfiniteAmmo )
								pPlayer->ammoCount[ kAmmoShells - kAmmoBase ] -= 1;
							WeaponFireShotgun( pPlayer, 1 );
							return;
					}
					break;
				}
			}
		}
		else
		{
			dprintf("Alternate out of ammo: setting weapon state to kFindWeapon\n");
			pPlayer->weaponState[nWeapon] = kFindWeapon;
			return;
		}
	}

	WeaponUpdateState(pPlayer);
}
