/*******************************************************************************
	FILE:			SHPLAY.CPP

	DESCRIPTION:	Sonic Holography 3D sound player

	AUTHOR:			Peter M. Freese
	CREATED:		11-09-95
	COPYRIGHT:		Copyright (c) 1995 Q Studios Corporation
*******************************************************************************/
#include <conio.h>
#include <dos.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string.h>
#include <pcx.h>
#include <palette.h>
#include <io.h>
#include <fcntl.h>

#include "typedefs.h"
#include "debug4g.h"
#include "misc.h"
#include "task_man.h"
#include "fx_man.h"
#include "music.h"
#include "key.h"
#include "getopt.h"
#include "inifile.h"
#include "textio.h"
#include "usrhooks.h"


#define kAspectAdjustX	1.2

#define kPi				3.141592654
#define kPi2			(kPi * 2)

#define kAngle360		kPi2
#define kAngle180		(kAngle360 / 2)
#define kAngle90		(kAngle360 / 4)
#define kAngle60		(kAngle360 / 6)
#define kAngle45		(kAngle360 / 8)
#define kAngle30		(kAngle360 / 12)
#define kAngle20		(kAngle360 / 18)
#define kAngle15		(kAngle360 / 24)
#define kAngle10		(kAngle360 / 36)
#define kAngle5			(kAngle360 / 72)

#define kTimerRate		30
#define kSampleRate		11110
#define kIntraAuralTime	0.0006
#define kVolScale		50


struct MENU_LINE
{
	int id;
	char *name;
	char *hint;
};


IniFile config("SHPLAY.INI");

MENU_LINE SoundCardMenu[] =
{
	{ -1,				"None",					"No sound card" },
	{ SoundBlaster,		"SoundBlaster",			"Sound Blaster, SB Pro, SB 16, or compatible" },
	{ ProAudioSpectrum,	"Pro Audio Spectrum",	"Media Vision Pro Audio Sprectrum" },
	{ SoundMan16,		"Sound Man 16",			"Logitech Sound Man 16" },
//	{ Adlib,			"Adlib" },
//	{ GenMidi,			"General Midi" },
//	{ SoundCanvas,		"Sound Canvas" },
	{ Awe32,			"Awe 32",				"Sound Blaster Awe 32" },
//	{ WaveBlaster,		"WaveBlaster" },
	{ SoundScape,		"SoundScape",			"Ensoniq Sound Scape" },
	{ UltraSound,		"Gravis UltraSound",	"Advanced Gravis UltraSound" },
	{ SoundSource,		"Disney Sound Source",	"Disney Sound Source" },
	{ TandySoundSource,	"Tandy Sound Source",	"Tandy Sound Source" },
	{ PC,				"PC speaker",			"PC Internal Speaker" },
};

MENU_LINE BitMenu[] =
{
	{ 8,	"8-bit",	"8-bit low quality mixing" },
	{ 16,	"16-bit",	"16-bit high quality mixing" },
};

MENU_LINE ChannelMenu[ 2 ] =
{
	{ 1,	"Mono",		"Mono sound, no stereo imaging" },
	{ 2,	"Stereo",	"Stereo sound, full stereo imaging" },
};

MENU_LINE VoiceMenu[] =
{
	{ 8,	"8",		"8 voice mixing" },
	{ 16,	"16",		"16 voice mixing" },
	{ 24,	"24",		"24 voice mixing" },
	{ 32,	"32",		"32 voice mixing" },
};

MENU_LINE MixRateMenu[] =
{
	{ 11110,	"11 KHz",	"Mix at 11 KHz" },
	{ 22220,	"22 KHz",	"Mix at 22 KHz" },
	{ 44440,	"44 KHz",	"Mix at 44 KHz" },
};

char soundName[11][80];
char *soundData[11];
int soundLen[11];

BYTE *background = NULL, *stage, *video = (BYTE *)0xA0000;

#define kNumBuffers 34
char *buffers[kNumBuffers];

int gSoundCard;
int gMixBits;
int gChannels;
int gMixVoices;
int gMixRate;
int gFXVolume;

int fHolography = TRUE;

double gPinnaAngle = kAngle45;
double nSat = 0.8;

static int lVoice, rVoice;
static int lPhase, rPhase, lVel, rVel, lVol, rVol;
int x, y, xold, yold;

volatile short gMouseX, gMouseY;
BOOL gMouseDragging = FALSE;
int gMouseShown = FALSE;


/*---------------------------------------------------------------------
   Function: USRHOOKS_GetMem

   Allocates the requested amount of memory and returns a pointer to
   its location, or NULL if an error occurs.  NOTE: pointer is assumed
   to be dword aligned.
---------------------------------------------------------------------*/

int USRHOOKS_GetMem( void **ptr, unsigned long size )
{
	void *memory;

	memory = malloc( size );
	if ( memory == NULL )
	{
		return( USRHOOKS_Error );
	}

	*ptr = memory;
	return( USRHOOKS_Ok );
}


/*---------------------------------------------------------------------
   Function: USRHOOKS_FreeMem

   Deallocates the memory associated with the specified pointer.
---------------------------------------------------------------------*/

int USRHOOKS_FreeMem( void *ptr )
{
	if ( ptr == NULL )
    {
		return( USRHOOKS_Error );
    }

	free( ptr );
	return( USRHOOKS_Ok );
}


int vidGetMode( void );
#pragma aux vidGetMode =\
	"mov	ah,0x0F",\
    "int    0x10,"\
    "and	eax,0xFF",\

int vidSetMode(int);
#pragma aux vidSetMode =\
	"int 	0x10",\
	parm [eax]\

void mouGetPos( void );
#pragma aux mouseGetPos =\
	"mov	ax,03h",\
	"int 	33h",\
	"shr	cx,1",\
	"mov	[gMouseX],cx",\
	"mov	[gMouseY],dx",\
	modify [eax ecx edx ebx]\

char mouseGetButtons( void );
#pragma aux mouseGetButtons =\
	"mov ax,03h",\
	"int 33h",\
	"xor eax,eax",\
	"mov al,bl",\
	modify [eax ebx ecx edx]


inline void pixel( BYTE *buffer, int x, int y, char c )
{
   	*(char *)(buffer + x + y * 320) = c;
}


/*---------------------------------------------------------------------
   Function: plot

   XORs a 4x4 block onto the screen.
---------------------------------------------------------------------*/

void plot( BYTE *buffer, int x, int y, char c )
{
   	int i;
	long d = (c << 24) | (c << 16) | (c << 8) | c;
	long *v = (long *)(buffer + x + y * 320);

	for( i = 0; i < 4; i++)
	{
		*v = d;
		v += 80;
	}
}


double Vol3d(double angle, double vol)
{
	if (fHolography)
		return vol * (1.0 - (1.0 - cos(angle)) * nSat / 2.0);
	else
		return vol * (cos(angle) * 0.5 + 0.5);
}


int Dist3d(int x, int y)
{
	return sqrt(x * x + y * y);
}


double CalcAngle( int x, int y )
{
	if (y < 0)
		return atan((double)x / (double)y);

	if (y > 0)
		return kPi + atan((double)x / (double)y);

	if (x < 0)
		return kPi / 2;

	return -kPi / 2;
}

/*******************************************************************************

Here is the equation for determining the frequency shift of a sound:

	                    
	             v + vL 
	    fL = fS   
	             v - vS 
	                    

Where:

	fL	= frequency as heard by the listener
	fS	= frequency of the source
	v	= velocity of sound
	vL	= velocity of listener
	vS	= velocity of source

Speed of sound = 343 m/s = 1126 ft/s.

*******************************************************************************/


#define kSoundVelocity	((double)kSampleRate / kTimerRate)
int Freq( int velocity )
{
	return kSampleRate * kSoundVelocity / (ClipLow(kSoundVelocity - velocity, 1));
}


// ear distance (in samples)
#define kEarLX (int)(-kIntraAuralTime / 2 * kSampleRate)
#define kEarRX (int)(+kIntraAuralTime / 2 * kSampleRate)

void MoveSource( void )
{
	int lDist, rDist;
	double angle, vol;

	xold = x;
	yold = y;
	x = gMouseX - 160;
	y = gMouseY - 100;

	angle = CalcAngle(x, y);

	if (fHolography)
	{
		lDist = Dist3d(x - kEarLX, y);
		lVel = Dist3d(xold - kEarLX, yold) - lDist;
		vol = kVolScale * 255 / ClipLow(lDist, 1);
		lVol = Vol3d(angle - gPinnaAngle, vol);
		lPhase = lDist;

		rDist = Dist3d(x - kEarRX, y);
		rVel = Dist3d(xold - kEarRX, yold) - rDist;
		vol = kVolScale * 255 / ClipLow(rDist, 1);
		rVol = Vol3d(angle + gPinnaAngle, vol);
		rPhase = rDist;
	}
	else
	{
		vol = kVolScale * 255 / ClipLow(Dist3d(x, y), 1);
		if (vol > 255) vol = 255;

		lVol = Vol3d(angle - kAngle90, vol);
		rVol = Vol3d(angle + kAngle90, vol);

		lVel = rVel = 0;
		lPhase = rPhase = 0;
	}
}


/*---------------------------------------------------------------------
   Function: PlaySound

   Starts playback of a sound.
---------------------------------------------------------------------*/

void PlaySound( int num )
{
	int nLength;
	char *vl, *vr;
	int id;

	if ( mouseGetButtons() )
		gMouseDragging = TRUE;

	nLength = soundLen[num];

	vl = (char *)malloc(nLength + lPhase);
	if (vl)
	{
		memset(vl, 128, lPhase);
		memcpy(&vl[lPhase], soundData[num], nLength);
	}

	vr = (char *)malloc(nLength + rPhase);
	if (vr)
	{
		memset(vr, 128, rPhase);
		memcpy(&vr[rPhase], soundData[num], nLength);
	}

	_disable();
	if (vl)
	{
		for (id = 0; buffers[id] != 0; id++);
		lVoice = FX_PlayRaw( vl, nLength, Freq(lVel), 0, lVol, lVol, 0, 1, id);
		if ( lVoice > FX_Ok )
		{
			buffers[id] = vl;
		}
		else
			free(vl);
	}

	if (vr)
	{
		for (id = 0; buffers[id] != 0; id++);
		rVoice = FX_PlayRaw( vr, nLength, Freq(rVel), 0, rVol, 0, rVol, 1, id);
		if ( rVoice > FX_Ok )
		{
			buffers[id] = vr;
		}
		else
			free(vr);
	}
	_enable();
}


void UpdateSound( void )
{
	if (lVoice > FX_Ok)
	{
		FX_SetPan(lVoice, lVol, lVol, 0);
		FX_SetFrequency(lVoice, Freq(lVel));
	}

	if (rVoice > FX_Ok)
	{
		FX_SetPan(rVoice, rVol, 0, rVol);
		FX_SetFrequency(rVoice, Freq(rVel));
	}
}


/*---------------------------------------------------------------------
   Function: DrawMark

   Displays the position the voice will appear in relation to the
   listener.
---------------------------------------------------------------------*/

void DrawMark( int fErase )
{
	static int  lastx = 0;
	static int  lasty = 0;
	static char c = 16;
	int newx, newy;
	BYTE *v;
	int i, j;

	newx = gMouseX - 1;
	if (newx < 0) newx = 0;
	if (newx > 317) newx = 317;

	newy = gMouseY - 1;
	if (newy < 0) newy = 0;
	if (newy > 197) newy = 197;

	// clear the old one
	if (fErase)
	{
		v = video + lastx + lasty * 320;
		for( i = 0; i < 3; i++)
		{
			for (j = 0; j < 3; j++)
				*v++ ^= c;
			v += 320 - 3;
		}
	}

	if (++c == 32)
		c = 16;

	// draw the new one
	v = video + newx + newy * 320;
	for( i = 0; i < 3; i++)
	{
		for (j = 0; j < 3; j++)
			*v++ ^= c;
		v += 320 - 3;
	}

	lastx = newx;
	lasty = newy;
}



/*---------------------------------------------------------------------
   Function: DrawDemoScreen

   Sets up our information screen for the user.  Displays last error,
   what sounds are playing, and the volume levels.
---------------------------------------------------------------------*/

void DrawDemoScreen( void )
{
	int x;
	double a;
	double lSat, rSat, nCos, nSin;

	// make sure the mouse doesn't get drawn while we're setting up
	gMouseShown = FALSE;

	// clear screen
	memcpy(stage, background, 320 * 200);

	for ( a = 0.0; a < kAngle360; a += kAngle5 )
	{
		if (fHolography)
		{
			lSat = Vol3d(a - gPinnaAngle, 25);
			rSat = Vol3d(a + gPinnaAngle, 25);
		}
		else
		{
			lSat = Vol3d(a - kAngle90, 25);
			rSat = Vol3d(a + kAngle90, 25);
		}

		nSin = sin(a);
		nCos = cos(a);

		pixel(stage, 160 - nSin * lSat, 100 - nCos * lSat, 8);
		pixel(stage, 160 - nSin * rSat, 100 - nCos * rSat, 8);
		pixel(stage, 160 - nSin * (lSat + rSat) * kAspectAdjustX, 100 - nCos * (lSat + rSat), 14);
	}

	// draw the holography toggle
	plot( stage, 135, 6, fHolography ? 14 : 8 );

	// Draw volume display
   	for( x = 0; x < 10; x++ )
	{
    	plot( stage, 240 + x * 7, 6, gFXVolume > x ? 10 : 8 );
	}

	// draw the stage
	memcpy(video, stage, 320 * 200);

	// draw the initial cursor
	DrawMark(0);

	// okay, we're ready for the mouse
	gMouseShown = TRUE;
}



/*---------------------------------------------------------------------
   Function: TimerFunc
---------------------------------------------------------------------*/

void TimerFunc( TASK * /* task */ )
{
	mouseGetPos();
	if ( gMouseDragging && !mouseGetButtons() )
		gMouseDragging = FALSE;

	if (gMouseShown)
	{
		DrawMark(1);
		MoveSource();

		if ( gMouseDragging )
		{
			UpdateSound();
		}
	}
}


/*---------------------------------------------------------------------
   Function: SoundCallback

   When a sound fx stops playing, this function is called and updates the
   displayed number of voices using that sample.
---------------------------------------------------------------------*/

void SoundCallback(unsigned long id)
{
	if (buffers[id] != NULL)
	{
   		free(buffers[id]);
		buffers[id] = NULL;
	}
	else
		dprintf("SoundCall() called for non-playing sample\n");
}


/*---------------------------------------------------------------------
   Function: CheckKeys

   Checks which keys are hit and responds appropriately.
---------------------------------------------------------------------*/

BYTE CheckKeys( void )
{
	BYTE key;

	key = keyGet();
	switch( key )
	{
		case KEY_SPACE:
	        PlaySound(0);
	        break;

		case KEY_1:
		case KEY_2:
		case KEY_3:
		case KEY_4:
		case KEY_5:
		case KEY_6:
		case KEY_7:
		case KEY_8:
		case KEY_9:
		case KEY_0:
	        PlaySound(key - KEY_1 + 1);
	        break;

		case KEY_MINUS:
			if (gFXVolume > 0)
			{
				gFXVolume--;
				FX_SetVolume(gFXVolume * 255 / 10);
				plot( video, 240 + gFXVolume * 7, 6, 8 );
			}
			break;

		case KEY_EQUALS:
			if (gFXVolume < 10)
			{
				plot( video, 240 + gFXVolume * 7, 6, 10 );
				gFXVolume++;
				FX_SetVolume(gFXVolume * 255 / 10);
			}
			break;

		case KEY_LBRACE:
			if ( gPinnaAngle > 0 )
			{
				gPinnaAngle -= kAngle5;
				DrawDemoScreen();
			}
			break;

		case KEY_RBRACE:
			if ( gPinnaAngle < kAngle90 )
			{
				gPinnaAngle += kAngle5;
				DrawDemoScreen();
			}
			break;

		case KEY_COMMA:
			if ( nSat > 0 )
			{
				nSat -= 0.05;
				DrawDemoScreen();
			}
			break;

		case KEY_PERIOD:
			if ( nSat < 1.0 )
			{
				nSat += 0.05;
				DrawDemoScreen();
			}
			break;

		case KEY_K:
        	 FX_StopAllSounds();
         	break;

      	case KEY_S:
        	fHolography = !fHolography;
         	DrawDemoScreen();
         	break;
	}
	return key;
}


/*---------------------------------------------------------------------
   Function: DoDemo

   Main loop for our demo.  Sets up a timer so that marker moves at
   same rate on any computer.
---------------------------------------------------------------------*/
void DoDemo( void )
{
	TASK *taskPtr = NULL;
	int   Timer;
	int   i;
	int nOldMode;
	int width = 320, height = 200;
	PALETTE palette;

	if (PCX_OKAY != ReadPCX("screen.pcx", palette, &width, &height, &background))
	{
		printf("Error reading screen.pcx\n");
		exit(1);
	}

	stage = (BYTE *)malloc(320 * 200);
	if (!stage)
	{
		printf("Couldn't allocate stage.\n");
		exit(1);
	}

	nOldMode = vidGetMode();
	vidSetMode(0x13);

	gSetDACRange(0, 256, palette);

	taskPtr = TS_ScheduleTask( TimerFunc, kTimerRate, 1, &Timer );
	TS_Dispatch();
	Timer = 0;

	// set the current volume of the sound fx
	FX_SetVolume( gFXVolume * 255 / 10 );

	// Set up my keyboard handler.
	keyInstall();

	for (i = 0; i < kNumBuffers; i++)
		buffers[i] = NULL;

	// Set up our fx callback so we can display the sounds that are playing
	FX_SetCallBack(SoundCallback);

	DrawDemoScreen();

	while ( CheckKeys() != KEY_ESC );

	// Stop the sound fx engine.
	FX_StopAllSounds();

	// Restore the system keyboard handler
	keyRemove();

	vidSetMode(nOldMode);

	// Terminate my timer.  Failure to do so could be fatal!!!
	TS_Terminate(taskPtr);
	TS_Shutdown();
}


/*---------------------------------------------------------------------
   Function: SelectMenu

   Draws the sound card selection screen and waits for the user to
   select a card.
---------------------------------------------------------------------*/

int SelectMenu( char *title, MENU_LINE list[], int nListItems, int nDefault )
{
	int cols = 0, rows = nListItems;
	int pos = 0;

//	cols = strlen(title);
	(void)title;

	// find widest menu item text
	for ( int i = 0; i < nListItems; i++)
	{
		int n = strlen(list[i].name);
		if (n > cols)
			cols = n;
	}

	cols += 5;
	rows += 3;

	char *saveUnder = (char *)malloc(rows * cols * 2);

	int left = (tioScreenCols - cols) / 2;
	int top = (tioScreenRows - rows) / 2;
	tioSaveWindow(saveUnder, top, left, rows, cols);

	tioFrame(top, left, rows - 1, cols - 1, kFrameSingle, kColorBlue * 16 | kColorCyan);
	tioFill(top + 1, left + 1, rows - 3, cols - 3, ' ', kColorBlue * 16 | kColorYellow);
	tioFillShadow(top + rows - 1, left + 1, 1, cols - 1);
	tioFillShadow(top + 1, left + cols - 1, rows - 2, 1);

	tioSetAttribute(kColorBlue * 16 | kColorYellow);
	for( i = 0; i < nListItems; i++ )
	{
		tioSetPos(top + i + 1, left + 2);
		tioPrint("%s", list[i].name);

		if ( list[i].id == nDefault )
			pos = i;
	}

	int ch = 0;
	while( ch != 13 && ch != 32 )
	{
		tioFillAttr(top + pos + 1, left + 1, 1, cols - 3, kColorLightGray * 16 | kColorBlack);
		tioCenterString(tioScreenRows - 1, 0, tioScreenCols - 1,
			list[pos].hint, kColorRed * 16 | kColorYellow);

		ch = getch();
		if ( ch == 0 )
		{
			ch = getch() + 256;
		}
		// Up arrow
		if ( ch == 72 + 256 )
		{
			tioFillAttr(top + pos + 1, left + 1, 1, cols - 3, kColorBlue * 16 | kColorYellow);
			pos--;
			if ( pos < 0 )
			{
				pos = nListItems - 1;
			}
		}
		// Down arrow
		if ( ch == 80 + 256 )
		{
			tioFillAttr(top + pos + 1, left + 1, 1, cols - 3, kColorBlue * 16 | kColorYellow);
			pos++;
			if ( pos >= nListItems )
			{
				pos = 0;
			}
		}
		// Esc
		if ( ch == 27 )
		{
			tioRestoreWindow(saveUnder, top, left, rows, cols);
			free(saveUnder);
			return -1;
		}
	}
	tioRestoreWindow(saveUnder, top, left, rows, cols);
	free(saveUnder);
	return list[pos].id;
}


/*---------------------------------------------------------------------
   Function: ReadSounds

   Reads the digitized sounds from disk.
---------------------------------------------------------------------*/

int ReadSounds( void )
{
	int i;

	for( i = 1; i <= 10; i++ )
	{
		char keyName[3];
		sprintf(keyName, "%d", i);
		strcpy(soundName[i], config.GetKeyString("Sounds", keyName, ""));
	}

	for ( i = 0; i < 11; i++ )
	{
		soundData[i] = NULL;

		if ( strlen(soundName[i]) > 0 )
		{
			int hFile = open(soundName[i], O_RDONLY | O_BINARY);

			if ( hFile == -1 )
			{
				printf("Error opening file '%s'.\n", soundName[i]);
				continue;
			}

			soundLen[i] = filelength(hFile);
			soundData[i] = (char *)malloc(soundLen[i]);

			if ( soundData[i] == NULL )
			{
				printf("Out of memory reading file '%s'.\n", soundName[i]);
				return FALSE;
			}

			if ( !FileRead(hFile, soundData[i], soundLen[i]) )
			{
				printf("Error reading file '%s'.\n", soundName[i]);
				return FALSE;
			}

			close(hFile);
		}
	}
	return TRUE;
}


/*---------------------------------------------------------------------
   Function: GetSoundFXCard

   Asks the user which card to use for sound fx and initializes the
   sound fx routines.
---------------------------------------------------------------------*/

BOOL SetupSoundCard( void )
{
	// Draw our config screen
	tioCenterString(0, 0, tioScreenCols - 1, "Q Studios Sonic Holography Setup", kColorBlue * 16 | kColorYellow);
	tioFill( 1, 0, tioScreenRows - 2, tioScreenCols, '', kColorBlack * 16 | kColorDarkGray);

	// Get Sound fx card
	gSoundCard = SelectMenu("Select a device for digital sound",
		SoundCardMenu, LENGTH(SoundCardMenu), gSoundCard);
	if ( gSoundCard < 0 )
		return FALSE;

	switch ( gSoundCard )
	{
		case SoundBlaster:
		case ProAudioSpectrum:
		case SoundMan16:
		case Awe32:
		case WaveBlaster:
		case SoundScape:
			gMixBits = SelectMenu("Select the number of bits to use for mixing",
				BitMenu, LENGTH(BitMenu), gMixBits);
			if ( gMixBits < 0 )
				return FALSE;

			gChannels = SelectMenu("Select the channels mode",
				ChannelMenu, LENGTH(ChannelMenu), gChannels);
			if ( gChannels < 0 )
				return FALSE;
			gMixVoices = SelectMenu("Select the number of voices to use for mixing",
				VoiceMenu, LENGTH(VoiceMenu), gMixVoices);
			break;

		default:
			gMixBits = 8;
			gChannels = 1;
			gMixVoices = 2;
			break;
	}

	gMixRate = SelectMenu("Select the rate to use for mixing",
		MixRateMenu, LENGTH(MixRateMenu), gMixRate);
	if ( gMixRate < 0 )
		return FALSE;

	config.PutKeyInt("Setup", "SoundCard", gSoundCard);
	config.PutKeyInt("Setup", "Bits", gMixBits);
	config.PutKeyInt("Setup", "Channels", gChannels);
	config.PutKeyInt("Setup", "Voices", gMixVoices);
	config.PutKeyInt("Setup", "MixRate", gMixRate);

	return TRUE;
}


void InitSoundCard( void )
{
	int status;
	fx_device device;

	status = FX_SetupCard( gSoundCard, &device );
	if ( status != FX_Ok )
	{
		printf( "%s\n", FX_ErrorString( status ) );
		exit( 1 );
	}

	status = FX_Init( gSoundCard, gMixVoices, gChannels, gMixBits, gMixRate );
	if ( status != FX_Ok )
	{
		printf( "%s\n", FX_ErrorString( status ) );
		exit( 1 );
	}
}


/*******************************************************************************
	FUNCTION:		ShowUsage()

	DESCRIPTION:	Display command-line parameter usage, then exit
*******************************************************************************/
void ShowUsage(void)
{
	tioPrint("Syntax: SHPLAY [options] file[.RAW]");
	tioPrint("  /setup    Reconfigure sound card");
	tioPrint("  /?        This help");
	tioPrint("");
	exit(0);
}


/***********************************************************************
 * Process command line arguments
 **********************************************************************/
void ParseOptions( void )
{
	enum {
		kSwitchHelp,
		kSwitchSetup,
		kSwitchMap,
		kSwitchOmit,
	};
	static SWITCH switches[] = {
		{ "?", kSwitchHelp, FALSE },
		{ "SETUP", kSwitchSetup, FALSE },
	};
	char buffer[256];
	int r;
	while ( (r = GetOptions(switches)) != GO_EOF )
	{
		switch (r)
		{
			case GO_INVALID:
				sprintf(buffer, "Invalid argument: %s", OptArgument);
				tioPrint(buffer);
				exit(1);

			case GO_FULL:
				strcpy(soundName[0], OptArgument);
				break;

			case kSwitchSetup:
				if ( !SetupSoundCard() )
					exit(0);
				break;

			case kSwitchHelp:
				ShowUsage();
				break;
		}
	}
}



/*---------------------------------------------------------------------
   Function: main

   Sets up sound cards, calls the demo, and then cleans up.
---------------------------------------------------------------------*/

void main( void )
{
	tioInit(0);

	// Select which cards to use
	gSoundCard = config.GetKeyInt("Setup", "SoundCard", -1);
	gMixBits = config.GetKeyInt("Setup", "Bits", 16);
	gChannels = config.GetKeyInt("Setup", "Channels", 2);
	gMixVoices = config.GetKeyInt("Setup", "Voices", 16);
	gMixRate = config.GetKeyInt("Setup", "MixRate", 22050);
	gFXVolume = config.GetKeyInt("Setup", "FXVolume", 8);

	if ( gSoundCard == -1 )
	{
		if ( !SetupSoundCard() )
			exit(0);
	}

	ParseOptions();

	if ( !ReadSounds() )
		exit( 1 );

	InitSoundCard();

	DoDemo();

	FX_Shutdown();

	tioClearWindow();
	tioPrint("Sonic Holography Player Version 2.0   Copyright (c) 1995 Q Studios Corporation");
	tioPrint("Written by Peter M. Freese");
	tioPrint("");
	tioPrint("Pinna angle: %3.0f degrees", gPinnaAngle * 360.0 / kAngle360);
	tioPrint("Saturation:  %0.2f", nSat);

	tioTerm();

	config.PutKeyInt("Setup", "FXVolume", gFXVolume);
	config.Save();
}




