#include <string.h>

#include "engine.h"
#include "multi.h"

#include "typedefs.h"
#include "misc.h"
#include "gameutil.h"
#include "trig.h"
#include "view.h"
#include "globals.h"
#include "textio.h"
#include "names.h"
#include "db.h"
#include "debug4g.h"
#include "resource.h"
#include "error.h"
#include "player.h"
#include "screen.h"
#include "options.h"
#include "tile.h"
#include "levels.h"
#include "actor.h"
#include "sectorfx.h"
#include "mirrors.h"
#include "fire.h"
#include "weapon.h"
#include "map2d.h"

#include <memcheck.h>

// absolute coordinates of new status bar
#define kStat2HeadX			0
#define kStat2HeadY			155

#define kStat2StatsX		50
#define kStat2StatsY		171

#define kStat2HealthX		54
#define kStat2HealthY		185

#define kStat2ArmorX		54
#define kStat2ArmorY		194

#define kStat2AmmoX			199
#define kStat2AmmoY			174

#define kStat2AmmoTypeX		210
#define kStat2AmmoTypeY		181

#define kStat2AmmoCountX	234
#define kStat2AmmoCountY	177

#define kStat2HeartX		271
#define kStat2HeartY		155

#define kStat2LustX			310
#define kStat2LustY			129

#define kStat2LustBarX		313
#define kStat2LustBarY		131

// old status bar
#define kStatHealthX 6
#define kStatHealthY 0

#define kStatHeadX 23
#define kStatHeadY -6

#define kStatHeartX 66
#define kStatHeartY 0

#define kStatArmorX 259
#define kStatArmorY 0

#define kStatFoeTextX 286
#define kStatFoeTextY 3

#define kStatScoreX 125
#define kStatScoreY 16

#define kStatAmmoX 210
#define kStatAmmoY 16

#define kStatFoesX 282
#define kStatFoesY 16

#define kStatWeaponX 202
#define kStatWeaponY 5

/***********************************************************************
 * Variables
 **********************************************************************/
int gViewIndex = 0;
int gViewSize = 1;
int gViewMode = kView3D;
VIEWPOS gViewPos = kViewPosCenter;
int gZoom = 1024;
int gViewX0, gViewY0, gViewX1, gViewY1;
int gViewXCenter, gViewYCenter;

static int gHeadPic = kPicJack;
static short statusHeight[MAXXDIM];
static int statusLow = 0;
static int pcBackground = 0, pcStatusBar = 0;

int gShowFrameRate = FALSE;
int gShowFrags = FALSE;

long gScreenTilt = 0;

int deliriumTilt = 0;
int deliriumTurn = 0;
int deliriumPitch = 0;

struct PLOCATION
{
	int x, y, z;
	int bobHeight, bobWidth;
	int swayHeight, swayWidth;
	int horiz;
	short ang;
} gPrevPlayerLoc[kMaxPlayers];

struct LOCATION
{
	int x, y, z;
	short ang;
} gPrevSpriteLoc[kMaxSprites];

int gInterpolate;

int gViewPosAngle[] =
{
	0,						// kViewPosCenter
	kAngle180,				// kViewPosBack
	kAngle180 - kAngle45,	// kViewPosLeftBack
	kAngle90,				// kViewPosLeft
	kAngle45,				// kViewPosLeftFront
	0,						// kViewPosFront
	-kAngle45,				// kViewPosRightFront
	-kAngle90,				// kViewPosRight
	-kAngle90 - kAngle45,	// kViewPosRightBack
};

void koutp(int, int);
#pragma aux koutp =\
	"out dx, al",\
	parm [edx][eax]\

void koutpw(int, int);
#pragma aux koutpw =\
	"out dx, ax",\
	parm [edx][eax]\

/*
void mywritesprite (int thex, int they, int nTile, int shade, int nPal,
	int cx1, int cy1, int cx2, int cy2)
{
	long sizeX, sizeY, x, x1, y1, x2, y2;
	long lm, rm, um, dm, i;
	long plane;

	BYTE *p, *palookupoffs, *palookupoff4[4];
	BYTE *tileoffs, *bufplc, *bufplc4[4];

	dassert(nTile >= 0 && nTile < kMaxTiles);

	setupmvlineasm(24);

	if ((tilesizx[nTile] <= 0) || (tilesizy[nTile] <= 0))
		return;

	shade = ClipRange(shade, 0, kPalLookups - 1);
	palookupoffs = palookup[nPal] + (shade << 8);

	tileoffs = tileLoadTile(nTile);
	sizeX = tilesizx[nTile];
	sizeY = tilesizy[nTile];

	x1 = thex;
	y1 = they;
	x2 = x1 + sizeX;
	y2 = y1 + sizeY;

	lm = x1; if (lm < 0) lm = 0;
	rm = x2; if (rm > xdim) rm = xdim;
	um = y1; if (um < 0) um = 0;
	dm = y2; if (dm > ydim) dm = ydim;

	if (lm < cx1) lm = cx1;
	if (rm > cx2+1) rm = cx2+1;
	if (um < cy1) um = cy1;
	if (dm > cy2+1) dm = cy2+1;

	if ((rm < 0) || (lm >= xdim) || (rm <= lm) || (dm < 0) || (um >= ydim) || (dm <= um))
		return;

	for (i = 0; i < 4; i++)
	{
		palookupoff4[i] = palookupoffs;
		vince[i] = 0x1000000;
	}

	if (vidoption == 0 && chainnumpages > 2)	// unchained mode, direct to display
	{
		for (plane = 0; plane < 4; plane++)
		{
			x = lm + plane;

			koutpw(0x3c4,2+(256<<(x&3)));
			p = (BYTE *)(chainplace + ylookup[um] + (x >> 2));
			bufplc = tileoffs + (x - x1) * sizeY + (um - y1);

			while (x < rm-12)
			{
				bufplc4[0] = bufplc;
				bufplc4[1] = bufplc4[0]+(sizeY<<2);
				bufplc4[2] = bufplc4[1]+(sizeY<<2);
				bufplc4[3] = bufplc4[2]+(sizeY<<2);
				bufplc = bufplc4[3]+(sizeY<<2);
				vplce[0] = vplce[1] = vplce[2] = vplce[3] = 0;
				mvlineasm4(dm - um, p);
				p += 4;
				x += 16;
			}
			while (x < rm)
			{
				mvlineasm1(0x1000000, palookupoffs, dm - um, 0L, bufplc, p);
				bufplc += (sizeY<<2);
				p++;
				x += 4;
			}
		}
	}
	else
	{
		x = lm;

		bufplc = tileoffs + (x-x1)*sizeY + (um-y1);

		p = frameplace + ylookup[um] + x;

		// dword align to video memory
		while (x < rm && (x & 3) )
		{
			mvlineasm1(0x1000000, palookupoffs, dm - um, 0L, bufplc, p);
			bufplc += sizeY;
			p++;
			x++;
		}

		while (x < rm-3)
		{
			bufplc4[0] 	= bufplc;
			bufplc4[1] 	= bufplc4[0]+sizeY;
			bufplc4[2] 	= bufplc4[1]+sizeY;
			bufplc4[3] 	= bufplc4[2]+sizeY;
			bufplc 		= bufplc4[3]+sizeY;
			vplce[0] = vplce[1] = vplce[2] = vplce[3] = 0;
			mvlineasm4(dm - um, p);
			p += 4;
			x += 4;
		}
		while (x < rm)
		{
			mvlineasm1(0x1000000, palookupoffs, dm - um, p);
			bufplc += sizeY;
			p++;
			x++;
		}
	}
}
*/

void mytilesprite( int nTile, int shade, int nPal, int cx1, int cy1, int cx2, int cy2 )
{
	long sizeX, sizeY, x, y, y0, y1;
	long lm, rm, um, dm, i;
	int nSize = tilesizx[nTile] * tilesizy[nTile];
	BYTE *p, *palookupoffs;
	BYTE *tileoffs, *bufplc, *wrap;

	setupvlineasm(16);

	dassert(nTile >= 0 && nTile < kMaxTiles);

	if ((tilesizx[nTile] <= 0) || (tilesizy[nTile] <= 0))
		return;

	shade = ClipRange(shade, 0, kPalLookups - 1);
	palookupoffs = palookup[nPal] + (shade << 8);

	tileoffs = tileLoadTile(nTile);
	wrap = tileoffs + nSize;
	sizeX = tilesizx[nTile];
	sizeY = tilesizy[nTile];

	lm = cx1;
	rm = cx2+1;
	um = cy1;
	dm = cy2+1;

	if ((rm < 0) || (lm >= xdim) || (rm <= lm) || (dm < 0) || (um >= ydim) || (dm <= um))
		return;

	for (i = 0; i < 4; i++)
	{
		palookupoffse[i] = palookupoffs;
		vince[i] = 0x00010000;
	}

	for (y = um; y < dm; y = IncBy(y, sizeY))
	{
		y0 = y;
		y1 = IncBy(y, sizeY) - 1;
		if (y1 >= dm) y1 = dm - 1;

		x = lm;

		bufplc = tileoffs + (x % sizeX) * sizeY + (y % sizeY);

		p = frameplace + ylookup[y] + x;

		// dword align to video memory
		while (x < rm && (x & 3) )
		{
			vlineasm1(0x00010000, palookupoffs, y1 - y0 + 1, 0L, bufplc, p);
			bufplc += sizeY;
			if (bufplc >= wrap) bufplc -= nSize;
			p++;
			x++;
		}

		while (x < rm-3)
		{
			for (i = 0; i < 4; i++)
			{
				bufplce[i] 	= bufplc;
				bufplc += sizeY;
				if (bufplc >= wrap) bufplc -= nSize;
				vplce[i] = 0;
			}
			vlineasm4(y1 - y0 + 1, p);
			p += 4;
			x += 4;
		}

		while (x < rm)
		{
			vlineasm1(0x00010000, palookupoffs, y1 - y0 + 1, 0L, bufplc, p);
			bufplc += sizeY;
			if (bufplc >= wrap) bufplc -= nSize;
			p++;
			x++;
		}
	}
}

void InitStatusBar( void )
{
	int i, j;
	int tileHeight;

	tioPrint("Building startmost table");

	tileLoadTile(kPicStatBar);

	tileHeight = tilesizy[kPicStatBar];

	for ( i = 0; tioGauge(i, tilesizx[kPicStatBar]); i++ )
	{
		uchar *p = waloff[kPicStatBar] + i * tileHeight;
		for ( j = 0; *p == 255; j++, p++ );
		statusHeight[i] = (short)j;
		if (j > statusLow)
			statusLow = j;
	}
}

/*
static void TileGauge(int x, int y, int nBackTile, int nForeTile, int n, int total)
{
	int nGauge = tilesizy[nForeTile] - (n * tilesizy[nForeTile] / total);

	mywritesprite(x, y, nBackTile, 0, kPLUNormal, 0, y, xdim-1, y+nGauge-1);
	mywritesprite(x, y, nForeTile, 0, kPLUNormal, 0, y+nGauge, xdim-1, y+tilesizy[nForeTile]);
}
*/


static void UpdateStatusBar(void)
{
	SPRITE *pSprite = gView->sprite;
	XSPRITE *pXSprite = gView->xsprite;

	if (gViewMode == kView2D || gViewSize > 0)		// status bar present?
	{
		int statY = ydim - tilesizy[kPicStatBar];

		if (pXSprite->health > 0)
		{
			int picStart = kPicJack + 40 -
				((ClipHigh(pXSprite->health >> 4, 100) - 1) / 20) * 10;
			switch ( pXSprite->moveState )
			{
				case kMoveFall:
					gHeadPic = picStart + 5;
					break;

				case kMoveLand:
					gHeadPic = picStart + 4;
					break;

				default:
					if (gView->godMode)
						gHeadPic = 2303;
					else if ( gView->run )
						gHeadPic = picStart;
					else
						gHeadPic = picStart + 7;
					break;

			}
		}
		else
			gHeadPic = kAnmJacksDeath + ClipHigh(gView->deathTime >> 4, 12);

		// head background
		rotatesprite(25 << 16, 177 << 16, 0x10000, 0, 2201, 0, 0, kRotateStatus, 0, 0, xdim-1, ydim-1);

		// overwrite face
		rotatesprite(25 << 16, 174 << 16, 0x10000, 0, (short)gHeadPic, 0, 0, kRotateStatus, 0, 0, xdim-1, ydim-1);

		// health and armor background
		rotatesprite(90 << 16, 185 << 16, 0x10000, 0, 2202, 0, 0, kRotateStatus, 0, 0, xdim-1, ydim-1);

		// health meter (needs to be clipped)
		rotatesprite(86 << 16, 186 << 16, 0x10000, 0, 2203, 0, 0, kRotateStatus | kRotateNoMask, 0, 0, xdim-1, ydim-1);

		// armor meter (needs to be clipped)
		rotatesprite(86 << 16, 195 << 16, 0x10000, 0, 2204, 0, 0, kRotateStatus | kRotateNoMask, 0, 0, xdim-1, ydim-1);

		// weapon and ammo background
		rotatesprite(235 << 16, 184 << 16, 0x10000, 0, 2205, 0, 0, kRotateStatus, 0, 0, xdim-1, ydim-1);

		// blood lust background
		rotatesprite(315 << 16, 164 << 16, 0x10000, 0, 2206, 0, 0, kRotateStatus | kRotateNoMask, 0, 0, xdim-1, ydim-1);

		// temporary hack just to see it pumping
		short nHeartPic = (short)(2234 + ((gFrameClock >> 4) & 3));

		// heart
		rotatesprite(290 << 16, 177 << 16, 0x10000, 0, nHeartPic, 0, 0, kRotateStatus, 0, 0, xdim-1, ydim-1);

		// blood lust meter (needs to be clipped)
		rotatesprite(315 << 16, 146 << 16, 0x10000, 0, 2207, 0, 0, kRotateStatus | kRotateNoMask, 0, 0, xdim-1, ydim-1);

		// display level name
		if ( gViewMode == kView2D )
			printext256(0, statY - 16, gStdColor[15], -1, gLevelDescription, 0);

// 		TileGauge(kStatHealthX, statY + kStatHealthY, kPicBlackMeter, kPicRedMeter, ClipHigh(pXSprite->health >> 4, 100), 100);
//		if ( gView->lifemode == kModeBeast )
//			overwritesprite(18, statY - 4, kPicBeast, 0, kOrientNormal, kPLUNormal);
//		else
//			overwritesprite(kStatHeadX, statY + kStatHeadY, (short)gHeadPic, 0, kOrientNormal, kPLUNormal);
//
	}

	if (gCacheMiss)
	{
		overwritesprite(xdim - 15, ydim - 15, kPicDisk, -128, kOrientNormal, kPLUNormal);
		pcStatusBar = numpages;
		gCacheMiss = ClipLow(gCacheMiss - kFrameTicks, 0);
	}

	// draw keys
	for (int i = 0; i < 7; i++)
	{
		short nKeyPic = gView->hasKey[i+1];
		if (nKeyPic)
			overwritesprite(48, 12 + i * 10, nKeyPic, -128, kOrientScale | kOrientXFlip, kPLUNormal);
	}
}


void viewInit(void)
{
	gViewXCenter = xdim / 2;
	gViewYCenter = ydim / 2;

	tioPrint("Initializing status bar");
	InitStatusBar();
}


void viewResizeView(int size)
{
	//int x0, y0, x1, y1;

	gViewSize = ClipRange(size, 0, 8);

	// full screen mode
	if (gViewSize < 2)
	{
		gViewX0 = 0;
		gViewY0 = 0;
		gViewX1 = xdim - 1;
		gViewY1 = ydim - 1;

		setview(gViewX0, gViewY0, gViewX1, gViewY1);
		return;
	}

	gViewX0 = 0;
	gViewY0 = 0;
	gViewX1 = xdim - 1;
	gViewY1 = ydim - 1; // - tilesizy[kPicStatBar] + statusLow;

	// full screen with status bar
	if (gViewSize == 1)
	{
		setview(gViewX0, gViewY0, gViewX1, gViewY1);

		// clip for status bar area
//		for (int i = 0; i < xdim; i++)
//			startdmost[i] -= statusLow - statusHeight[i];

		pcStatusBar = numpages;
		return;
	}

	gViewX0 += (gViewSize - 1) * xdim / 16;
	gViewX1 -= (gViewSize - 1) * xdim / 16;
	gViewY0 += (gViewSize - 1) * ydim / 16;
	gViewY1 -= (gViewSize - 1) * ydim / 16;

	setview(gViewX0, gViewY0, gViewX1, gViewY1);

	pcBackground = numpages;
	pcStatusBar = numpages;
}


#define kBackTile 230
void viewDrawInterface( void )
{
	int x0, y0, x1, y1;

	if (gViewMode == kView3D && gViewSize > 1)
	{
		if ( pcBackground )
		{
			x0 = gViewX0;
			y0 = gViewY0;
			x1 = gViewX1;
			y1 = gViewY1;

			mytilesprite(kBackTile, 20, kPLUNormal, 0, 0, xdim-1, y0-4); 		// top
			mytilesprite(kBackTile, 20, kPLUNormal, 0, y1+4, xdim-1, ydim-1);	// bottom
			mytilesprite(kBackTile, 20, kPLUNormal, 0, y0-3, x0-4, y1+3);		// left
			mytilesprite(kBackTile, 20, kPLUNormal, x1+4, y0-3, xdim-1, y1+3);	// right

			mytilesprite(kBackTile, 40, kPLUNormal, x0-3, y0-3, x0-1, y1);		// left
			mytilesprite(kBackTile, 40, kPLUNormal, x0, y0-3, x1+3, y0-1); 		// top
			mytilesprite(kBackTile, 0, kPLUNormal, x1+1, y0, x1+3, y1+3);		// right
			mytilesprite(kBackTile, 0, kPLUNormal, x0-3, y1+1, x1, y1+3);		// bottom

			pcBackground--;
		}
	}

	UpdateStatusBar();

	powerupDraw( gView );
}


void viewSet3dMode( void )
{
	scrSetGameMode();
	scrNextPage();

	viewResizeView(gViewSize);
}

short viewInsertTSprite( short nSector, short nStatus )
{
	short nTSprite = -1;

	if ( spritesortcnt < kMaxViewSprites)
	{
		nTSprite = (short)spritesortcnt;
		SPRITE *pTSprite = &tsprite[nTSprite];
		memset(pTSprite, 0, sizeof(SPRITE));
		pTSprite->sectnum = nSector;
		pTSprite->statnum = nStatus;
		pTSprite->cstat = kSpriteOriginAlign;
		pTSprite->xrepeat = 64;
		pTSprite->yrepeat = 64;
		pTSprite->owner = -1;
		pTSprite->extra = -1;
		spritesortcnt++;
	}
	return nTSprite;
}


enum VIEW_EFFECT {
	kViewEffectShadow = 0,
	kViewEffectFlare,
	kViewEffectTorchHigh,
	kViewEffectTorchLow,
	kViewEffectSmokeHigh,
	kViewEffectSmokeLow,
	kViewEffectFlame,
	kViewEffectSpear,
	kViewEffectMax,
};


short viewAddEffect( int nTSprite, VIEW_EFFECT nViewEffect )
{
	dassert( nViewEffect >= 0 && nViewEffect < kViewEffectMax );
	SPRITE *pTSprite = &tsprite[nTSprite];

	int nTEffect = viewInsertTSprite( pTSprite->sectnum, 0x7FFF );

	if ( nTEffect >= 0 )
	{
		SPRITE *pTEffect = &tsprite[nTEffect];
		pTEffect->x = pTSprite->x;
		pTEffect->y = pTSprite->y;
		pTEffect->owner = pTSprite->owner;	// (short)nTSprite;
		pTEffect->ang = pTSprite->ang;

		switch (nViewEffect)
		{
			case kViewEffectFlame:
			{
				pTEffect->z = pTSprite->z;
				pTEffect->cstat = (uchar)(pTSprite->cstat & ~kSpriteBlocking);
				pTEffect->shade = kMaxShade;
				pTEffect->xrepeat = pTEffect->yrepeat =
					(uchar)(tilesizx[pTSprite->picnum] * pTSprite->xrepeat / 64);
				pTEffect->picnum = kAnmFlame3;
				pTEffect->statnum = kStatDefault; // show up in front of burning objects
				break;
			}

			case kViewEffectSmokeHigh:
			{
				int zTop, zBot;
				GetSpriteExtents(pTSprite, &zTop, &zBot);
				pTEffect->z = zTop;
				pTEffect->picnum = kAnmSmoke2;
				pTEffect->cstat = (short)(pTSprite->cstat | kSpriteTranslucent & ~kSpriteBlocking);
				pTEffect->shade = 8;
				pTEffect->xrepeat = pTSprite->xrepeat;
				pTEffect->yrepeat = pTSprite->yrepeat;
				break;
			}

			case kViewEffectSmokeLow:
			{
				int zTop, zBot;
				GetSpriteExtents(pTSprite, &zTop, &zBot);
				pTEffect->z = zBot;
				pTEffect->picnum = kAnmSmoke2;
				pTEffect->cstat = (short)(pTSprite->cstat | kSpriteTranslucent & ~kSpriteBlocking);
				pTEffect->shade = 8;
				pTEffect->xrepeat = pTSprite->xrepeat;
				pTEffect->yrepeat = pTSprite->yrepeat;
				break;
			}

			case kViewEffectTorchHigh:
			{
				int zTop, zBot;
				GetSpriteExtents(pTSprite, &zTop, &zBot);
				pTEffect->z = zTop;
				pTEffect->picnum = kAnmFlame2;
				pTEffect->cstat = (short)(pTSprite->cstat & ~kSpriteBlocking);
				pTEffect->shade = kMaxShade;
				pTEffect->xrepeat = pTEffect->yrepeat =
					(uchar)(tilesizx[pTSprite->picnum] * pTSprite->xrepeat / 32);
				break;
			}

			case kViewEffectTorchLow:
			{
				int zTop, zBot;
				GetSpriteExtents(pTSprite, &zTop, &zBot);
				pTEffect->z = zBot;
				pTEffect->picnum = kAnmFlame2;
				pTEffect->cstat = (short)(pTSprite->cstat & ~kSpriteBlocking);
				pTEffect->shade = kMaxShade;
				pTEffect->xrepeat = pTEffect->yrepeat =
					(uchar)(tilesizx[pTSprite->picnum] * pTSprite->xrepeat / 32);
				break;
			}

			case kViewEffectShadow:
				if (gShowDetails)
				{
					// insert a shadow
					pTEffect->z = getflorzofslope(pTSprite->sectnum, pTEffect->x, pTEffect->y);

					pTEffect->cstat = (uchar)(pTSprite->cstat | kSpriteTranslucent);
					pTEffect->shade = kMinShade;
					pTEffect->xrepeat = pTSprite->xrepeat;
					pTEffect->yrepeat = (uchar)(pTSprite->yrepeat >> 2);
					pTEffect->picnum = pTSprite->picnum;
					int nTile = pTEffect->picnum;

					// position it so it's based on the floor
					pTEffect->z -= (tilesizy[nTile] - (tilesizy[nTile] / 2 + picanm[nTile].ycenter)) * (pTEffect->yrepeat << 2);
				}
				break;

			case kViewEffectFlare:
			{
				// insert a lens flare halo effect around flare missiles
				pTEffect->z = pTSprite->z;
				pTEffect->cstat = (uchar)(pTSprite->cstat | kSpriteTranslucent);
				pTEffect->shade = kMaxShade;
				pTEffect->pal = kPLURed;
				pTEffect->xrepeat = pTSprite->xrepeat;
				pTEffect->yrepeat = pTSprite->yrepeat;
				pTEffect->picnum = kAnmFlareHalo;

				if (gShowDetails)
				{
					int zDist;

					// create a glowing reflection on the floor and ceiling
					SECTOR *pSector = &sector[pTSprite->sectnum];
					zDist = (pTSprite->z - pSector->ceilingz) >> 8;

					if ( !(pSector->ceilingstat & kSectorParallax) && zDist < 64 )
					{
						short nTEffect = viewInsertTSprite( pTSprite->sectnum, 0x7FFF );
						SPRITE *pHalo = &tsprite[nTEffect];
						pHalo->x = pTSprite->x;
						pHalo->y = pTSprite->y;
						pHalo->z = pSector->ceilingz;
						pHalo->cstat = (uchar)(pTSprite->cstat | kSpriteTranslucent | kSpriteFloor | kSpriteOneSided | kSpriteFlipY);
						pHalo->shade = (schar)(-64 + zDist);
						pHalo->pal = kPLURed;
						pHalo->xrepeat = 64;
						pHalo->yrepeat = 64;
						pHalo->picnum = kAnmGlowSpot1;
						pHalo->ang = pTSprite->ang;
						pHalo->owner = pTSprite->owner;	//(short)nTSprite;
					}

					zDist = (pSector->floorz - pTSprite->z) >> 8;
					if ( !(pSector->floorstat & kSectorParallax) && zDist < 64 )
					{
						short nTEffect = viewInsertTSprite( pTSprite->sectnum, 0x7FFF );
						SPRITE *pHalo = &tsprite[nTEffect];
						pHalo->x = pTSprite->x;
						pHalo->y = pTSprite->y;
						pHalo->z = pSector->floorz;
						pHalo->cstat = (uchar)(pTSprite->cstat | kSpriteTranslucent | kSpriteFloor | kSpriteOneSided);
						pHalo->shade = (schar)(-32 + zDist);
						pHalo->pal = kPLURed;
						pHalo->xrepeat = (uchar)zDist;
						pHalo->yrepeat = (uchar)zDist;
						pHalo->picnum = kAnmGlowSpot1;
						pHalo->ang = pTSprite->ang;
						pHalo->owner = pTSprite->owner;	//(short)nTSprite;
					}
				}
				break;
			}

			case kViewEffectSpear:
				pTEffect->z = pTSprite->z;
				pTEffect->cstat = (uchar)((pTSprite->cstat & ~kSpriteRMask) | kSpriteFloor);
				pTEffect->shade = pTSprite->shade;
				pTEffect->xrepeat = pTSprite->xrepeat;
				pTEffect->yrepeat = pTSprite->yrepeat;
				pTEffect->picnum = pTSprite->picnum;
				pTSprite->ang += kAngle180;
				pTSprite->ang &= kAngleMask;
				break;

		}
	}
	return (short)nTEffect;
}


void viewProcessSprites( int nViewX, int nViewY, int nViewZ )
{
	int nTSprite, nOctant;
	short nXSprite;
	long dx, dy;

	// count down so we don't process shadows
	for (nTSprite = spritesortcnt - 1; nTSprite >= 0; nTSprite--)
	{
		SPRITE *pTSprite = &tsprite[nTSprite];
		XSPRITE *pTXSprite = NULL;
		nXSprite = pTSprite->extra;

		if ( nXSprite > 0 )
			pTXSprite = &xsprite[nXSprite];

		// KEN'S CODE TO SHADE SPRITES
		// Add relative shading code for textures, and possibly sector cstat bit to
		// indicate where shading comes from (i.e. shadows==floorshade, sky==ceilingshade)
		// ALSO: This code does not affect sprite effects added to the tsprite list

		pTSprite->shade = (schar)ClipRange( pTSprite->shade + 6, -128, 127 );
		if ( (sector[pTSprite->sectnum].ceilingstat & 1)
		&& !(sector[pTSprite->sectnum].floorstat & kSectorFloorShade) )
			pTSprite->shade = (schar)ClipRange( pTSprite->shade + sector[pTSprite->sectnum].ceilingshade, -128, 127 );
		else
			pTSprite->shade = (schar)ClipRange( pTSprite->shade + sector[pTSprite->sectnum].floorshade, -128, 127 );

		int nTile = pTSprite->picnum;
		if (nTile < 0 || nTile >= kMaxTiles)
		{
			dprintf("tsprite[].cstat = %i\n", pTSprite->cstat);
			dprintf("tsprite[].shade = %i\n", pTSprite->shade);
			dprintf("tsprite[].pal = %i\n", pTSprite->pal);
			dprintf("tsprite[].picnum = %i\n", pTSprite->picnum);
			dprintf("tsprite[].ang = %i\n", pTSprite->ang);
			dprintf("tsprite[].owner = %i\n", pTSprite->owner);
			dprintf("tsprite[].sectnum = %i\n", pTSprite->sectnum);
			dprintf("tsprite[].statnum = %i\n", pTSprite->statnum);
			dprintf("tsprite[].type = %i\n", pTSprite->type);
			dprintf("tsprite[].flags = %i\n", pTSprite->flags);
			dprintf("tsprite[].extra = %i\n", pTSprite->extra);
			dassert(nTile >= 0 && nTile < kMaxTiles);
		}

		// only interpolate certain moving things
		if ( pTSprite->flags & kAttrMove )
		{
			long x = gPrevSpriteLoc[pTSprite->owner].x;
			long y = gPrevSpriteLoc[pTSprite->owner].y;
			long z = gPrevSpriteLoc[pTSprite->owner].z;
			short nAngle = gPrevSpriteLoc[pTSprite->owner].ang;

			// interpolate sprite position
			x += mulscale16(pTSprite->x - gPrevSpriteLoc[pTSprite->owner].x, gInterpolate);
			y += mulscale16(pTSprite->y - gPrevSpriteLoc[pTSprite->owner].y, gInterpolate);
			z += mulscale16(pTSprite->z - gPrevSpriteLoc[pTSprite->owner].z, gInterpolate);
			nAngle += mulscale16(((pTSprite->ang - gPrevSpriteLoc[pTSprite->owner].ang + kAngle180) & kAngleMask) - kAngle180, gInterpolate);

			pTSprite->x = x;
			pTSprite->y = y;
			pTSprite->z = z;
			pTSprite->ang = nAngle;
		}

		int nFrames = picanm[nTile].frames + 1;

		switch ( picanm[nTile].view )
		{
			case kSpriteViewSingle:

				if (nXSprite > 0)
				{
					dassert(nXSprite < kMaxXSprites);
					switch ( pTSprite->type )
					{
						case kSwitchToggle:
						case kSwitchMomentary:
							if ( xsprite[nXSprite].state )
								pTSprite->picnum = (short)(nTile + nFrames);
								break;

						case kSwitchCombination:
							pTSprite->picnum = (short)(nTile + xsprite[nXSprite].data1 * nFrames);
							break;

					}
				}
				break;

			case kSpriteView5Full:

				// Calculate which of the 8 angles of the sprite to draw (0-7)
				dx = nViewX - pTSprite->x;
				dy = nViewY - pTSprite->y;
				RotateVector(&dx, &dy, -pTSprite->ang + kAngle45 / 2);
				nOctant = GetOctant(dx, dy);

				if (nOctant <= 4)
				{
					pTSprite->picnum = (short)(nTile + nOctant);
					pTSprite->cstat &= ~4;   // clear x-flipping bit
				}
				else
				{
					pTSprite->picnum = (short)(nTile + 8 - nOctant);
					pTSprite->cstat |= 4;   // set x-flipping bit
				}
				break;

			case kSpriteView8Full:
				// Calculate which of the 8 angles of the sprite to draw (0-7)
				dx = nViewX - pTSprite->x;
				dy = nViewY - pTSprite->y;
				RotateVector(&dx, &dy, -pTSprite->ang + kAngle45 / 2);
				nOctant = GetOctant(dx, dy);

				pTSprite->picnum = (short)(nTile + nOctant);
				break;

			case kSpriteView5Half:
				break;
		}

		if ( spritesortcnt < kMaxViewSprites )
		{
			if (pTSprite->statnum == kStatDude || pTSprite->statnum == kStatThing || pTSprite->statnum == kStatMissile)
			{
				if ( pTXSprite && actGetBurnTime(pTXSprite) > 0 )
					viewAddEffect( nTSprite, kViewEffectFlame );

				if ( pTSprite->flags & kAttrSmoke )
					viewAddEffect(nTSprite, kViewEffectSmokeHigh);

				if (pTSprite->type >= kDudePlayer1 && pTSprite->type <= kDudePlayer8)
				{
					PLAYER *pPlayer = &gPlayer[pTSprite->type - kDudePlayer1];

					if ( powerupCheck( pPlayer, kItemLtdInvisibility - kItemBase ) > 0)
					{
						pTSprite->cstat |= kSpriteTranslucent; // kSpriteInvisible;
						pTSprite->pal = kPLUGray;
					}
					// if not invisible, show the temporary player ID
					if ( !(pTSprite->cstat & kSpriteTranslucent) )
						pTSprite->pal = (uchar )(kPLUPlayer1 + ( pPlayer->teamID & 3 ));
				}

				switch (pTSprite->type)
				{
					case kMissileSpear:
						viewAddEffect( nTSprite, kViewEffectSpear );
						break;

					case kMissileFlare:
						viewAddEffect( nTSprite, kViewEffectFlare );
						break;

					default:
						// only add shadow if will be below the view z
						if ( getflorzofslope(pTSprite->sectnum, pTSprite->x, pTSprite->y) >= nViewZ)
						{
							// don't add to kThings already on the ground
							if ( pTSprite->type >= kThingBase && pTSprite->type < kThingMax)
							{
								int zTop, zBot;

								GetSpriteExtents(pTSprite, &zTop, &zBot);
								if ( zBot == getflorzofslope(pTSprite->sectnum, pTSprite->x, pTSprite->y) )
									break;
							}
							viewAddEffect( nTSprite, kViewEffectShadow );
						}
						break;
				}
			}
			else switch ( pTSprite->type )
			{
				case kMiscTorch:
					if ( pTXSprite )
					{
						if ( pTXSprite->state > 0 )
						{
							pTSprite->picnum++;
							viewAddEffect( nTSprite, kViewEffectTorchHigh );
						} else {
							viewAddEffect( nTSprite, kViewEffectSmokeHigh );
						}
					}
					else
					{
						pTSprite->picnum++;
						viewAddEffect( nTSprite, kViewEffectTorchHigh );
					}
					break;
				case kMiscHangingTorch:
					if ( pTXSprite )
					{
						if ( pTXSprite->state > 0 )
						{
							pTSprite->picnum++;
							viewAddEffect( nTSprite, kViewEffectTorchLow );
						} else {
							viewAddEffect( nTSprite, kViewEffectSmokeLow );
						}
					}
					else
					{
						pTSprite->picnum++;
						viewAddEffect( nTSprite, kViewEffectTorchLow );
					}
					break;
			}
		}
	}
}


#define kViewDistance	(80 << 4)
extern void viewCalcPosition( SPRITE *pSprite, long *px, long *py, long *pz, short *pAngle, short *pSector )
{
	int nAngle = (*pAngle + gViewPosAngle[gViewPos]) & kAngleMask;
	int dx = mulscale30(kViewDistance, Cos(nAngle));
	int dy = mulscale30(kViewDistance, Sin(nAngle));

	*pAngle = (short)((nAngle + kAngle180) & kAngleMask);

	ushort cstat = pSprite->cstat;
	pSprite->cstat &= ~kSpriteBlocking;
	ClipMove(px, py, pz, pSector, dx, dy, 4 << 4, 1, 1, 0);
	pSprite->cstat = cstat;
}


void viewBackupPlayerLoc( int nPlayer )
{
	SPRITE *pSprite = gPlayer[nPlayer].sprite;
	PLOCATION *pPLocation = &gPrevPlayerLoc[nPlayer];
	pPLocation->x = pSprite->x;
	pPLocation->y = pSprite->y;
	pPLocation->z = pSprite->z;
	pPLocation->ang = pSprite->ang;
	pPLocation->horiz = gPlayer[nPlayer].horiz;
	pPLocation->bobHeight = gPlayer[nPlayer].bobHeight;
	pPLocation->bobWidth = gPlayer[nPlayer].bobWidth;
	pPLocation->swayHeight = gPlayer[nPlayer].swayHeight;
	pPLocation->swayWidth = gPlayer[nPlayer].swayWidth;
}


void viewBackupSpriteLoc( int nSprite, SPRITE *pSprite )
{
	LOCATION *pLocation = &gPrevSpriteLoc[nSprite];
	pLocation->x = pSprite->x;
	pLocation->y = pSprite->y;
	pLocation->z = pSprite->z;
	pLocation->ang = pSprite->ang;
}


void viewBackupAllSpriteLoc( void )
{
	SPRITE *pSprite = &sprite[0];
	LOCATION *pLocation = &gPrevSpriteLoc[0];
	for (int i = 0; i < kMaxSprites; i++, pSprite++, pLocation++)
	{
		if ( pSprite->flags & kAttrMove )
		{
			pLocation->x = pSprite->x;
			pLocation->y = pSprite->y;
			pLocation->z = pSprite->z;
			pLocation->ang = pSprite->ang;
		}
	}
}


/***********************************************************************
 * viewDrawSprite()
 *
 * Draws tiles to the screen using a unified set of flags
 * similar to those in overwritesprite/rotatesprite
 **********************************************************************/
void viewDrawSprite( long sx, long sy,
	long nZoom, short nAngle, short nTile, schar nShade, char nPal, ushort nFlags,
	long wx1, long wy1, long wx2, long wy2 )
{
	// convert x-flipping and y-flipping bits accordingly
	nAngle = (short)((long)((nAngle + ((nFlags & kDrawXFlip) << 2)) & kAngleMask));
	nFlags = (ushort)(nFlags + (nFlags & kDrawYFlip) ^ ((nFlags & kDrawXFlip) >> 6));

	// call rotatesprite passing only compatible bits in nFlags
	rotatesprite( sx, sy, nZoom, nAngle, nTile, nShade, nPal,
		(char)nFlags, wx1, wy1, wx2, wy2 );
}

#define kMaxViewFlames	9

static struct {
	short	tile;
	uchar	flags;
	char	pal;
	long	zoom;
	short	x;
	short	y;
} burnTable[ kMaxViewFlames ] = {
	{	kAnmFlame2,	(uchar)(kDrawScale),			0,	0x22000,	10,		200 },
	{	kAnmFlame2,	(uchar)(kDrawScale|kDrawXFlip),	0,	0x20000,	40,		200 },
	{	kAnmFlame2,	(uchar)(kDrawScale),			0,	0x19000,	85,		200 },
	{	kAnmFlame2,	(uchar)(kDrawScale),			0,	0x16000,	120,	200 },
	{	kAnmFlame2,	(uchar)(kDrawScale|kDrawXFlip),	0,	0x14000,	160,	200 },
	{	kAnmFlame2,	(uchar)(kDrawScale|kDrawXFlip),	0,	0x17000,	200,	200 },
	{	kAnmFlame2,	(uchar)(kDrawScale),			0,	0x18000,	235,	200 },
	{	kAnmFlame2,	(uchar)(kDrawScale),			0,	0x20000,	275,	200 },
	{	kAnmFlame2,	(uchar)(kDrawScale|kDrawXFlip),	0,	0x23000,	310,	200 },
};


/***********************************************************************
 * viewBurnTime()
 *
 * NOTES: i'd like to scale the zoom based on the remaining burnTime,
 * but only for the last five seconds of burnTime.
 *
 **********************************************************************/
void viewBurnTime( PLAYER */*pView*/ )
{
	for ( int i = 0; i < kMaxViewFlames; i++ )
	{
		// fake a sprite type for animateoffs to derive an index into an animation sequence
		short nTile = burnTable[i].tile;
		nTile += animateoffs( nTile, (ushort)(0x8000 + i));
		long nZoom = burnTable[i].zoom;
		viewDrawSprite( burnTable[i].x << 16, burnTable[i].y << 16, nZoom, 0, nTile, 0,
			burnTable[i].pal, burnTable[i].flags, windowx1, windowy1, windowx2, windowy2);
	}
}


void viewDrawScreen( void )
{
	static lastDacUpdate = 0;

	if ((gViewMode == kView3D) || (gViewMode == kView2DIcon) || (gOverlayMap))
	{
	 	DoSectorLighting();
		DoSectorPanning();
	}
	if ((gViewMode == kView3D) || (gOverlayMap))
	{
		int theView = gViewIndex;
		if ( powerupCheck(gMe, kItemCrystalBall - kItemBase) )
			theView = myconnectindex;

		short nSector = gView->sprite->sectnum;

		if (gInterpolate < 0 || gInterpolate > 0x10000)
			dprintf("gInterpolate = %x\n", gInterpolate);
		gInterpolate = ClipRange(gInterpolate, 0, 0x10000);

		long x = gPrevPlayerLoc[theView].x;
		long y = gPrevPlayerLoc[theView].y;
		long z = gPrevPlayerLoc[theView].z;
		short nAngle = gPrevPlayerLoc[theView].ang;
		int nHoriz = gPrevPlayerLoc[theView].horiz;
		int bobWidth = gPrevPlayerLoc[theView].bobWidth;
		int bobHeight = gPrevPlayerLoc[theView].bobHeight;
		int swayWidth = gPrevPlayerLoc[theView].swayWidth;
		int swayHeight = gPrevPlayerLoc[theView].swayHeight;

		// interpolate player position
		x += mulscale16(gView->sprite->x - gPrevPlayerLoc[theView].x, gInterpolate);
		y += mulscale16(gView->sprite->y - gPrevPlayerLoc[theView].y, gInterpolate);
		z += mulscale16(gView->sprite->z - gPrevPlayerLoc[theView].z, gInterpolate);
		nAngle += mulscale16(((gView->sprite->ang - gPrevPlayerLoc[theView].ang + kAngle180) & kAngleMask) - kAngle180, gInterpolate);
		nHoriz += mulscale16(gView->horiz - gPrevPlayerLoc[theView].horiz, gInterpolate);
		bobWidth += mulscale16(gView->bobWidth - gPrevPlayerLoc[theView].bobWidth, gInterpolate);
		bobHeight += mulscale16(gView->bobHeight - gPrevPlayerLoc[theView].bobHeight, gInterpolate);
		swayWidth += mulscale16(gView->swayWidth - gPrevPlayerLoc[theView].swayWidth, gInterpolate);
		swayHeight += mulscale16(gView->swayHeight - gPrevPlayerLoc[theView].swayHeight, gInterpolate);

		if ( gViewPos == kViewPosCenter )
		{
			x -= mulscale30(bobWidth, Sin(nAngle)) >> 4;
			y += mulscale30(bobWidth, Cos(nAngle)) >> 4;
			z += bobHeight + gView->compression - gView->eyeAboveZ + nHoriz * 10;
		}
		else
		{
			viewCalcPosition( gView->sprite, &x, &y, &z, &nAngle, &nSector );
			z -= gView->eyeAboveZ + (16 << 8);
		}

// 		should modify the z to handle earthquake type effects

//		something like this might come in handy for the crystal ball
//		need to mod based on number of players, and exclude self
//		int nEyes = (gGameGlock & 0x700) >> 8; // show through new "eyes" every two seconds
//		then render the other view to a tile and draw the tile (after the screen) using a lens effect

		long tiltLock = gScreenTilt;

		BOOL bDelirious   = FALSE;
		BOOL bClairvoyant = FALSE;

		bDelirious = ( powerupCheck(gView, kItemShroomDelirium - kItemBase) > 0 );
		if ( tiltLock != 0 || bDelirious )
		{
			if ( waloff[ TILTBUFFER ] == NULL )
				allocatepermanenttile( TILTBUFFER, 320L, 320L );

			dassert( waloff[ TILTBUFFER ] != NULL );
			setviewtotile( TILTBUFFER, 320L, 320L );
		}
		else
		{
			bClairvoyant = ( powerupCheck(gView, kItemCrystalBall - kItemBase) > 0 );
			if ( bClairvoyant && gViewIndex != theView )
			{
				dprintf("CLAIR SETUP\n");
				PLAYER *pViewed = &gPlayer[gViewIndex];
				if ( waloff[ BALLBUFFER ] == NULL )
					allocatepermanenttile( BALLBUFFER, 128L, 128L );
				dassert( waloff[ BALLBUFFER ] != NULL );
				setviewtotile( BALLBUFFER, 128L, 128L );
				long x = gPrevPlayerLoc[gViewIndex].x;
				long y = gPrevPlayerLoc[gViewIndex].y;
				long z = gPrevPlayerLoc[gViewIndex].z;
				short nAngle = gPrevPlayerLoc[gViewIndex].ang;
				int bobWidth = gPrevPlayerLoc[gViewIndex].bobWidth;
				int bobHeight = gPrevPlayerLoc[gViewIndex].bobHeight;
				int swayWidth = gPrevPlayerLoc[gViewIndex].swayWidth;
				int swayHeight = gPrevPlayerLoc[gViewIndex].swayHeight;
				x += mulscale16(pViewed->sprite->x - gPrevPlayerLoc[gViewIndex].x, gInterpolate);
				y += mulscale16(pViewed->sprite->y - gPrevPlayerLoc[gViewIndex].y, gInterpolate);
				z += mulscale16(pViewed->sprite->z - gPrevPlayerLoc[gViewIndex].z, gInterpolate);
//				nAngle += mulscale16(((pViewed->sprite->ang - gPrevPlayerLoc[gViewIndex].ang + kAngle180) & kAngleMask) - kAngle180, gInterpolate);
//				bobWidth += mulscale16(pViewed->bobWidth - gPrevPlayerLoc[gViewIndex].bobWidth, gInterpolate);
//				bobHeight += mulscale16(pViewed->bobHeight - gPrevPlayerLoc[gViewIndex].bobHeight, gInterpolate);
//				swayWidth += mulscale16(pViewed->swayWidth - gPrevPlayerLoc[gViewIndex].swayWidth, gInterpolate);
//				swayHeight += mulscale16(pViewed->swayHeight - gPrevPlayerLoc[gViewIndex].swayHeight, gInterpolate);
//				viewCalcPosition( pViewed->sprite, &x, &y, &z, &nAngle, &nSector );
//				z -= pViewed->eyeAboveZ + (16 << 8);
				DrawMirrors(x, y, z, nAngle, kHorizDefault + pViewed->horiz );
				drawrooms(x, y, z, nAngle, kHorizDefault + pViewed->horiz, nSector);
				viewProcessSprites(x, y, z);
				drawmasks();
				setviewback();
			}
		}

		if ( !bDelirious )
		{
			deliriumTilt = 0;
			deliriumTurn = 0;
			deliriumPitch = 0;
		}

		nAngle = (short)((nAngle + deliriumTurn) & kAngleMask);

		// prevent the player from being drawn (not taken care of automatically if x,y different)
		ushort oldcstat = gView->sprite->cstat;
		if ( gViewPos == kViewPosCenter )
			gView->sprite->cstat |= kSpriteInvisible;

		DrawMirrors(x, y, z, nAngle, kHorizDefault + nHoriz + deliriumPitch);
		drawrooms(x, y, z, nAngle, kHorizDefault + nHoriz + deliriumPitch, nSector);
		viewProcessSprites(x, y, z);
		drawmasks();

		gView->sprite->cstat = oldcstat;

		// process gotpics
		if ( GotPic(kPicFire) )
		{
			FireProcess();
			ClearGotPic(kPicFire);
		}

		if ( tiltLock != 0 || bDelirious )
		{
			dassert( waloff[ TILTBUFFER ] != NULL );

			setviewback();

			uchar nFlags = kRotateScale | kRotateYFlip | kRotateNoMask;
			if ( bDelirious )
				nFlags |= kRotateTranslucent;
			rotatesprite( 320 << 15, 200 << 15, 0x10000, (short)(tiltLock + kAngle90), TILTBUFFER, 0, 0, nFlags,
				gViewX0, gViewY0, gViewX1, gViewY1);
		}
		else
		{
			if ( bClairvoyant && gViewIndex != theView )
			{
				dprintf("CLAIR DRAW\n");
				rotatesprite(0,0,0x10000,0,BALLBUFFER,0,0,0,gViewX0, gViewY0, gViewX1, gViewY1);
			}
		}

		if ( gViewPos == kViewPosCenter )
		{
			x = 160 + (swayWidth >> 8);
			y = 200 + gPosture[gView->lifemode].swayV + (swayHeight >> 8) + (gView->compression >> 8);
			int nShade = sector[nSector].floorshade;
			WeaponDraw(gView, nShade, x, y);
		}

		if ( gViewPos == kViewPosCenter && actGetBurnTime(gView->xsprite) > (kTimerRate / 2) )
			viewBurnTime( gView );
	}

	if ( (gViewMode == kView2D || gViewMode == kView2DIcon) && !gOverlayMap)
		clearview(0);  // Clear screen to specified color

	if ( gViewMode == kView2DIcon )
		drawmapview(gView->sprite->x, gView->sprite->y, gZoom >> 2, gView->sprite->ang);

	if ( gViewMode == kView2D /*|| gViewMode == kView2DIcon */)
	{
//		drawoverheadmap(gView->sprite->x, gView->sprite->y, gZoom, gView->sprite->ang);
		DrawMap(gView->sprite->x, gView->sprite->y, gView->sprite->ang, gZoom);
	}
	viewDrawInterface();

	scrDisplayMessage(gStdColor[kColorWhite]);

	CalcFrameRate();
	if ( gShowFrameRate )
	{
		char buffer[16];
		sprintf(buffer, "%3i", gFrameRate);
		printext256(gViewX1 - 12, gViewY0, 31, -1, buffer, 1);
	}
	if ( gShowFrags )
	{
		for ( int i = 0; i < numplayers; i++)
		{
			char buffer[16];
			sprintf(buffer, "%i:%i", i, gPlayer[i].fragCount);
			printext256(gViewX0 + 12, gViewY0+(i*8), 31, -1, buffer, 1);
		}
	}

	scrSetDac(gGameClock - lastDacUpdate);
	lastDacUpdate = gGameClock;

	scrNextPage();
}
