#include <stdio.h>
#include <string.h>

#include "engine.h"
#include "ai.h"
#include "db.h"
#include "trig.h"
#include "misc.h"
#include "actor.h"
#include "player.h"
#include "globals.h"
#include "gameutil.h"
#include "multi.h"
#include "dude.h"
#include "debug4g.h"
#include "seq.h"

/****************************************************************************
** LOCAL CONSTANTS
****************************************************************************/

#define kAIThinkRate		16	// how often high level AI is sampled (in frames)
#define kAIThinkMask		(kAIThinkRate - 1)
#define kAIThinkTime		(kAIThinkRate * kFrameTicks)

#define kAxeZombieMeleeDist			M2X(2.0)	//M2X(1.6)
#define kFatZombieMeleeDist			M2X(2.0)	//M2X(1.6)
#define kFatZombiePukeDist1			M2X(10)
#define kFatZombiePukeDist2			M2X(6)

#define kHoundMeleeDist				M2X(1.6)
#define kCultistTFireDist			M2X(100)
#define kCultistSFireDist			M2X(10)
#define kCultistThrowDist1			M2X(16)
#define kCultistThrowDist2			M2X(10)

#define kFGargoyleMeleeDist		M2X(2.0)	//M2X(1.6)
#define kFGargoyleThrowDist1	M2X(12)		// used for bone
#define kFGargoyleThrowDist2	M2X(6)

#define kSGargoyleMeleeDist		M2X(2.0)	//M2X(1.6)
#define kSGargoyleBlastDist1	M2X(20)	// used for paralyzing blast
#define kSGargoyleBlastDist2	M2X(14)

#define kHoundBlastDist1		M2X(8)	// used for fire blast
#define kHoundBlastDist2		M2X(4)

#define kTentacleActivateDist	M2X(5)	// activates and stays on until target reaches deactivate distance?
#define kTentacleDeactivateDist	M2X(9)
#define kTentacleMeleeDist		M2X(2)

#define kPodActivateDist		M2X(8)
#define kPodDeactivateDist		M2X(14)
#define kPodFireDist1			M2X(12)
#define kPodFireDist2			M2X(8)

#define kGillBeastMeleeDist		M2X(1.6)
#define kGillBeastSummonDist1	M2X(16)
#define kGillBeastSummonDist2	M2X(12)

#define kEelMeleeDist		M2X(1)
#define kRatMeleeDist		M2X(1)
#define kHandMeleeDist		M2X(1)

#define kCerberusMeleeDist	M2X(2)
#define kCerberusBlastDist1	M2X(14)	// used for fireball
#define kCerberusBlastDist2	M2X(10)

#define kPhantasmMeleeDist	M2X(1.6)
#define kPhantasmThrowDist1	M2X(16)
#define kPhantasmThrowDist2	M2X(12)


static int cumulDamage[kMaxXSprites];	// cumulative damage per frame


typedef void (*STATEFUNC)( SPRITE *sprite, XSPRITE *xsprite);


struct STATE
{
	int				seqId;
	SEQCALLBACK		seqCallback;
	int				ticks;
	STATEFUNC		enter;
	STATEFUNC		move;
	STATEFUNC		think;
	STATE			*next;
};


static void CultistTFireCallback( int /* type */, int nXIndex );
static void CultistSFireCallback( int /* type */, int nXIndex );
static void CultistThrowCallback( int /* type */, int nXIndex );

static void AxeZombieAttackCallback( int /* type */, int nXIndex );
static void FatZombieAttackCallback( int /* type */, int nXIndex );
static void FatZombiePukeCallback( int /* type */, int nXIndex );

static void FGargoyleSlashCallback( int /* type */, int nXIndex );	// flesh gargoyle callbacks
static void FGargoyleThrowCallback( int /* type */, int nXIndex );
static void tFGargoyleSearch( SPRITE *pSprite, XSPRITE *pXSprite );
static void tFGargoyleGoto( SPRITE *pSprite, XSPRITE *pXSprite );
static void tFGargoyleChase( SPRITE *pSprite, XSPRITE *pXSprite );

//static void SGargoyleSlashCallback( int /* type */, int nXIndex );	// stone gargoyle callbacks
//static void SGargoyleThrowCallback( int /* type */, int nXIndex );

static void HoundAttackCallback( int /* type */, int nXIndex );		// hell hound callbacks

static void mForward( SPRITE *pSprite, XSPRITE *pXSprite );
static void mDodge( SPRITE *pSprite, XSPRITE *pXSprite );
static void tFindTarget( SPRITE *pSprite, XSPRITE *pXSprite );

static void tCultistSearch( SPRITE *pSprite, XSPRITE *pXSprite );
static void tCultistGoto( SPRITE *pSprite, XSPRITE *pXSprite );
static void tCultistChaseThink( SPRITE *pSprite, XSPRITE *pXSprite );

static void tFatZombieSearch( SPRITE *pSprite, XSPRITE *pXSprite );
static void tFatZombieGoto( SPRITE *pSprite, XSPRITE *pXSprite );
static void tFatZombieChase( SPRITE *pSprite, XSPRITE *pXSprite );

static void tAxeZombieSearch( SPRITE *pSprite, XSPRITE *pXSprite );
static void tAxeZombieGoto( SPRITE *pSprite, XSPRITE *pXSprite );
static void tAxeZombieChase( SPRITE *pSprite, XSPRITE *pXSprite );

static void eEarthZombie( SPRITE *pSprite, XSPRITE *pXSprite );
static void eFleshStatue( SPRITE *pSprite, XSPRITE *pXSprite );

static STATE genIdle	= { kSeqDudeIdle, NULL, 0, NULL, NULL, NULL, NULL };
static STATE genRecoil	= { kSeqDudeRecoil, NULL, 20, NULL, NULL, NULL, &genIdle };

static STATE cultistIdle	= { kSeqDudeIdle, NULL, 0, NULL, NULL, tFindTarget, NULL };
static STATE tCultistChase	= { kSeqCultistWalk, NULL, 0, NULL, mForward, tCultistChaseThink, NULL };
static STATE cultistDodge	= { kSeqCultistWalk, NULL, 90, NULL, mDodge, NULL, &tCultistChase };
static STATE cultistGoto	= { kSeqCultistWalk, NULL, 3600, NULL, mForward, tCultistGoto, &cultistIdle };
static STATE cultistThrow	= { kSeqCultistAttack2, CultistThrowCallback, 120, NULL, NULL, NULL, &tCultistChase };
static STATE cultistSearch	= { kSeqCultistWalk, NULL, 3600, NULL, mForward, tCultistSearch, &cultistIdle };
static STATE cultistSFire	= { kSeqCultistAttack1, CultistSFireCallback, 60, NULL, NULL, NULL, &tCultistChase };
static STATE cultistTFire	= { kSeqCultistAttack1, CultistTFireCallback, 0, NULL, NULL, tCultistChaseThink, &cultistTFire };
static STATE cultistRecoil	= { kSeqDudeRecoil, NULL, 0, NULL, NULL, NULL, &cultistDodge };

static STATE fatZombieIdle		= { kSeqDudeIdle, NULL, 0, NULL, NULL, tFindTarget, NULL };
static STATE fatZombieChase		= { kSeqFatZombieWalk, NULL, 0, NULL, mForward, tFatZombieChase, NULL };
static STATE fatZombieGoto		= { kSeqFatZombieWalk, NULL, 3600, NULL, mForward, tFatZombieGoto, &fatZombieIdle };
static STATE fatZombieHack		= { kSeqFatZombieAttack, FatZombieAttackCallback, 120, NULL, NULL, NULL, &fatZombieChase };
static STATE fatZombiePuke		= { kSeqFatZombieAttack, FatZombiePukeCallback, 120, NULL, NULL, NULL, &fatZombieChase };
static STATE fatZombieSearch	= { kSeqFatZombieWalk, NULL, 3600, NULL, mForward, tFatZombieSearch, &fatZombieIdle };
static STATE fatZombieRecoil	= { kSeqDudeRecoil, NULL, 0, NULL, NULL, NULL, &fatZombieChase };

static STATE axeZombieIdle		= { kSeqDudeIdle, NULL, 0, NULL, NULL, tFindTarget, NULL };
static STATE axeZombieChase		= { kSeqAxeZombieWalk, NULL, 0, NULL, mForward, tAxeZombieChase, NULL };
static STATE axeZombieGoto		= { kSeqAxeZombieWalk, NULL, 3600, NULL, mForward, tAxeZombieGoto, &axeZombieIdle };
static STATE axeZombieHack		= { kSeqAxeZombieAttack, AxeZombieAttackCallback, 120, NULL, NULL, NULL, &axeZombieChase };
static STATE axeZombieSearch	= { kSeqAxeZombieWalk, NULL, 3600, NULL, mForward, tAxeZombieSearch, &axeZombieIdle };
static STATE axeZombieRecoil	= { kSeqDudeRecoil, NULL, 0, NULL, NULL, NULL, &axeZombieChase };

static STATE earthZombieIdle		= { kSeqEarthZombieIdle, NULL, 0, NULL, NULL, tFindTarget, NULL };
static STATE earthZombieUp2			= { kSeqDudeIdle, NULL, 1, eEarthZombie, NULL, NULL, &axeZombieSearch };
static STATE earthZombieUp			= { kSeqEarthZombieUp, NULL, 180, NULL, NULL, NULL, &earthZombieUp2 };

static STATE fleshGargoyleIdle		= { kSeqDudeIdle, NULL, 0, NULL, NULL, tFindTarget, NULL };
static STATE fleshGargoyleChase		= { kSeqDudeIdle, NULL, 0, NULL, mForward, tFGargoyleChase, NULL };
//static STATE fleshGargoyleDodge		= { kSeqDudeIdle, NULL, 90, NULL, mDodge, NULL, &fleshGargoyleChase };
static STATE fleshGargoyleGoto		= { kSeqDudeIdle, NULL, 3600, NULL, mForward, tFGargoyleGoto, &fleshGargoyleIdle };
//static STATE fleshGargoyleSwoop		= { kSeqGargoyleFly, NULL, 0, NULL, mForward, tFGargoyleChase, NULL };
static STATE fleshGargoyleSlash		= { kSeqGargoyleAttack, FGargoyleSlashCallback, 120, NULL, NULL, NULL, &fleshGargoyleChase };
static STATE fleshGargoyleThrow		= { kSeqGargoyleAttack, FGargoyleThrowCallback, 120, NULL, NULL, NULL, &fleshGargoyleChase };
static STATE fleshGargoyleRecoil	= { kSeqDudeRecoil, NULL, 0, NULL, NULL, NULL, &fleshGargoyleChase };
static STATE fleshGargoyleSearch	= { kSeqDudeIdle, NULL, 3600, NULL, mForward, tFGargoyleSearch, &fleshGargoyleIdle };

static STATE fleshStatueMorph2		= { -1, NULL, 0, eFleshStatue, NULL, NULL, &fleshGargoyleSearch };
static STATE fleshStatueMorph		= { kSeqFleshStatueTurn, NULL, 0, NULL, NULL, NULL, &fleshStatueMorph2 };

STATE houndAttack	= { kSeqHoundAttack1, HoundAttackCallback, 0, NULL, NULL, NULL, NULL };

/****************************************************************************
** GLOBALS
****************************************************************************/


/****************************************************************************
** LOCALS
****************************************************************************/


static void CultistTFireCallback( int /* type */, int nXIndex )
{
	XSPRITE *pXSprite = &xsprite[nXIndex];
	int nSprite = pXSprite->reference;
	SPRITE *pSprite = &sprite[nSprite];

	int dx = Cos(pSprite->ang) >> 16;
	int dy = Sin(pSprite->ang) >> 16;
	int dz = 0;

	// dispersal modifiers here
	dx += BiRandom(2000);
	dy += BiRandom(2000);
	dz += BiRandom(2000 << 4);

	actFireVector(nSprite, pSprite->z, dx, dy, dz, 0,
		kDamageBullet, 2 );
}


static void CultistSFireCallback( int /* type */, int nXIndex )
{
	XSPRITE *pXSprite = &xsprite[nXIndex];
	int nSprite = pXSprite->reference;
	SPRITE *pSprite = &sprite[nSprite];

	int dx = Cos(pSprite->ang) >> 16;
	int dy = Sin(pSprite->ang) >> 16;
	int dz = 0;

	// aim modifiers
	dx += BiRandom(4000);
	dy += BiRandom(4000);
	dz += BiRandom(4000 << 4);

	for ( int i = 0; i < 8; i++ )
	{
		actFireVector(nSprite, pSprite->z, dx + BiRandom(2000), dy + BiRandom(2000),
			dz + BiRandom(2000 << 4), 0, kDamageBullet, 4 );
	}
}


static void CultistThrowCallback( int /* type */, int nXIndex )
{
	XSPRITE *pXSprite = &xsprite[nXIndex];
	int nSprite = pXSprite->reference;
	SPRITE *pSprite = &sprite[nSprite];

	actFireThing(nSprite, pSprite->z, 0, kThingTNTStick);
//	pXSprite->stateTimer += Random(60);
}


static void AxeZombieAttackCallback( int /* type */, int nXIndex )
{
	XSPRITE *pXSprite = &xsprite[nXIndex];
	int nSprite = pXSprite->reference;
	SPRITE *pSprite = &sprite[nSprite];

	int dx = Cos(pSprite->ang) >> 16;
	int dy = Sin(pSprite->ang) >> 16;
	int dz = 0;

	actFireVector(nSprite, pSprite->z, dx, dy, dz, kAxeZombieMeleeDist,
		kDamageStab, 20 );
}


static void FatZombieAttackCallback( int /* type */, int nXIndex )
{
	XSPRITE *pXSprite = &xsprite[nXIndex];
	int nSprite = pXSprite->reference;
	SPRITE *pSprite = &sprite[nSprite];

	int dx = Cos(pSprite->ang) >> 16;
	int dy = Sin(pSprite->ang) >> 16;
	int dz = 0;

	actFireVector(nSprite, pSprite->z, dx, dy, dz, kFatZombieMeleeDist,
		kDamageStab, 10 );
}


static void FatZombiePukeCallback( int /* type */, int nXIndex )
{
	XSPRITE *pXSprite = &xsprite[nXIndex];
	int nSprite = pXSprite->reference;
	SPRITE *pSprite = &sprite[nSprite];

	int dx = Cos(pSprite->ang) >> 16;
	int dy = Sin(pSprite->ang) >> 16;
	int dz = 0;

	actFireThing(nSprite, pSprite->z, 0, kThingTNTStick);
}


static void FGargoyleSlashCallback( int /* type */, int nXIndex )
{
	XSPRITE *pXSprite = &xsprite[nXIndex];
	int nSprite = pXSprite->reference;
	SPRITE *pSprite = &sprite[nSprite];

	int dx = Cos(pSprite->ang) >> 16;
	int dy = Sin(pSprite->ang) >> 16;
	int dz = 0;

	actFireVector(nSprite, pSprite->z, dx, dy, dz, kFGargoyleMeleeDist,
		kDamageStab, 10 );
}


static void FGargoyleThrowCallback( int /* type */, int nXIndex )
{
	XSPRITE *pXSprite = &xsprite[nXIndex];
	int nSprite = pXSprite->reference;
	SPRITE *pSprite = &sprite[nSprite];

	actFireThing(nSprite, pSprite->z, 0, kThingBoneClub);
//	pXSprite->stateTimer += Random(60);
}


static void HoundAttackCallback( int /* type */, int nXIndex )
{
	XSPRITE *pXSprite = &xsprite[nXIndex];
	int nSprite = pXSprite->reference;
	SPRITE *pSprite = &sprite[nSprite];

	int dx = Cos(pSprite->ang) >> 16;
	int dy = Sin(pSprite->ang) >> 16;
	int dz = 0;

	actFireVector(nSprite, pSprite->z, dx, dy, dz, kHoundMeleeDist,
		kDamageStab, 10 );
}


static void NewState( SPRITE *pSprite, XSPRITE *pXSprite, STATE *pState )
{
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];
	pXSprite->stateTimer = pState->ticks;
	pXSprite->aiState = pState;

	if ( pState->seqId >= 0 )
	{
		if ( pDudeInfo->pSeq[pState->seqId] == NULL )
		{
			dprintf("NULL sequence, dudeType = %d, seqId = %d\n", pSprite->type, pState->seqId);
			return;
		}
		seqSpawn(pDudeInfo->pSeq[pState->seqId], SS_SPRITE, pSprite->extra, pState->seqCallback);
	}

	// call the enter function if defined
	if ( pState->enter )
		pState->enter(pSprite, pXSprite);
}


BOOL CanMove( SPRITE *pSprite, int nTarget, int ang, int dist )
{
	long x, y, z;
	short nSector;
	int zTop, zBot;
	int r;

	GetSpriteExtents(pSprite, &zTop, &zBot);

	x = pSprite->x;
	y = pSprite->y;
	z = pSprite->z;
	nSector = pSprite->sectnum;

	int dx = mulscale30(dist, Cos(ang));
	int dy = mulscale30(dist, Sin(ang));

	HITINFO hitInfo;
	HitScan(pSprite, pSprite->z, dx, dy, 0, &hitInfo);
	int hitDist = qdist(pSprite->x - hitInfo.hitx, pSprite->y - hitInfo.hity);
	if ( hitDist < dist + (pSprite->clipdist << 2) )
	{
		// okay to be blocked by target
		if ( (r & 0xC000) == 0xC000 )
		{
			if ( (r & 0x3FFF) == nTarget )
				return TRUE;
		}
		return FALSE;
	}

	updatesector(pSprite->x + dx, pSprite->y + dy, &nSector);

//	long ceilZ, ceilHit, floorZ, floorHit;
//	getzrange( x, y, z, nSector, &ceilZ, &ceilHit, &floorZ, &floorHit,
//		pSprite->clipdist << 2, 0);
	long floorZ = sector[nSector].floorz;

	// this should go into the dude table and be time relative
	if ( floorZ - zBot - z > M2Z(1.5) )
		return FALSE;

	return TRUE;
}


static void ChooseDirection( SPRITE *pSprite, XSPRITE *pXSprite, int ang )
{
	dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];

	int dang = ((kAngle180 + ang - pSprite->ang) & kAngleMask) - kAngle180;

	long sin = Sin(pSprite->ang);
	long cos = Cos(pSprite->ang);

	// find vel and svel relative to current angle
	long vel = dmulscale30(pSprite->xvel, cos, pSprite->yvel, sin);
	long svel = dmulscale30(pSprite->xvel, sin, -pSprite->yvel, cos);

	int avoidDist = vel * kAIThinkTime >> 4;
	int turnTo = kAngle60;
	if (dang < 0 )
		turnTo = -turnTo;

	// clear movement toward target?
	if ( CanMove(pSprite, pXSprite->target, pSprite->ang + dang, avoidDist) )
	{
		pXSprite->goalAng = (pSprite->ang + dang) & kAngleMask;
	}
	// clear movement partially toward target?
	else if ( CanMove(pSprite, pXSprite->target, pSprite->ang + dang / 2, avoidDist) )
	{
		pXSprite->goalAng = (pSprite->ang + dang / 2) & kAngleMask;
	}
	// try turning in target direction
	else if ( CanMove(pSprite, pXSprite->target, pSprite->ang + turnTo, avoidDist) )
	{
		pXSprite->goalAng = (pSprite->ang + turnTo) & kAngleMask;
	}
	// clear movement straight?
	else if ( CanMove(pSprite, pXSprite->target, pSprite->ang, avoidDist) )
	{
		pXSprite->goalAng = pSprite->ang;
	}
	// try turning away
	else if ( CanMove(pSprite, pXSprite->target, pSprite->ang - turnTo, avoidDist) )
	{
		pXSprite->goalAng = (pSprite->ang - turnTo) & kAngleMask;
	}
	else
	{
	// just turn around
		pXSprite->goalAng = (pSprite->ang + kAngle180) & kAngleMask;
	}

	// choose dodge direction
	pXSprite->dodgeDir = (Random(2) == 0) ? 1 : -1;
	if ( !CanMove(pSprite, pXSprite->target, pSprite->ang + kAngle90 * pXSprite->dodgeDir,
		pDudeInfo->sideSpeed * 90 >> 4) )
	{
		pXSprite->dodgeDir = -pXSprite->dodgeDir;
		if ( !CanMove(pSprite, pXSprite->target, pSprite->ang + kAngle90 * pXSprite->dodgeDir,
			pDudeInfo->sideSpeed * 90 >> 4) )
			pXSprite->dodgeDir = 0;
	}

//	pSprite->zvel = (short)(dz >> 8);
}


static void eEarthZombie( SPRITE *pSprite, XSPRITE */*pXSprite*/ )
{
	pSprite->flags |= kAttrMove;
	pSprite->type = kDudeAxeZombie;
}


static void eFleshStatue( SPRITE *pSprite, XSPRITE */*pXSprite*/ )
{
	pSprite->type = kDudeFleshGargoyle;
}


static void mForward( SPRITE *pSprite, XSPRITE *pXSprite )
{
	dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];

	pXSprite->moveState = kMoveWalk;

	int dang = ((kAngle180 + pXSprite->goalAng - pSprite->ang) & kAngleMask) - kAngle180;
	int maxTurn = pDudeInfo->angSpeed * kFrameTicks >> 4;

	pSprite->ang = (short)((pSprite->ang + ClipRange(dang, -maxTurn, maxTurn)) & kAngleMask);
	int accel = pDudeInfo->frontAccel * kFrameTicks;

	// don't move forward if trying to turn around
	if ( qabs(dang) > kAngle60 )
		return;

	long sin = Sin(pSprite->ang);
	long cos = Cos(pSprite->ang);

	// find vel and svel relative to current angle
	long vel = dmulscale30(pSprite->xvel, cos, pSprite->yvel, sin);
	long svel = dmulscale30(pSprite->xvel, sin, -pSprite->yvel, cos);

	// acceleration
	if ( accel > 0 )
	{
		if ( vel < pDudeInfo->frontSpeed )
			vel = ClipHigh(vel + accel, pDudeInfo->frontSpeed);
	}
	else
	{
		if ( vel > -pDudeInfo->backSpeed )
			vel = ClipLow(vel + accel, pDudeInfo->backSpeed);
	}

	// reconstruct x and y velocities
	pSprite->xvel = (short)dmulscale30(vel, cos, svel, sin);
	pSprite->yvel = (short)dmulscale30(vel, sin, -svel, cos);
}


static void mDodge( SPRITE *pSprite, XSPRITE *pXSprite )
{
	dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];

	pXSprite->moveState = kMoveWalk;

	int dang = ((kAngle180 + pXSprite->goalAng - pSprite->ang) & kAngleMask) - kAngle180;
	int maxTurn = pDudeInfo->angSpeed * kFrameTicks >> 4;
	pSprite->ang = (short)((pSprite->ang + ClipRange(dang, -maxTurn, maxTurn)) & kAngleMask);

	if ( pXSprite->dodgeDir == 0 )
		return;

	int accel = pDudeInfo->sideAccel * kFrameTicks;
	long sin = Sin(pSprite->ang);
	long cos = Cos(pSprite->ang);

	// find vel and svel relative to current angle
	long vel = dmulscale30(pSprite->xvel, cos, pSprite->yvel, sin);
	long svel = dmulscale30(pSprite->xvel, sin, -pSprite->yvel, cos);

	if ( pXSprite->dodgeDir > 0 )
	{
		if ( svel < pDudeInfo->sideSpeed )
			svel = ClipHigh(svel + accel, pDudeInfo->sideSpeed);
	}
	else
	{
		if ( svel > -pDudeInfo->sideSpeed )
			svel = ClipLow(svel - accel, -pDudeInfo->sideSpeed);
	}

	// reconstruct x and y velocities
	pSprite->xvel = (short)dmulscale30(vel, cos, svel, sin);
	pSprite->yvel = (short)dmulscale30(vel, sin, -svel, cos);
}


void ActivateDude( SPRITE *pSprite, XSPRITE *pXSprite )
{
	dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];


	if ( pXSprite->state == 0 )
	{
		// this doesn't take into account sprites triggered w/o a target location....
		int nAngle = getangle(pXSprite->targetX - pSprite->x, pXSprite->targetY - pSprite->y);
		ChooseDirection(pSprite, pXSprite, nAngle);
		pXSprite->state = 1;
	}

	switch ( pSprite->type )
	{
		case kDudeBrownSpider:
		case kDudeRedSpider:
		case kDudeBlackSpider:
			pSprite->cstat &= ~kSpriteFlipY;
			break;

		case kDudeTommyCultist:
		case kDudeShotgunCultist:
			if (pXSprite->target == -1)
				NewState(pSprite, pXSprite, &cultistSearch);
			else
				NewState(pSprite, pXSprite, &tCultistChase);
			break;

		case kDudeAxeZombie:
			if (pXSprite->target == -1)
				NewState(pSprite, pXSprite, &axeZombieSearch);
			else
				NewState(pSprite, pXSprite, &axeZombieChase);
			break;

		case kDudeEarthZombie:
			dassert(pXSprite->aiState == &earthZombieIdle);
			NewState(pSprite, pXSprite, &earthZombieUp);
			break;

		case kDudeFatZombie:
			if (pXSprite->target == -1)
				NewState(pSprite, pXSprite, &fatZombieSearch);
			else
				NewState(pSprite, pXSprite, &fatZombieChase);
			break;

		case kDudeFleshStatue:
			NewState(pSprite, pXSprite, &fleshStatueMorph);
			break;

		case kDudeStoneStatue:
			seqSpawn(pDudeInfo->pSeq[kSeqStoneStatueTurn], SS_SPRITE, pSprite->extra);
			pSprite->type = kDudeStoneGargoyle;
			break;
	}
}


/*******************************************************************************
	FUNCTION:		SetTarget()

	DESCRIPTION:	Target a location (as opposed to a sprite)
*******************************************************************************/
static void SetTarget( XSPRITE *pXSprite, int x, int y, int z )
{
	pXSprite->target = -1;
	pXSprite->targetX = x;
	pXSprite->targetY = y;
	pXSprite->targetZ = z;
}


static void SetTarget( XSPRITE *pXSprite, int nTarget )
{
	dassert(nTarget >= 0 && nTarget < kMaxSprites);
	SPRITE *pTarget = &sprite[nTarget];

	if (pTarget->type < kDudeBase || pTarget->type >= kDudeMax)
		return;

	dassert(pTarget->type >= kDudeBase && pTarget->type < kDudeMax);
	DUDEINFO *pTargetInfo = &dudeInfo[pTarget->type - kDudeBase];

	pXSprite->target = nTarget;
	pXSprite->targetX = pTarget->x;
	pXSprite->targetY = pTarget->y;
	pXSprite->targetZ = pTarget->z - (pTargetInfo->eyeHeight * pTarget->yrepeat << 2);
}


void aiDamageSprite( SPRITE *pSprite, XSPRITE *pXSprite, int nSource, DAMAGE_TYPE nDamageType, int nDamage )
{
	(void)nDamageType;

	dassert(nSource < kMaxSprites);

	if (pXSprite->health == 0)
		return;

	cumulDamage[pSprite->extra] += nDamage;	// add to cumulative damage

	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];

	if (nSource >= 0 && nSource != pXSprite->reference)
	{
		if (pXSprite->target == -1)
		{
			// give a dude a target
			SetTarget(pXSprite, nSource);
			ActivateDude(pSprite, pXSprite);
		}
		else if (nSource != pXSprite->target)
		{
			// retarget
			int nThresh = nDamage;

			if ( sprite[nSource].type == pSprite->type )
				nThresh *= pDudeInfo->changeTargetKin;
			else
				nThresh *= pDudeInfo->changeTarget;

			if ( Random(0x10000) < nThresh )
			{
				SetTarget(pXSprite, nSource);
				ActivateDude(pSprite, pXSprite);
			}
		}

		// you DO need special processing here or somewhere else (your choice) for dodging
		switch ( pSprite->type )
		{
			case kDudeTommyCultist:
			case kDudeShotgunCultist:
//					if (nDamage >= (pDudeInfo->hinderDamage << 2))
					NewState(pSprite, pXSprite, &cultistDodge);
				break;

			default:
				break;
		}
	}
}


void RecoilDude( SPRITE *pSprite, XSPRITE *pXSprite )
{
	dprintf("Recoiling dude\n");
	switch ( pSprite->type )
	{
		case kDudeTommyCultist:
		case kDudeShotgunCultist:
			NewState(pSprite, pXSprite, &cultistRecoil);
			break;

		case kDudeFatZombie:
			NewState(pSprite, pXSprite, &fatZombieRecoil);
			break;

		case kDudeAxeZombie:
		case kDudeEarthZombie:
			NewState(pSprite, pXSprite, &axeZombieRecoil);
			break;

		case kDudeFleshGargoyle:
			NewState(pSprite, pXSprite, &fleshGargoyleRecoil);
			break;

		default:
			NewState(pSprite, pXSprite, &genRecoil);
			break;
	}
}


static void tFindTarget( SPRITE *pSprite, XSPRITE *pXSprite )
{
	dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];

	for (int i = 0; i < numplayers; i++)
	{
		PLAYER *pPlayer = &gPlayer[i];
		if ( powerupCheck( pPlayer, kItemLtdInvisibility - kItemBase ) > 0 )
			continue;

		int x = pPlayer->sprite->x;
		int y = pPlayer->sprite->y;
		int z = pPlayer->sprite->z;
		short nSector = pPlayer->sprite->sectnum;

		int dx = x - pSprite->x;
		int dy = y - pSprite->y;

		int dist = qdist(dx, dy);

		if ( dist <= pDudeInfo->seeDist || dist <= pDudeInfo->hearDist )
		{
			int eyeAboveZ = pDudeInfo->eyeHeight * pSprite->yrepeat << 2;

			// is there a line of sight to the player?
			if ( cansee(x, y, z, nSector, pSprite->x, pSprite->y, pSprite->z - eyeAboveZ,
				pSprite->sectnum) )
			{
				int nAngle = getangle(dx, dy);
				int losAngle = ((kAngle180 + nAngle - pSprite->ang) & kAngleMask) - kAngle180;

				// is the player visible?
				if ( dist < pDudeInfo->seeDist && qabs(losAngle) <= pDudeInfo->periphery )
				{
					SetTarget( pXSprite, pPlayer->nSprite );
					ActivateDude( pSprite, pXSprite );
					return;
				}

				// we may want to make hearing a function of sensitivity, rather than distance
				if ( dist < pDudeInfo->hearDist )
				{
					SetTarget(pXSprite, x, y, z);
					ActivateDude( pSprite, pXSprite );
					return;
				}
			}
		}
	}
}


static BOOL TargetNearExplosion( SPRITE *pTarget )
{
	for (short nSprite = headspritesect[pTarget->sectnum]; nSprite >= 0; nSprite = nextspritesect[nSprite])
	{
		// check for TNT sticks or explosions in the same sector as the target
		if (sprite[nSprite].type == kThingTNTStick || sprite[nSprite].statnum == kStatExplosion)
			return TRUE; // indicate danger
	}
	return FALSE;
}


static void tCultistSearch( SPRITE *pSprite, XSPRITE *pXSprite )
{
	ChooseDirection(pSprite, pXSprite, pXSprite->goalAng);
	tFindTarget(pSprite, pXSprite);
}


static void tCultistGoto( SPRITE *pSprite, XSPRITE *pXSprite )
{
	int dx, dy, dist;

	dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];

	SPRITE *pTarget = &sprite[pXSprite->target];
	XSPRITE *pXTarget = &xsprite[pTarget->extra];

	dx = pXSprite->targetX - pSprite->x;
	dy = pXSprite->targetY - pSprite->y;

	int nAngle = getangle(dx, dy);
	dist = qdist(dx, dy);

	ChooseDirection(pSprite, pXSprite, nAngle);

	// if reached target, change to search mode
	if ( dist < M2X(1.0) && qabs(pSprite->ang - nAngle) < pDudeInfo->periphery )
		NewState(pSprite, pXSprite, &cultistSearch);

	tFindTarget(pSprite, pXSprite);
}


static void tCultistChaseThink( SPRITE *pSprite, XSPRITE *pXSprite )
{
	if ( pXSprite->target == -1)
	{
		NewState(pSprite, pXSprite, &cultistGoto);
		return;
	}

	int dx, dy, dist;

	dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];

	dassert(pXSprite->target >= 0 && pXSprite->target < kMaxSprites);
	SPRITE *pTarget = &sprite[pXSprite->target];
	XSPRITE *pXTarget = &xsprite[pTarget->extra];

	// check target
	dx = pTarget->x - pSprite->x;
	dy = pTarget->y - pSprite->y;

	ChooseDirection(pSprite, pXSprite, getangle(dx, dy));

	if ( pXTarget->health == 0 )
	{
		// target is dead
		NewState(pSprite, pXSprite, &cultistSearch);
		return;
	}

	if ( IsPlayerSprite( pTarget ) )
	{
		PLAYER *pPlayer = &gPlayer[ pTarget->type - kDudePlayer1 ];
		if ( powerupCheck( pPlayer, kItemLtdInvisibility - kItemBase ) > 0 )
		{
			NewState(pSprite, pXSprite, &cultistSearch);
			return;
		}
	}

	dist = qdist(dx, dy);
	if ( dist <= pDudeInfo->seeDist )
	{
		int nAngle = getangle(dx, dy);
		int losAngle = ((kAngle180 + nAngle - pSprite->ang) & kAngleMask) - kAngle180;
		int eyeAboveZ = pDudeInfo->eyeHeight * pSprite->yrepeat << 2;

		// is there a line of sight to the target?
		if ( cansee(pTarget->x, pTarget->y, pTarget->z, pTarget->sectnum,
			pSprite->x, pSprite->y, pSprite->z - eyeAboveZ, pSprite->sectnum) )
		{
			// is the target visible?
			if ( dist < pDudeInfo->seeDist && qabs(losAngle) <= pDudeInfo->periphery )
			{
				SetTarget(pXSprite, pXSprite->target);

				// check to see if we can attack
				switch ( pSprite->type )
				{
					case kDudeTommyCultist:
						if ( dist < kCultistThrowDist1 && dist > kCultistThrowDist2 && qabs(losAngle) < kAngle15
						&& !TargetNearExplosion(pTarget) && (pTarget->flags & kAttrFall)
						&& !( IsPlayerSprite(pXSprite->target) && gPlayer[pTarget->type - kDudePlayer1].run ))
							NewState(pSprite, pXSprite, &cultistThrow);
						else if ( dist < kCultistTFireDist && qabs(losAngle) < kAngle15 )
							NewState(pSprite, pXSprite, &cultistTFire);
						break;

					case kDudeShotgunCultist:
						if ( dist < kCultistThrowDist1 && dist > kCultistThrowDist2 && qabs(losAngle) < kAngle15
						&& !TargetNearExplosion(pTarget) && (pTarget->flags & kAttrFall)
						&& !( IsPlayerSprite(pXSprite->target) && gPlayer[pTarget->type - kDudePlayer1].run ))
							NewState(pSprite, pXSprite, &cultistThrow);
						else if ( dist < kCultistSFireDist && qabs(losAngle) < kAngle15 )
							NewState(pSprite, pXSprite, &cultistSFire);
						break;
				}
				return;
			}
		}
	}
	dprintf("Cultist %d lost sight of target %d\n", pXSprite->reference, pXSprite->target);
	NewState(pSprite, pXSprite, &cultistGoto);
	pXSprite->target = -1;
}


static void tFatZombieSearch( SPRITE *pSprite, XSPRITE *pXSprite )
{
	ChooseDirection(pSprite, pXSprite, pXSprite->goalAng);
	tFindTarget(pSprite, pXSprite);
}


static void tFatZombieGoto( SPRITE *pSprite, XSPRITE *pXSprite )
{
	int dx, dy, dist;

	dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];

	SPRITE *pTarget = &sprite[pXSprite->target];
	XSPRITE *pXTarget = &xsprite[pTarget->extra];

	dx = pXSprite->targetX - pSprite->x;
	dy = pXSprite->targetY - pSprite->y;

	int nAngle = getangle(dx, dy);
	dist = qdist(dx, dy);

	ChooseDirection(pSprite, pXSprite, nAngle);

	// if reached target, change to search mode
	if ( dist < M2X(1.0) && qabs(pSprite->ang - nAngle) < pDudeInfo->periphery )
		NewState(pSprite, pXSprite, &fatZombieSearch);

	tFindTarget(pSprite, pXSprite);
}


static void tFatZombieChase( SPRITE *pSprite, XSPRITE *pXSprite )
{
	if ( pXSprite->target == -1)
	{
		NewState(pSprite, pXSprite, &fatZombieGoto);
		return;
	}

	int dx, dy, dist;

	dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];

	dassert(pXSprite->target >= 0 && pXSprite->target < kMaxSprites);
	SPRITE *pTarget = &sprite[pXSprite->target];
	XSPRITE *pXTarget = &xsprite[pTarget->extra];

	// check target
	dx = pTarget->x - pSprite->x;
	dy = pTarget->y - pSprite->y;

	ChooseDirection(pSprite, pXSprite, getangle(dx, dy));

	if ( pXTarget->health == 0 )
	{
		// target is dead
		NewState(pSprite, pXSprite, &fatZombieSearch);
		return;
	}

	if ( IsPlayerSprite( pTarget ) )
	{
		PLAYER *pPlayer = &gPlayer[ pTarget->type - kDudePlayer1 ];
		if ( powerupCheck( pPlayer, kItemLtdInvisibility - kItemBase ) > 0
		|| powerupCheck( pPlayer, kItemDeathMask - kItemBase ) > 0 )
		{
			NewState(pSprite, pXSprite, &fatZombieSearch);
			return;
		}
	}

	dist = qdist(dx, dy);

	if ( dist <= pDudeInfo->seeDist )
	{
		int nAngle = getangle(dx, dy);
		int losAngle = ((kAngle180 + nAngle - pSprite->ang) & kAngleMask) - kAngle180;
		int eyeAboveZ = pDudeInfo->eyeHeight * pSprite->yrepeat << 2;

		// is there a line of sight to the target?
		if ( cansee(pTarget->x, pTarget->y, pTarget->z, pTarget->sectnum,
			pSprite->x, pSprite->y, pSprite->z - eyeAboveZ, pSprite->sectnum) )
		{
			// is the target visible?
			if ( dist < pDudeInfo->seeDist && qabs(losAngle) <= pDudeInfo->periphery )
			{
				SetTarget(pXSprite, pXSprite->target);

				// check to see if we can attack
				if ( dist < kFatZombiePukeDist1 && dist > kFatZombiePukeDist2 && qabs(losAngle) < kAngle15 )
					NewState(pSprite, pXSprite, &fatZombiePuke);
				else
				if ( dist < kFatZombieMeleeDist && qabs(losAngle) < kAngle15 )
					NewState(pSprite, pXSprite, &fatZombieHack);
				return;
			}
		}
	}
	dprintf("Fat zombie %d lost sight of target %d\n", pXSprite->reference, pXSprite->target);
	NewState(pSprite, pXSprite, &fatZombieSearch);
	pXSprite->target = -1;
}


static void tAxeZombieSearch( SPRITE *pSprite, XSPRITE *pXSprite )
{
	ChooseDirection(pSprite, pXSprite, pXSprite->goalAng);
	tFindTarget(pSprite, pXSprite);
}


static void tAxeZombieGoto( SPRITE *pSprite, XSPRITE *pXSprite )
{
	int dx, dy, dist;

	dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];

	SPRITE *pTarget = &sprite[pXSprite->target];
	XSPRITE *pXTarget = &xsprite[pTarget->extra];

	dx = pXSprite->targetX - pSprite->x;
	dy = pXSprite->targetY - pSprite->y;

	int nAngle = getangle(dx, dy);
	dist = qdist(dx, dy);

	ChooseDirection(pSprite, pXSprite, nAngle);

	// if reached target, change to search mode
	if ( dist < M2X(1.0) && qabs(pSprite->ang - nAngle) < pDudeInfo->periphery )
		NewState(pSprite, pXSprite, &axeZombieSearch);

	tFindTarget(pSprite, pXSprite);
}


static void tAxeZombieChase( SPRITE *pSprite, XSPRITE *pXSprite )
{
	if ( pXSprite->target == -1)
	{
		NewState(pSprite, pXSprite, &axeZombieGoto);
		return;
	}

	int dx, dy, dist;

	dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];

	dassert(pXSprite->target >= 0 && pXSprite->target < kMaxSprites);
	SPRITE *pTarget = &sprite[pXSprite->target];
	XSPRITE *pXTarget = &xsprite[pTarget->extra];

	// check target
	dx = pTarget->x - pSprite->x;
	dy = pTarget->y - pSprite->y;

	ChooseDirection(pSprite, pXSprite, getangle(dx, dy));

	if ( pXTarget->health == 0 )
	{
		// target is dead
		NewState(pSprite, pXSprite, &axeZombieSearch);
		return;
	}

	if ( IsPlayerSprite( pTarget ) )
	{
		PLAYER *pPlayer = &gPlayer[ pTarget->type - kDudePlayer1 ];
		if ( powerupCheck( pPlayer, kItemLtdInvisibility - kItemBase ) > 0
		|| powerupCheck( pPlayer, kItemDeathMask - kItemBase ) > 0 )
		{
			NewState(pSprite, pXSprite, &axeZombieSearch);
			return;
		}
	}

	dist = qdist(dx, dy);

	if ( dist <= pDudeInfo->seeDist )
	{
		int nAngle = getangle(dx, dy);
		int losAngle = ((kAngle180 + nAngle - pSprite->ang) & kAngleMask) - kAngle180;
		int eyeAboveZ = pDudeInfo->eyeHeight * pSprite->yrepeat << 2;

		// is there a line of sight to the target?
		if ( cansee(pTarget->x, pTarget->y, pTarget->z, pTarget->sectnum,
			pSprite->x, pSprite->y, pSprite->z - eyeAboveZ, pSprite->sectnum) )
		{
			// is the target visible?
			if ( dist < pDudeInfo->seeDist && qabs(losAngle) <= pDudeInfo->periphery )
			{
				SetTarget(pXSprite, pXSprite->target);

				// check to see if we can attack
				if ( dist < kAxeZombieMeleeDist && qabs(losAngle) < kAngle15 )
					NewState(pSprite, pXSprite, &axeZombieHack);
				return;
			}
		}
	}
	dprintf("Axe zombie %d lost sight of target %d\n", pXSprite->reference, pXSprite->target);
	NewState(pSprite, pXSprite, &axeZombieSearch);
	pXSprite->target = -1;
}


static void tFGargoyleSearch( SPRITE *pSprite, XSPRITE *pXSprite )
{
	ChooseDirection(pSprite, pXSprite, pXSprite->goalAng);
	tFindTarget(pSprite, pXSprite);
}


static void tFGargoyleGoto( SPRITE *pSprite, XSPRITE *pXSprite )
{
	int dx, dy, dist;

	dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];

	SPRITE *pTarget = &sprite[pXSprite->target];
	XSPRITE *pXTarget = &xsprite[pTarget->extra];

	dx = pXSprite->targetX - pSprite->x;
	dy = pXSprite->targetY - pSprite->y;

	int nAngle = getangle(dx, dy);
	dist = qdist(dx, dy);

	ChooseDirection(pSprite, pXSprite, nAngle);

	// if reached target, change to search mode
	if ( dist < M2X(1.0) && qabs(pSprite->ang - nAngle) < pDudeInfo->periphery )
		NewState(pSprite, pXSprite, &fleshGargoyleSearch);

	tFindTarget(pSprite, pXSprite);
}


static void tFGargoyleChase( SPRITE *pSprite, XSPRITE *pXSprite )
{
	if ( pXSprite->target == -1)
	{
		NewState(pSprite, pXSprite, &fleshGargoyleGoto);
		return;
	}

	int dx, dy, dist;

	dassert(pSprite->type >= kDudeBase && pSprite->type < kDudeMax);
	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];

	dassert(pXSprite->target >= 0 && pXSprite->target < kMaxSprites);
	SPRITE *pTarget = &sprite[pXSprite->target];
	XSPRITE *pXTarget = &xsprite[pTarget->extra];

	// check target
	dx = pTarget->x - pSprite->x;
	dy = pTarget->y - pSprite->y;

	ChooseDirection(pSprite, pXSprite, getangle(dx, dy));

	if ( pXTarget->health == 0 )
	{
		// target is dead
		NewState(pSprite, pXSprite, &fleshGargoyleSearch);
		return;
	}

	if ( IsPlayerSprite( pTarget ) )
	{
		PLAYER *pPlayer = &gPlayer[ pTarget->type - kDudePlayer1 ];
		if ( powerupCheck( pPlayer, kItemLtdInvisibility - kItemBase ) > 0 )
		{
			NewState(pSprite, pXSprite, &fleshGargoyleSearch);
			return;
		}
	}

	dist = qdist(dx, dy);
	if ( dist <= pDudeInfo->seeDist )
	{
		int nAngle = getangle(dx, dy);
		int losAngle = ((kAngle180 + nAngle - pSprite->ang) & kAngleMask) - kAngle180;
		int eyeAboveZ = pDudeInfo->eyeHeight * pSprite->yrepeat << 2;

		// is there a line of sight to the target?
		if ( cansee(pTarget->x, pTarget->y, pTarget->z, pTarget->sectnum,
			pSprite->x, pSprite->y, pSprite->z - eyeAboveZ, pSprite->sectnum) )
		{
			// is the target visible?
			if ( dist < pDudeInfo->seeDist && qabs(losAngle) <= pDudeInfo->periphery )
			{
				SetTarget(pXSprite, pXSprite->target);

				// check to see if we can attack
				switch ( pSprite->type )
				{
					case kDudeFleshGargoyle:
						if ( dist < kFGargoyleThrowDist1 && dist > kFGargoyleThrowDist2 && qabs(losAngle) < kAngle15 )
							NewState(pSprite, pXSprite, &fleshGargoyleThrow);
						else if ( dist < kFGargoyleMeleeDist && qabs(losAngle) < kAngle15 )
							NewState(pSprite, pXSprite, &fleshGargoyleSlash);
						break;
				}
				return;
			}
		}
	}
	dprintf("Gargoyle %d lost sight of target %d\n", pXSprite->reference, pXSprite->target);
	NewState(pSprite, pXSprite, &fleshGargoyleGoto);
	pXSprite->target = -1;
}


void aiProcessDudes( void )
{
	// process active sprites
	for (short nSprite = headspritestat[kStatDude]; nSprite >= 0; nSprite = nextspritestat[nSprite])
	{
		SPRITE *pSprite = &sprite[nSprite];
		int nXSprite = pSprite->extra;
		XSPRITE *pXSprite = &xsprite[nXSprite];
	 	DUDEINFO *pDudeInfo = &dudeInfo[pSprite->type - kDudeBase];

		// don't manipulate players!
		if ( pSprite->type >= kDudePlayer1 && pSprite->type <= kDudePlayer8 )
			continue;

		pXSprite->stateTimer = ClipLow(pXSprite->stateTimer - kFrameTicks, 0);

		if ( pXSprite->aiState->move )
			pXSprite->aiState->move(pSprite, pXSprite);

		if ( pXSprite->aiState->think && (gFrame & kAIThinkMask) == (nSprite & kAIThinkMask) )
			pXSprite->aiState->think(pSprite, pXSprite);

		if ( pXSprite->stateTimer == 0 && pXSprite->aiState->next != NULL )
		{
			if ( pXSprite->aiState->ticks > 0 )
				NewState(pSprite, pXSprite, pXSprite->aiState->next);
			else if ( seqGetStatus(SS_SPRITE, nXSprite) < 0 )
				NewState(pSprite, pXSprite, pXSprite->aiState->next);
		}

		// process dudes for recoil
		if ( cumulDamage[nXSprite] >= pDudeInfo->hinderDamage << 4 )
			RecoilDude(pSprite, pXSprite);
	}

	// reset the cumulative damages for the next frame
	memset(cumulDamage, 0, sizeof(cumulDamage));

/*
		// special processing for converting dude types
		switch ( pSprite->type )
		{
			case kDudeCerberus:
				if ( pXSprite->health <= 0 && seqGetStatus(SS_SPRITE, nXSprite) < 0 )	// head #1 finished dying?
				{
					pXSprite->health = dudeInfo[kDudeCerberus2 - kDudeBase].startHealth << 4;
					pSprite->type = kDudeCerberus2;		// change his type
//						ActivateDude(pSprite, pXSprite);	// reactivate him
				}
				break;
		}
*/
}


/*******************************************************************************
	FUNCTION:		aiInit()

	DESCRIPTION:

	PARAMETERS:		void

	RETURNS:		void

	NOTES:
*******************************************************************************/
void aiInit( void )
{
	for (short nSprite = headspritestat[kStatDude]; nSprite >= 0; nSprite = nextspritestat[nSprite])
	{
		SPRITE *pDude = &sprite[nSprite];
		XSPRITE *pXDude = &xsprite[pDude->extra];

		switch ( pDude->type )
		{
			case kDudeTommyCultist:
			case kDudeShotgunCultist:
				NewState(pDude, pXDude, &cultistIdle);
				break;

			case kDudeFatZombie:
				NewState(pDude, pXDude, &fatZombieIdle);
				break;

			case kDudeAxeZombie:
				NewState(pDude, pXDude, &axeZombieIdle);
				break;

			case kDudeEarthZombie:
				NewState(pDude, pXDude, &earthZombieIdle);
				pDude->flags &= ~kAttrMove;
				break;

			case kDudeFleshGargoyle:
				NewState(pDude, pXDude, &fleshGargoyleIdle);
				break;

			default:
				NewState(pDude, pXDude, &genIdle);
		}

		SetTarget(pXDude, 0, 0, 0);

		pXDude->stateTimer	= 0;

		switch ( pDude->type )
		{
			case kDudeBrownSpider:
			case kDudeRedSpider:
			case kDudeBlackSpider:
				if ( pDude->cstat & kSpriteFlipY )
					pDude->flags &= ~kAttrFall;
				break;

			case kDudeBat:
			case kDudeFleshGargoyle:
			case kDudeStoneGargoyle:
				pDude->flags &= ~kAttrFall;
				break;
		}
	}
}

