// scriplib.h

#ifndef _scriplib_public
#define _scriplib_public

#define        MAXTOKEN    128
extern	char	token[MAXTOKEN];

//====================================================================
//
// ScripLib
//
//====================================================================
// NOTE: all section names and entry names are case sensitive,
// so be careful when mixing and matching names, Music is not equivalent
// to MUSIC.

/*
=============================================================================
=
= Tokenized functions
=
=============================================================================
*/

//
// Resets a script to the first line when accessing it in a tokenized format
//
void ResetScript (void);

//
// Signals that the current token was not used, and should be reported
// for the next GetToken.  Note that
//
// GetToken (true);
// UnGetToken ();
// GetToken (false);
//
// could cross a line boundary.
//
void UnGetToken (void);

//
// Get the next token
// crossline determines whether or not to go to the next line or stay on the
// current one.
// entireline determines whether or not to grab the entire line or just
// one token
//

void GetToken (boolean crossline, boolean entireline);

//
// whether or not their are any more tokens available
// if crossline is true lines are crossed
//
boolean TokenAvailable (boolean crossline);

//
// Token form of get integer
//
void GetInteger(int32 * number, boolean crossline);

//
// Token form of get double
//
void GetDouble(double * number, boolean crossline);

//
// Look for a specific token and skip over the script until
// we find it
//
void GetSpecificToken(char * );

//
// Count the number of lines in the file with valid data (non-comments)
//
int32 CountLinesInFile( void );

/*
=============================================================================
=
= Section/Entry functions
=
=============================================================================
*/

//
// saves a non-tokenized script to disk
//
void SaveScript (char * filename);

//
// Number of sections in script
//

int32 NumberOfScriptSections( void );

//
// A specific section name by number
//
char * ScriptSection( int32 which );

//
// Number of entries in a particular section
//
int32 NumberOfScriptEntries( char * sectionname );

//
// A specific entry by number for a section
//
char * ScriptEntry( char * sectionname, int32 which );

//
// returns pointer to value of entry, NULL if it doesn't exist
//
char * GetScriptRaw( char * sectionname, char * entryname );

//
// returns string denoted by entry, or unmodified if not found
//
void GetScriptString( char * sectionname, char * entryname, char * dest );

//
// returns double strings denoted by entry, or unmodified if not found
//
void GetScriptDoubleString( char * sectionname, char * entryname, char * dest1, char * dest2 );

//
// returns number denoted by entry, or unmodified if not found
// hexadecimal and decimal are automatically parsed.
// returns true if value is the default value
//
boolean GetScriptNumber
   (
   char * sectionname,
   char * entryname,
   int32 * number
   );

//
// returns a floating point double pointed to by entryname
//
void GetScriptDouble
   (
   char * sectionname,
   char * entryname,
   double * number
   );

//
// Write out printf style comment
//
void PutScriptMultiComment( char * comment, ... );

//
// Write out a raw entry no special characters are added
//
void PutScriptRaw( char * sectionname, char * entryname, char * raw );

//
// Write out a string which is surround by string terminators
//
void PutScriptString( char * sectionname, char * entryname, char * string );

//
// Write out a double string which is surround by string terminators
//
void PutScriptDoubleString( char * sectionname, char * entryname, char * string1, char * string2 );

//
// Write out a Number either decimal or hexadecimal or a default value
//
void PutScriptNumber
   (
   char * sectionname,
   char * entryname,
   int32 number,
   boolean hexadecimal,
   boolean defaultvalue
   );

//
// Write out a floating point double
//
void PutScriptDouble
   (
   char * sectionname,
   char * entryname,
   double number,
   boolean defaultvalue
   );

//
// Write out a section header which causes all subsequent entries to be
// part of that section
//
void PutScriptSectionHeader( char * sectionname );

//
// Write out a comment
//
void PutScriptComment( char * comment );
//
// Write out a linefeed
//
void PutScriptEOL( void );


/*
=============================================================================
=
= Shared functions
=
=============================================================================
*/

void InitScript( void );

//
// copies a script into memory for READ functions
// if tokenized is true script isn't grabbed in a typical Windows format,
// instead it is accessible through GetToken, TokenAvailable and ResetScript.
//
void ParsePreloadedScript ( char *data, int32 length, boolean tokenized );


//
// loads a script into memory for READ functions
// if tokenized is true script isn't grabbed in a typical Windows format,
// instead it is accessible through GetToken, TokenAvailable and ResetScript.
//
void LoadScript ( char * filename, boolean tokenized );

//
// frees a script from memory. Subsequent Loadscripts will first call
// FreeScript if a script is already in memory.
//
void FreeScript ( void );

int32 GetScriptLineNumber( void );

#endif
