#include <string.h>
#include <dos.h>

#include "sound.h"
#include "debug4g.h"
#include "error.h"
#include "inifile.h"
#include "misc.h"
#include "globals.h"
#include "music.h"
#include "fx_man.h"

#include <memcheck.h>

#define kSampleRate		11110

static BOOL sndActive = FALSE;

int gMusicDevice;
int gMusicVolume;
int gMidiPort;
int gFXDevice;
int gMixBits;
int gChannels;
int gMixVoices;
int gMixRate;
int gFXVolume;
fx_blaster_config gSBConfig;


RESHANDLE hSong;
BYTE *pSong = NULL;
int songSize;


/*---------------------------------------------------------------------
   Function: LoadMidi

   Loads the midi file from disk.
---------------------------------------------------------------------*/
void sndPlaySong( char *songName, BOOL loopflag )
{
	if ( pSong )
		sndStopSong();

	if ( strlen(songName) == 0 )
		return;

	hSong = gSysRes.Lookup(songName, ".MID");
	if (hSong == NULL)
	{
		dprintf("Could not load song %s\n", songName);
		return;
	}

	songSize = gSysRes.Size(hSong);

	pSong = (BYTE *)gSysRes.Lock(hSong);
	dpmiLockMemory(FP_OFF(pSong), songSize);

	MUSIC_PlaySong(pSong, loopflag);
}


BOOL sndIsSongPlaying( void )
{
	return (BOOL)MUSIC_SongPlaying();
}


void sndStopSong( void )
{
	if ( pSong )
	{
		MUSIC_StopSong();
		dpmiUnlockMemory(FP_OFF(pSong), songSize);
		gSysRes.Unlock(hSong);
		pSong = NULL;
	}
}


static void SoundCallback(ulong id)
{
	RESHANDLE hSample = (RESHANDLE)id;

	// unlock the sound resource
	gSysRes.Unlock(hSample);
}


void sndPlaySample( char *sampleName, int nVol )
{
	if ( strlen(sampleName) == 0 )
		return;

	RESHANDLE hSample = gSysRes.Lookup(sampleName, ".RAW");
	if (hSample == NULL)
	{
		dprintf("Could not load sample %s\n", sampleName);
		return;
	}

	int sampleSize = gSysRes.Size(hSample);

	char *pSample = (char *)gSysRes.Lock(hSample);

	FX_PlayRaw(pSample, sampleSize, kSampleRate, 0, nVol, nVol, nVol, 1, (ulong)hSample);
}


static void GetSettings( void )
{
	FX_GetBlasterSettings(&gSBConfig);

	// Read current setup
	gMusicDevice	= BloodINI.GetKeyInt("Sound Setup", "MusicDevice", -1);
	gMusicVolume	= BloodINI.GetKeyInt("Sound Setup", "MusicVolume", 200);
	gMidiPort		= BloodINI.GetKeyHex("Sound Setup", "MidiPort", gSBConfig.Midi);
	gFXDevice		= BloodINI.GetKeyInt("Sound Setup", "FXDevice", -1);
	gMixBits		= BloodINI.GetKeyInt("Sound Setup", "Bits", 16);
	gChannels		= BloodINI.GetKeyInt("Sound Setup", "Channels", 2);
	gMixVoices		= BloodINI.GetKeyInt("Sound Setup", "Voices", 16);
	gMixRate		= BloodINI.GetKeyInt("Sound Setup", "MixRate", 22050);
	gFXVolume		= BloodINI.GetKeyInt("Sound Setup", "FXVolume", 200);

	gSBConfig.Address	= BloodINI.GetKeyInt("Sound Setup", "BlasterAddress", gSBConfig.Address);
	gSBConfig.Type		= BloodINI.GetKeyInt("Sound Setup", "BlasterType", gSBConfig.Type);
	gSBConfig.Interrupt	= BloodINI.GetKeyInt("Sound Setup", "BlasterInterrupt", gSBConfig.Interrupt);
	gSBConfig.Dma8		= BloodINI.GetKeyInt("Sound Setup", "BlasterDma8", gSBConfig.Dma8);
	gSBConfig.Dma16		= BloodINI.GetKeyInt("Sound Setup", "BlasterDma16", gSBConfig.Dma16);
	gSBConfig.Emu		= BloodINI.GetKeyInt("Sound Setup", "BlasterEmu", gSBConfig.Emu);
}


void InitSoundDevice(void)
{
	int status;

	// Do special Sound Blaster, AWE32 stuff
	if ( gFXDevice == SoundBlaster || gFXDevice == Awe32 )
	{
		int MaxVoices;
		int MaxBits;
		int MaxChannels;

		status = FX_SetupSoundBlaster(gSBConfig, &MaxVoices, &MaxBits, &MaxChannels);
	}
	else
	{
		fx_device device;
		status = FX_SetupCard( gFXDevice, &device );
	}

	if ( status != FX_Ok )
		ThrowError(FX_ErrorString(status), ES_ERROR);

	status = FX_Init( gFXDevice, gMixVoices, gChannels, gMixBits, gMixRate );
	if ( status != FX_Ok )
		ThrowError(FX_ErrorString(status), ES_ERROR);

	FX_SetVolume(gFXVolume);

	// Set up our fx callback so we can unlock sound resources
	FX_SetCallBack(SoundCallback);
}


void InitMusicDevice(void)
{
	int status;

	status = MUSIC_Init( gMusicDevice, gMidiPort);
	if ( status != MUSIC_Ok )
		ThrowError(MUSIC_ErrorString( status ), ES_ERROR);

	RESHANDLE hTimbres = gSysRes.Lookup("GMTIMBRE", ".TMB");
	if ( hTimbres != NULL )
		MUSIC_RegisterTimbreBank((BYTE *)gSysRes.Load(hTimbres));

	MUSIC_SetVolume(gMusicVolume);
}


/***********************************************************************
 * Remove sound/music drivers
 **********************************************************************/
void sndTerm(void)
{
	// prevent this from shutting down more than once
	if ( sndActive )
	{
		sndStopSong();

		dprintf("MUSIC_Shutdown()\n");
		MUSIC_Shutdown();

		FX_Shutdown();

		sndActive = FALSE;
	}
}

/***********************************************************************
 * Initialize sound/music drivers
 **********************************************************************/
void sndInit(void)
{
	GetSettings();
	InitSoundDevice();
	InitMusicDevice();
	atexit(sndTerm);
	sndActive = TRUE;
}
