#include "engine.h"
#include "ai.h"
#include "db.h"
#include "gameutil.h"
#include "misc.h"
#include "globals.h"

/****************************************************************************
** LOCAL CONSTANTS
****************************************************************************/

#define kTurn					kAngle90
#define kHear					128 << 4
#define kSpeed					8 << 4
#define kPeriph					kAngle90
#define kAccel					4

/****************************************************************************
** GLOBALS
****************************************************************************/
static char buffer[256];

/****************************************************************************
** LOCALS
****************************************************************************/

#define kWidth 64<<4
short CanMove(int nSprite, int ang, int dist)
{
	long x, y, z, sin, cos, dx1, dy1, dx2, dy2;
	long hitx, hity, hitz;
	short hitsect, hitwall, hitsprite;
	SPRITE *pSprite;

	pSprite = &sprite[nSprite];
	x = pSprite->x;
	y = pSprite->y;
	z = pSprite->z;
	sin = Sin(ang);
	cos = Cos(ang);

	pSprite->cstat &= ~1; 	// clear hitscan flag

	dx1 = ((long)2 * kWidth * sin) >> 14;
	dy1 = ((long)-2 * kWidth * cos) >> 14;
	dx2 = (dist * cos) >> 14;
	dy2 = (dist * sin) >> 14;

	// try left vect
	x = pSprite->x - dx1 / 2;
	y = pSprite->y - dy1 / 2;
	hitscan(x, y, z, pSprite->sectnum, cos, sin, 0,
		&hitsect, &hitwall, &hitsprite, &hitx, &hity, &hitz);
	if (qdist(hitx - x, hity - y) < dist)
		return 0;

	// try right vect
	x = pSprite->x + dx1 / 2;
	y = pSprite->y + dy1 / 2;
	hitscan(x, y, z, pSprite->sectnum, cos, sin, 0,
		&hitsect, &hitwall, &hitsprite, &hitx, &hity, &hitz);
	if (qdist(hitx - x, hity - y) < dist)
		return 0;

	// try top vect
	x = pSprite->x + dx2 - dx1 / 2;
	y = pSprite->y + dy2 - dy1 / 2;
	hitscan(x, y, z, pSprite->sectnum, sin, -cos, 0,
		&hitsect, &hitwall, &hitsprite, &hitx, &hity, &hitz);
	if (qdist(hitx - x, hity - y) < dist)
		return 0;

	pSprite->cstat |= 1;		// set hitscan flag
	return 1;
}

short MyMoveSprite( short nSprite )
{
	long x, y, z, dx, dy, dz, vel;
	short ang;
	SPRITE *pSprite;
	long ceilDist = 0, floorDist = 0;

	pSprite = &sprite[nSprite];

	x = pSprite->x;
	y = pSprite->y;
	z = pSprite->z;
	vel = pSprite->xvel;
	ang = pSprite->ang;
	dx = mulscale14(vel, Cos(ang));
	dy = mulscale14(vel, Sin(ang));
	dz = pSprite->zvel;

	if ( movesprite(nSprite, dx, dy, dz, ceilDist, floorDist, 0, gFrameTicks) )
		return 1;

	return 0;
}

void WalkMonster( XSPRITE *pXSprite )
{
	int dx, dy, dz, dist, avoidDist;
	int ang, dang, dang2, maxTurn;


	short nSprite = (short)pXSprite->reference;
	SPRITE *pSprite = &sprite[nSprite];

	// calculate some useful values ahead of time
	dx = pXSprite->targetX - pSprite->x;
	dy = pXSprite->targetY - pSprite->y;
	dz = pXSprite->targetZ - pSprite->z;

	dist = qdist(dx,dy);
	ang = getangle(dx,dy);
	pSprite->zvel = (short)((dz << 8) / dist);
	dang = ((kAngle180 + ang - pSprite->ang) & kAngleMask) - kAngle180;

	// clip turning speed
	maxTurn = kTurn / ((pSprite->xvel >> 4) + 2);
	dang2 = dang;
	if (dang < -maxTurn) dang2 = -maxTurn;
	if (dang > maxTurn) dang2 = maxTurn;

	avoidDist = (pSprite->xvel / kAccel + 1) * (pSprite->xvel / 2);

	if (pSprite->xvel == 0 && qabs(dang) > kAngle90)
	{
		pSprite->ang = (short)((pSprite->ang + dang2) & kAngleMask);
		return;
	}

	// clear movement toward target?
	if (CanMove(nSprite, pSprite->ang + dang2, avoidDist))
	{
		pSprite->ang = (short)((pSprite->ang + dang2) & kAngleMask);
		MyMoveSprite(nSprite);
		pSprite->xvel = (short)ClipHigh(pSprite->xvel + kAccel, kSpeed);
		return;
	}

	// clear movement straight?
	if (CanMove(nSprite, pSprite->ang, avoidDist))
	{

		// slow down if target is behind monster
		if (labs(dang) > kAngle90)
			pSprite->xvel = (short)ClipLow(pSprite->xvel - 2 * kAccel, 0);

		if (MyMoveSprite(nSprite))
			return;
	}

	// try turning left
	if (CanMove(nSprite, pSprite->ang - kAngle45, avoidDist)) {
		pSprite->ang = (short)((pSprite->ang - maxTurn / 2) & kAngleMask);
		if (MyMoveSprite(nSprite))
			return;
	}

	// try turning right
	if (CanMove(nSprite, pSprite->ang + kAngle45, avoidDist)) {
		pSprite->ang = (short)((pSprite->ang + maxTurn / 2) & kAngleMask);
		if (MyMoveSprite(nSprite))
			return;
	}

	// uh-oh, hard turn
	pSprite->xvel = (short)ClipLow(pSprite->xvel - kAccel, 0);

	// try turning left
	if (CanMove(nSprite, pSprite->ang - kAngle90, avoidDist)) {
		pSprite->ang = (short)((pSprite->ang - maxTurn) & kAngleMask);
		if (MyMoveSprite(nSprite))
			return;
	}

	// try turning right
	if (CanMove(nSprite, pSprite->ang + kAngle90, avoidDist)) {
		pSprite->ang = (short)((pSprite->ang + maxTurn) & kAngleMask);
		if (MyMoveSprite(nSprite))
			return;
	}

	if ( !MyMoveSprite(nSprite) ) {
		// crashed!
		pSprite->xvel = (short)-pSprite->xvel;
		MyMoveSprite(nSprite);
		pSprite->xvel = 0;
		pXSprite->moveState = kMoveStill;
		pXSprite->aiState = kAIScanSector;
	}
}


void FX_MoveSprites(void)
{
	long dx,dy,dist;
	long lsprite, lnext;
	short los, losang, dang, i;
	SPRITE *pSprite;

	// PROCESS ACTIVE SPRITES
	lsprite = headspritestat[kStatDude];
	while (lsprite != -1)
	{
		lnext = nextspritestat[ lsprite ];
		pSprite = &sprite[lsprite];
		int nXSprite = pSprite->extra;
		XSPRITE *pXSprite = &xsprite[nXSprite];

		switch(	pSprite->type )
		{
			// MONSTERS
			case kDudeFleshGargoyle:
				switch (pXSprite->moveState)
				{
					case kMoveWalk:
						WalkMonster(pXSprite);
						break;
				}

				switch (pXSprite->aiState)
				{
					case kAIScanSector:
						if ((totalclock & 1 == 1) || (cursectnum == pSprite->sectnum)) {

							dx = posx - pSprite->x;
							dy = posy - pSprite->y;
							dist = ldist(dx,dy);
							losang = getangle(dx,dy);
							dang = ((kAngle180 + losang - pSprite->ang) & kAngleMask) - kAngle180;

							los = cansee(posx, posy, posz, cursectnum, pSprite->x, pSprite->y,
								pSprite->z - (tilesizy[pSprite->picnum]<<8), pSprite->sectnum);

							// is the player visible?
							if (los && labs(dang) < kPeriph)
							{
								pXSprite->target = -1; // player
								pXSprite->targetX = posx;
								pXSprite->targetY = posy;
								pXSprite->targetZ = posz+heightoffceiling;
								pXSprite->aiState = kAIFollow;
								pXSprite->moveState = kMoveWalk;
								break;
							}

							if (dist < kHear)
							{
								pXSprite->target = 0; // no target
								pXSprite->targetX = posx;
								pXSprite->targetY = posy;
								pXSprite->targetZ = posz+heightoffceiling;
								pXSprite->aiState = kAIGoto;
								pXSprite->moveState = kMoveWalk;
								break;
							}
						}
						break;

					case kAIGoto:	// basic movement algorithm

						if (pXSprite->moveState == kMoveStill) {
							pXSprite->aiState = kAIScanSector;
							break;
						}

						dx = posx - pSprite->x;
						dy = posy - pSprite->y;
						dist = ldist(dx,dy);
						losang = getangle(dx,dy);
						dang = ((kAngle180 + losang - pSprite->ang) & kAngleMask) - kAngle180;

						los = cansee(posx, posy, posz, cursectnum, pSprite->x, pSprite->y,
							pSprite->z - (tilesizy[pSprite->picnum]<<8), pSprite->sectnum);

						// is the player visible?
						if (los && labs(dang) < kPeriph) {
							pXSprite->target = -1; // player
							pXSprite->targetX = posx;
							pXSprite->targetY = posy;
							pXSprite->targetZ = posz+heightoffceiling;
							pXSprite->aiState = kAIFollow;
							pXSprite->moveState = kMoveWalk;
							break;
						}

						dx = pXSprite->targetX - pSprite->x;
						dy = pXSprite->targetY - pSprite->y;
						dist = ldist(dx, dy);

						break;

					case kAIFollow:

						if (pXSprite->moveState == kMoveStill) {
							pXSprite->aiState = kAIScanSector;
							break;
						}

						dx = posx - pSprite->x;
						dy = posy - pSprite->y;
						dist = ldist(dx,dy);
						losang = getangle(dx,dy);
						dang = ((kAngle180 + losang - pSprite->ang) & kAngleMask) - kAngle180;

						los = cansee(posx, posy, posz, cursectnum, pSprite->x, pSprite->y,
							pSprite->z - (tilesizy[pSprite->picnum]<<8), pSprite->sectnum);

						// is the player visible?
						if (los && labs(dang) < kPeriph) {
							pXSprite->targetX = posx;
							pXSprite->targetY = posy;
							pXSprite->targetZ = posz+heightoffceiling;
						}
						else
						{
							pXSprite->aiState = kAIGoto;
						}
						break;
				}

				/* choose rotated bitmap if with 1024 pixels of the sprite
				 * This code should eventually be in a draw callback
				 */
				if (dist < 16384) {
					// Calculate which of the 8 angles of the sprite to draw (0-7)
					i = ((ang - pSprite->ang + 1152) & kAngleMask) >> 8;
					if (i <= 4) {
						pSprite->picnum = 286 + 3 * i;
						pSprite->cstat &= ~4;   //clear x-flipping bit
					}
					else {
						pSprite->picnum = 286 + 3 * (8-i);
						pSprite->cstat |= 4;     //set x-flipping bit
					}
				}
				break;
		}
		lsprite = lnext;
	}
}

/****************************************************************************
** FX_InitSprites
**
** ARGUMENTS:
**	none
**
** RETURNS:
**	none
**
****************************************************************************/
void FX_InitSprites(void)
{
	long  lsprite;
	short gMonsterCnt;

	for (lsprite=0, gMonsterCnt=0; lsprite<MAXSPRITES; lsprite++)
	{
		if (sprite[lsprite].statnum < MAXSTATUS) //if the sprite exists
		{
			switch(	(short)sprite[lsprite].tag )
			{
				default:
					break;
				case kTagGargoyle:
				case kTagRat:
				case kTagBat:
				case kTagSpider:
				case kTagGator:
				case kTagWisp:
				case kTagZombie:
				case kTagMummy:
				case kTagSkeleton:
				case kTagWaterDemon:
				case kTagDemon:
				case kTagPhantasm:
				case kTagShadow:
				case kTagWerewolf:
				case kTagDemonDog:
				case kTagLich:
				case kTagBoss2:
				case kTagBoss3:
					sprite[lsprite].cstat |= kSpriteBlocking;	// hitscan sensitivity
					gMonsters[gMonsterCnt].sprite = lsprite;
					gMonsters[gMonsterCnt].aiState  = kAIScanSector;
					gMonsters[gMonsterCnt].moveState   = kMoveStill;
					gMonsters[gMonsterCnt].health = 100;
					gMonsters[gMonsterCnt].target = SHRT_MIN;
					sprite[lsprite].extra         = &gMonsters[gMonsterCnt];
					gMonsterCnt++;
					changespritestat(lsprite,kListActive);
					break;
			}
		}
	}
}


